
/*****************************************************************************
 *                                    M L X                                  *
 *                 Rendering Library for Accelerated 3d Hardware             *
 *                           (C) SuSE GmbH 1997, 1998                        *
 *****************************************************************************/
/* author: simon pogarcic, sim@suse.de */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <linux/pci.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#if 1
#define LOG_DEBUG
#endif

#define LOG_DBGINFO "<MLXagent> "
#include "log_debug.h"

#include "mlx.h"



MLXDriverCtxPtr MLXdriverctx[MAX_CARDS_ALOWED];
MLXDriverCtxPtr	MLXfirstdrvctx = NULL;

Tubyte *MLXDriversList[] = {
	"-none-",				/*  0 */
	"3DLabs GLINT 500TX/MX driver",         /*  1 */
	"3DLabs Permedia driver",               /*  2 */
	"3DLabs Permedia 2 driver"              /*  3 */
};



/*****************************************************************************
	ACL API INITIALISATION
 *****************************************************************************/

static ACLbool	MLXentered = FALSE;

ACLint
AclEnter( ACLvoid )
{
    if(MLXentered) {
	MSG("MLX ALREADY INITIALIZED.\n");
    	return 0;
    }

    if( gpcli_enter_app() )
	return 1;

    if( mlx_glint_init()) {
	gpcli_exit_app(0, FALSE);
	return 1;
    }

    mlx_list_drivers();

    MLXentered = TRUE;
    return 0;
}



ACLint
AclQueryCard( ACLint drvidx, ACLenum hw, ACLenum pos )
{
    MLXDriverCtxPtr drvctx;

    if(!MLXentered)
    	return (-1);

    if ( pos==ACL_GET_NEXT || pos==ACL_GET_PREV ) {

	if( !(drvctx = mlx_drvidx_to_drvctx( drvidx )) )
	    return (-1);

	hw = drvctx->hw;
	
	if(pos==ACL_GET_NEXT) {
	    drvctx = drvctx->next;
	    pos = ACL_GET_FIRST;
	}
	else {
	    drvctx = drvctx->prev;
	    pos = ACL_GET_LAST;
	}

	if(drvctx && drvctx->hw == hw) {
	    drvidx = drvctx->card_idx;
	}
	else {
	    drvctx = mlx_search_driver( hw, pos );
	    drvidx = drvctx->card_idx;
	}
    }

    else if ( pos==ACL_GET_FIRST || pos==ACL_GET_LAST ) {

	if( hw < ACL_DRIVER_NONE || hw > ACL_DRIVER_LAST ) {
	    ERR("Requested hardware not supported (%d)\n", hw);
	    return (-1);
	}

	drvctx = mlx_search_driver( hw, pos );

	if(!drvctx) {
	    ERR("Neither drivers nor hardware found...\n");
	    return (-1);
	}

	if (drvctx->hw != hw) {
	    ERR("%s: Requested hardware not found\n", MLXDriversList[hw]);
	    return (-1);
	}

	drvidx = drvctx->card_idx;
    }

    else {
    	ERR("Bad query position param specified\n");
	return (-1);
    }

    MSG("\n");
    MSG("[QUERY] %s\n", drvctx->drv->drvinfo);
    MSG("    Card %s in PCI slot %d\n", drvctx->card_info, drvctx->card_slot);

    return drvidx;
}



ACLContext
AclCreateContext ( ACLint drvidx )
{
    MLXDriverCtxPtr drvctx;

    ACLModules mod = NULL;
    ACLAccelMod accelmod = NULL;
    ACLDepthMod depthmod = NULL;
    ACLTexMod texmod = NULL;
    ACLContext aclctx = NULL;

    ACLuint mflags;

    if(!MLXentered)
    	return NULL;

    if( !(drvctx = mlx_drvidx_to_drvctx( drvidx )) )
	return NULL;
	
    mflags = drvctx->drv->supmod;

LOG("Allocating ACLContext RAM...\n");
    aclctx = (ACLContext) malloc( sizeof(acl_context) );
    if(!aclctx) {
	ERR("AclCreateContext(): malloc (aclctx) failed\n");
	return NULL;
    }

LOG("Allocating modules RAM...\n");    
    mod = (ACLModules) malloc( sizeof(acl_modules) );
    if(!mod) {
	ERR("AclCreateContext(): malloc (mod) failed\n");
	free (aclctx);
	return NULL;
    }
    
    if(mflags & ACL_MODULE_ACCEL) {
LOG("Allocating ACLAccelMod RAM...\n");    
	accelmod = (ACLAccelMod) malloc( sizeof(acl_accel_mod) );
	if(!accelmod) {
	    ERR("AclCreateContext(): malloc (accelmod) failed\n");
	    free (aclctx);
	    free (mod);
	    return NULL;
	}
	mod->Accel = accelmod;
    }

    if(mflags & ACL_MODULE_DEPTH) {
LOG("Allocating ACLDepthMod RAM...\n");    
	depthmod = (ACLDepthMod) malloc( sizeof(acl_depth_mod) );
	if(!depthmod){
	    ERR("AclCreateContext(): malloc (depthmod) failed\n");
	    free (aclctx);
	    free (mod);
	    if(accelmod) free (accelmod);
	    return NULL;
	}
	mod->Depth = depthmod;
    }

    if(mflags & ACL_MODULE_TEX) {
LOG("Allocating ACLTexMod RAM...\n");    
	texmod = (ACLTexMod) malloc( sizeof(acl_tex_mod) );
	if(!texmod){
	    ERR("AclCreateContext(): malloc (texmod) failed\n");
	    free (aclctx);
	    free (mod);
	    if(accelmod) free (accelmod);
	    if(depthmod) free (depthmod);
	    return NULL;
	}
	mod->Tex = texmod;
    }

    aclctx->drvidx = drvidx;
    aclctx->mod = mod;

LOG("Connecting the modules to driver...\n");
    drvctx->drv->bind_client( aclctx );

    return aclctx;
}



ACLvoid
AclExit( ACLContext ctx )
{

    if(!MLXentered) {
    	return;
    }

    gpcli_exit_app(0, FALSE);

    MLXentered = FALSE;
    MSG("MLX DEINITIALIZED.\n");
}



/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

MLXDriverCtxPtr
mlx_drvidx_to_drvctx( ACLint drvidx )
{
    MLXDriverCtxPtr drvctx;

    if( drvidx < 0 || drvidx >= GpcliCardsFound ) {
	ERR("Bad drv context descriptor (%d)\n", drvidx);
	return NULL;
    }

    if( !(drvctx = MLXdriverctx[drvidx]) ) {
	ERR("driver not initialized yet (%d)\n", drvidx);
	return NULL;
    }

    return drvctx;
}



ACLvoid
mlx_list_drivers( ACLvoid )
{
    MLXDriverCtxPtr drvctx = MLXfirstdrvctx;

    MSG("\n");
    if(!drvctx)
	ERR("[R] * NO DRIVERS FOUND *\n");

    for(; drvctx; drvctx = drvctx->next) {
	MSG("[R] %s\n", drvctx->drv->drvinfo);
	MSG("    Chipset: %s, card #%d (%d), slot %u\n", drvctx->card_info,
	    drvctx->card_idx, drvctx->tween_idx, drvctx->card_slot );
    }
    MSG("\n\n");
}



/*
 * searches for registered drivers; if 'hw' specified driver was not
 * found, it returns the first or last driver record from whole drivers
 * area. Only ACL_GET_FIRST or ACL_GET_LAST can be used as parameter.
 */

MLXDriverCtxPtr
mlx_search_driver( ACLenum hw, ACLenum pos )
{
    MLXDriverCtxPtr drvctx, lastdrvctx;

    if( pos == ACL_GET_FIRST || pos == ACL_GET_LAST ) {
	lastdrvctx = MLXfirstdrvctx;
	for( drvctx = MLXfirstdrvctx; drvctx ; drvctx = drvctx->next) {
	    if (drvctx->hw == hw) {
		if (pos == ACL_GET_FIRST) {
		    return drvctx;
		}
		else {
		    lastdrvctx = drvctx;
		    for( drvctx = drvctx->next; drvctx && drvctx->hw == hw;
			drvctx = drvctx->next)	lastdrvctx = drvctx;
		    return lastdrvctx;
		}
	    }
	    lastdrvctx = drvctx;
	}
	if (pos == ACL_GET_FIRST)
	    return MLXfirstdrvctx;
	else
	    return lastdrvctx;
    }

    return NULL;
}

