
/*
 * -*- mode: C; tab-width:8;  -*-
 * 
 * aclX.c - ACL DD Mesa interface to Xlib
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(ACL)

#include "aclDrv.h"



extern ACLSetup aclMesaHwSetup;
extern ACLContext aclMesaCurrentCtx;

aclXVidModeDataPtr aclMesaCurrentVidMode = NULL;



/*
 * Compare Function for sorting Video Modes
 */
static int
aclXCompareVidModes( const void *first, const void *second)
{
  return ( ((XF86VidModeModeInfo *) first)->hdisplay -
	   ((XF86VidModeModeInfo *) second)->hdisplay);
}



ACLint
aclXInitVidModes( Display *dpy, ACLuint needW, ACLuint needH )
{
    aclXVidModeDataPtr vidmod;
    ACLuint scrW, scrH, scrBPP, virtualH, virtualW; 

    char *env_dga = getenv("ACL_MESA_DGA");
    char *env_vm = getenv("ACL_MESA_VIDMODE");
    
LOG("aclXInitVidModes() Start: vidmod (malloc)\n");
    vidmod = (aclXVidModeDataPtr) malloc(sizeof(aclXVidModeData));
    if( !vidmod ) {
	return 1;
    }


/******************************************************************************
 * Get default screen data
 ******************************************************************************/
    /* color depth */
#if 0    
    scrBPP = DefaultDepth(dpy, XDefaultScreen(dpy));
#endif

    /* better get the bpp value from pixmap formats */
    {
      XPixmapFormatValues *pmf;
      int count, i;
     
      /* get all pixmap formats */
      pmf = XListPixmapFormats (dpy, &count);

      /* initial bpp value */
      scrBPP = pmf[0].bits_per_pixel;      

      /* take the biggest bpp value */
      for (i=1; i<count; i++) {
	if (pmf[i].bits_per_pixel > scrBPP)
	  scrBPP = pmf[i].bits_per_pixel;      
      }
      
LOG("color depth: %d bpp\n", scrBPP);
      
      XFree ((char *) pmf);
    }

    /* Virtual Screen geometry */
    virtualW = DisplayWidth(dpy, XDefaultScreen(dpy));
    virtualH = DisplayHeight(dpy, XDefaultScreen(dpy));

    /* we need the vidmode entries, not the screen entries 
       screenwidth/screenheight are always the values of the biggest 
       resolution of all vidmodes */
    
    {
      XF86VidModeModeLine modeline;
      int dotclock;

      XF86VidModeGetModeLine(dpy,XDefaultScreen(dpy),
			     &dotclock,&modeline);
      
      scrW = modeline.hdisplay;
      scrH = modeline.vdisplay;
      LOG("current resolution: %dx%d\n", scrW, scrH);
    }

    /* remember Display */
    vidmod->dpy = dpy;

/******************************************************************************
 * Switch to new video mode if required
 *****************************************************************************/
    if (env_vm && !strcmp(env_vm,"1") ) {

      XF86VidModeModeInfo **vm_modelines;
      XF86VidModeModeInfo *new_vm_modelines;
      int vm_count;
      int i;
      
      vidmod->Vflags |= ACLMESA_VIDMODE_ON;

      /* change the current X11 video mode */
      XF86VidModeGetAllModeLines(dpy,XDefaultScreen(dpy),
				 &vm_count,&vm_modelines);
      
      /* save current modeline */
      vidmod->vm_current_modeline= (void *)malloc(sizeof(XF86VidModeModeInfo));
      if (!vidmod->vm_current_modeline) {
	free( vidmod );  
	return 1;
      }

      memcpy(vidmod->vm_current_modeline,(void *)(vm_modelines[0]),
	     sizeof(XF86VidModeModeInfo));

LOG("vidmode: x11 modes (unsorted):\n");
      for (i = 0; i < vm_count; i++) {
	if (i == 0) {
LOG("vidmode: \t%d: %dx%d (current mode)\n",
    i,vm_modelines[i]->hdisplay,vm_modelines[i]->vdisplay);
	} else {
LOG("vidmode: \t%d: %dx%d\n",
    i,vm_modelines[i]->hdisplay,vm_modelines[i]->vdisplay);
	}
      }
      
      /* new array to be able to use qsort */
      new_vm_modelines=(XF86VidModeModeInfo *) 
	malloc (vm_count * sizeof(XF86VidModeModeInfo));
	if (!new_vm_modelines) {
	  free( vidmod->vm_current_modeline );
	  free( vidmod );  
	  return 1;
      }
      
      /* copy modeline entries to new array */
      for (i = 0; i < vm_count; i++) 
	memcpy(new_vm_modelines+i,*(vm_modelines+i),
	       sizeof(XF86VidModeModeInfo));
      
      /* sort modelines ascending */
      qsort(new_vm_modelines, vm_count, sizeof(XF86VidModeModeInfo), 
	    aclXCompareVidModes);
      
      /* copy new modeline entries back to old array */
      for (i = 0; i < vm_count; i++) 
	memcpy(*(vm_modelines+i),new_vm_modelines+i,
	       sizeof(XF86VidModeModeInfo));
      
LOG("vidmode: x11 modes (sorted):\n");
      for (i = 0; i < vm_count; i++) 
LOG("vidmode: \t%d: %dx%d\n",
    i,vm_modelines[i]->hdisplay,vm_modelines[i]->vdisplay);
      
      /* find best resolution */
      for (i = 0; i < vm_count; i++) {
	if ((needW <= vm_modelines[i]->hdisplay) &&
	    (needH <= vm_modelines[i]->vdisplay)) {
LOG("vidmode: \tswitching to best resolution: %d\n",i);
	  XF86VidModeSwitchToMode(dpy,XDefaultScreen(dpy),
				  vm_modelines[i]);
	  scrW = vm_modelines[i]->hdisplay;
	  scrH = vm_modelines[i]->vdisplay;

	    break;
	}
	/* largest and last mode */
	if (i == vm_count-1) {
LOG("vidmode: \tswitching to largest and last resolution: %d\n",i);
	  XF86VidModeSwitchToMode(dpy,XDefaultScreen(dpy),
				  vm_modelines[i]);
	  scrW = vm_modelines[i]->hdisplay;
	  scrH = vm_modelines[i]->vdisplay;
	}
      }
    }

    XFlush(dpy);

/******************************************************************************
 * Setup DGA, if required
 *****************************************************************************/
    if (env_dga &&
	!strcmp(env_dga ,"1") ) {
      vidmod->Vflags |= ACLMESA_DGA_ON;

      vidmod->sonpid=fork();
      
      if (!vidmod->sonpid)
      {
        /**********************************/
	/* Son ... only used for DGA mode */
        /**********************************/
	
	int factor;
	int event_base, error_base;
	int flags;
	char* addr;
	int width, banksize, memsize;
	int viewport_width, viewport_height;
	char dummy;
	

	/* check if DGA is available */
	if (XF86DGAQueryExtension(dpy,&event_base,&error_base)) {
	  XF86DGAQueryDirectVideo(dpy,XDefaultScreen(dpy),&flags);
	  if ( (flags & XF86DGADirectPresent) == 0) {
LOG("dga: no direct present, exiting\n");
	    exit(1);
	  }
	}
	
	/* get Video parameters */
	XF86DGAGetVideo(dpy,XDefaultScreen(dpy),&addr, 
			&width,&banksize,&memsize);
LOG("dga: addr=%p width=%d banksize=%d memsize=%d\n",
    addr,width,banksize,memsize);
	
	/* get visible screen area */
	XF86DGAGetViewPortSize(dpy,XDefaultScreen(dpy),
			       &viewport_width, &viewport_height);
LOG("dga: viewport size: %dx%d\n",
    viewport_width,viewport_height);
	
	/* move the viewport to the origin */
	XF86DGASetViewPort(dpy,XDefaultScreen(dpy),0,0);
	
	/* enable DGA */
	XF86DGADirectVideo(dpy,XDefaultScreen(dpy),
			   XF86DGADirectGraphics | 
			   XF86DGADirectMouse | 
			   XF86DGADirectKeyb );
	
	if (scrBPP==15)
	  factor=2;
	else
	  factor=scrBPP/8;
	
	/* write something into framebuffer directly (blank screen)  */
	memset((void*)addr,0,(viewport_width*viewport_height)*factor);

	/* inform father that DGA mode is established */
	kill(getppid(), SIGUSR1);

	pause();

LOG("dga: disable DGA mode\n");
	XF86DGADirectVideo(dpy,XDefaultScreen(dpy),0);

	exit(0);
      } /******************
      	 * end of son code
	 ******************/
    }


    if (env_dga &&
	!strcmp(env_dga ,"1") ) {
      /* wait until son has established DGA mode */
      pause();
    }

    aclMesaCurrentVidMode = vidmod;

    aclMesaHwSetup->VisibleWidth = scrW;
    aclMesaHwSetup->VisibleHeight = scrH;
    aclMesaHwSetup->VirtualWidth = virtualW;
    aclMesaHwSetup->VirtualHeight = virtualH;
    aclMesaHwSetup->ScreenBpp = scrBPP;


    return 0;
}

 

ACLvoid
aclXSetDefaultVidModes( ACLvoid )
{
    Display * dpy;

    if( !aclMesaCurrentVidMode )
    	return;

    dpy = aclMesaCurrentVidMode->dpy;

/******************************************************************************
 * kill DGA child softly if required
 *****************************************************************************/
    if ( aclMesaCurrentVidMode->Vflags & ACLMESA_DGA_ON) {
	LOG("dga: Killing DGA child softly\n");
	kill( aclMesaCurrentVidMode->sonpid, SIGUSR1);
    }

/******************************************************************************
 * Switch to previous video mode if required
 *****************************************************************************/
    if ( aclMesaCurrentVidMode->Vflags & ACLMESA_VIDMODE_ON) {
	/*
	 * restore current modeline
	 */

      LOG("vidmode: switching to previous modeline %ix%i\n",
	  aclMesaCurrentVidMode->vm_current_modeline->hdisplay,
	  aclMesaCurrentVidMode->vm_current_modeline->vdisplay);

      /* doesn't work for now, I really don't know why :-( */
      XF86VidModeSwitchToMode(dpy,XDefaultScreen(dpy),
			      aclMesaCurrentVidMode->vm_current_modeline);

/* you forgott this, Mr. Soremouth !!! :^) */
      XFlush(dpy);

      free(aclMesaCurrentVidMode->vm_current_modeline);
    }
}



#else

/*
 * Need this to provide at least one external definition.
 */

int
gl_pm_dummy_function_api(void)
{
    return 0;
}

#endif /* ACL */

