
#ifndef _TRACKER_H_
#define _TRACKER_H_

/*
 * Author: Simon Pogarcic (sim@suse.de)
 *
 *
 *
 * INTRODUCTION:
 * -------------
 *
 * - This small utility should help debugging the function calls, especialy
 *   for functions which are very often called (like graphic primitives) and
 *   where you want to keep track how many times those functions are called.
 *   The debuging messages appears than periodicaly in time interval which
 *   can be specified with TRACKER_INTERVAL (see tracker.c); 3 sec is default.
 *
 * - The counter of function calls which you keep track is stored in an array.
 *   I call this array "function slots". The number of function slots is hard
 *   coded, so you have to know of how many functions you want to keep track
 *   and to take care to not to overlap the indexes of slots. However, the
 *   care must be taken only considering different source files, because the
 *   initialisation of slots in single source file happens automaticly.
 *   Therefore, in every source file you have to use the TINIT() macro,
 *   which you call with the first index of unused slots as parameter,
 *   continuing upwards. So, just consider that every source file has its own
 *   area of slots, which first slot is determined with TINIT(first_slot)
 *   macro. The slot areas of different files should not overlap, although it
 *   doesn't matter if there are unused slots.
 *
 * - It can happen that you work with different librarys and object modules
 *   which both require debuging. If you have to link those together, you
 *   get double symbol errors or wrong count output. In that case you have to
 *   do the folowing:
 *
 *	1. Put some other parameter (prefix) in TRACKERProto macro call (here)
 *	2. Put the same prefix in TRACKERCode macro call (in tracker.c)
 *	3. Change the prefix of '_track_register()' in TLOG macro (here)
 *
 * - If you don't have enough slots, redefine TRACKER_MAXFUNC (tracker.c)
 *
 * - There is also simple debugging feature, with normal fprintf call and pid
 *   information, if enabled. Process ID output is enabled if you define
 *   TPRINTPID
 *
 * - The module information text can be defined with TINFO
 *
 * - The periodical debuging can be enabled/disabled with TPOFF/TPON macro.
 * 
 * - Normal debuging can be enabled/disabled with TOFF/TON macro.
 *
 * - The compilation is switched off/on with TPDEBUG and TDEBUG defines.
 *   Using those two defines you can actualy easy remove all TRACKER junk.
 *
 *
 *
 * USAGE EXAMPLE:
 * --------------
 *
 * #define TDEBUG
 * #define TPDEBUG
 * #define TPRINTPID
 *
 * #include "tracker.h"
 * TINIT(20)
 *
 * ...
 *
 * TPON
 *
 * #undef TINFO
 * #define TINFO "[my_info]: "
 * some_often_called_function()
 * {
 *	TLOG("some_often_called_function()\n")
 *	...
 * }
 *
 * ...
 *
 * TON
 *
 * #undef TINFO
 * #define TINFO "[my_other_info]: "
 * some_normal_called_function()
 * {
 *	LOG("Called some_normal_called_function with %d\n", par);
 *	...
 * }
 *
 * TOFF
 *
 * ...
 *
 */

#include <stdio.h>



#ifndef TINFO
#define TINFO "<info>: "
#endif



#ifdef TPDEBUG

static char TPIsOn=1;
# define TINIT(s) static int TslotStart = s;
# define TPON TPIsOn = 1;
# define TPOFF TPIsOn = 0;
# define TLOG(t) \
  { \
	static int MyTSlot = -1; \
	if (MyTSlot == -1) MyTSlot = TslotStart++; \
/**************************************************************/ \
/* PUT SOME OTHER PREFIX HERE IF YOU GOT DOUBLE SYMBOL ERROR! */ \
/******************VV******************************************/ \
	if(TPIsOn) mesa_track_register(MyTSlot, TINFO, t); \
/******************^^******************************************/ \
  }
# define TRACKERProto(pref) \
  extern void pref ## _track_print( char * ); \
  extern void pref ## _track_register( unsigned int, char *, char * ); \
  extern void pref ## _track_init( int );

#else

# define TPON
# define TPOFF
# define TINIT(s)
# define TLOG(t)
# define TRACKERProto(pref)

#endif



#ifdef TDEBUG

static char TIsOn=1;
# define TON TIsOn=1;
# define TOFF TIsOn=0;

# ifdef __KERNEL__
#  define LOG(f, a...)		if(TIsOn) printk(TINFO f, ## a)
# else
#  define LOG(f, a...)		if(TIsOn) fprintf(stderr, TINFO f, ## a)
# endif

#else

# define TON
# define TOFF
# define LOG(f, a...)

#endif



#ifdef __KERNEL__
# define ERR(f, a...)		printk(TINFO "ERR:" f, ## a)
#else
# define ERR(f, a...)		fprintf(stderr, TINFO "ERR: " f, ## a)
#endif



#ifdef __KERNEL__
# define MSG(f, a...)		printk(TINFO f, ## a)
#else
# define MSG(f, a...)		fprintf(stderr, TINFO f, ## a)
#endif



/**************************************************************/
/* PUT SOME OTHER PREFIX HERE IF YOU GOT DOUBLE SYMBOL ERROR! */
/************VV************************************************/
TRACKERProto(mesa)
/************^^************************************************/



#endif
