/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.apache.webapp.admin.server;

import java.util.Locale;
import java.io.IOException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.struts.action.Action;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.ActionMessages;

import javax.management.Attribute;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.webapp.admin.ApplicationServlet;
import org.apache.struts.util.MessageResources;

/**
 * Implementation of <strong>Action</strong> that saves server properties.
 *
 * @author Jazmin Jonson
 * @author Manveen Kaur
 * @version $Revision: 479042 $ $Date: 2006-11-24 16:02:41 -0700 (Fri, 24 Nov 2006) $
 */

public final class SaveServerAction extends Action {
    
    /**
     * The MBeanServer we will be interacting with.
     */
    private MBeanServer mBServer = null;
    
    // --------------------------------------------------------- Public Methods
    
    
    /**
     * Process the specified HTTP request, and create the corresponding HTTP
     * response (or forward to another web component that will create it).
     * Return an <code>ActionForward</code> instance describing where and how
     * control should be forwarded, or <code>null</code> if the response has
     * already been completed.
     *
     * @param mapping The ActionMapping used to select this instance
     * @param actionForm The optional ActionForm bean for this request (if any)
     * @param request The HTTP request we are processing
     * @param response The HTTP response we are creating
     *
     * @exception IOException if an input/output error occurs
     * @exception ServletException if a servlet exception occurs
     */
    public ActionForward execute(ActionMapping mapping,
    ActionForm form,
    HttpServletRequest request,
    HttpServletResponse response)
    throws IOException, ServletException {
        
       // Acquire the resources that we need
        HttpSession session = request.getSession();
        Locale locale = getLocale(request);
        MessageResources resources = getResources(request);
        
        // Acquire a reference to the MBeanServer containing our MBeans
        try {
            mBServer = ((ApplicationServlet) getServlet()).getServer();
        } catch (Throwable t) {
            throw new ServletException
            ("Cannot acquire MBeanServer reference", t);
        }
        
        ActionMessages errors = new ActionMessages();
        
        // Report any errors we have discovered back to the original form
        if (!errors.isEmpty()) {
            saveErrors(request, errors);
            return (new ActionForward(mapping.getInput()));
        }
        
        ServerForm sform = (ServerForm) form;
        String sObjectName = sform.getObjectName();
        // Acquire a reference to the Server MBean
        ObjectName soname = null;
        try {            
            soname = new ObjectName(sObjectName);
        } catch (Throwable t) {
            throw new ServletException
            ("Cannot acquire Server MBean reference ", t);
        }
        

        // Perform attribute updates as requested
        String attribute = null;
        try{          
            attribute = "port";
            int port = 0;
            try {
                port = Integer.parseInt(sform.getPortNumberText());
            } catch (Throwable t) {
                port = 0;
            }
            mBServer.setAttribute(soname,
                                  new Attribute("port", new Integer(port)));   
            // set port warning as port < 1024 requires
            // special software capabilities
            if (port < 1024) {    
                request.setAttribute("warning", "server.port.warning");
            }
            
            attribute = "shutdown";
            mBServer.setAttribute(soname,
                                  new Attribute("shutdown", sform.getShutdownText()));
            
        } catch(Exception e){
            getServlet().log
                (resources.getMessage(locale, "users.error.attribute.set",
                                      attribute), e);
            response.sendError
                (HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
                 resources.getMessage(locale, "users.error.attribute.set",
                                      attribute));
            return (null);
       }
        
        // Forward to the success reporting page
        session.removeAttribute(mapping.getAttribute());
        return (mapping.findForward("Save Successful"));        
    }  
}
