/*************************************************************************\
*   Copyright (C) 2009 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA            *
\*************************************************************************/

//--- LOCAL ---
#include "../config.h"
#include "countrymap.h"
#include "dlgaddcity.h"
#include "ionlistmodel.h"
#include "utils.h"
#include "yawpdefines.h"
#include "logger/streamlogger.h"

//--- QT4 ---
#include <QPalette>
#include <QPushButton>

//--- KDE4 ---
#include <KIcon>
#include <KProgressDialog>
#include <Plasma/DataEngine>
#include <Plasma/BusyWidget>

#define CityRole		Qt::UserRole
#define CountryRole		Qt::UserRole + 1
#define CountryCodeRole		Qt::UserRole + 2
#define ExtraDataRole		Qt::UserRole + 3
#define ProviderRole		Qt::UserRole + 4


DlgAddCity::DlgAddCity( Yawp::Storage * pStorage, QWidget * parent )
	: QDialog( parent )
{
	m_pStorage = pStorage;
	
	setupUi( this );
	
	setAttribute(Qt::WA_DeleteOnClose);

	bttnFind->setIcon( KIcon("edit-find") );
	buttonBox->button( QDialogButtonBox::Apply )->setIcon( KIcon("dialog-ok") );
	buttonBox->button( QDialogButtonBox::Close )->setIcon( KIcon("dialog-close") );
	
	//--- workarround for the not working accept signal ---
	connect(buttonBox,     SIGNAL(clicked(QAbstractButton *)),   this, SLOT(slotApplySelection(QAbstractButton *)));
	connect(bttnFind,      SIGNAL(released()),                   this, SLOT(slotFindLocations()));
	connect(editLocation,  SIGNAL(textChanged(const QString &)), this, SLOT(slotValidateTextInput(const QString &)));

	comboProvider->clear();
	comboProvider->setModel( m_pStorage->ionListModel() );
	slotValidateTextInput(editLocation->text());
	enableApply();

	busyWidget->hide();
	errorIcon->setPixmap( KIcon("dialog-warning").pixmap(32, 32, QIcon::Normal, QIcon::On) );
}

DlgAddCity::~DlgAddCity()
{
}

void
DlgAddCity::sendSelectedCity()
{
	dStartFunct();
	const QListWidgetItem * item = resultList->currentItem();
	if( item )
	{
		CityWeather * cityInfo = new CityWeather();

		cityInfo->setCity(        QUrl::fromPercentEncoding(item->data(CityRole).toString().toUtf8()) );
		cityInfo->setCountry(     QUrl::fromPercentEncoding(item->data(CountryRole).toString().toUtf8()) );
		cityInfo->setCountryCode( QUrl::fromPercentEncoding(item->data(CountryCodeRole).toString().toUtf8()) );
		cityInfo->setExtraData(   QUrl::fromPercentEncoding(item->data(ExtraDataRole).toString().toUtf8()) );
		cityInfo->setProvider(    QUrl::fromPercentEncoding(item->data(ProviderRole).toString().toUtf8()) );
		
		dTracing() << "Requested preselected timezones";
		QStringList vTimeZones( Utils::GetTimeZones( *cityInfo, m_pStorage ) );
		if( vTimeZones.count() == 1 )
			cityInfo->setTimeZone( vTimeZones.at(0) );

		dDebug() << cityInfo->city() << cityInfo->country() << cityInfo->countryCode() << cityInfo->extraData() << cityInfo->provider();
		emit citySelected(cityInfo);
	}

	dEndFunct();
}

void
DlgAddCity::updateLocations(const QString & ion, const QString & location, const QStringList & vTokens )
{
	Q_UNUSED(ion);
	
	dStartFunct();

	const QLatin1String valid("valid");
	const QLatin1String place("place");
	const QLatin1String extra("extra");
	const QLatin1String stationType("stationtype");
	const QLatin1String distance("distance");
	
	if (vTokens.count() >= 3 && vTokens.at(1).compare(valid) == 0)
	{
		if( vTokens.at(2).compare("single") == 0 ||
		    vTokens.at(2).compare("multiple") == 0 )
		{
			int iPos = 3;
			QString sLocation, sExtra, sItemText;

//			dDebug() << "Tokenized search result:" << vTokens;

			//--- go through all places and extract all informations ---
			while( iPos+1 < vTokens.count() && vTokens.at(iPos).compare(place) == 0 )
			{
				sLocation = vTokens.at(iPos+1);
				sItemText = sLocation;
				sExtra.clear();

				//--- go through all attributes that belongs to the current place ---
				iPos+=2;
				while( iPos+1 < vTokens.count() && vTokens.at(iPos).compare(place) != 0 )
				{
					if (vTokens.at(iPos).compare(extra) == 0)
					{
						sExtra = vTokens.at(iPos+1);
					}
					else if (vTokens.at(iPos).compare(stationType) == 0)
					{
						if (vTokens.at(iPos+1).compare(QLatin1String("airport"), Qt::CaseInsensitive) == 0)
							sItemText = QString("%1: %2").arg( i18n(vTokens.at(iPos+1).toUtf8().constData()) ).arg( sItemText );
					}
					else if (vTokens.at(iPos).compare(distance) == 0)
					{
						sItemText.append( QString(" (%1 %2)").arg(i18n("distance:")).arg(vTokens.at(iPos+1)) );
					}
					iPos+=2;
				}

				if( !sLocation.isEmpty() )
				{
					QListWidgetItem * item = new QListWidgetItem;
					item->setText(sItemText);

					/*  We can not change the value sLocation, since some
					 *  providers or dataengines use this as a key to find the specific city!!!
					 */
					item->setData( CityRole, sLocation );
					item->setData( ExtraDataRole, sExtra );
					item->setData( ProviderRole, vTokens.at(0) );
					
					QString sCity, sDistrict, sCountry, sCountryCode;
					Utils::ExtractLocationInfo(sLocation, sCity, sDistrict, sCountry);
					if( Utils::GetCountryCode(sCountry, sCountryCode, m_pStorage) )
					{
						item->setData( CountryCodeRole, sCountryCode );
						item->setData( CountryRole, sCountry );
						
						QPixmap flag = m_pStorage->countryMap()->getPixmapForCountryCode( sCountryCode );
						item->setIcon(QIcon(flag));
					}
					resultList->addItem( item );
				}
			}
		}
		showResultList();
	}
	
	/*  The message boxes will crash the plasma environment :)
	 */
	else if( vTokens.count() >= 2 && vTokens.at(1).compare("timeout") == 0 )
	{
		QString message = i18n("The applet was not able to contact the server, please try again later.");
		dDebug() << message;
		showErrorMessage(message);
	}
	else if( vTokens.count() >= 2 && vTokens.at(1).compare("malformed") == 0 )
	{
		QString message = i18n("Ion has rejected invalid or malformed command.");
		dDebug() << message;
		showErrorMessage(message);
	} 
	else if( vTokens.count() >= 4 )
	{
		QString message = i18n("The place '%1' is not valid. The weather-service is not able to find this place.", location);
		dDebug() << message;
		showErrorMessage(message);
	}
	else
	{
		QString message = i18n("The applet was not able to contact the server, due to an unknown error.");
		dDebug() << message;
		showErrorMessage(message);
	}
	dEndFunct();
}

void
DlgAddCity::enableApply()
{
	dStartFunct();
	QPushButton * bttnApply = buttonBox->button(QDialogButtonBox::Apply);
	bttnApply->setEnabled( resultList->count() > 0 );
	dEndFunct();
}

void
DlgAddCity::slotValidateTextInput(const QString & text)
{
	if (!busyWidget->isVisible())
		bttnFind->setEnabled( !text.isEmpty() );
}

void
DlgAddCity::slotFindLocations()
{
	dStartFunct();
	if( comboProvider->count() == 0 || editLocation->text().isEmpty() )
	{
		showErrorMessage( i18n("You have to enter a city to search for.") );
	}
	else
	{
		resultList->clear();
		busyWidget->show();
		enableApply();
		bttnFind->setEnabled(false);
		
		int iProviderIndex = comboProvider->currentIndex();
		dTracing() << "CurrentProvider: " << iProviderIndex;
		
		QString sProvider = comboProvider->itemData( iProviderIndex >= 0 ? iProviderIndex : 0 ).toString();
		QString sLocation = editLocation->text();
		
		m_sCurrentAction  = QString("%1|validate|%2").arg(sProvider).arg(sLocation);
		
		dTracing() << "Using command: " << m_sCurrentAction;
		
		m_pStorage->ionListModel()->engine()->connectSource(m_sCurrentAction, this);
	}
	dEndFunct();
}

void
DlgAddCity::dataUpdated(const QString & sAction, const Plasma::DataEngine::Data & data)
{
	dStartFunct();
	
	if (m_sCurrentAction.length() > 0 && m_sCurrentAction.compare(sAction) == 0)
	{
		m_sCurrentAction.clear();
		QLatin1String validate = QLatin1String( "validate" );
		
		QStringList vTokens;
		if (data.contains(validate))
		{
			vTokens = data[validate].toString().split(QLatin1Char( '|' ));
			
			// Now we go through all special keys.
			// Right now only ION wunderground will use special keys to bypass information,
			// because standard weather applets does not support further weather information
			// besides "place" and "extra".
			const QList<QString> keys = data.keys();
			foreach(QString key, keys)
			{
				int keyIndex = vTokens.indexOf(key);
				if (keyIndex > 3)
				{
					QStringList vExtraValues = data[key].toString().split(QLatin1Char( '|' ));
					while (vExtraValues.length() >= 2)
					{
						vTokens.insert(keyIndex+1, vExtraValues.takeFirst());
						vTokens.insert(keyIndex+2, vExtraValues.takeFirst());
						keyIndex += 2;
					}
				}
			}
		}

		m_pStorage->ionListModel()->engine()->disconnectSource(sAction, this);
		
		QStringList vAction = sAction.split(QLatin1Char( '|' ));
		updateLocations(vAction[0], vAction[2], vTokens);
		busyWidget->hide();
		bttnFind->setEnabled(true);
		enableApply();
	}

	dEndFunct();
}

void
DlgAddCity::slotApplySelection( QAbstractButton * button )
{
	switch( buttonBox->buttonRole( button ) )
	{
	case QDialogButtonBox::ApplyRole:
		sendSelectedCity();
		accept();
		break;
	case QDialogButtonBox::RejectRole:
		reject();
		break;
	default:
		break;
	}
}

void
DlgAddCity::showErrorMessage(const QString & message)
{
      stackedWidget->setCurrentWidget(errorMessageBox);
      errorText->setText(message);
}

void
DlgAddCity::showResultList()
{
      stackedWidget->setCurrentWidget(resultList);
}
