/******************************************************************************
*                                PlayWolf                                     *
*******************************************************************************
*                                                                             *
*                   Copyright (C) 2008-2009 Giulio Camuffo		      *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License along   *
*   with this program; if not, write to the Free Software Foundation, Inc.,   *
*   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA                *
*                                                                             *
*                                                                             *
*                                                                             *
*   For further information contact me at giuliocamuffo@gmail.com	      *
******************************************************************************/

#include "playercontroller.h"

#include <QtDBus>
#include <QDBusReply>
#include <QtCore/QStringList>
#include <KService>
#include <KRun>
Q_DECLARE_METATYPE(Status)

PlayerController* PlayerController::instance = 0;        //Address of the singleton
int PlayerController::m_playWolfInstancesCounter = 0;

/* DBus arguments need to handle the << and >> operators */

const QDBusArgument & operator<<(QDBusArgument &arg, const Status &change) {
    arg.beginStructure();
    arg << change.int1 << change.int2 << change.int3 << change.int4;
    arg.endStructure();

    return arg;
}

const QDBusArgument & operator>>(const QDBusArgument &arg, Status &change) {
    arg.beginStructure();
    arg >> change.int1 >> change.int2 >> change.int3 >> change.int4;
    arg.endStructure();

    return arg;
}

//----------------------------------------------------------------------

PlayerController* PlayerController::self() {
    if (instance == 0)
	instance = new PlayerController();

    return instance;
}

PlayerController::PlayerController() {
    qDBusRegisterMetaType<Status>();

    amarokTracklistDbus = new QDBusInterface("org.kde.amarok", "/TrackList", "org.freedesktop.MediaPlayer");
    amarokPlayerDbus = new QDBusInterface( "org.kde.amarok", "/Player", "org.freedesktop.MediaPlayer" );

    amarokPlayerDbus->connection().connect( "org.kde.amarok", "/Player", "org.freedesktop.MediaPlayer",
	"StatusChange", this, SLOT(statusChange(const Status &)) );
    amarokPlayerDbus->connection().connect( "org.kde.amarok", "/Player", "org.freedesktop.MediaPlayer",
	"TrackChange", this, SLOT(trackChange(const QMap<QString, QVariant> &)) );
    amarokTracklistDbus->connection().connect( "org.kde.amarok", "/TrackList",
	"org.freedesktop.MediaPlayer", "TrackListChange", this, SLOT(updatePlaylist(const int&)));

    int status_interval=1000;
    timer.connect(&timer,SIGNAL(timeout(void)), this,SLOT(updatePosition()));
    timer.setInterval(status_interval);
}

void PlayerController::registerPlayWolfInstance() {
	++m_playWolfInstancesCounter;
}

void PlayerController::destroy() {
	if (m_playWolfInstancesCounter == 1)
		delete instance;
	else
		--m_playWolfInstancesCounter;
}

PlayerController::~PlayerController() {
}

void PlayerController::playerQuery(const QString &query)  {
  amarokPlayerDbus->call(query);
}

void PlayerController::playerQuery(const QString &query, const int arg) {
  amarokPlayerDbus->call(query, arg);
}

int PlayerController::playerIntQuery(const QString &query) {
    QDBusReply<int> reply = amarokPlayerDbus->call( query );
    if ( reply.isValid() )
	return reply.value();

    return -1;
}

int PlayerController::playlistIntQuery(const QString &query) {
    QDBusReply<int> reply = amarokTracklistDbus->call( query );
    if ( reply.isValid() )
	return reply.value();

    return -1;
}

Status PlayerController::getStatus(void) {
    qDBusRegisterMetaType<Status>();

    QDBusReply<Status> reply = amarokPlayerDbus->call("GetStatus");
    if (reply.isValid())
	return reply.value();

    Status st;
    st.int1 = -1;
    st.int2 = -1;
    st.int3 = -1;
    st.int4 = -1;
    return st;
}

int PlayerController::getSongPosition() {
    return playlistIntQuery("GetCurrentTrack");
}

QVariantMap PlayerController::getSongMetadata(int song) {
    QDBusReply<QMap<QString, QVariant> > argumentList = amarokTracklistDbus->call("GetMetadata", song);
    if (argumentList.isValid())
	return (argumentList.value());

    return QVariantMap();
}

QVariantMap PlayerController::getMetadata(void) {
    return getSongMetadata(getSongPosition());
}

QStringList PlayerController::getPlaylist(int start, int numSongs) {
    QStringList titleList;
    int i = 0;
    do {
	QVariantMap meta = getSongMetadata(start);
	if (meta != QVariantMap())
	    titleList << meta.value("title").toString();
	else
	    break;
	++start;
	++i;
	if ((i == numSongs) and (numSongs != -1)) {
	    break;
	}
    }
    while (true);

    return titleList;
}

void PlayerController::updatePlaylist(const int& size) {
    Q_UNUSED(size);
    emit playlistChanged(getPlaylist(getSongPosition()+1, 3));
}

void PlayerController::playPause() {

    if (getStatus().int1 == -1){
	executeAmarok();

	if(getStatus().int1 != 0)
		playerQuery("Play");
	}

    if (getStatus().int1 == 0)
	playerQuery("Pause");
    else
	playerQuery("Play");

}

void PlayerController::prev() {
    playerQuery("Prev");
}

void PlayerController::next() {
    playerQuery("Next");
}

void PlayerController::stop() {
    playerQuery("Stop");
}

void PlayerController::showAmarok() {
    QDBusInterface amarok( "org.kde.amarok", "/amarok/MainWindow", "org.kde.amarok.MainWindow" );

    if (getStatus().int1 == -1)
	executeAmarok();
    else
	amarok.call("showHide");
}

void PlayerController::executeAmarok() {
    const KService::Ptr amarokService = KService::serviceByStorageId("amarok");

    if (!useCustomAmarokBinPath) {
	if (amarokService) {
	    new KRun(KUrl(amarokService->entryPath()), 0);
	}
    }
    else {
	new KRun(KUrl(customAmarokBinPath), 0);
    }
}

void PlayerController::seek(const int pos) {
    playerQuery("PositionSet",pos);
}

void PlayerController::seekForward(int seconds) {
    seek(playerIntQuery("PositionGet")+seconds*1000);
}

void PlayerController::seekBack(int seconds) {
    int time = playerIntQuery("PositionGet")-seconds*1000;
    if (time >= 0)
	seek(time);
    else
	seek(1);
}

void PlayerController::setVolume(int volume) {
    playerQuery("VolumeSet", volume);
}

int PlayerController::volume() {
    return playerIntQuery("VolumeGet");
}

bool PlayerController::p() {
	 QDBusReply<Status> reply = amarokPlayerDbus->call( "isPlaying" );
    if ( reply.isValid() )
	return true;

    return false;

}


void PlayerController::volumeUp() {
    setVolume(volume() + 5);
}

void PlayerController::volumeDown() {
    setVolume(volume() -5);
}

int PlayerController::rating() {
    metadata = getMetadata();

    return metadata.value("rating").toInt();
}

QString PlayerController::album() {
    metadata = getMetadata();

    return metadata.value("album").toString();
}

void PlayerController::updateInfos(void) {
    statusChange(getStatus());
}

void PlayerController::trackChange(const QVariantMap &meta) {
    metadata = meta;
    emit metadataChanged(metadata);
    updatePosition();
    updatePlaylist();
}

void PlayerController::statusChange(const Status &status){
    int stat=status.int1;
    emit statusChanged(stat);
    if (stat == 0) {
	timer.start();
	trackChange(getMetadata());
    }
    else if (stat == 1) {
	timer.stop();
	trackChange(getMetadata());
    }
    else {
	timer.stop();
	emit clear();
    }
}

int PlayerController::getPosition(void) {
    return playerIntQuery("PositionGet");
}

void PlayerController::updatePosition(void) {
    emit positionChanged(getPosition(), metadata.value("mtime").toInt());
}

void PlayerController::setAmarokPath(bool useCustom, QString& path) {
    useCustomAmarokBinPath = useCustom;
    customAmarokBinPath = path;
}

QString& PlayerController::amarokPath() {
    return customAmarokBinPath;
}

bool PlayerController::useCustomAmarokPath() {
    return useCustomAmarokBinPath;
}

#include "playercontroller.moc"
