/*
 * シェルスクリプトエンジン
 */

#ifndef SAPHIRE_H
#define SAPHIRE_H

#include "saphire/saphire_debug.h"
#include "saphire/saphire_vector.h"
#include "saphire/saphire_list.h"
#include "saphire/saphire_string.h"
#include "saphire/saphire_kanji.h"
#include "saphire/saphire_curses.h"
#include "saphire/saphire_hash.h"
#include "saphire/saphire_extra.h"

/*
    使い方のサンプルはmain.cを参考にしてください
*/

extern enum eKanjiCode gKanjiCode;
    // saphireが使うデフォルトの漢字コード
    // 初期値はUTF8
    // (端末の漢字コードとは別でかまわない)
enum eLineField { kLF, kCRLF, kCR, kBel };
extern enum eLineField gLineField;
    // saphireが文字列処理の内部コマンドで使う改行コード
enum eAppType { kATOptC, kATXApp, kATCursesApp, kATConsoleApp };
    // アプリケーションの種類
    //
    // kATOptC コマンドラインから使う場合
    // kATXApp Xを使っているアプリ
    // kATCursesApp 端末制御ライブラリを使っているコンソールアプリ
    // kATConsoleApp 端末制御ライブラリを使わないコンソールアプリ

enum eRuntimeScript { kRSNoRead, kRSSource, kRSObject };
    // ランタイムスクリプトをどうするのかの設定

    // kRSNoRead 読み込まない
    // kRSSource ソースファイルを読み込む
    // kRSObject オブジェクトファイルを読み込む

typedef struct 
{
    vector_obj* mStatments;
    string_obj* mSource;
} sStatments;

void sStatments_delete(sStatments* self);

#if defined(MDEBUG)
sStatments* sStatments_new_debug(const char* fname, int line, const char* func_name);
#define STATMENTS_NEW() sStatments_new_debug(__FILE__, __LINE__, __FUNCTION__)
sStatments* sStatments_new2_debug(sStatments* statments, const char* fname, int line, const char* func_name);
#define STATMENTS_NEW2(o) sStatments_new2_debug(o, __FILE__, __LINE__, __FUNCTION__)
#else
sStatments* sStatments_new();
#define STATMENTS_NEW() sStatments_new()
sStatments* sStatments_new2(sStatments* statments);
#define STATMENTS_NEW2(o) sStatments_new2(o)
#endif

typedef struct _sClass {
    string_obj* name;      // クラス名
    sStatments* statments; // 命令表
    struct _sClass* parent;        // オーバーライド先
    BOOL input;             // 実行時の情報
    BOOL output;

    int mRefCount;
} sClass;

sClass* sClass_new(char* name, sStatments* statments, sClass* parent, int ref_count);
void sClass_delete(sClass* self);
  
void sClass_save(sClass* self, int fd);
sClass* sClass_load(int fd);
void sClass_view(sClass* self);
/// オブジェクトの定義
typedef struct _sObject {
    string_obj* mName;               /// 自分の名前
    string_obj* mClassName;          /// もしあるなら属しているクラス名
    hash_obj* mGlobals;
    hash_obj* mHashs;
    hash_obj* mArrays;
    hash_obj* mObjects;
    hash_obj* mRefs;
    hash_obj* mClasses;

    struct _sObject* mParent;

    hash_obj* mMethods;

    int mRefCount;
} sObject;

sObject* sObject_new(char* name, char* klass, sObject* parent, int ref_count);
void sObject_delete(sObject* self);

sObject* saphire_get_object(char* name, sObject* object, BOOL global, BOOL local);
    // 引数の名前のオブジェクトを返す

/// メモリチェッカー
void memchecker_init();
enum eMemCheckerKind { kMCNothing, kMCRef, kMCArray, kMCObject, kMCHash, kMCVar, kMCGlobal, kMCFunction, kMCClass };

void* memchecker_GC_malloc(int size, enum eMemCheckerKind obj_kind);
    // メモリチェッカーを使ったGCのメモリ領域確保
int memchecker_is_enable_mem(void* obj);
    // オブジェクトの種類を返す 0:無し 1:string_obj*

typedef struct {
    void* mMem;
    enum eMemCheckerKind mKind;
} sRef;

sRef* sRef_new(void* mem, enum eMemCheckerKind kind);
void sRef_delete(sRef* self);
void sRef_inc_refcount(sRef* self);
void sRef_dec_refcount(sRef* self);

sRef* saphire_get_ref(char* name);
    // 引数の名前のリファレンスを返す

vector_obj* saphire_methods_names(sObject* object);
    // 引数のオブジェクトのメソッドの一覧を返す

typedef struct {
    int mFd;
    char* mBuffer;
    char* mP;
    int mMallocSize;
} sWFd;

#ifndef MDEBUG
sWFd* sWFd_new(int fd);

#define WFD_NEW(o) sWFd_new(o)

#else
sWFd* sWFd_new_debug(int fd, const char* fname, int line, const char* func);
#define WFD_NEW(o) sWFd_new_debug(o, __FILE__, __LINE__, __FUNCTION__)

#endif

void sWFd_delete(sWFd* self);
BOOL sWFd_push_back(sWFd* self, char* str);
BOOL sWFd_flash(sWFd* self);
BOOL sWFd_close(sWFd* self);

/*
typedef struct {
    int mFd;
    string_obj* mBuffer;
} sRFd;
*/

typedef struct {
    int mFd;
    char* mBuffer;
    char* mPtr;
    int mMallocSize;
} sRFd;

#if defined(MDEBUG)

sRFd* sRFd_new_debug(int fd, const char* fname, int line, const char* func_name);
#define RFD_NEW(o) sRFd_new_debug(o, __FILE__, __LINE__, __FUNCTION__)

sRFd* sRFd_new2_debug(int fd, char* str, const char* fname, int line, const char* func_name);
#define RFD_NEW2(o, o2) sRFd_new2_debug(o, o2, __FILE__, __LINE__, __FUNCTION__)

#else

sRFd* sRFd_new(int fd);
#define RFD_NEW(o) sRFd_new(o)

sRFd* sRFd_new2(int fd, char* str);
#define RFD_NEW2(o, o2) sRFd_new2(o, o2)

#endif

sRFd* sRFd_new3(int fd, MANAGED char* buffer, int malloc_size);
#define RFD_NEW3(o, o2, o3) sRFd_new3(o, o2, o3)
sRFd* sRFd_new_from_wfd(int fd, sWFd* nextout);

void sRFd_delete(sRFd* self);
BOOL sRFd_close(sRFd* self);
int sRFd_read_all(sRFd* self, string_obj* str);
int sRFd_read_all_preserve(sRFd* self, string_obj* str);
int sRFd_read_oneline(sRFd* self, string_obj* str, enum eLineField lf);
int sRFd_read_oneline_num(sRFd* self, string_obj* str, int line_num, enum eLineField lf);
int sRFd_read_oneline_preserve(sRFd* self, string_obj* str, enum eLineField lf);
int sRFd_read_oneline_preserve_num(sRFd* self, string_obj* str, int line_num, enum eLineField lf);
int sRFd_read_onechar_num(sRFd* self, string_obj* str, int char_num);
int sRFd_read_onechar_preserve_num(sRFd* self, string_obj* str, int char_num);
int sRFd_read_all_to_buffer(sRFd* self);
void sRFd_set_buffer(sRFd* self, char* buffer);

void saphire_init(enum eAppType app_type, BOOL job_control, enum eRuntimeScript runtime_script, BOOL run_user_runtimescript);
    // 初期化。saphire_shellなどを実行する前に必ず必要
    // アプリケーションの種別とジョブコントロールの有無とランタイムスクリプトをどうするかを読み込むかを設定してください
void saphire_final();
    // 終了化。終了時に必要
void saphire_set_signal();
    // saphireを実行中に使うシグナルの設定を設定する。
    // saphire_shell, saphie_shell3などのsaphireのコードを実行する前
    // に必ず呼んでください。
    // 呼ばれないと正しくコマンドは実行されません。

    // 処理しているのは
    // SIGCHLD, SIGINT, SIGCONT, SIGWINCH, SIGUSR1
    // 無視しているのは
    // SIGTTOU, SIGTTIN, SIGTSTP, SIGQUIT, SIGPIPE
void saphire_set_signal_optc();
    // saphireを実行中に使うシグナルの設定を設定する。
    // -c時に使われる

    // 処理しているのは
    // SIGCHLD, SIGINT, SIGCONT, SIGWINCH, SIGUSR1
    // 無視しているのは
    // SIGTTOU, SIGTTIN, SIGTSTP, SIGQUIT, SIGPIPE
void saphire_restore_signal_default();
    // シグナルを初期値に戻す
extern void (*saphire_set_signal_other)();
    // もし関数ポインタが代入されていたら
    // saphire_set_signalで実行される
extern string_obj* gErrMsg;
    // エラーが起こった時はこの文字列にメッセージ入っているので出力してください
int saphire_shell(char* command, char* title, sWFd* pipeout, sRFd* pipein, int pipeerr);
    // 文字列を読み込んで実行
    // commandは実行するコマンドの文字列。titleはタイトル
    // pipeoutは出力先。pipeinは入力先
    // pipeerrはエラー先（ファイルディスクリプタ)
    // 戻り値はプログラムの終了コードだが、0以下だと
    // saphire内でエラーが起こっているのでgErrMsgをどこかに
    // 出力してください。
int saphire_shell3(sWFd* pipeout, char* command, char* title, sRFd* pipein);
    // 文字列を読み込んで実行。コマンドの出力はpipeout->mBuffer
    // 戻り値はプログラムの終了コードだが、0以下だとsaphire内でエラーが
    // 起こっている
    // のでgErrMsgをどこかに出力してください。
    // resultはstring_newで領域を確保しておかないといけない
int saphire_load(char* fname, sWFd* pipeout, sRFd* pipein, int pipeerr);
    // スクリプトファイルを読み込んで実行
    // pipeoutは出力先。pipeinは入力先
    // pipeerrはエラー先（ファイルディスクリプタ)
    // 戻り値はプログラムの終了コードだが、0以下だとsaphire内で
    // エラーが起こっている
    // のでgErrMsgをどこかに出力してください。

BOOL saphire_compile(char* fname, char* out_fname);
    // fnameのファイルをコンパイルしてout_fnameに保存
    // スクリプトファイルを読み込んでコンパイルして
    // ファイルとして書き込む関数。
    // エラー時はgErrMsgにエラーメッセージが入っている
int saphire_load_obj(char* fname, sWFd* pipeout, sRFd* pipein, int pipeerr, vector_obj* parent_blocks);
    // コンパイル済みのファイルfnameを実行
    // pipeoutは出力先。pipeinは入力先
    // pipeerrはエラー先（ファイルディスクリプタ)
    // 戻り値はプログラムの終了コードだが、0以下だとsaphire内で
    // エラーが起こっている
    // のでgErrMsgをどこかに出力してください。
    // parent_blocksは親のブロック引数。無いならNULL
void saphire_wait_background_job();
    // バックグランドジョブの状態のチェック
    // ジョブ管理をする場合メインループ中に定期的に実行してください
extern void (*saphire_job_done)(int job_num, char* job_title);
    // ジョブ管理しているときにジョブが終わったら実行する処理
    // job_numに終わったジョブの番号が入いれて呼ばれる。
    // job_titleに終わったジョブのタイトルを入れて呼ばれる
int saphire_job_num();
    // 現在のジョブの数を返す
char* saphire_job_title(int num);
    // num番号のジョブのタイトルを返す。
void saphire_kill_job(int num);
    // num番号のジョブを消す
void saphire_kill_all_jobs();
    // 全てのジョブを強制消去
    // アプリケーションを終了する前に実行してください
    // ただし大抵のジョブは終わっていないと問題が起こると
    // 思われるのでsaphire_job_numで終わっていないジョブがあると
    // アプリケーションを終われないようにしとくほうがいいかもしれません。

void saphire_init_stack_frame();
    // スタックフレームの初期化
    // ローカル変数を初期化します

extern vector_obj* gSaphireProgCompletions;               // 補完候補

extern hash_obj* gGlobals;
    // グローバル変数 キーは変数名 内容はstring_obj*
extern hash_obj* gArrays;
    // グローバル変数の配列 キーは変数名
    // 内容は配列(vector_obj*) 配列の中にはstring_obj*
    // が入っている
extern hash_obj* gHashs;
    // グローバル変数の配列 キーは変数名
    // 内容はハッシュ(hash_obj*) ハッシュの中にはstring_obj*
    // が入っている

void saphire_rehash();
    // プログラム補完候補の一覧の再読み込み
    // $PATHから読み込む
    // gSaphireProgCompletions(動的配列型vector_obj*)に文字列型(string_obj*)として入る
void saphire_rehash2();
    // プログラム補完候補の一覧の再読み込み
    // 外部コマンドはりハッシュしない

extern BOOL gKitutukiExit;
        // exit内部コマンドが呼ばれたら終了コードが入る
        // メインループで>=0ならプログラムを終了してください。
extern void saphire_get_quoted_fname(char* fname, string_obj* quoted_fname);
        // saphireシェルで使われる特殊な文字をクォートした文字列を返す
        // &,|,>など

typedef BOOL (*fInnerCommand)(int* rcode, vector_obj* argv, vector_obj* blocks, vector_obj* parent_blocks, sWFd* nextout, sRFd* nextin, int nexterr, char* title, BOOL input, char* sname, int sline);
        // 内部コマンドの実行コード
        // 戻り値は実行に成功したらTRUE, 実行に失敗したらFALSEを返してください。
        // rcodeはリターンコード. 何も入力しなかったら最初は1が入っている
        // コマンドが成功したら*rcode = 0などとしてコマンドの成功を
        // 示してください。
        // argvはstring_obj*（文字列）のvector_obj*(動的配列)。
        // コマンドライン引数が入っている。(0番目はコマンド名)
        // blocksはsStatments*(文)のvector_obj*(動的配列)
        // 引数のブロックが入っている
        // parent_blocksはsStatments*(文)のvector_obj*(動的配列)
        // 親のブロックが入っている
        // 親がなければNULL
        // nextoutは出力先 (パイプやファイルや端末など適切に処理される)
        // nextinは入力元
        // nexterrはエラー出力先
        // titleはコマンドのタイトルが入っている
        // inputはパイプ処理を行う場合はTRUEになっている
        // ls | command などのcommand
        // | command (コンテキストパイプ)とかのcommandの場合はTRUE
        // command | lessのcommandだとFALSE
        // sname,sline ソースファイルの位置

void saphire_add_inner_command(char* name, fInnerCommand fun);
       // 内部コマンドの追加。上のfInnerCommandを実装して第二引数に
       // 関数ポインタを入力。第一引数はコマンド名

extern volatile BOOL gKitutukiSigInt;  // CTRL-Cが押されたらTRUE
extern volatile BOOL gKitutukiSigTstp;  // CTRL-Zが押されたらTRUE
extern volatile BOOL gKitutukiSigCont;  // SITCONT食らったらTRUE

string_obj* saphire_get_local_var(char* name, sObject* object);
    // 現在のスタックフレームでのローカル変数を返す
    // 無ければNULLが返る

BOOL saphire_set_local_var(char* name, char* value, sObject* object);
    // 現在のスタックフレームにローカル変数を追加する
    // FALSEが帰ったらinvalid refferenceが起こっている
void saphire_delete_local_var(char* name);
    // 現在のスタックフレームのローカル変数を削除する

string_obj* saphire_get_global_var(char* name, sObject* object, BOOL global);
    // グローバル変数を返す
    // 無ければNULLが返る
BOOL saphire_set_global_var(char* name, char* value, sObject* object);
    // グローバル変数を設定する

typedef struct {
   void** mTable;
   int mTableSize;
   
   int mCount;

   int mRefCount;
} sAry;

sAry* saphire_get_array(char* name, sObject* object, BOOL global, BOOL local);
    // 配列を得る

typedef struct {
   hash_it** mTable;
   int mTableSize;

   hash_it* mEntryIt;

   int mCounter;                // 全走査用
   int mRefCount;
} sHash;
 
sHash* saphire_get_hash(char* name, sObject* object, BOOL global, BOOL local);
    // ハッシュを得る

void saphire_clear_hash_element(sHash* h);
    // ハッシュをクリアする

sClass* saphire_get_class(char* name, sObject* object, BOOL global);
    // クラスを得る

typedef struct _sFunction {
    string_obj* name;       // コマンド名
    string_obj* arg_name;
    //vector_obj* args;       // リファレンスの引数
    sStatments* statments; // 命令表
    BOOL input;             // 実行時の情報
    BOOL output;

    struct _sFunction* parent;     // オーバーライドしたときの親

    int mRefCount;
} sFunction;

sFunction* saphire_get_fun(char* name, sObject* object, BOOL global);
    // 関数を得る

void saphire_set_array(char* name, vector_obj* array);
    // 配列を設定する;
void saphire_insert_array_element(sAry* ary, int n, string_obj* item);
void saphire_push_back_array_element(sAry* ary, string_obj* item);

BOOL saphire_compile2(char* cmdline, char* fname, sStatments* statments);
    // cmdlineのソースをコンパイルして結果をstatmentsに格納する。
    // コンパイル結果のstatmentsはsaphire_runによって実行できる。
    // statmentsはsStatments_newによって初期化されないといけない。
    // エラー時はgErrMsgにエラー内容が入っている
int saphire_run(sStatments* statments, char* title, sWFd* pipeout, sRFd* pipein, int pipeerr);

    // サンプル
    /*

    sStatments statments = sStatments_new();

    saphire_compile2("ls -al | less; pwd", "title", statments);

    saphire_run(statments, "run", STDOUT_FILENO, STDIN_FILENO, STDERR_FILENO, FALSE);

    */

// 出力をCTRL-Cで止めたいので、この関数を経由して書き込む
// 戻り値 FALSE --> CTRL-Cで止められた
// 戻り値 TRUE --> 正常終了
BOOL statment_tree_internal_commands_write_nextout(sWFd* nextout, char* str);

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
int statment_tree_internal_commands_read_nextin(sRFd* nextin, string_obj* str);

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
// 読み込んでもバッファを読み込んだ分のバッファを消さない
int statment_tree_internal_commands_read_nextin_preserve(sRFd* nextin, string_obj* str);

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 一行読み込む
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
int statment_tree_internal_commands_read_nextin_oneline(sRFd* nextin, string_obj* str, enum eLineField lf);

// 出力をCTRL-Cで止めたいので、この関数を経由して読み込む
// 一行読み込み
// 戻り値 -1 --> CTRL-Cで止められた
// 戻り値 0 --> 正常終了
// 戻り値 1 --> EOF
// 読み込んでも読み込んだ分のバッファを消さない
int statment_tree_internal_commands_read_nextin_oneline_preserve(sRFd* nextin, string_obj* str, enum eLineField lf);

void saphire_sweep();
    // ごみ掃除
    // sapire_shellやsaphire_loadが呼ばれた後呼び出してください
int saphire_gc_sweep(sWFd* nextout);
    // GCのごみ掃除

extern vector_obj* gSaphireVarCompletions;
// 変数補完候補

void saphire_make_var_completion_list();          
// 変数補完候補を更新

typedef struct {
    sStatments* mStatments;
    vector_obj* mArgs;
    int mCount;
} sBlock;
// ブロック

void err_msg(char* msg, char* sname, int line);
    // エラーメッセージを設定
 
extern BOOL gRunningOnTerminal;
    // 端末上で動いているかどうか

/// cancelable
BOOL vector_sort_cancelable(vector_obj* self, sort_if fun);
BOOL memcpy_cancelable(char* mem, char* mem2, size_t size);
BOOL string_put_cancelable(string_obj* self, char* str);
BOOL string_push_back_cancelable(string_obj* self, char* str);
string_obj* string_new_cancelable(char* str);
int str_kanjilen_cancelable(enum eKanjiCode code, char* mbs);

#endif

