# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"

require "rast_test"
require "rast/filter"
require "rast/database-generatable"

module Rast
  module Filter
    load_mime_filter("mp3.rb")

    class AudioMpegTest < Test::Unit::TestCase
      include DatabaseGeneratable

      def test_invoke
        invoke_test_simple_v1
        invoke_test_simple_v2_3
        invoke_test_error
      end

      def invoke_test_simple_v1
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        path = File.join(MakeVariable::TOP_SRCDIR, "tests", "data",
                         "filter", "ja", "mp3tagv1.mp3")
        file = File.open(path)
        bucket = FileBucket.new(file)
        brigade.insert_tail(bucket)
        brigade.insert_tail(EOSBucket.new)
        filter_module = AudioMpeg.new
        filter_module.invoke(filter, brigade, "audio/mpeg")

        buf = ""
        filter.passed[0].brigade.each do |bucket|
          buf.concat(bucket.read)
        end

        assert_equal("text/plain", filter.passed[0].mime_type)
        assert_match(/^title: 曲名$/m, buf)
        assert_match(/^artist: アーティスト$/m, buf)
        assert_match(/^album: アルバム$/, buf)
        assert_match(/^comments: コメント$/, buf)
        assert_equal("曲名", filter.properties["title"])
        assert_equal("アーティスト", filter.properties["artist"])
        assert_equal("アルバム", filter.properties["album"])
        assert_equal(2005, filter.properties["year"])
        assert_equal("コメント", filter.properties["comments"])
        assert_equal(1, filter.properties["tracknum"])
        assert_equal("JPop", filter.properties["genre"])
        assert_equal(146, filter.properties["genre_id"])
        assert_equal("アーティスト", filter.properties["author"])
        assert_equal(9, filter.properties.length)
        assert_equal(1, filter.passed.length)
      end

      def invoke_test_simple_v2_3
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        path = File.join(MakeVariable::TOP_SRCDIR, "tests", "data",
                         "filter", "ja", "mp3tagv2_3.mp3")
        file = File.open(path)
        bucket = FileBucket.new(file)
        brigade.insert_tail(bucket)
        brigade.insert_tail(EOSBucket.new)
        filter_module = AudioMpeg.new
        filter_module.invoke(filter, brigade, "audio/mpeg")

        buf = ""
        filter.passed[0].brigade.each do |bucket|
          buf.concat(bucket.read)
        end

        assert_equal("text/plain", filter.passed[0].mime_type)
        assert_match(/^title: mp3テスト$/m, buf)
        assert_match(/^artist: アーティスト$/m, buf)
        assert_match(/^album: アルバム$/, buf)
        assert_match(/^comments: これはmp3ファイルです$/, buf)
        assert_equal("mp3テスト", filter.properties["title"])
        assert_equal("アーティスト", filter.properties["artist"])
        assert_equal("アルバム", filter.properties["album"])
        assert_equal(2005, filter.properties["year"])
        assert_equal("これはmp3ファイルです", filter.properties["comments"])
        assert_equal(1, filter.properties["tracknum"])
        assert_equal("JPop", filter.properties["genre"])
        assert_equal(146, filter.properties["genre_id"])
        assert_equal("アーティスト", filter.properties["author"])
        assert_equal(9, filter.properties.length)
        assert_equal(1, filter.passed.length)
      end

      def invoke_test_error
        create_options = {
          "encoding" => "utf8",
          "preserve_text" => true,
          "properties" => [],
        }
        db_name = generate_db_name
        LocalDB.create(db_name, create_options)

        LocalDB.open(db_name, Rast::DB::RDWR) do |db|
          doc = db.create_document
          chain = FilterChain.new(doc)
          brigade = Brigade.new
          brigade.insert_tail(TransientBucket.new("invalid type file"))
          brigade.insert_tail(EOSBucket.new)
          assert_raise(RubyError) do
            chain.invoke(brigade, "audio/mpeg")
          end
        end
      end
    end
  end
end
