/***************************************************************************
 *   Copyright (C) 2006-2009 by Robert Keevil                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; under version 2 of the License.         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#ifndef LIBSCROBBLE_H
#define LIBSCROBBLE_H

#include <string>
#include <vector>
#include <ctime>

#ifdef WIN32
#define CURL_STATICLIB
#endif

#include <curl/curl.h>

#if defined(HAVE_WMDM) && defined(HAVE_LIBMTP)
#error Cannot have both WMDM and Libmtp at the same time
#endif

#if defined(HAVE_WMDM) || defined(HAVE_LIBMTP)
#define HAVE_MTP       // to enable MTP support
#endif

#define HAVE_ITUNESDB  // to enable ipod support
#define HAVE_LOG       // to enable portable player log support

#ifdef HAVE_WMDM
#include <windows.h>
#include "initguid.h"
#include "mswmdm_i.c"
#include "mswmdm.h"
#include "sac.h"
#include "scclient.h"
#endif
#ifdef HAVE_LIBMTP
#include <libmtp.h>
#endif

#define CLIENT_ID "qts"
#define CLIENT_VERSION "0.10"
#define HANDSHAKE_HOST "post.audioscrobbler.com"
#define PROTOCOL_VERSION "1.2"

#define MHOD_TITLE 1
#define MHOD_ALBUM 3
#define MHOD_ARTIST 4

using std::string;

// N.B.: The following strings are UTF-8
struct scrob_entry{
    string artist;
    string album;
    string title;
    string tracknum;
    int length;
    char played;
    int when;    // times in GMT
    string mb_track_id;
};

enum {
    SCROBBLE_NONE = -1,
    SCROBBLE_LOG,
    SCROBBLE_IPOD,
#ifdef HAVE_MTP
    SCROBBLE_MTP,
#endif
    // Add new method above this
    SCROBBLE_NUM_METHODS
};

enum LOG_LEVEL {
    LOG_ERROR,
    LOG_INFO,
    LOG_DEBUG,
    LOG_NUM_LEVELS
};

const string LOG_LEVEL_NAMES[LOG_NUM_LEVELS] = { "ERROR", "INFO", "DEBUG" };

struct log_entry {
    LOG_LEVEL level;
    string msg;
};

string MD5Digest(const char*);

class Scrobble {
private:
    string error_str;
    bool have_mb;
    int scrobble_method;

    // handshaking stuff
    bool handshake();
    bool need_handshake;
    string sessionid;
    string nowplay_url;
    string submit_url;
    bool real_submit();

    string username;
    string password_hash;

    bool scrob_init;

    std::vector<scrob_entry> entries;
    std::vector<log_entry> log_messages;

    void add_log(LOG_LEVEL, string);
    LOG_LEVEL log_print_level;

    int is_dst;
    string zonename;
    int offset;

    bool do_proxy();
    string proxy_host;
    long proxy_port;
    string proxy_user;
    string proxy_pass;
    bool proxy_winauth;

    string int2string(int);
    int string2int(string);

    bool mtp_connected;

#ifdef HAVE_WMDM
    IComponentAuthenticate *mtp_ICompAuth;
    CSecureChannelClient *mtp_pSacClient;
    IWMDeviceManager3 *mtp_pIdvMgr;
    std::vector<IWMDMDevice3*> mtp_devices;
    void mtp_do_storage(bool);
    void mtp_recurse_storage(IWMDMEnumStorage*, bool);
    HRESULT mtp_get_metadata(IWMDMStorage *pStorage, bool);
    void mtp_device_name(IWMDMDevice3*);
    string wstring_to_utf(const std::wstring&);
#endif
#ifdef HAVE_LIBMTP
    LIBMTP_mtpdevice_t *mtp_device_list;
    void mtp_trackinfo(LIBMTP_mtpdevice_t *, LIBMTP_track_t *, bool);
    void mtp_iterate(bool);
#endif

public:
    Scrobble();
    ~Scrobble();

    bool init(int, int);
    bool parse_file(const string&, int);
    bool parse_db(const string&, int);
    void recalc_dt(int);
    void recalc_now();
    size_t get_num_tracks();
    scrob_entry get_track(size_t);
    void remove_track(size_t);
    void update_track(scrob_entry, size_t);
    bool submit(const string&, const string&);
    string get_error_str() { return error_str; };
    bool mb_present() { return have_mb; };
    int get_last_method() { return scrobble_method; };
    void clear_method();
    void set_proxy(const string&, long, const string&, const string&, bool);
    void cleanup_tracks(void);
    size_t get_num_logs();
    log_entry get_log(size_t);
    void clear_log();
    void set_log_level(LOG_LEVEL);
    bool check_age();

    void cancel_submission();
    double get_u_tot();
    double get_u_now();

#ifdef HAVE_MTP
    // generic functions, used for both implementations
    bool mtp_connect();
    void mtp_disconnect();
    void mtp_get_tracks();
    void mtp_clear_tracks();
#endif

    int get_dst() { return is_dst; };
    int get_offset() {return offset; };
    string get_zonename() { return zonename; };
    string offset_str();
};

#endif
