
%{ /* -*-C-*- */

/* qmlex.l: lexical analyzer for Q interpreter commands */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* make sure we have flex */

#ifndef FLEX_SCANNER
#error "Sorry, this program requires flex."
#endif

#include "qdefs.h"
#include "qmparse.h"

int		pmode = -1;	/* parse mode indicator */
static int	lexinit = 0;	/* initial state */
static char	_sflag;		/* set when input comes from string */
static char    *_s;		/* string buffer */
static char    *_sp;		/* pointer to end of last token */
static FILE    *_fp;		/* input file */
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
static iconv_t *_ic;		/* associated iconv handle */
#endif

static int abufsz = 0, bufp = 0;
static char *buf = NULL;

static THREAD *thr;

static int checkid(int tok, char *name);
static bigint();
static string();
static utf8_qualid();

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)

#define CHUNKSZ 128

static inline char *
ictoutf8(iconv_t ic[2], char *s)
{
  if (ic[0] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[0] = iconv_open("UTF-8", codeset);
    else
      ic[0] = (iconv_t)-1;
  }
  if (ic[0] == (iconv_t)-1)
    return NULL;
  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic[0], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return NULL;
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

#endif

static char *getbuf(FILE *fp)
{
  static char *mybuf = NULL;
  static size_t mybufsz = 0;

  char *t, *r;
  int l;

  if (!mybuf) {
    mybuf = malloc(MAXSTRLEN);
    if (!mybuf) return NULL;
    mybufsz = MAXSTRLEN;
  }

  *mybuf = 0; t = mybuf;
  while((r = fgets(t, MAXSTRLEN, fp)) && *t &&
	t[(l = strlen(t))-1] != '\n') {
    /* try to enlarge the buffer: */
    int k = t-mybuf+l;
    char *mybuf1;
    if (mybuf1 = (char*)realloc(mybuf, mybufsz+MAXSTRLEN)) {
      mybuf = mybuf1;
      t = mybuf+k;
      mybufsz += MAXSTRLEN;
    } else
      return NULL;
  }
  return mybuf;
}

static char *mybuf = NULL, *mybufptr = NULL;

static inline int getch(void)
{
  if (_sflag)
    if (*_s)
      return *_s++;
    else
      return EOF;
 read:
  if (mybufptr && *mybufptr)
    return *mybufptr++;
  if (mybuf) free(mybuf);
  mybufptr = NULL;
  mybuf = getbuf(_fp);
  if (!mybuf) return EOF;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  {
    char *buf = ictoutf8(_ic, mybuf);
    if (buf)
      mybuf = buf;
    else
      mybuf = strdup(mybuf);
  }
#else
  mybuf = strdup(mybuf);
#endif
  if (!mybuf) return EOF;
  mybufptr = mybuf;
  if (!*mybuf) return EOF;
  goto read;
}

/* redefined input macro: */

#undef YY_INPUT
#define YY_INPUT(buf,result,max_size)\
	{\
	int c = getch();\
	result = (c == EOF)?YY_NULL:(buf[0]=c,1);\
	}

#define tok(x) if(_sflag)_sp=_s;BEGIN(0);return x;
#define tok2(x) if(_sflag)_sp=_s;BEGIN(arg);return x;
#define tok3(x) if(_sflag)_sp=_s;BEGIN(arg2);return x;
#define tok4(x) if(_sflag)_sp=_s;BEGIN(arg3);return x;

char *my_yytext, yylastsym[MAXSTRLEN];
static char my_yytextbuf[MAXSTRLEN];

#define YY_USER_ACTION my_yytext = NULL;

%}

%s cmd arg arg2 arg3

L			[A-Za-z_]
O			[0-7]
D			[0-9]
X			[0-9A-Fa-f]
SF			([Ee][+-]?{D}+)
WS			[ \t\f\r]
NWS			([^ \t\f\r;]|\\.)
NWSNQ			([^ \t\f\r;"']|\\.)
DELIM			[\"()\[\]{},;]
NDELIM			[^0-9\1-\40\"()\[\]{},;]
SYM			({L}({L}|{D})*({NDELIM}|::)?|{NDELIM}|::)

%%

	if (lexinit) {
	  lexinit = 0;
	  if (pmode == INTERACT || pmode == SOURCE)
	    BEGIN(cmd);
	  yy_set_bol(1);
	  if(_sflag)_sp=_s;
	  return pmode;
	}

<cmd>^"#!"		{ skip(); BEGIN(0); }
<cmd>^{WS}*"@"		;
<cmd>{WS}+		;
<cmd>\?			BEGIN(0);
<cmd>!			tok(yytext[0]);
<cmd>break		tok3(_BREAK_);
<cmd>cd			tok2(_CHDIR_);
<cmd>chdir		tok2(_CHDIR_);
<cmd>clear		tok4(_CLEAR_);
<cmd>completion_matches	tok2(_COMPLETION_MATCHES_);
<cmd>copying		tok2(_COPYING_);
<cmd>cstacksize		tok2(_CSTACKSIZE_);
<cmd>debug		tok2(_DEBUG_);
<cmd>dec		tok2(_DEC_);
<cmd>def		tok(_DEF_);
<cmd>echo		tok2(_ECHO_);
<cmd>edit		tok2(_EDIT_);
<cmd>fix		tok2(_FIX_);
<cmd>format		tok2(_FORMAT_);
<cmd>help		tok2(_HELP_);
<cmd>hex		tok2(_HEX_);
<cmd>histfile		tok2(_HISTFILE_);
<cmd>histsize		tok2(_HISTSIZE_);
<cmd>import		tok2(_IMPORT_);
<cmd>imports		tok2(_IMPORTS_);
<cmd>load		tok2(_LOAD_);
<cmd>ls			tok2(_LS_);
<cmd>memsize		tok2(_MEMSIZE_);
<cmd>modules		tok2(_MODULES_);
<cmd>oct		tok2(_OCT_);
<cmd>path		tok2(_PATH_);
<cmd>profile		tok(_PROFILE_);
<cmd>prompt		tok2(_PROMPT_);
<cmd>pwd		tok2(_PWD_);
<cmd>run		tok2(_RUN_);
<cmd>save		tok2(_SAVE_);
<cmd>sci		tok2(_SCI_);
<cmd>\./{WS}		tok2(_SOURCE_);
<cmd>source		tok2(_SOURCE_);
<cmd>stacksize		tok2(_STACKSIZE_);
<cmd>stats		tok2(_STATS_);
<cmd>std		tok2(_STD_);
<cmd>tbreak		tok(_TBREAK_);
<cmd>undef		tok(_UNDEF_);
<cmd>unparse		tok2(_UNPARSE_);
<cmd>var		tok(_VAR_);
<cmd>which		tok2(_WHICH_);
<cmd>who		tok2(_WHO_);
<cmd>whos		tok(_WHOS_);
<cmd>whois		tok(_WHOIS_);
<arg>{WS}+		;
<arg>{NWSNQ}{NWS}*	{ static char *buf1 = NULL;
			  char *buf = NULL;
			  if (buf1) free(buf1);
			  buf1 = NULL;
			  if(thr->qmstat != MEM_OVF &&
			     (buf1 = (char*)malloc((yyleng+1)*sizeof(char)))
			     != NULL &&
			     (buf = (char*)malloc((yyleng+1)*sizeof(char)))
			     != NULL) {
			    /* avoid globbering yytext */
			    strcpy(buf, yytext);
			    scanstr(buf1, buf);
			    free(buf);
			    yylval.sval = buf1;
			    tok2(ARG);
			  } else {
			    if (buf1) free(buf1);
			    if (buf) free(buf);
			    buf1 = NULL;
			    thr->qmstat = MEM_OVF;
			    tok2(ERRTOK);
			  }
			}
<arg>["']		{ static char *buf1 = NULL;
			  if (buf1) free(buf1);
			  string(yytext[0]);
			  if(thr->qmstat == STR_ERR) {
			    tok2(ARGERR);
			  } else if(thr->qmstat != MEM_OVF &&
			     (buf1 = (char*)malloc((strlen(buf)+1)*
						   sizeof(char)))
			     != NULL) {
			    scanstr(buf1, buf);
			    yylval.sval = buf1;
			    tok2(ARG);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok2(ERRTOK);
			  }
			}

  /* Special case: on/off arg of break must be treated as a string arg,
     everything else is parsed like an ordinary expression. */

<arg2>on|off		|
<arg2>\'on\'|\'off\'	|
<arg2>\"on\"|\"off\"	{ if (strcmp(yytext, "on") &&
			      strcmp(yytext, "\'on\'") &&
			      strcmp(yytext, "\"on\""))
			    yylval.sval = "off";
			  else
			    yylval.sval = "on";
			  tok2(ARG);
			}

  /* Special case: break/profile arg of clear must be treated as a keyword,
     everything else is parsed like an ordinary expression. */

<arg3>break		tok(_BREAK_);
<arg3>profile		tok(_PROFILE_);

as			tok(AS);
const			tok(CONST);
def			tok(DEF);
else			tok(ELSE);
extern			tok(EXTERN);
from			tok(FROM);
if			tok(IF);
import			tok(IMPORT);
include			tok(INCLUDE);
otherwise		tok(OTHERWISE);
private			tok(PRIVATE);
public			tok(PUBLIC);
special			tok(SPECIAL);
then			tok(THEN);
type			tok(TYPE);
undef			tok(UNDEF);
var			tok(VAR);
virtual			tok(VIRTUAL);
where			tok(WHERE);
"<<"[^>\n]+">>"		tok(XID);

  /* There is a potential mem leak here -- when scanning an integer or string,
     memory is allocated dynamically which might not be claimed back when the
     parser runs into a syntax error before having processed the token.
     However, with the current grammar this shouldn't happen since *any* legal
     token sequence can be followed by a primary. At least I hope so. ;-) */

0{O}+			{ if (bigint(yylval.zval)) {
			    tok(INT);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
                        }
0{O}+/\.\.		{ if (bigint(yylval.zval)) {
			    tok(INT);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
                        }
0{D}+			{ tok(ERRTOK); }
0{D}+/\.\.		{ tok(ERRTOK); }
{D}+			{ if (bigint(yylval.zval)) {
			    tok(INT);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
                        }
{D}+/\.\.		{ if (bigint(yylval.zval)) {
			    tok(INT);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
                        }
0[xX]{X}+		{ if (bigint(yylval.zval)) {
			    tok(INT);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
                        }
0[xX]{X}+/\.\.		{ if (bigint(yylval.zval)) {
			    tok(INT);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
                        }
{D}+{SF}		|
{D}+\.{D}*{SF}?		|
{D}*\.{D}+{SF}?		{ yylval.fval = my_strtod(yytext, NULL); tok(FLOAT); }
\"			{ char *buf1 = NULL;
			  string('"');
			  if(thr->qmstat == STR_ERR) {
			    tok2(STRERR);
			  } else if(thr->qmstat != MEM_OVF &&
			     (buf1 = (char*)malloc((strlen(buf)+1)
						   *sizeof(char)))
			     != NULL) {
			    if (scanstr(buf1, buf)) {
			      yylval.sval = buf1;
			      tok(STR);
			    } else {
			      free(buf1);
			      thr->qmstat = BAD_ESC;
			      tok2(STRERR);
			    }
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
			}
{WS}			|
"//".*			/* line-oriented comments allowed */;
\\\n			;
";"			{ if(_sflag)_sp=_s; BEGIN(cmd); return yytext[0]; }
{DELIM}			return(yytext[0]);
{SYM}			{ int ret, idtok = NID;
			  yyless(0);
			  ret = utf8_qualid();
			  if (ret && thr->qmstat != MEM_OVF) {
			    if (strlen(buf) >= MAXSTRLEN) {
			      thr->qmstat = BAD_SYM;
			      tok(ERRTOK);
			    }
			    if (ret&8) idtok = QID;
			    strcpy(yylastsym, buf);
			    strcpy(my_yytextbuf, buf);
			    my_yytext = my_yytextbuf;
			    tok(checkid(idtok, buf));
			  } else {
			    tok(ERRTOK);
			  }
			}
.			|
\n			tok(yytext[0]);

%%

static int op_tok[] = { OP0, OP1, OP2, OP3, OP4, OP5, OP6, OP7, OP8, OP9 };

typedef struct {
  char *name;
  int tok;
} keyword;

static keyword kwtable[] = {
  {"as",	AS},
  {"const",	CONST},
  {"def",	DEF},
  {"else",	ELSE},
  {"extern",	EXTERN},
  {"from",	FROM},
  {"if",	IF},
  {"import",	IMPORT},
  {"include",	INCLUDE},
  {"otherwise",	OTHERWISE},
  {"private",	PRIVATE},
  {"public",	PUBLIC},
  {"special",	SPECIAL},
  {"then",	THEN},
  {"type",	TYPE},
  {"undef",	UNDEF},
  {"var",	VAR},
  {"virtual",	VIRTUAL},
  {"where",	WHERE},
  {"..",	DOTDOT},
  {":",		':'},
  {"|",		'|'},
  {"=",		'='},
  {"==",	EQUIV},
  {"-",		'-'},
  {"\\",	'\\'},
  {".",		'.'},
  {"@",		'@'},
  {"~",		'~'},
};

static int compkw(const void *k1, const void *k2)
{
  keyword *kw1 = (keyword*) k1;
  keyword *kw2 = (keyword*) k2;
  return strcmp(kw1->name, kw2->name);
}

static void initkws(void)
{
  qsort(kwtable, sizeof(kwtable)/sizeof(keyword), sizeof(keyword), compkw);
}

static int kwtok(char *name)
{
  keyword key, *res;
  key.name = name;
  res = bsearch(&key, kwtable, sizeof(kwtable)/sizeof(keyword),
		sizeof(keyword), compkw);
  if (res)
    return res->tok;
  else
    return NONE;
}

static int checkid(int tok, char *name)
{
  int fno, tok1 = kwtok(name);
  if (tok1 != NONE) return tok1;
  fno = mksym(name);
  if (fno == NONE) return ERRTOK;
  yylval.ival = fno;
  if (symtb[fno].prec != NONE)
    return op_tok[symtb[fno].prec];
  else
    return tok;
}

static int checksym(const char *sym)
{
  char s[MAXSTRLEN];
  int modno = (mainno>=0)?mainno:0;
  strcpy(s, sym);
  return kwtok(s) != NONE || getsym(s, modno) != NONE;
}

static initbuf()
{
  bufp = 0;
}

static addbuf(char c)
{
  if (bufp >= abufsz)
    if (!(buf = (char*)arealloc(buf, abufsz, 100, sizeof(char)))) {
      thr->qmstat = MEM_OVF;
      return;
    } else
      abufsz += 100;
  buf[bufp++] = c;
}

static lookahead(const char *s)
{
  register long c;
  const char *t = s;
  int ret;
  while (*t && (c = input()) == *t)
    t++;
  ret = !*t;
  if (*t) {
    if (c != EOF) unput(c);
  }
  while (t > s)
    unput(*--t);
  return ret;
}

#define DELIM "\"()[]{},;_"

#ifdef HAVE_UNICODE

#define ERRC (-99)

static inline long
u8getc(void)
{
  size_t n;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  int ch;
  for (n = 0; n == 0 && (ch = input()) != EOF; ) {
    unsigned char uc = (unsigned char)ch;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return ERRC;
    }
  }
  if (n == 1)
    return c;
  else
    return EOF;
}

static inline char *
u8encode(char *t, unsigned long c)
{
  unsigned char *uc = (unsigned char*)t;
  if (c < 0x80) {
    uc[1] = 0;
    uc[0] = c;
  } else if (c < 0x800) {
    uc[2] = 0;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xc0 | c;
  } else if (c < 0x10000) {
    uc[3] = 0;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xe0 | c;
  } else {
    uc[4] = 0;
    uc[3] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xf0 | c;
  }
  return t;
}

static inline void
u8ungetc(long c)
{
  if (c && c != EOF) {
    int i;
    char s[5];
    u8encode(s, (unsigned long)c);
    for (i = strlen(s)-1; i >= 0; --i)
      unput(s[i]);
  }
}

static inline void
u8addbuf(long c)
{
  if (c != EOF) {
    char s[5], *t;
    u8encode(s, (unsigned long)c);
    for (t = s; *t && thr->qmstat != MEM_OVF; t++)
      addbuf(*t);
  }
}

static utf8_ident_or_sym()
{
  register long c;
  int save_bufp = bufp, ret = 1, k = bufp, lastk = bufp;
  while ((c = u8getc()) && c != EOF) {
    if (c == ERRC) {
      addbuf('\0');
      return 0;
    }
    if (bufp == save_bufp)
      if (u_isalpha(c) || c == '_') {
	if (u_isupper(c))
	  ret |= 2; /* indicates a capitalized identifier */
	u8addbuf(c);
      } else if (u_ispunct(c) && (c >= 128 || !strchr(DELIM, c))) {
	u8addbuf(c); addbuf('\0');
	if (checksym(buf)) k = strlen(buf);
	bufp--;
	ret |= 4; /* indicates a symbol */
      } else
	break;
    else if (ret & 4) {
      if (u_ispunct(c) && (c >= 128 || !strchr(DELIM, c))) {
	int l = bufp-1;
	u8addbuf(c); addbuf('\0');
	if (strcmp(buf+l, "::") == 0 ||
	    strcmp(buf+l, "//") == 0 || strcmp(buf+l, "/*") == 0) {
	  if (k > l) k = lastk;
	  bufp--;
	  goto out;
	}
	if (checksym(buf)) {
	  lastk = k;
	  k = strlen(buf);
	}
	bufp--;
      } else
	break;
    } else if (u_isalnum(c) || c == '_')
      u8addbuf(c);
    else
      break;
  }
  if (c && c != EOF)
    u8ungetc(c);
 out:
  if (ret & 4) {
    while (bufp > k)
      unput(buf[--bufp]);
  }
  addbuf('\0');
  if (strcmp(buf, "_") == 0) ret |= 2;
#if 0
  printf("got symbol '%s' (flags: %d)\n", buf, ret);
#endif
  return (*buf)?ret:0;
}

#else

static utf8_ident_or_sym()
{
  register long c;
  int save_bufp = bufp, ret = 1, k = bufp, lastk = bufp;
  while ((c = input()) && c != EOF) {
    if (bufp == save_bufp)
      if (isalpha(c) || c == '_') {
	if (isupper(c))
	  ret |= 2; /* indicates a capitalized identifier */
	addbuf(c);
      } else if (ispunct(c) && (c >= 128 || !strchr(DELIM, c))) {
	addbuf(c); addbuf('\0');
	if (checksym(buf)) k = strlen(buf);
	bufp--;
	ret |= 4; /* indicates a symbol */
      } else
	break;
    else if (ret & 4) {
      if (ispunct(c) && (c >= 128 || !strchr(DELIM, c))) {
	int l = bufp-1;
	addbuf(c); addbuf('\0');
	if (strcmp(buf+l, "::") == 0 ||
	    strcmp(buf+l, "//") == 0 || strcmp(buf+l, "/*") == 0) {
	  if (k > l) k = lastk;
	  bufp--;
	  goto out;
	}
	if (checksym(buf)) {
	  lastk = k;
	  k = strlen(buf);
	}
	bufp--;
      } else
	break;
    } else if (isalnum(c) || c == '_')
      addbuf(c);
    else
      break;
  }
  if (c && c != EOF)
    unput(c);
 out:
  if (ret & 4) {
    while (bufp > k)
      unput(buf[--bufp]);
  }
  addbuf('\0');
  if (strcmp(buf, "_") == 0) ret |= 2;
#if 0
  printf("got symbol '%s' (flags: %d)\n", buf, ret);
#endif
  return (*buf)?ret:0;
}

#endif

static utf8_qualid()
{
  register long c;
  int ret, ret2;
  initbuf();
  ret = utf8_ident_or_sym();
  if (ret && (ret & 4))
    return ret;
  else if (lookahead("::")) {
    bufp--;
    addbuf(input()); addbuf(input());
    ret2 = utf8_ident_or_sym();
    if (ret2)
      return ret2 | 8; /* indicates a qualified id */
    else {
      unput(':'); unput(':');
      bufp--; bufp--;
      buf[bufp] = 0;
      return ret;
    }
  } else
    return ret;
}

static string(int d)
{
  register int    c;

  initbuf();
  while ((c = input()) && c != EOF && thr->qmstat != MEM_OVF) {
    if (c == d)
      break;
    else if (c == '\\') {
      if ((c = input()) != '\n') {
	addbuf('\\');
	addbuf(c);
      }
    } else if (c == '\n') {
      break;
    } else
      addbuf(c);
  }
  addbuf('\0');
  if (c != d) {
    thr->qmstat = STR_ERR;
    skip();
  }
}

static char *
skipz(char *s)
{
  while (*s == '0') s++;
  return s;
}

static bigint(z)
     mpz_t z;
{
  int sz;
  if (strncmp(yytext, "0x", 2) == 0 ||
      strncmp(yytext, "0X", 2) == 0)
    sz = 4*strlen(skipz(yytext+2));
  else if (*yytext == '0')
    sz = 3*strlen(skipz(yytext+1));
  else
    sz = log(10)/log(2)*strlen(skipz(yytext))+1;
  sz = sz/(CHAR_BIT*sizeof(mp_limb_t)) + 2;
  mpz_init(z); 
  if (z->_mp_d && my_mpz_realloc(z, sz)) {
    int sz1;
    mpz_set_str(z, yytext, 0);
    sz1 = mpz_size(z);
    if (sz1 < sz && !my_mpz_realloc(z, sz1)) {
      thr->qmstat = MEM_OVF;
      tok(ERRTOK);
    }
    tok(INT);
  } else {
    thr->qmstat = MEM_OVF;
    tok(ERRTOK);
  }
}

#undef yywrap

yywrap()
{
  return 1;
}

peek()
{
  int c = input();
  unput(c);
  return c;
}

skip()
/* skip remainder of input line */
{
  if (_sflag)
    _s += strlen(_s);
  else {
    register int c;
    while ((c = input()) && c != EOF && c != '\n')
      if (c == '\\' && (!(c = input()) || c == EOF))
	break;
  }
}

getln(s)
     char	       *s;
/* get remainder of input line */
{
  if (_sflag) {
    strcpy(s, _s);
    _s += strlen(_s);
  } else {
    register int c;
    while ((c = input()) && c != EOF && c != '\n')
      if (c == '\\') {
	*s++ = c;
	if (!(c = input()) || c == EOF)
	  break;
	else
	  *s++ = c;
      } else
	*s++ = c;
    *s = 0;
  }
}

static struct {
  int	   pmode, lexinit, start;
  char	   _sflag;
  char    *_s, *_sp;
  FILE    *_fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  iconv_t *_ic;
#endif
  YY_BUFFER_STATE state;
  THREAD *thr;
  char *mybuf, *mybufptr;
} lexstack[MAXSTACK], *lexp = NULL;

static lexpush()
{
  if (!lexp)
    lexp = lexstack;
  else {
    /* scanner already running, push state */
    if (lexp - lexstack >= MAXSTACK)
      return 0;
    lexp->pmode = pmode;
    lexp->lexinit = lexinit;
    lexp->start = YYSTATE;
    lexp->_sflag = _sflag;
    lexp->_s = _s;
    lexp->_sp = _sp;
    lexp->_fp = _fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    lexp->_ic = _ic;
#endif
    lexp->state = YY_CURRENT_BUFFER;
    lexp->thr = thr;
    lexp->mybuf = mybuf;
    lexp->mybufptr = mybufptr;
    mybuf = mybufptr = NULL;
    yy_switch_to_buffer(yy_create_buffer(NULL, YY_BUF_SIZE));
    lexp++;
  }
  yyrestart(NULL);
  return 1;
}

static lexpop()
{
  if (lexp > lexstack) {
    --lexp;
    pmode = lexp->pmode;
    lexinit = lexp->lexinit;
    _sflag = lexp->_sflag;
    _s = lexp->_s;
    _sp = lexp->_sp;
    _fp = lexp->_fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    _ic = lexp->_ic;
#endif
    thr = lexp->thr;
    if (mybuf) free(mybuf);
    mybuf = lexp->mybuf;
    mybufptr = lexp->mybufptr;
    yy_delete_buffer(YY_CURRENT_BUFFER);
    yy_switch_to_buffer(lexp->state);
    BEGIN(lexp->start);
  } else {
    if (mybuf) free(mybuf);
    mybuf = mybufptr = NULL;
    lexp = NULL;
    yyrestart(NULL);
  }
}

int initlex(void *source, void *x, int mode)
{
  static int init0 = 0;
  if (!init0) {
    initkws();
    init0 = 1;
  }
  if (!lexpush())
    return 0;
  thr = get_thr();
  BEGIN(0);
  pmode = mode;
  lexinit = 1;
  switch (mode) {
  case STRING:
  case INTERACT:
  case SOURCE:
    _sflag = 1;
    _s = _sp = (char*) source;
    break;
  case LINE:
    _sflag = 0;
    _fp = (FILE*) source;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    _ic = (iconv_t*) x;
#endif
    break;
  }
  return 1;
}

void finilex(void)
{
  lexinit = 0;
  lexpop();
  if (abufsz > 10000) {
    free(buf); buf = NULL; abufsz = 0;
  }
}

char *actchar(void)
{
  if (_sflag)
    return _sp;
  else
    return NULL;
}

char *actbuf(void)
{
  return buf;
}
