
/* qdefs.h: header file for the Q interpreter */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qbase.h"
#include "expr.h"

extern char    *code;		/* the code file name (default: "q.out") */
extern char    *which;		/* the real source file name (full pathname) */
extern bool	eflag, qflag, iflag, hflag, Vflag, gflag, norl, gc_flag, gc_v;
				/* flags */
extern double   gc_tol;		/* gc tolerance */
extern bool	debug;		/* global debug flag */
extern int	debug_lock;	/* number of threads with debug lock */
extern int lastblksz, maxnblks;	/* memory sizes */
extern bool	init_mode;	/* set during initialization */
extern int	nbreak, nprof;	/* number of active break/profile points */

/* maximum number of nested files (source command) */
#define MAXSTACK 256

/* maximum number of POSIX threads */
#ifdef USE_THREADS
#define MAXTHREAD 1024
#else
#define MAXTHREAD 1
#endif

/* number of entries to leave in the symbol table for temporary function
   symbols (adjusted dynamically) */
#define TMPTBSZ 1024

/* space to leave in the string space to store print names of tmp function
   symbols (adjusted dynamically) */
#define TMPSPSZ 4096

extern int      tmpspsz, tmptbsz;	/* actual temp table sizes */
extern int      atmpspsz, atmptbsz;	/* allocated temp table sizes */


/* basic definitions of the Q machine */

#define MAXARGS 3		/* max arg count for built-ins */
#define XBLKSZ 4096		/* blocksize for EXPR heap management */

#define NSIGNALS 64		/* signal numbers go from 1 to NSIGNALS-1 */

/* the following table sizes are adjusted dynamically: */

#define XSTSZ 16384		/* expression stack */
#define ASTSZ 8192		/* activation stack */

/* error codes (don't change the order of the BREAK..XCEPT errors, the
   catch function depends on it): */

enum qmstats {

	OK,			/* no error */

	/* Q machine errors (these can be handled): */

	BREAK,			/* break called */
	HALT,			/* halt called */
	QUIT,			/* quit called */
	MEM_OVF,		/* heap overflow */
	XST_OVF,		/* expression stack overflow */
	AST_OVF,		/* activation stack overflow */
	SYMTB_OVF,		/* symbol table overflow */
	COND_ERR,		/* error in conditional */
	EXT_ERR,		/* error in external function */
	MATCH_ERR,		/* value mismatch in definition */

	/* exception handling: */

	XCEPT,			/* unhandled exception */
	XCEPT_FAIL,		/* fail called */
	XCEPT_FAIL2,		/* _FAIL_ called */
	XCEPT_CATCH,		/* catch caught an exception */

	/* code file errors: */

	FILE_NOT_FOUND,
	CODE_FILE_ERR,
	FILE_FORMAT_ERR,

	/* interpreter error codes: */
	SYNTAX_ERR,		/* syntax error (parser) */
	STR_ERR,		/* unterminated string constant (parser) */
	BAD_ESC,		/* invalid character escape (parser) */
	BAD_OBJ,		/* invalid <<...>> object (parser) */
	BAD_SYM,		/* unknown symbol */
	BAD_REF,		/* ambiguous symbol reference */
	BAD_DEF,		/* attempt to redefine function symbol */
	BAD_REDEF,		/* attempt to redefine built-in variable */
	BAD_UNDEF,		/* attempt to undefine built-in variable */
	BAD_FORMAT,		/* bad format string */
	BAD_DIR,		/* bad directory name */
	STACK_OVF,		/* source stack overflow */
	FILE_ERR,		/* error writing file */
	COMPILE_ERR,		/* error compiling script */
	ARGS_ERR,		/* too many args (run command) */

	/* debugger error codes: */

	BAD_VAR,		/* unknown local variable */
	INVALID_COMMAND,	/* just as it looks like */

	/* others */

	THIS_CANT_HAPPEN	/* CAN happen, but shouldn't */
};

extern char    *qmmsg[];	/* error messages */

/* dll interface */

typedef struct {
  void (*prepare)(void);
  void (*parent)(void);
  void (*child)(void);
} thread_atfork_t;

extern char **dll_name;
extern lt_dlhandle *dll_handle;
extern void (**dll_init)(), (**dll_fini)();
extern thread_atfork_t *dll_atfork;

/* The activation stack records state information of invoked rules (the base
   pointer must be recorded as an index relative to the beginning of the stack
   since the expression stack may be reallocated during evaluation). */

typedef struct arec {
  short         fno;
  EXPR         *lvals[2];
  int		modno, lineno, info_addr;
  byte		info_offs;
  int	       *rp;
  int           rc;
  OPREC	       *ip;
  long		xbp;
}               AREC;

/* The mark stack stores the necessary data for exception handling. */

typedef struct mark {
  long xp; /* stack index */
  EXPR *h; /* exception handler */
} Mark;

/* The temporary variable table is used by the debugger to unparse variables
   bound in where clauses. */

typedef struct {
  char         *pname;	/* print name */
  byte          offs;
  byte          plen;
  PATH          p;
} VARREC;

/* Expression list data structure. */

typedef struct exprl {
  EXPR *x;
  struct exprl *next;
} EXPRL;

/* status information: */

#ifdef USE_THREADS
/* internal mutex locks */
extern pthread_mutex_t global_mutex, tty_mutex, parse_mutex, reads_mutex;
#endif

/* Thread-local stuff: */

typedef struct thread_struct {
#ifdef USE_THREADS
  pthread_t id;		/* OS id of the executing thread */
  /* exit mutex and condition */
  pthread_mutex_t exit_mutex;
  pthread_cond_t exit_cond;
#endif

  struct thread_struct
  *next;		/* next pointer in free list */
  bool used:1;		/* flag indicating used entry */
  bool active:1;	/* flag indicating whether the thread is active */
  bool released:1;	/* flag indicating temporary release of the mutex */
  bool sticky:1;	/* flag indicating "sticky" deallocation mode */

  int qmstat;		/* status of Q machine */
  int qmstat_save;	/* saved qmstat (qmcatch) */
  bool debug;		/* set for debugging mode */
  bool debug_lock;	/* set when thread has acquired terminal for debug */
  bool tty_lock;	/* set when thread has acquired terminal input */
  bool mode;		/* quote mode flag of Q machine */
  bool brkflag;		/* local brkflag (qmbreak) */
  bool brkdbg;		/* local break flag */
  int nsig;		/* number of pending signals */
  byte sig[NSIGNALS];	/* signal queue */
  bool sigpend:1;	/* set if signal is pending */
  bool sigblk:1;	/* set to block further signals */
  long level;		/* eval recursion level */
  long stoplevel;	/* debugger stoplevel */

  int maxxstsz;		/* max stack sizes (stackmax) */
  int maxastsz;

  int xstsz;		/* size of expression stack */
  EXPR **xst;		/* expression stack */
  EXPR **xsp;		/* stack pointer */

  int astsz;		/* activation stack size */
  AREC **ast;		/* activation stack */
  AREC **asp;		/* activation stack pointer */

  Mark *mark, *markp;	/* the mark stack and the pointer */
  int marksz;		/* allocated size */

  EXPRL *sentinels;	/* sentinel queue */

  VARREC *vartb;	/* the variable table */
  int nvarsyms, avarsyms;
  int lastaddr;		/* last code address used for var table */

  EXPR   *self, **args;	/* pass arguments to builtins and externals */

  /* stats data */

  bool stats_init, stats_fini;
  clock_t starttime, endtime;
  unsigned long maxexprs, nexprs, nredns;

  /* C stack base pointer, used to prevent C stack overflows */

  char *baseptr;
} THREAD;

extern THREAD threads[], *thr0;
extern short nthreads, nused;

/* Globals: */

extern volatile bool brkflag;	/* SIGINT */
extern volatile bool quitflag;	/* SIGTERM */
extern bool	brkdbg;		/* set for debug on break */
extern bool	debug_long;	/* set for debug_long mode */
extern int      maxargs;	/* max argcount for builtins and externals */

/* Expression heap: */

typedef struct xblk {
	struct xblk    *next;	/* pointer to next block */
	EXPR            x[XBLKSZ];
				/* data */
}		XBLK;

extern int	xnblks;		/* current number of allocated blocks */
extern XBLK    *xblk;		/* pointer to current block on the heap */
extern EXPR    *xheap;		/* pointer to next free EXPR in xblk */
extern EXPR    *xfreep;		/* head of free list */

/* Function prototypes. */

/* q.c ***********************************************************************/

extern void fatal(char *s);
extern void echo(char *s);
extern void flush_shift(void);

extern void *my_mpz_realloc(mpz_ptr m, mp_size_t new_size);
extern char **sym_completion (char *text, int start, int end);
extern void new_xprompt(void);
extern char *mygetline(FILE *fp, char *prompt, int expand);
extern void switch_history(void);
extern void list_completions(char *s);

extern RETSIGTYPE break_handler(int), quit_handler(int), sig_handler(int);
extern void push_sigint(RETSIGTYPE (*new_handler)());
extern void pop_sigint(void);

extern int rerun(const char *path, int argc, char *const *argv);
extern void gcmd(char *name);
extern void gcmd_i(char *name, int i);
extern void gcmd_b(char *name, int b);
extern void gcmd_s(char *name, char *s);

/* qmcode.c ******************************************************************/

extern int iscode(FILE *fp);
extern void readtables(void);

/* qm.c **********************************************************************/

extern void error(char *s);

extern int push_mark(THREAD *thr, EXPR *h);
extern void pop_mark(THREAD *thr);
extern int get_mark(THREAD *thr, long *xp, EXPR **h);
extern int have_mark(THREAD *thr);

extern void process_sentinels(THREAD *thr);

extern int add_signal(THREAD *thr, int sig, EXPR *x);
extern void raise_signal(THREAD *thr, int sig);

extern int active_threads(void);
extern int init_thread(void);
extern void exit_thread(int id);
extern void fini_thread(int id);
extern THREAD *get_thr(void);
extern void kill_threads(void);
extern void wait_threads(void);
extern int this_thread(void);
extern int have_lock(void);
extern void release_lock(void);
extern void acquire_lock(void);
extern void acquire_tty(void);
extern void release_tty(void);
extern void start_init(void);
extern void end_init(void);

extern void init(void);
extern void reinit(void);

extern void qmfree(THREAD *thr, EXPR *x);
extern EXPR *qmnew(EXPR *x);
extern void clear(int force_gc);

extern EXPR *intexpr(THREAD *thr, long i);
extern EXPR *uintexpr(THREAD *thr, unsigned long i);
extern EXPR *mpzexpr(THREAD *thr, mpz_t z);
extern EXPR *floatexpr(THREAD *thr, double f);
extern EXPR *strexpr(THREAD *thr, char *s);
extern EXPR *fileexpr(THREAD *thr, FILE *fp);
extern EXPR *pipeexpr(THREAD *thr, FILE *fp);
extern EXPR *vectexpr(THREAD *thr, int n, EXPR **xv);
extern EXPR *usrexpr(THREAD *thr, int type, void *vp);
extern EXPR *funexpr(THREAD *thr, int fno);
extern EXPR *consexpr(THREAD *thr, int fno, EXPR *x1, EXPR *x2);

extern int push(THREAD *thr, EXPR *x);
extern int pushint(THREAD *thr, long i);
extern int pushuint(THREAD *thr, unsigned long i);
extern int pushmpz(THREAD *thr, mpz_t z);
extern int pushfloat(THREAD *thr, double f);
extern int pushstr(THREAD *thr, char *s);
extern int pushfile(THREAD *thr, FILE *fp);
extern int pushpipe(THREAD *thr, FILE *fp);
extern int pushvect(THREAD *thr, int n, EXPR **xv);
extern int pushfun(THREAD *thr, int fno);

extern int setvar(int vno, EXPR *x);
extern int evaldef(int offs, int *modno, int *lineno);
extern int eval(THREAD *thr, EXPR *x);

extern int mksym(char *s);
extern int newsym(char *s);
extern int getsym(char *s, int modno);
extern int gettype(char *s, int modno);
extern int getmodno(char *name);

/* qmfuns.c ******************************************************************/

/* built-in function table */

extern int      (*funtb[BUILTIN]) ();
extern int      nargs[BUILTIN];

/* builtin lambda support */

extern void qmlambda_dtor(void *vp);

/* qmmatch.c *****************************************************************/

extern int matchp(THREAD *thr, int s, EXPR **x);
extern int matchtype(THREAD *thr, int fno, int type);
extern int match(THREAD *thr, int fno, EXPR **x, int **rp, int *rc);

/* qmprint.c *****************************************************************/

extern bool unparseflag;
extern int maxlevel, maxlist, maxchars;

extern char *sprintx(EXPR *x);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
extern int fprintx(FILE *fp, iconv_t *ic, EXPR *x);
#else
extern int fprintx(FILE *fp, EXPR *x);
#endif
extern int printx(EXPR *x);

/* qmlex.l *******************************************************************/

extern int initlex(void *source, void *arg, int mode);
extern void finilex(void);
extern char *actchar(void), *actbuf(void);

/* qmparse.y *****************************************************************/

extern int sparsex(char *s);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
extern int fparsex(FILE *fp, iconv_t *ic);
#else
extern int fparsex(FILE *fp);
#endif
extern int parsesrc(char *fname, int chk);
extern int parsex(char *s);
