(** Overloading of the standard OCaml numeric types. *)

open Camlp4.PreCast
module FV = Camlp4.Struct.FreeVars.Make(Ast)
open Printf
open Delimited_overloading

let not_interactive = not(!Sys.interactive)

(* FIXME: do we add *_set/_*get rules so that we do not recurse in
   accessors indexes, so [Float.(a.(1 + i) <- x + 3)] has the
   "expected" effect.  Or, better, we can "call" Int.() for these so
   that the +,... have their usual meaning even if redefined. *)

(***********************************************************************
 *                      Specialized operations
 ***********************************************************************)

(* Several specialized operations require caching functions defined by
   [external].  We define here the functions to perform this
   definition on demand. *)

(** [expr_for decl] returns a lazy lid [i] that was used to create the
    declaration [decl i] that is cached at the beginning of the
    file. *)
let expr_for op decl : Loc.t -> Ast.expr =
  if not_interactive then
    let lid = new_lid() in
    let first_usage = ref true in
    fun _loc ->
      if !first_usage then (
        add_to_beginning_of_file (decl lid);
        first_usage := false;
      );
      <:expr< $lid:lid$ >>
  else
    (* FIXME: either provide an alternative expression or insert the
       declaration in the toplevel at the point of need. *)
    (fun _loc -> <:expr< $lid:op$ >>)

(** [lid_in_expr op expr] check whether the lid [op] is the same as
    the one contained in the expression [expr] -- e.g. generated by
    [expr_for]. *)
let lid_in_expr op expr = match expr with
  | <:expr< $lid:op1$ >> when op = op1 -> true
  | _ -> false


(***********************************************************************
 *                        Generic operations
 ***********************************************************************)

(* Introduce a variable name for [e] if needed. [f v] is the
   expression using the variable [v] (so [$v$] can be used several
   times in the output expression with no harm).  FIXME: something
   like this should be available in the standard overloading
   module.  *)
let bind _loc e f =
  if is_immediate e then f e
  else
    let v = new_lid() in
    let v_expr = <:expr< $lid:v$ >> in
    <:expr< let $lid:v$ = $e$ in $f v_expr$ >>  ;;

(** Return an expression computing the maximum of the list of
    expressions [el] for the [>] given by [gt loc].  *)
let max_list _loc gt el =
  let m = new_lid() in
  let set_max ex e =
    let do_e = bind _loc e
      (fun e -> <:expr< (if $gt _loc$ $e$ (Pervasives.( ! ) $lid:m$) then
                          Pervasives.( := ) $lid:m$ $e$) >> ) in
    <:expr< $ex$; $do_e$ >> in
  match el with
  | [] -> Loc.raise _loc (Stream.Error "Max/min of an empty list not defined")
  | [e] -> e
  | e :: tl ->
      <:expr< (let $lid:m$ = ref $e$ in
               $List.fold_left set_max <:expr< >> tl$;
               Pervasives.( ! ) $lid:m$) >>

(***********************************************************************
 *                 Overloadings for the module Int
 ***********************************************************************)

(* This seem to be void but is important to be able to recover the
   usual meaning of symbols inside an expression
   (e.g. [Float.(... Int.(...) ...)].  It also specialises comparison
   operators so they are efficiently compiled *)

module Int =
struct
  let eq = expr_for "=" (fun i -> <:str_item@here<
                         external $i$ : int -> int -> bool = "%equal" >>)
  let ne = expr_for "<>" (fun i -> <:str_item@here<
                         external $i$ : int -> int -> bool = "%notequal">>)
  let lt = expr_for "<" (fun i -> <:str_item@here<
                         external $i$ : int -> int -> bool = "%lessthan">>)
  let le = expr_for "<=" (fun i -> <:str_item@here<
                         external $i$ : int -> int -> bool = "%lessequal">>)
  let gt = expr_for ">" (fun i -> <:str_item@here<
                         external $i$ : int -> int -> bool = "%greaterthan" >>)
  let ge = expr_for ">=" (fun i -> <:str_item@here<
                         external $i$ : int -> int -> bool = "%greaterequal" >>)
  let compare = expr_for "compare" (fun i -> <:str_item@here<
                         external $i$ : int -> int -> int = "%compare" >>)

  let is_zero = function
    | <:expr< $int:x$ >> when int_of_string x = 0 -> true
    | _ -> false

  let is_one = function
    | <:expr< $int:x$ >> when int_of_string x = 1 -> true
    | _ -> false

  let is_literal = function <:expr< $int:_$ >> -> true | _ -> false

  let int_of_lit = function
    | <:expr< $int:x$ >> -> int_of_string x
    | _ -> assert false

  let overloading =
    let ops = ["+"; "-"; "*"; "/" ; "~-"; "abs"; "asr";
               "float"; "float_of_int"; "land"; "lnot"; "lor";
               "lsl"; "lsr"; "lxor"; "max_int"; "min_int"; "mod"; "pred";
               "string_of_int"; "succ"; "truncate"; "incr"; "decr" ] in
    let t = openin empty ~remove:true in (* Int not real module *)
    let t =
      let reloc id = fun _ _loc -> <:expr< Pervasives. $lid:id$ >> in
      let ov t id = lid t id (reloc id) ~cache:false in
      List.fold_left ov t ops in
    (* In case [of_int] and [to_int] are used as functions. *)
    let t = lid t "of_int" ~cache:false
      (fun _ _loc -> <:expr< fun (x:int) -> x >>) in
    let t = lid t "to_int" ~cache:false
      (fun _ _loc -> <:expr< fun (x:int) -> x >>) in
    let t = lid t "of_int32" ~cache:false
      (fun _ _loc -> <:expr< Int32.to_int >>) in
    let t = lid t "to_int32" ~cache:false
      (fun _ _loc -> <:expr< Int32.of_int >>) in
    let t = lid t "of_int64" ~cache:false
      (fun _ _loc -> <:expr< Int64.to_int >>) in
    let t = lid t "to_int64" ~cache:false
      (fun _ _loc -> <:expr< Int64.of_int >>) in
    let t = lid t "of_nativeint" ~cache:false
      (fun _ _loc -> <:expr< Nativeint.to_int >>) in
    let t = lid t "to_nativeint" ~cache:false
      (fun _ _loc -> <:expr< Nativeint.of_int >>) in
    let t = lid t "to_string" ~cache:false
      (fun _ _loc -> <:expr< Pervasives.string_of_int >>) in
    let t = lid t "of_string" ~cache:false
      (fun _ _loc -> <:expr< Pervasives.int_of_string >>) in
    let t = lid t "="  (fun _ loc -> eq loc) ~cache:false in
    let t = lid t "<>" (fun _ loc -> ne loc) ~cache:false in
    let t = lid t "<"  (fun _ loc -> lt loc) ~cache:false in
    let t = lid t "<=" (fun _ loc -> le loc) ~cache:false in
    let t = lid t ">"  (fun _ loc -> gt loc) ~cache:false in
    let t = lid t ">=" (fun _ loc -> ge loc) ~cache:false in
    let t = lid t "compare" (fun _ loc -> compare loc) ~cache:false in
    (* Optimize when one knows the argument. *)
    let optimize tr expr = match expr with
      | <:expr< of_int $e$ >> | <:expr< to_int $e$ >> -> (self tr)#expr e
      | <:expr@_loc< $e1$ + $e2$ >>
      | <:expr@_loc< Pervasives.( + ) $e1$ $e2$ >> ->
        let e1 = (self tr)#expr e1 and e2 = (self tr)#expr e2 in
        if is_zero e1 then e2
        else if is_zero e2 then e1
        else if is_literal e1 && is_literal e2 then
          <:expr< $`int:(int_of_lit e1 + int_of_lit e2)$ >>
        else <:expr< Pervasives.( + ) $e1$ $e2$ >>
      | <:expr@_loc< $e1$ - $e2$ >>
      | <:expr@_loc< Pervasives.( - ) $e1$ $e2$ >> ->
        let e1 = (self tr)#expr e1 and e2 = (self tr)#expr e2 in
        if is_zero e1 then <:expr< ~- e2 >> (* could try simpl. more *)
        else if is_zero e2 then e1
        else if is_literal e1 && is_literal e2 then
          <:expr< $`int:(int_of_lit e1 - int_of_lit e2)$ >>
        else <:expr< Pervasives.( - ) $e1$ $e2$ >>
      | <:expr@_loc< $e1$ * $e2$ >>
      | <:expr@_loc< Pervasives.( * ) $e1$ $e2$ >> ->
        let e1 = ((self tr)#expr e1) and e2 = ((self tr)#expr e2) in
        if is_zero e1 || is_zero e2 then <:expr< 0 >>
        else if is_one e1 then e2
        else if is_one e2 then e1
        else if is_literal e1 && is_literal e2 then
          <:expr< $`int:(int_of_lit e1 * int_of_lit e2)$ >>
        else <:expr< Pervasives.( * ) $e1$ $e2$ >>
      | <:expr@_loc< $e1$ / $e2$ >>
      | <:expr@_loc< Pervasives.( / ) $e1$ $e2$ >> ->
        let e1 = ((self tr)#expr e1) and e2 = ((self tr)#expr e2) in
        if is_zero e1 then <:expr< 0 >>
        else if is_zero e2 then
          Loc.raise _loc (Stream.Error("Division by literal 0."))
        else if is_one e2 then e1
        else if is_literal e1 && is_literal e2 then
          <:expr< $`int:(int_of_lit e1 / int_of_lit e2)$ >>
        else <:expr< Pervasives.( / ) $e1$ $e2$ >>
      | <:expr@_loc< max ($e1$, $e2$) >> ->
        (* max over an explicit tuple of arguments *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc gt e
      | <:expr@_loc< min ($e1$, $e2$) >> ->
        (* min over an explicit tuple of arguments *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc lt e
      | _ -> super tr expr in
    let t = expr t optimize in
    (* [max] and [min] are bound to optimized implementations.  With
       caching, the functions will be defined once only. *)
    let t = lid t "max" begin fun _ _loc ->
      <:expr< fun x y -> if (x:int) >= y then x else y >>
    end in
    lid t "min" begin fun _ _loc ->
      <:expr< fun x y -> if (x:int) <= y then x else y >>
    end


  let () =
    associate overloading "Int"
end

(***********************************************************************
 *                Overloadings for the integer modules
 ***********************************************************************)

let std_funs =
  ["mod", "rem";  "succ", "succ"; "pred", "pred"; "abs", "abs";
   "land", "logand"; "lor", "logor"; "lxor", "logxor";
   "lnot", "lognot"; "lsl", "shift_left"; "asr", "shift_right";
   "lsr", "shift_right_logical";
   "truncate", "of_float";  "float", "to_float"; "compare", "compare";
   "of_int", "of_int";  "to_int", "to_int";
   "of_string", "of_string";  "to_string", "to_string";
   "max_int", "max_int";  "min_int", "min_int";
   (* We want the arithmetic operators to be qualified by the proper
      module name and that it DOES NOT CHANGE when the overloadings
      are "inherited". *)
   ("+","add"); ("-","sub"); ("~-", "neg"); ("*","mul"); ("/","div");
  ]

(* Make sure the functions are qualified by the module.  Other
   operations specifically depend on the module, so do not allow to
   change the qualification for these either. *)
let std_funs_for m =
  let ov t (i0,i1) = lid t i0 (fun _ l -> qualify_lid i1 m l) ~cache:false in
  List.fold_left ov empty std_funs

module Int32 =
struct
  let m = Macro.Module_longident.of_string "Int32"

  let eq = expr_for "=" (fun i -> <:str_item@here<
                     external $i$ : int32 -> int32 -> bool = "%equal" >>)
  let ne = expr_for "<>" (fun i -> <:str_item@here<
                     external $i$ : int32 -> int32 -> bool = "%notequal">>)
  let lt = expr_for "<" (fun i -> <:str_item@here<
                     external $i$ : int32 -> int32 -> bool = "%lessthan">>)
  let le = expr_for "<=" (fun i -> <:str_item@here<
                     external $i$ : int32 -> int32 -> bool = "%lessequal">>)
  let gt = expr_for ">" (fun i -> <:str_item@here<
                     external $i$ : int32 -> int32 -> bool = "%greaterthan" >>)
  let ge = expr_for ">=" (fun i -> <:str_item@here<
                     external $i$ : int32 -> int32 -> bool = "%greaterequal" >>)

  let overloading =
    let t = std_funs_for m in
    let t = (int t ~cache:false
               (fun i _ _loc -> <:expr< $`int32:Int32.of_int(i)$ >>)) in
    let t = lid t "="  (fun _ loc -> eq loc) ~cache:false in
    let t = lid t "<>" (fun _ loc -> ne loc) ~cache:false in
    let t = lid t "<"  (fun _ loc -> lt loc) ~cache:false in
    let t = lid t "<=" (fun _ loc -> le loc) ~cache:false in
    let t = lid t ">"  (fun _ loc -> gt loc) ~cache:false in
    let t = lid t ">=" (fun _ loc -> ge loc) ~cache:false in
    let t = lid t "incr" (fun _ _loc ->
                            <:expr< fun x -> x := Int32.add x 1l >>) in
    let t = lid t "decr" (fun _ _loc ->
                            <:expr< fun x -> x := Int32.sub x 1l >>) in
    let t = lid t "of_int32" ~cache:false
      (fun _ _loc -> <:expr< fun (x:int32) -> x >>) in
    let t = lid t "to_int32" ~cache:false
      (fun _ _loc -> <:expr< fun (x:int32) -> x >>) in
    let t = lid t "of_int64" ~cache:false
      (fun _ _loc -> <:expr< Int64.to_int32 >>) in
    let t = lid t "to_int64" ~cache:false
      (fun _ _loc -> <:expr< Int64.of_int32 >>) in
    let t = lid t "of_nativeint" ~cache:false
      (fun _ _loc -> <:expr< Nativeint.to_int32 >>) in
    let t = lid t "to_nativeint" ~cache:false
      (fun _ _loc -> <:expr< Nativeint.of_int32 >>) in
    let optimize tr expr = match expr with
        (* Remove the convertion of/to int32 if one knows the argument: *)
      | <:expr< of_int32 $e$ >> | <:expr< to_int32 $e$ >> -> (self tr)#expr e
        (* No automatic conversion of int literals for shifts: *)
      | <:expr@_loc< $e$ lsl $int:i$ >> ->
        <:expr< Int32.shift_left $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< $e$ lsr $int:i$ >> ->
        <:expr< Int32.shift_right_logical $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< $e$ asr $int:i$ >> ->
        <:expr< Int32.shift_right $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< max ($e1$, $e2$) >> ->
        (* max over an explicit tuple of arguments *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc gt e
      | <:expr@_loc< min ($e1$, $e2$) >> ->
        (* min over an explicit tuple of arguments *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc lt e
      | _ -> super tr expr in
    expr t optimize
end

module Int64 =
struct
  let m = Macro.Module_longident.of_string "Int64"

  let eq = expr_for "=" (fun i -> <:str_item@here<
                         external $i$ : int64 -> int64 -> bool = "%equal" >>)
  let ne = expr_for "<>" (fun i -> <:str_item@here<
                         external $i$ : int64 -> int64 -> bool = "%notequal">>)
  let lt = expr_for "<" (fun i -> <:str_item@here<
                     external $i$ : int64 -> int64 -> bool = "%lessthan">>)
  let le = expr_for "<=" (fun i -> <:str_item@here<
                     external $i$ : int64 -> int64 -> bool = "%lessequal">>)
  let gt = expr_for ">" (fun i -> <:str_item@here<
                     external $i$ : int64 -> int64 -> bool = "%greaterthan" >>)
  let ge = expr_for ">=" (fun i -> <:str_item@here<
                     external $i$ : int64 -> int64 -> bool = "%greaterequal" >>)

  let overloading =
    let t = std_funs_for m in
    let t = (int t ~cache:false
               (fun i _ _loc -> <:expr<$`int64:Int64.of_int(i)$>>)) in
    let t = lid t "="  (fun _ loc -> eq loc) ~cache:false in
    let t = lid t "<>" (fun _ loc -> ne loc) ~cache:false in
    let t = lid t "<"  (fun _ loc -> lt loc) ~cache:false in
    let t = lid t "<=" (fun _ loc -> le loc) ~cache:false in
    let t = lid t ">"  (fun _ loc -> gt loc) ~cache:false in
    let t = lid t ">=" (fun _ loc -> ge loc) ~cache:false in
    let t = lid t "incr" (fun _ _loc ->
                            <:expr< fun x -> x := Int64.add x 1L >>) in
    let t = lid t "decr" (fun _ _loc ->
                            <:expr< fun x -> x := Int64.sub x 1L >>) in
    let t = lid t "of_int32" ~cache:false
      (fun _ _loc -> <:expr< Int64.of_int32 >>) in
    let t = lid t "to_int32" ~cache:false
      (fun _ _loc -> <:expr< Int64.to_int32 >>) in
    let t = lid t "of_int64" ~cache:false
      (fun _ _loc -> <:expr< fun (x:int64) -> x >>) in
    let t = lid t "to_int64" ~cache:false
      (fun _ _loc -> <:expr< fun (x:int64) -> x >>) in
    let t = lid t "of_nativeint" ~cache:false
      (fun _ _loc -> <:expr< Int64.of_nativeint >>) in
    let t = lid t "to_nativeint" ~cache:false
      (fun _ _loc -> <:expr< Int64.to_nativeint >>) in
    (* Remove the convertion of/to int64 if one knows the argument: *)
    let optimize tr expr = match expr with
      | <:expr< of_int64 $e$ >> | <:expr< to_int64 $e$ >> -> (self tr)#expr e
        (* No automatic conversion of int literals for shifts: *)
      | <:expr@_loc< $e$ lsl $int:i$ >> ->
        <:expr< Int64.shift_left $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< $e$ lsr $int:i$ >> ->
        <:expr< Int64.shift_right_logical $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< $e$ asr $int:i$ >> ->
        <:expr< Int64.shift_right $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< max ($e1$, $e2$) >> ->
        (* max over an explicit tuple of arguments *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc gt e
      | <:expr@_loc< min ($e1$, $e2$) >> ->
        (* min over an explicit tuple of arguments *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc lt e
      | _ -> super tr expr in
    expr t optimize
end

module Nativeint =
struct
  let m = Macro.Module_longident.of_string "Nativeint"

  let eq = expr_for "=" (fun i -> <:str_item@here<
             external $i$ : nativeint -> nativeint -> bool = "%equal" >>)
  let ne = expr_for "<>" (fun i -> <:str_item@here<
             external $i$ : nativeint -> nativeint -> bool = "%notequal">>)
  let lt = expr_for "<" (fun i -> <:str_item@here<
             external $i$ : nativeint -> nativeint -> bool = "%lessthan">>)
  let le = expr_for "<=" (fun i -> <:str_item@here<
             external $i$ : nativeint -> nativeint -> bool = "%lessequal">>)
  let gt = expr_for ">" (fun i -> <:str_item@here<
             external $i$ : nativeint -> nativeint -> bool = "%greaterthan" >>)
  let ge = expr_for ">=" (fun i -> <:str_item@here<
             external $i$ : nativeint -> nativeint -> bool = "%greaterequal" >>)

  let overloading =
    let t = std_funs_for m in
    let t = (int t ~cache:false
               (fun i _ _loc -> <:expr<$`nativeint:Nativeint.of_int(i)$>>)) in
    let t = lid t "="  (fun _ loc -> eq loc) ~cache:false in
    let t = lid t "<>" (fun _ loc -> ne loc) ~cache:false in
    let t = lid t "<"  (fun _ loc -> lt loc) ~cache:false in
    let t = lid t "<=" (fun _ loc -> le loc) ~cache:false in
    let t = lid t ">"  (fun _ loc -> gt loc) ~cache:false in
    let t = lid t ">=" (fun _ loc -> ge loc) ~cache:false in
    let t = lid t "incr" (fun _ _loc ->
                            <:expr< fun x -> x := Nativeint.add x 1n >>) in
    let t = lid t "decr" (fun _ _loc ->
                            <:expr< fun x -> x := Nativeint.sub x 1n >>) in
    let t = lid t "of_int32" ~cache:false
      (fun _ _loc -> <:expr< Nativeint.of_int32 >>) in
    let t = lid t "to_int32" ~cache:false
      (fun _ _loc -> <:expr< Nativeint.to_int32 >>) in
    let t = lid t "of_int64" ~cache:false
      (fun _ _loc -> <:expr< Int64.to_nativeint >>) in
    let t = lid t "to_int64" ~cache:false
      (fun _ _loc -> <:expr< Int64.of_nativeint >>) in
    let t = lid t "of_nativeint" ~cache:false
      (fun _ _loc -> <:expr< fun (x:nativeint) -> x >>) in
    let t = lid t "to_nativeint" ~cache:false
      (fun _ _loc -> <:expr< fun (x:nativeint) -> x >>) in
    (* Remove the convertion of/to nativeint if one knows the argument: *)
    let optimize tr expr = match expr with
      | <:expr< of_nativeint $e$ >> | <:expr< to_nativeint $e$ >> ->
        (self tr)#expr e
          (* No automatic conversion of int literals for shifts: *)
      | <:expr@_loc< $e$ lsl $int:i$ >> ->
        <:expr< Nativeint.shift_left $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< $e$ lsr $int:i$ >> ->
        <:expr< Nativeint.shift_right_logical $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< $e$ asr $int:i$ >> ->
        <:expr< Nativeint.shift_right $(self tr)#expr e$ $int:i$ >>
      | <:expr@_loc< max ($e1$, $e2$) >> ->
        (* max over an explicit tuple of arguments *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc gt e
      | <:expr@_loc< min ($e1$, $e2$) >> ->
        (* min over an explicit tuple of arguments *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc lt e
      | _ -> super tr expr in
    expr t optimize
end

let () =
  associate Int32.overloading "Int32";
  associate Int64.overloading "Int64";
  associate Nativeint.overloading "Nativeint"

(***********************************************************************
 *                Overloadings for the module Float
 ***********************************************************************)

(* [normalize_float x] returns a string representing the same float
   but without trailing zeros.  All the constants of this file should
   respect the same format. *)
let normalize_float x = string_of_float(float_of_string x)

module Float =
struct
  (* $`flo:x$ is to be avoided because it may generate an important
     loss of precision due to an insufficient number of digits: try
     e.g. [<:expr< $`flo: 1. /. 3.$ >>].  See OCaml bug #0004345. *)
  let to_expr x _loc =
    let s = sprintf "%.20g" x in
    (* [sprintf] does not put a dot if it is an integer.  Infinities
       and nan are printed fine with [sprintf].  %F cannot be used, it
       does not respect the modifier ".20" (see OCaml bug
       #0004768). (FIXME: nan loose their sign) *)
    let s =
      if s = "inf" || s = "-inf" || s = "nan" || String.contains s '.' then s
      else s ^ "." in
    <:expr< $flo:s$ >>
  ;;

  (* Float optimized operations, even if the standard identifiers have
     been redefined. *)
  let eq = expr_for "=" (fun i -> <:str_item@here<
                         external $i$ : float -> float -> bool = "%equal" >>)
  let ne = expr_for "<>" (fun i -> <:str_item@here<
                         external $i$ : float -> float -> bool = "%notequal" >>)
  let lt = expr_for "<" (fun i -> <:str_item@here<
                     external $i$ : float -> float -> bool = "%lessthan" >>)
  let le = expr_for "<=" (fun i -> <:str_item@here<
                     external $i$ : float -> float -> bool = "%lessequal" >>)
  let gt = expr_for ">" (fun i -> <:str_item@here<
                     external $i$ : float -> float -> bool = "%greaterthan" >>)
  let ge = expr_for ">=" (fun i -> <:str_item@here<
                     external $i$ : float -> float -> bool = "%greaterequal" >>)
  let compare = expr_for "compare" (fun i -> <:str_item@here<
                     external $i$ : float -> float -> int = "%compare" >>)
  let neg = expr_for "~-." (fun i -> <:str_item@here<
                     external $i$ : float -> float = "%negfloat" >>)
  let add = expr_for "+." (fun i -> <:str_item@here<
                     external $i$ : float -> float -> float = "%addfloat" >>)
  let sub = expr_for "-." (fun i -> <:str_item@here<
                     external $i$ : float -> float -> float = "%subfloat" >>)
  let mul = expr_for "*." (fun i -> <:str_item@here<
                     external $i$ : float -> float -> float = "%mulfloat" >>)
  let div = expr_for "/." (fun i -> <:str_item@here<
                     external $i$ : float -> float -> float = "%divfloat" >>)
  let pow = expr_for "**" (fun i -> <:str_item@here<
                     external $i$ : float -> float -> float
                       = "caml_power_float" "pow" "float" >>)
  let abs = expr_for "abs_float" (fun i -> <:str_item@here<
                     external $i$ : float -> float = "%absfloat" >>)

  let max = expr_for "max" (fun i -> <:str_item@here<
    let $lid:i$ x y = if (x: float) >= y then x
                  else if (y:float) <> y then x else y >>)

  let min = expr_for "min" (fun i -> <:str_item@here<
    let $lid:i$ x y = if (x: float) <= y then x
                  else if (y:float) <> y then x else y >>)

  let pi _loc = (<:expr< 3.141592653589793238462643383279502884197 >>)

  let hypot = expr_for "hypot" (fun i -> <:str_item@here<
    let $lid:i$ (x: float) (y: float) =
      let x = Pervasives.abs_float x
      and y = Pervasives.abs_float y in
      if x < y then let xy = x /. y in y *. sqrt(1. +. xy *. xy)
      else (* x >= y *) if x = 0. then 0. (* 0 = x >= y >= 0 *)
      else let yx = y /. x in x *. sqrt(1. +. yx *. yx)
      >>)

  (* [pow_float_literal _loc e f] outputs a more efficient reprensentation
     of [e**f] for particular float literals [f]. *)
  let pow_float_literal _loc e f =
    (* Make sure only local variables (that do not escape their
       scope) are used in the substitutions. *)
      match normalize_float f with
      | "0." -> <:expr< 1. >>
      | "1." -> e
      | "2." -> bind _loc e (fun x -> <:expr< $mul _loc$ $x$ $x$ >>)
      | "3." -> bind _loc e (fun x -> <:expr<
                                $mul _loc$ $x$ ($mul _loc$ $x$ $x$) >>)
      | "4." -> bind _loc e (fun x ->
                                <:expr< let x2 = $mul _loc$ $x$ $x$ in
                                $mul _loc$ x2 x2 >>)
      | "5." -> bind _loc e  begin fun x ->
          <:expr< let x2 = $mul _loc$ $x$ $x$ in
          $mul _loc$ x2 ($mul _loc$ x2 $x$) >>
        end
      | "6." -> bind _loc e  begin fun x ->
          <:expr< let x2 = $mul _loc$ $x$ $x$ in
          $mul _loc$ x2 ($mul _loc$ x2 x2) >>
        end
      | _ -> <:expr< $pow _loc$ $e$ $flo:f$ >>  ;;

  let is_zero = function
    | <:expr< $int:x$ >> | <:expr< $flo:x$ >> -> float_of_string x = 0.
    | _ -> false

  let is_one = function
    | <:expr< $int:x$ >> | <:expr< $flo:x$ >> -> float_of_string x = 1.
    | _ -> false

  let is_literal = function
    | <:expr< $int:_$ >> | <:expr< $flo:_$ >> -> true
    | _ -> false

  let float_of_lit = function
    | <:expr< $int:x$ >> | <:expr< $flo:x$ >> -> float_of_string x
    | _ -> assert false

  let overloading =
    let t = openin empty ~remove:true in (* Float is not a real module *)
    let t = int t ~cache:false
      (fun i _ _loc -> to_expr(float_of_int i) _loc) in
    let t = lid t "~-" (fun _ loc -> neg loc) ~cache:false in
    let t = lid t "+"  (fun _ loc -> add loc) ~cache:false in
    let t = lid t "-"  (fun _ loc -> sub loc) ~cache:false in
    let t = lid t "*"  (fun _ loc -> mul loc) ~cache:false in
    let t = lid t "/"  (fun _ loc -> div loc) ~cache:false in
    let t = lid t "**" (fun _ loc -> pow loc) ~cache:false in
    let t = lid t "abs" (fun _ loc -> abs loc) ~cache:false in
    let t = lid t "to_string" ~cache:false
      (fun _ _loc -> <:expr< Pervasives.string_of_float >>) in
    let t = lid t "of_string" ~cache:false
      (fun _ _loc -> <:expr< Pervasives.float_of_string >>) in
    (* We qualify with Pervasives the functions below to make sure they
       have their usual meaning even if they were redefined in the
       source file. *)
    let f = ["abs_float"; "acos"; "asin"; "atan"; "atan2"; "ceil";
             "cos"; "cosh"; "epsilon_float"; "exp"; "float"; "floor"; "frexp";
             "infinity"; "int_of_float"; "log"; "log10"; "max_float";
             "min_float"; "mod_float"; "modf"; "nan"; "neg_infinity";
             "sin"; "sinh"; "sqrt"; "string_of_float"; "tan"; "tanh";
             "truncate" ] in
    let t =
      let reloc id = fun _ _loc -> <:expr< Pervasives. $lid:id$ >> in
      List.fold_left (fun t id -> lid t id (reloc id) ~cache:false) t f in
    let t = lid t "incr" (fun _ _loc -> <:expr< fun x -> x := x +. 1. >>) in
    let t = lid t "decr" (fun _ _loc -> <:expr< fun x -> x := x -. 1. >>) in
    (* Specialized comparisons.  This is actually the only
       case I've seen where caching does not bring optimal performance.
       Since we do not want to have even a small efficiency loss, we use
       here a specialized binary operator overloading (type annotations
       are not enough because there may a partial evaluation of "="). *)
    let t = lid t "="  (fun _ loc -> eq loc) ~cache:false in
    let t = lid t "<>" (fun _ loc -> ne loc) ~cache:false in
    let t = lid t "<"  (fun _ loc -> lt loc) ~cache:false in
    let t = lid t "<=" (fun _ loc -> le loc) ~cache:false in
    let t = lid t ">"  (fun _ loc -> gt loc) ~cache:false in
    let t = lid t ">=" (fun _ loc -> ge loc) ~cache:false in
    let t = lid t "compare" (fun _ loc -> compare loc) ~cache:false in
    let t = lid t "is_nan" begin fun _ _loc ->
      <:expr< fun x -> $ne _loc$ x x >>
    end in
    let t = lid t "max" (fun _ loc -> max loc) ~cache:false in
    let t = lid t "min" (fun _ loc -> min loc) ~cache:false in
    let t = lid t "pi"  (fun _ loc -> pi loc)  ~cache:false in
    let t = lid t "hypot" (fun _ loc -> hypot loc) ~cache:false in
    let optimize tr expr =
      match expr with
      | (<:expr@_loc< $e1$ + $e2$ >> | <:expr@_loc< $e1$ +. $e2$ >>) ->
        let e1 = ((self tr)#expr e1) and e2 = ((self tr)#expr e2) in
        if is_zero e1 then e2
        else if is_zero e2 then e1
        else if is_literal e1 && is_literal e2 then
          <:expr< $`flo:(float_of_lit e1 +. float_of_lit e2)$ >>
        else <:expr< $add _loc$ $e1$ $e2$ >>
      | <:expr@_loc< $e1$ - $e2$ >> | <:expr@_loc< $e1$ -. $e2$ >> ->
        let e1 = (self tr)#expr e1 and e2 = (self tr)#expr e2 in
        if is_zero e1 then <:expr< ~-. e2 >> (* could try simpl. more *)
        else if is_zero e2 then e1
        else if is_literal e1 && is_literal e2 then
          <:expr< $`flo:(float_of_lit e1 -. float_of_lit e2)$ >>
        else <:expr< $sub _loc$ $e1$ $e2$ >>
      | <:expr@_loc< $e1$ * $e2$ >> | <:expr@_loc< $e1$ *. $e2$ >> ->
        let e1 = ((self tr)#expr e1) and e2 = ((self tr)#expr e2) in
        if is_zero e1 || is_zero e2 then <:expr< 0. >>
        else if is_one e1 then e2
        else if is_one e2 then e1
        else if is_literal e1 && is_literal e2 then
          <:expr< $`flo:(float_of_lit e1 *. float_of_lit e2)$ >>
        else <:expr< $mul _loc$ $e1$ $e2$ >>
      | <:expr@_loc< $e1$ / $e2$ >> ->
        let e1 = ((self tr)#expr e1) and e2 = ((self tr)#expr e2) in
        if is_one e2 then e1
        else if is_literal e1 && is_literal e2 then
          <:expr< $`flo:(float_of_lit e1 /. float_of_lit e2)$ >>
        else <:expr< $div _loc$ $e1$ $e2$ >>

      | <:expr@loc< $e$ ** $int:i$ >> ->
        pow_float_literal loc ((self tr)#expr e) (i ^ ".")
      | <:expr@loc< $e$ ** $flo:r$ >> ->
        pow_float_literal loc ((self tr)#expr e) r
      | <:expr@_loc< is_nan $e$ >> ->
        bind _loc ((self tr)#expr e) (fun e -> <:expr< $ne _loc$ $e$ $e$ >>)
      | <:expr@_loc< max $e1$ $e2$ >> ->
        (* Inline the def of "max" if its arguments are explicitly given *)
        bind _loc ((self tr)#expr e1)  begin fun e1 ->
          bind _loc ((self tr)#expr e2)  begin fun e2 ->
            <:expr< (if $ge _loc$ $e1$ $e2$ then $e1$
                     else if $ne _loc$ $e2$ $e2$ then $e1$ else $e2$) >>
          end
        end
      | <:expr@_loc< min $e1$ $e2$ >> ->
        (* Inline the def of "min" if its arguments are explicitly given *)
        bind _loc ((self tr)#expr e1)  begin fun e1 ->
          bind _loc ((self tr)#expr e2)  begin fun e2 ->
            <:expr< (if $le _loc$ $e1$ $e2$ then $e1$
                     else if $ne _loc$ $e2$ $e2$ then $e1$ else $e2$) >>
          end
        end
      | <:expr@_loc< max ($e1$, $e2$) >> ->
        (* max over an explicit tuple of arguments, ignore nan *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc gt (<:expr< neg_infinity >> :: e)
      | <:expr@_loc< min ($e1$, $e2$) >> ->
        (* min over an explicit tuple of arguments, ignore nan *)
        let e = Ast.list_of_expr e1 (Ast.list_of_expr e2 []) in
        let e = List.map (fun e -> ((self tr)#expr e)) e in
        max_list _loc lt (<:expr< infinity >> :: e)
      | <:expr@_loc< hypot $e1$ $e2$ >> ->
        (* Inline the definition of hypot if arguments are given *)
        <:expr<
          let x = Pervasives.abs_float $(self tr)#expr e1$
          and y = Pervasives.abs_float $(self tr)#expr e2$ in
          if x < y then let xy = x /. y in y *. sqrt(1. +. xy *. xy)
          else (* x >= y *) if x = 0. then 0. (* 0 = x >= y >= 0 *)
          else let yx = y /. x in x *. sqrt(1. +. yx *. yx)
        >>
      | _ -> super tr expr in
    expr t optimize

  let () =
    associate overloading "Float"

end

(***********************************************************************
 *                Overloadings for the module Complex
 ***********************************************************************)

(* We transform complex expressions by inlining operations (which
   avoids boxings and allows the compiler to inline float operations).
   This requires binding subexpressions.  Thus the transformation will
   be of the type

   (bindings, e) -> (bindings', e')

   The order of the bindings is important, binding subexpressions must
   happen before the bindings for the expression using them.
*)
module Complex =
struct
  (** Represents the knowledge that we have about a complex
      expression.  Specializations must be functions [t -> t]. *)
  type t =
    | C of number * number (* complex: a + bi *)
    | Unknown of Ast.expr (* an expression which type we do not know,
                             e.g. [f x] for an unknown function [f] *)
  and number = Ast.expr option
      (* None : syntactically, no such component exists; in a complex
                expression, is interpreted as 0.
         Some f : float expression *)

  (** [(s, e)] represents the binding of the expression [e] to the
      variable [s]. *)
  type bindings = (string * Ast.expr) list

  type opt_expr = bindings * t
      (** Specialization of complex operations.  We see binary
          operators as functions [opt_expr -> opt_expr -> opt_expr],
          which allows to specialize them if we know that the
          rexpression represent a purely real or imaginary number. *)

  let real r = C(Some r, None)
  let imag r = C(None, Some r)

  let expr_of_number _loc = function
    | None -> <:expr< 0. >>
    | Some f -> f

  let number_of_float _loc f = Some(<:expr< $flo:normalize_float f$ >>)

  let expr_of_t _loc e = match e with
    | C(r,i) -> <:expr< { Complex.re = $expr_of_number _loc r$;
                          Complex.im = $expr_of_number _loc i$ } >>
    | Unknown e -> e

  let float_of_t _loc e = match e with
    | C(r, None) -> expr_of_number _loc r
    | Unknown e -> <:expr< $e$ >>
    | _ -> Loc.raise _loc (Stream.Error "Expected a real number but the \
            expression seems to be complex.  Maybe you forgot to append .re \
            or .im?")

  (* The list of bindings is in reverse order: [([(v0,c0); ...; (vN,
     cN)], expr)] means [let vN = cN in ... let v0 = c0 in expr] *)
  let to_expr _loc (b,e) =
    let e0 = expr_of_t _loc e in
    List.fold_left (fun e (v,c) -> <:expr< let $lid:v$ = $c$ in $e$ >>) e0 b

  (** [add_binding b e] returns [(b', v)] where [b'] is the new
      binding list and [v] is the identifier to which the expression
      [e] is bound. *)
  let add_binding (b: bindings) e = match e with
    | _ when is_immediate e -> (b, e)
    | <:expr< $lid:_$.Complex.re >> | <:expr< $lid:_$.Complex.im >>
    | <:expr< ($lid:_$ : $_$).Complex.re >>
    | <:expr< ($lid:_$ : $_$).Complex.im >>
      (* Avoid to introduce bindings for e.g. [x.Complex.re]. *)
      -> (b, e)
    | _ ->
        let v = new_lid() in
        let _loc = Ast.loc_of_expr e in ((v, e) :: b, <:expr< $lid:v$ >>)

  let add_binding_number b r = match r with
    | None -> (b, r)
    | Some e -> let b, e1 = add_binding b e in (b, Some e1)

  (** Add a binding for [z] (if needed) and decompose the complex into
      its real and imaginary parts.  Add a type annotation for the
      type system to provide early feedback. *)
  let components b z =
    let _loc = Ast.loc_of_expr z in
    let b, z = add_binding b <:expr< ($z$:Complex.t) >> in
    (b, Some <:expr< $z$.Complex.re >>, Some <:expr< $z$.Complex.im >>)

  let components_of_t ?(bind=true) b t = match t with
    | C(r, i) ->
        if bind then
          let b, r = add_binding_number b r in
          let b, i = add_binding_number b i in
          b, r, i
        else b, r, i (* OK if the components are used at one place only *)
    | Unknown z -> components b z

  (** Return the real and imaginary parts of [t] as float expressions.
      Can be useful for the general case of optimizations. *)
  let float_components_of_t ?(bind=true) b _loc t =
    let b, r, i = components_of_t ~bind b t in
    (b, expr_of_number _loc r, expr_of_number _loc i)

  (** Expression to test whether the variable [r] is [>0] or is [+0.].
      FIXME: Could be faster if we had [copysign]. *)
  let pos_sign b r =
    let _loc = Ast.loc_of_expr r in
    let b, r = add_binding b r in
    (b, <:expr< Pervasives.( || ) ($Float.gt _loc$ $r$ 0.)
                ($Float.gt _loc$ ($Float.div _loc$ 1. $r$) 0.)
       >> );;

  (* Optimize boolean operators *)
  let both _loc e1 e2 = match e1, e2 with
    | _, <:expr< true >> -> e1
    | <:expr< true >>, _ -> e2
    | <:expr< false >>, _ | _, <:expr< false >> -> <:expr< false >>
    | _ -> <:expr< e1 && e2 >>  ;;

  let oneof _loc e1 e2 = match e1, e2 with
    | _, <:expr< false >> -> e1
    | <:expr< false >>, _ -> e2
    | <:expr< true >>, _ | _, <:expr< true >> -> <:expr< false >>
    | _ -> <:expr< e1 || e2 >>  ;;

  (* Order relations
   ***********************************************************************)

  (* The sign of zero does not matter for equality. *)
  let eq_float _loc r1 r2 = match r1, r2 with
    | None, None -> <:expr< true >>
    | Some x, None | None, Some x -> <:expr< $Float.eq _loc$ $x$ 0. >>
    | Some x1, Some x2 -> (<:expr< $Float.eq _loc$ $x1$ $x2$ >>)

  let eq _loc (b1,e1) (b2,e2) : opt_expr =
    (* bindings [b2] are not supposed to shadow [b1] bindings. *)
    let b = b1 @ b2 in
    match e1, e2 with
    | C(r1, i1), C(r2, i2) ->
        (b, Unknown(both _loc (eq_float _loc r1 r2) (eq_float _loc i1 i2)))
    | C(r1, i1), Unknown z | Unknown z, C(r1, i1) ->
        let b, r2, i2 = components b z in
        (b, Unknown(both _loc (eq_float _loc r1 r2) (eq_float _loc i1 i2)))
    | Unknown z1, Unknown z2 ->
        let b, r1, i1 = components b z1 in
        let b, r2, i2 = components b z2 in
        (b, Unknown(both _loc (eq_float _loc r1 r2) (eq_float _loc i1 i2)))

  (* The sign of zero does not matter for the inequality. *)
  let ne_float _loc r1 r2 = match r1, r2 with
    | None, None -> <:expr< false >>
    | Some x, None | None, Some x -> <:expr< $Float.ne _loc$ $x$ 0. >>
    | Some x1, Some x2 -> <:expr< $Float.ne _loc$ $x1$ $x2$ >> ;;

  let ne _loc (b1,e1) (b2,e2) : opt_expr =
    (* bindings [b2] are not supposed to shadow [b1] bindings. *)
    let b = b1 @ b2 in
    match e1, e2 with
    | C(r1, i1), C(r2, i2) ->
        (b, Unknown(oneof _loc (ne_float _loc r1 r2) (ne_float _loc i1 i2)))
    | C(r1, i1), Unknown z | Unknown z, C(r1, i1) ->
        let b, r2, i2 = components b z in
        (b, Unknown(oneof _loc (ne_float _loc r1 r2) (ne_float _loc i1 i2)))
    | Unknown z1, Unknown z2 ->
        let b, r1, i1 = components b z1 in
        let b, r2, i2 = components b z2 in
        (b, Unknown(oneof _loc (ne_float _loc r1 r2) (ne_float _loc i1 i2)))

  let ord_float cmp0 cmp _loc r1 r2 = match r1, r2 with
    | None, None -> <:expr< $uid:cmp0$ >>
    | Some x, None -> <:expr< $cmp _loc$ $x$ 0. >>
    | None, Some x -> <:expr< $cmp _loc$ 0. $x$ >>
    | Some x1, Some x2 -> <:expr< $cmp _loc$ $x1$ $x2$ >> ;;

  let ord cmp0 cmp cmp_name _loc (b1,e1) (b2,e2) : opt_expr =
    let b = b1 @ b2 in
    let o e1 e2 = (b, Unknown(ord_float cmp0 cmp _loc e1 e2)) in
    match e1, e2 with
    | C(r1, None), C(r2, None) -> o r1 r2
    | C(r1, None), Unknown r2 -> o r1 (Some r2)
    | Unknown r1, C(r2, None) -> o (Some r1) r2
    | Unknown r1, Unknown r2 -> o (Some r1) (Some r2)
    | C(_, Some _), _ | _, C(_, Some _) ->
        (* Not obvious real numbers (from the syntax). *)
        Loc.raise _loc (Stream.Error("pa_do: You can only compare ("
                                     ^ cmp_name ^ ") real numbers."))

  let lt = ord "False" Float.lt "<"
  let le = ord "True"  Float.le "<="
  let gt = ord "False" Float.gt ">"
  let ge = ord "True"  Float.ge ">="

  (* Arithmetic
   ***********************************************************************)

  (* Unary negation *)
  let neg_float = function
    | None -> None
    | Some(<:expr< $lid:op$ $e$ >>)
        when lid_in_expr op (Float.neg Loc.ghost) -> Some(e)
    | Some(<:expr@_loc< $flo:f$ >>) ->
        Some(Float.to_expr (-. float_of_string f) _loc)
    | Some e ->
        let _loc = Ast.loc_of_expr e in
        Some(<:expr< $Float.neg _loc$ $e$ >>)

  let neg _loc (b,e) =
    match e with
    | C(r, i) -> (b, C(neg_float r, neg_float i))
    | Unknown z -> (* [z] assumed to be a complex *)
        let b, r, i = components b z in
        (b, C(neg_float r, neg_float i))

  (** [binop op] return a function performing the operation on values
      of type [opt_expr] from the operation [op] specified on components. *)
  let binop op _loc (b1,e1) (b2,e2) : opt_expr =
    (* bindings [b2] are supposed not to shadow [b1] bindings. *)
    let b = b1 @ b2 in
    match e1, e2 with
    | C(r1, i1), C(r2, i2) ->
        let r, i = op _loc r1 i1 r2 i2 in (b, C(r, i))
    | C(r1, i1), Unknown z ->
        let b, r2, i2 = components b z in
        let r, i = op _loc r1 i1 r2 i2 in (b, C(r, i))
    | Unknown z, C(r2, i2) ->
        let b, r1, i1 = components b z in
        let r, i = op _loc r1 i1 r2 i2 in (b, C(r, i))
    | Unknown z1, Unknown z2 ->
        let b, r1, i1 = components b z1 in
        let b, r2, i2 = components b z2 in
        let r, i = op _loc r1 i1 r2 i2 in (b, C(r, i));;

  let add_float _loc r1 r2 = match r1, r2 with
    | None, None -> None
    | Some _, None -> r1 (* we do not try to imitate Complex.add on
                            zeros, this is because the user does not
                            expect computations on a component he did
                            not specify. *)
    | None, Some _ -> r2
    | Some <:expr< $flo:x1$ >>, Some <:expr< $flo:x2$ >> ->
      let sum = float_of_string x1 +. float_of_string x2 in
      Some(Float.to_expr sum _loc)
    | Some e1, Some e2 -> Some(<:expr< $Float.add _loc$ $e1$ $e2$ >>)

  let add = binop (fun _loc r1 i1 r2 i2 ->
                     (add_float _loc r1 r2, add_float _loc i1 i2))

  let sub_float _loc r1 r2 = match r1, r2 with
    | None, None -> None
    | Some _, None -> r1
    | None, Some _ -> neg_float r2
    | Some <:expr< $flo:x1$ >>, Some <:expr< $flo:x2$ >> ->
      let diff = float_of_string x1 -. float_of_string x2 in
      Some(Float.to_expr diff _loc)
    | Some e1, Some e2 -> Some(<:expr< $Float.sub _loc$ $e1$ $e2$ >>)

  let sub = binop (fun _loc r1 i1 r2 i2 ->
                     (sub_float _loc r1 r2, sub_float _loc i1 i2))

  let mul_float _loc r1 r2 = match r1, r2 with
    | None, _ | _, None -> None
    | Some(<:expr< 1. >>), _ -> r2
    | _, Some(<:expr< 1. >>) -> r1
    | Some(<:expr< -1. >>), _ -> neg_float r2
    | _, Some(<:expr< -1. >>) -> neg_float r1
    | Some <:expr< $flo:x1$ >>, Some <:expr< $flo:x2$ >> ->
      let prod = float_of_string x1 *. float_of_string x2 in
      Some(Float.to_expr prod _loc)
    | Some e1, Some e2 -> Some(<:expr< $Float.mul _loc$ $e1$ $e2$ >>)

  let mul _loc (b1,e1) (b2,e2) =
    let b = b1 @ b2 in
    let b, r1, i1 = components_of_t b e1 in
    let b, r2, i2 = components_of_t b e2 in
    let r = sub_float _loc (mul_float _loc r1 r2) (mul_float _loc i1 i2)
    and i = add_float _loc (mul_float _loc r1 i2) (mul_float _loc i1 r2) in
    (b, C(r (* r1 r2 - i1 i2 *), i (* r1 i2 + i1 r2 *) ))

  let inv _loc (b,e) = match e with
    | C(None, None) ->
        (* FIXME: To be coherent with the current [Complex.div]
           behavior but maybe not the better for the long term... *)
        let nan = Some <:expr< Pervasives.nan >> in ([], C(nan, nan))
    | C(Some <:expr< 1. >>, None) | C(Some <:expr< -1. >>, None) ->
        (b, e)
    | C(Some <:expr@_loc< $flo:x$ >>, None) ->
        let invx = 1. /. float_of_string x in
        (b, C(Some(Float.to_expr invx _loc), None))
    | C(None, Some <:expr< 1. >>) ->  (b, C(None, Some <:expr< -1. >>))
    | C(None, Some <:expr< -1. >>) -> (b, C(None, Some <:expr< 1. >>))
    | C(Some r, None) ->
        (b, C(Some(<:expr< $Float.div _loc$ 1. $r$ >>), None))
    | C(None, Some i) ->
        (b, C(None, Some <:expr< $Float.div _loc$ (-1.) $i$ >>))
    | _ ->
        let b, r, i = components_of_t b e in
        match r, i with
        | Some r, Some i ->
            (* We need to use references to have multiple alloc in the "if"
               -- unkown from the outside anyway. *)
            let b, re = add_binding b <:expr< ref Pervasives.nan >> in
            let b, im = add_binding b <:expr< ref Pervasives.nan >> in
            let b, _ = add_binding b <:expr<
              if $Float.ge _loc$ ($Float.abs _loc$ $r$) ($Float.abs _loc$ $i$)
              then (
                let q = $Float.div _loc$ $i$ $r$ in
                let d = $Float.div _loc$ 1.
                  ($Float.add _loc$ $r$ ($Float.mul _loc$ q $i$)) in
                $re$ := d;
                $im$ := $Float.neg _loc$ ($Float.mul _loc$ q d)
              )
              else (
                let q = $Float.div _loc$ $r$ $i$ in
                let d = $Float.div _loc$ 1.
                  ($Float.add _loc$ $i$ ($Float.mul _loc$ q $r$)) in
                $re$ := $Float.mul _loc$ q d;
                $im$ := $Float.neg _loc$ d
              ) >> in
            (b, C(Some <:expr< Pervasives.( ! ) $re$ >>,
                  Some <:expr< Pervasives.( ! ) $im$ >>))
        | None, _ | _, None ->
            assert false (* treated before with simpler formulas *)
  ;;

  let div_float _loc (r1:number) (r2:number) = match r1, r2 with
    | None, None -> None
    | Some _, None -> None
    | None, Some d -> None (* (r + None I) / (Some 3 + None I) *)
    | Some n, Some(<:expr< 1. >>) -> r1
    | Some n, Some(<:expr< -1. >>) -> neg_float r1
    | Some <:expr< $flo:x1$ >>, Some <:expr< $flo:x2$ >> ->
      let quo = float_of_string x1 /. float_of_string x2 in
      Some(Float.to_expr quo _loc)
    | Some n, Some d -> Some(<:expr< $Float.div _loc$ $n$ $d$ >>)

  let div _loc (b1,e1) (b2,e2 as t2) =
    let b = b1 @ b2 in
    match e1, e2 with
    | _, C(None, None) ->
        let nan = Some <:expr< Pervasives.nan >> in ([], C(nan, nan))
    | C(Some <:expr< 1. >>, None), _ ->
        (* This is *not* the same as [Complex.div Complex.one _] on
           the signs of zeros but is done so that [1 / z] has the
           expected behavior. *)
        inv _loc t2
    | C(Some <:expr< -1. >>, None), _ -> neg _loc (inv _loc t2)
    | C(r1, i1), C((Some _ as r2), None) ->
        (* When the imaginary part of the denominator is statically
           known as not being presentq, we do not consider ourselves
           as being bound to produce the same code as [Complex.div].
           This is e.g. in order that [z/(2 + 3)] has the expected
           behavior. *)
        let b, r2 = add_binding_number b r2 in
        (b, C(div_float _loc r1 r2, div_float _loc i1 r2))
    | C(r1, i1), C(None, (Some _ as i2)) ->
        let b, i2 = add_binding_number b i2 in
        (b, C(div_float _loc i1 i2, div_float _loc (neg_float r1) i2))
    | _, _ ->
        let b, r1, i1 = float_components_of_t b _loc e1 in
        let b, r2, i2 = float_components_of_t b _loc e2 in
        let b, re = add_binding b <:expr< ref Pervasives.nan >> in
        let b, im = add_binding b <:expr< ref Pervasives.nan >> in
        let b, _ = add_binding b <:expr<
          if $Float.ge _loc$ ($Float.abs _loc$ $r2$) ($Float.abs _loc$ $i2$)
          then (
            let q = $Float.div _loc$ $i2$ $r2$ in
            let d = $Float.add _loc$ $r2$ ($Float.mul _loc$ q $i2$) in
            $re$ := $Float.div _loc$
              ($Float.add _loc$ $r1$ ($Float.mul _loc$ q $i1$)) d;
            $im$ := $Float.div _loc$
              (Pervasives.( -. ) $i1$ ($Float.mul _loc$ q $r1$)) d;
          )
          else (
            let q = $Float.div _loc$ $r2$ $i2$ in
            let d = $Float.add _loc$ $i2$ ($Float.mul _loc$ q $r2$) in
            $re$ := $Float.div _loc$
              ($Float.add _loc$ ($Float.mul _loc$ q $r1$) $i1$) d;
            $im$ := $Float.div _loc$
              (Pervasives.( -. ) ($Float.mul _loc$ q $i1$) $r1$) d;
          ) >> in
        (b, C(Some <:expr< Pervasives.( ! ) $re$ >>,
              Some <:expr< Pervasives.( ! ) $im$ >>))

  let conj _loc (b,e) = match e with
    | C(r, i) -> (b, C(r, neg_float i))
    | Unknown z -> let b, r, i = components b z in (b, C(r, neg_float i))

  let norm _loc (b,e) = match e with
    | C(None, None) -> ([], C(None, None))
    | C(Some x, None) | C(None, Some x) ->
        (b, real(<:expr< $Float.abs _loc$ $x$ >>))
    | C(Some r, Some i) ->
        let n = <:expr<
          let r = $Float.abs _loc$ $r$ and i = $Float.abs _loc$ $i$ in
          if $Float.eq _loc$ r 0. then i
          else if $Float.eq _loc$ i 0. then r
          else if $Float.ge _loc$ r i then
            let q = $Float.div _loc$ i r in
            $Float.mul _loc$ r (Pervasives.sqrt($Float.add _loc$ 1.
                                                 ($Float.mul _loc$ q q)))
          else
            let q = $Float.div _loc$ r i in
            $Float.mul _loc$ i (Pervasives.sqrt($Float.add _loc$ 1.
                                                 ($Float.mul _loc$ q q))) >> in
        (b, real n)
    | Unknown z -> (b, real <:expr< Complex.norm $z$ >>)

  let norm2 _loc (b,e) = match e with
    | C(None, None) -> ([], C(None, None))
    | C(Some x, None) | C(None, Some x) ->
        let b, x = add_binding b x in
        (b, real(<:expr< $Float.mul _loc$ $x$ $x$ >>))
    | C(Some r, Some i) ->
        let b, r = add_binding b r in
        let b, i = add_binding b i in
        (b, real(<:expr< $Float.add _loc$ ($Float.mul _loc$ $r$ $r$)
                   ($Float.mul _loc$ $i$ $i$) >>))
    | Unknown z ->
        let b, z = add_binding b z in
        (b, real(<:expr< $Float.add _loc$
                   ($Float.mul _loc$ $z$.Complex.re $z$.Complex.re)
                   ($Float.mul _loc$ $z$.Complex.im $z$.Complex.im) >>))

  let pi = <:expr@here< 3.141592653589793238462643383279502884197 >>
  let neg_pi = <:expr@here< -3.141592653589793238462643383279502884197 >>
  let half_pi = <:expr@here< 1.570796326794896619231321691639751442099 >>
  let neg_half_pi = <:expr@here< -1.570796326794896619231321691639751442099 >>
  ;;

  let arg _loc (b,e) = match e with
    | C(None, None) -> ([], e)
    | C(Some r, None) ->
        (* Syntactically no imaginary part. *)
        let b, pos_r = pos_sign b r in
        (b, real <:expr< if $pos_r$ then 0. else $pi$ >>)
    | C(None, Some i) ->
        let b, i = add_binding b i in
        (* arg((0)I) = 0 *)
        (b, real <:expr<
           if $Float.eq _loc$ $i$ 0. then $i$
           else if $Float.gt _loc$ $i$ 0. then $half_pi$ else $neg_half_pi$ >>)
    | C(Some r, Some i) ->
        (b, real <:expr< Pervasives.atan2 $i$ $r$ >>)
    | Unknown z ->
        let b, z = add_binding b z in
        (b, real <:expr< Pervasives.atan2 $z$.Complex.im $z$.Complex.re >>)
  ;;

  let on_number op _loc r : number = match r with
    | None -> None
    | Some r -> Some <:expr< Pervasives.$lid:op$ $r$ >>
  let sin_float = on_number "sin"
  let cos_float = on_number "cos"
  let sinh_float = on_number "sinh"
  let cosh_float = on_number "cosh"
  let exp_float = on_number "exp"

  let sin _loc (b,e) = match e with
    | C(None, None) -> ([], e)
    | C(Some r, None) -> (b, real <:expr< Pervasives.sin $r$ >>)
    | C(None, Some i) -> (b, imag <:expr< Pervasives.sinh $i$ >>)
    | _ ->
        let b, r, i = components_of_t b e in
        (b, C(mul_float _loc (sin_float _loc r) (cosh_float _loc i),
              mul_float _loc (cos_float _loc r) (sinh_float _loc i)))

  let cos _loc (b,e) = match e with
    | C(None, None) -> ([], e)
    | C(Some r, None) -> (b, real <:expr< Pervasives.cos $r$ >>)
    | C(None, Some i) -> (b, real <:expr< Pervasives.cosh $i$ >>)
    | _ ->
        let b, r, i = components_of_t b e in
        (b, C(mul_float _loc (cos_float _loc r) (cosh_float _loc i),
              mul_float _loc (sin_float _loc r) (sinh_float _loc i)))

  let exp _loc (b,e) = match e with
    | C(None, None) -> ([], e)
    | C(Some r, None) -> (b, real <:expr< Pervasives.exp $r$ >>)
    | C(None, Some i) ->
        let b, i = add_binding b i in
        (b, C(Some <:expr< Pervasives.cos $i$ >>,
              Some <:expr< Pervasives.sin $i$ >>))
    | C(Some r, Some i) ->
        let b, exp_r = add_binding b <:expr< Pervasives.exp $r$ >> in
        let b, i = add_binding b i in
        (b, C(Some <:expr< $Float.mul _loc$ $exp_r$ (Pervasives.cos $i$) >>,
              Some <:expr< $Float.mul _loc$ $exp_r$ (Pervasives.sin $i$) >>))
    | Unknown z ->
        let b, z = add_binding b z in
        let b, expr = add_binding b <:expr< Pervasives.exp $z$.Complex.re >> in
        let r = <:expr<
          $Float.mul _loc$ $expr$ (Pervasives.cos $z$.Complex.im) >>
        and i = <:expr<
          $Float.mul _loc$ $expr$ (Pervasives.sin $z$.Complex.im) >> in
        (b, C(Some r, Some i))

  let log _loc (b,e) = match e with
    | C(None, None) -> ([], e)
    | C(Some r, None) ->
        (* Syntactically real number but may have a complex logarithm *)
        let b, r = add_binding b r in
        let b, pos_r = pos_sign b r in
        (b, C(Some <:expr< Pervasives.log($Float.abs _loc$ $r$) >>,
              Some <:expr< if $pos_r$ then 0. else $pi$ >>))
    | C(None, Some i) ->
        let b, i = add_binding b i in
        let b, pos_i = pos_sign b i in
        (b, C(Some <:expr< Pervasives.log($Float.abs _loc$ $i$) >>,
              Some <:expr< if $pos_i$ then $half_pi$ else $neg_half_pi$ >>))
    | C(Some r, Some i) ->
        let b, r = add_binding b r in
        let b, i = add_binding b i in
        let b, norm_expr = norm _loc (b, C(Some r, Some i)) in
        (b, C(Some <:expr< Pervasives.log($float_of_t _loc norm_expr$) >>,
              Some <:expr< Pervasives.atan2 $i$ $r$ >>))
    | Unknown z ->
        let b, z = add_binding b z in
        (b, C(Some <:expr< Pervasives.log(Complex.norm $z$) >>,
              Some <:expr< Pervasives.atan2 $z$.Complex.im $z$.Complex.re >>))


  let pow _loc (b1,e1 as x) (b2,e2 as y) =
    let b = b1 @ b2 in
    match e1, e2 with
    | C(None, None), _ -> ([], C(None, None))
    | _, C(None, None) -> ([], real <:expr< 1. >>)
    | C(Some r1, None), C(Some r2, None) ->
        let r = match r2 with
          | <:expr< $flo:f$ >> -> Float.pow_float_literal _loc r1 f
          | _ -> <:expr< $Float.pow _loc$ $r1$ $r2$ >> in
        (b, real r)
    | C(None, Some i1), C(Some r2, None) ->
        let b, pos_i1 = pos_sign b i1 in
        let b, n = add_binding b <:expr<
          $Float.pow _loc$ ($Float.abs _loc$ $i1$) $r2$ >> in
        let b, a = add_binding b <:expr<
          if $pos_i1$ then $Float.mul _loc$ $r2$ $half_pi$
          else $Float.mul _loc$ $r2$ $neg_half_pi$ >> in
        (b, C(Some <:expr< $Float.mul _loc$ $n$ (Pervasives.cos $a$) >>,
              Some <:expr< $Float.mul _loc$ $n$ (Pervasives.sin $a$) >>))
    | _, C(Some r, None) ->
        (* Complex powered to a real number *)
        (match r with
         | <:expr< 0. >> -> ([], C(Some <:expr< 1. >>, None))
         | <:expr< 1. >> -> (b, e1)
         | <:expr< -1. >> -> inv _loc (b, e1)
         | <:expr< 2. >> ->
           let b, r1, i1 = components_of_t b e1 in
           mul _loc (b, C(r1, i1)) ([], C(r1, i1)) (* do not repeat env [b] *)
         | <:expr< 3. >> ->
           let b, r1, i1 = float_components_of_t b _loc e1 in
           let b, r1_2 = add_binding b <:expr< $Float.mul _loc$ $r1$ $r1$ >> in
           let b, i1_2 = add_binding b <:expr< $Float.mul _loc$ $i1$ $i1$ >> in
           let r = <:expr< $Float.mul _loc$ $r1$
             ($Float.sub _loc$ $r1_2$ ($Float.mul _loc$ 3. $i1_2$)) >>
           and i = <:expr< $Float.mul _loc$ $i1$
             ($Float.sub _loc$ ($Float.mul _loc$ 3. $r1_2$) $i1_2$) >> in
           (b, C(Some r, Some i))
         | _ ->
             let b, r = add_binding b r in
             let b, r1, i1 = float_components_of_t b _loc e1 in
             let b, norm_expr = norm _loc (b, C(Some r1, Some i1)) in
             let b, n = add_binding b <:expr<
               $Float.pow _loc$ $float_of_t _loc norm_expr$ $r$ >> in
             let b, a = add_binding b <:expr<
               $Float.mul _loc$ (Pervasives.atan2 $i1$ $r1$) $r$ >> in
             (b, C(Some <:expr< $Float.mul _loc$ $n$ (Pervasives.cos $a$) >>,
                   Some <:expr< $Float.mul _loc$ $n$ (Pervasives.sin $a$) >>))
        )
    | _, _ ->
        exp _loc (mul _loc y (log _loc x))
  ;;

  let overloading =
    let t = lid_subst empty ["/","div"; "sqrt","sqrt"; "norm","norm"; "arg","arg";
                             "exp", "exp"; "log", "log"] in
    (* Equality *)
    let t = lid t "="  begin fun _ _loc ->
      <:expr< fun z1 z2 -> Pervasives.( && )
        ($Float.eq _loc$ z1.Complex.re z2.Complex.re)
        ($Float.eq _loc$ z1.Complex.im z2.Complex.im) >>
    end in
    let t = lid t "<>" begin fun _ _loc ->
      <:expr< fun z1 z2 -> Pervasives.( || )
        ($Float.ne _loc$ z1.Complex.re z2.Complex.re)
        ($Float.ne _loc$ z1.Complex.im z2.Complex.im) >>
    end in
    let t = lid t "to_string" begin fun _ _loc ->
      <:expr< fun z -> string_of_float z.Complex.re ^ "+"
        ^ string_of_float z.Complex.im ^ "I" >>
    end in
    let t = lid t "re" begin fun _ _loc ->
      <:expr< fun r -> { Complex.re = r; Complex.im = 0. } >>
    end in
    let t = lid t "im" begin fun _ _loc ->
      <:expr< fun i -> { Complex.re = 0.; Complex.im = i } >>
    end in

    (* The [let_var] (let-bound variables) will be carried down the
       AST thanks to [fold_expr]. *)
    let rec inline_complex ~self ~super let_var tr expr =
      to_expr (Ast.loc_of_expr expr) (optimize ~super let_var tr expr)
    and optimize ~super let_var tr expr : opt_expr =
      (* [let_var] is a list of the let-bound variables with their
         corresponding type [t].
         Return [(b,e)] where [b] are bindings and [e] is of type [t]. *)
      let _loc = Ast.loc_of_expr expr in
      match expr with
        (* Do not recurse in already overloaded expressions, but
           remove the overloading protection. *)
      | <:expr< $lid:id$ $e$ >> when id = overloaded -> ([], Unknown e)
      | <:expr< $lid:id$ $id:_$ $_$ >> when id = suspended -> ([], Unknown expr)

      (* Complex constants, "I" notation, float constants. *)
      | <:expr< { Complex.re = $flo:r$; im = $flo:i$ } >>
      | <:expr< { Complex.re = $flo:r$; Complex.im = $flo:i$ } >> ->
        (* If explicit zeros are present, we assume the programmer
           wants them so we do not try to optimize (this way the
           generated code better reflects the intent of the programmer). *)
        ([], C(number_of_float _loc r, number_of_float _loc i))
      | <:expr< { Complex.re = $r$; im = $i$ } >>
      | <:expr< { Complex.re = $r$; Complex.im = $i$ } >> ->
        ([], C(Some r, Some i))
      | <:expr< $flo:i$ $uid:"I"$ >> -> ([], C(None, number_of_float _loc i))
      | <:expr< $int:i$ $uid:"I"$ >> ->
        ([], C(None, Some <:expr< $flo:i ^ "."$ >>))
      | <:expr< $uid:"I"$ >> -> ([], C(None, Some <:expr< 1. >>))
      | <:expr< $flo:r$ >> ->   ([], C(number_of_float _loc r, None))
      | <:expr< $int:r$ >> ->   ([], C(number_of_float _loc (r ^ "."), None))

      (* Specialize [re] and [im] function whenever possible.  Do not
         recurse inside the expression since it is not complex. *)
      | <:expr< re $r$ >> -> ([], real r)
      | <:expr< im $i$ >> -> ([], imag i)
      | <:expr< float $e$ >> ->
        let b, e = optimize ~super let_var tr e in
        (* FIXME: should we add a [`Float] tag? *)
        (b, Unknown(float_of_t _loc e))

      (* Projections on the real and imaginary parts. *)
      | <:expr< $e$.re >> ->
        let b, e = optimize ~super let_var tr e in
        (match e with
         | C(r, _) -> (b, C(r, None))
         | Unknown z -> (b, real <:expr< $z$.Complex.re >>) )
      | <:expr< $e$.im >> ->
        let b, e = optimize ~super let_var tr e in
        (match e with
         | C(_, i) -> (b, C(i, None))
         | Unknown z -> (b, real <:expr< $z$.Complex.im >>) )

      (* Optimize order relations *)
      | <:expr< ( = ) $e1$ $e2$ >> ->
        eq _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( <> ) $e1$ $e2$ >> ->
        ne _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( < ) $e1$ $e2$ >> ->
        lt _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( <= ) $e1$ $e2$ >> ->
        le _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( > ) $e1$ $e2$ >> ->
        gt _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( >= ) $e1$ $e2$ >> ->
        ge _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)

      (* Operators *)
      | <:expr< ( ~- ) $e$ >> -> neg _loc (optimize ~super let_var tr e)
      | <:expr< ( + ) $e1$ $e2$ >> ->
        add _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( - ) $e1$ $e2$ >> ->
        sub _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( * ) $e1$ $e2$ >> ->
        mul _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( / ) $e1$ $e2$ >> ->
        div _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< ( ** ) $e1$ $e2$ >> ->
        pow _loc (optimize ~super let_var tr e1) (optimize ~super let_var tr e2)
      | <:expr< conj $e$ >> -> conj _loc (optimize ~super let_var tr e)
      | <:expr< norm $e$ >> | <:expr< abs $e$ >> ->
        norm _loc (optimize ~super let_var tr e)
      | <:expr< norm2 $e$ >> | <:expr< abs2 $e$ >> ->
        norm2 _loc (optimize ~super let_var tr e)
      | <:expr< arg $e$ >> -> arg _loc (optimize ~super let_var tr e)

      | <:expr< sin $e$ >> -> sin _loc (optimize ~super let_var tr e)
      | <:expr< cos $e$ >> -> cos _loc (optimize ~super let_var tr e)
      | <:expr< exp $e$ >> -> exp _loc (optimize ~super let_var tr e)
      | <:expr< log $e$ >> -> log _loc (optimize ~super let_var tr e)

      (* Let bindings.  Without this the [3] in [let x = 3 in sin x]
         will reach the [let] at which point it will be converted to a
         complex number (which is not intuitive). *)
      | <:expr< let $lid:x$ = $e1$ in $e2$ >> ->
        let b, opt_e1 = optimize ~super let_var tr e1 in
        let b, let_var = match opt_e1 with
          | C(None, None) ->
              (b,  (x, opt_e1) :: let_var) (* should not happen*)
          | C(Some r, None) ->
              (* Bind [x] to the float part of the complex number: *)
              ((x, r) :: b,  (x, C(Some <:expr< $lid:x$ >>, None)) :: let_var)
          | C(None, Some i) ->
              let b, i = add_binding b i in
              (b, (x, C(None, Some i)) :: let_var)
          | C(Some r, Some i) ->
              let b, r = add_binding b r in
              let b, i = add_binding b i in
              (b, (x, C(Some r, Some i)) :: let_var)
          | Unknown z ->
              (* Do not introduce a new var *)
              let _loc = Ast.loc_of_expr e1 in
              ((x, z) :: b,  (x, Unknown <:expr< $lid:x$ >>) :: let_var) in
        let b', e2 = optimize ~super let_var tr e2 in
        (b' @ b, e2)

      (* Variables -- use the information saved in let_bindings, if any.
         Bindings introduce new vars at the beginning => scoping handled *)
      | <:expr< $lid:x$ >> ->
        ([], try List.assoc x let_var with Not_found -> Unknown expr)

      (* Unkown construction.  Restore the new bindings put aside in
         [let_var]. *)
      | _ ->
          (* First tranfform the expression: this is because some
             variables may disappear as a result of optimizing the
             expression (e.g. [Complex.(let a = I in f a)]). *)
          let expr = super let_var tr expr in
          let fv = FV.free_vars FV.S.empty expr in
          let rec add_bindings = function
            | [] -> []
            | (v, (C(_, None) | Unknown _)) :: tl -> add_bindings tl
                (* No new variables were introduced. *)
            | (v, C(None, Some i)) :: tl ->
                if FV.S.mem v fv then
                  (* [v] is free in [expr] => restore binding. *)
                  let _loc = Ast.loc_of_expr i in
                  let e = <:expr< { Complex.re = 0.; Complex.im = $i$} >> in
                  (v, e) :: add_bindings tl
                else
                  add_bindings tl (* no [v] in [expr] *)
            | (v, C(Some r, Some i)) :: tl ->
                if FV.S.mem v fv then
                  let _loc =
                    Loc.merge (Ast.loc_of_expr r) (Ast.loc_of_expr i) in
                  let e = <:expr< { Complex.re = $r$; Complex.im = $i$} >> in
                  (v, e) :: add_bindings tl
                else
                  add_bindings tl
          in
          (add_bindings let_var,  Unknown(expr))
    in
    expr_fold t inline_complex []


  let () = associate overloading "Complex"
end


(***********************************************************************
 *                Overloadings for the module Hashtbl
 ***********************************************************************)

let overloading_hashtbl =
  let t = lid_subst empty ["clear", "clear"; "create", "create";
                           "fold", "fold";
                           "iter", "iter"; "length", "length"] in
  let get h k _ _loc = <:expr< Hashtbl.find $h$ $k$ >> in
  let t = array_get t get in
  let set h k v _ _loc = <:expr< Hashtbl.replace $h$ $k$ $v$ >> in
  array_set t set

let () = associate overloading_hashtbl "Hashtbl"

(***********************************************************************
 *                Overloadings for the module String
 ***********************************************************************)
(* Escape string overloading by other overloadings sets. *)

(* There is not overloading here, it will leave the strings and
   constructions [s.[i]] and [s.[i] <- v] as they are (still
   responsive to -unsafe).  FIXME: What could be nice is some kind of
   optimisation regarding Buffer. *)
let overloading_string =
  let t = lid empty "compare" ~cache:false
    (fun m _loc -> <:expr< String.compare >>) in
  t


let () = associate overloading_string "String"


(***********************************************************************
 *                Overloadings for the module Weak
 ***********************************************************************)

let overloadings_weak =
  let t = lid_subst empty ["blit", "blit"; "create", "create";
                           "fill", "fill"; "length", "length"] in
  let get a i _ _loc = <:expr< Weak.get $a$ $i$ >> in
  let t = array_get t get in
  let set a i v _ _loc = <:expr< Weak.set $a$ $i$ $v$ >> in
  array_set t set


let () = associate overloading_string "Weak"


(* Local Variables: *)
(* compile-command: "omake --no--progress" *)
(* End: *)
