(***********************************************************************)
(*                                                                     *)
(*                              OCamlP3l                               *)
(*                                                                     *)
(*                  projet Estime, INRIA-Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 2003-2007 INRIA.                                         *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  All rights reserved. This file is distributed only by permission.  *)
(*                                                                     *)
(*  Francois Clement <Francois.Clement@inria.fr>,                      *)
(*  Pierre Weis <Pierre.Weis@inria.fr>.                                *)
(*                                                                     *)
(***********************************************************************)

(* $Id: gnuplot.ML,v 1.6 2007/01/23 15:58:18 weis Exp $ *)

open Config;;

(* The gnuplot version number is set by an external shell script *)
let version = XXGPLXX;;
let gnuplot_version_number = version;;

let check_graphical_output = function
| No -> ()
| _ as go -> 
  if version <= 0. then begin
    prerr_endline
      (Printf.sprintf
         "Graphical output %s needs Gnuplot." (string_of_graphical_output go));
    exit 4 end;;

(* The syntax of some gnuplot commands depends on the version number *)
let unset, pointsize =
  let us, ps =
    if version >= 4.0 then
      (fun s -> Printf.sprintf "\nunset %s" s),
      (fun () -> Printf.sprintf "")
    else
      (fun s -> Printf.sprintf "\nset no%s" s),
      (fun () -> Printf.sprintf " pointsize 0.5") in
  (fun () -> us),
  (fun () -> ps);;

(* Generating the gnuplot command *)

let make_isosample () (nsbdx, nsbdy) =
  Printf.sprintf "set isosample %d, %d" (nsbdx + 1) (nsbdy + 1)

and make_term_or_title () (output, res, iter) =
  match output with
  | Postscript ->
      Printf.sprintf
        "%a%a%a%a\
        \nset terminal postscript eps color solid\
        \nset output '%s%d.eps'"
        unset "xtics"
        unset "ytics"
        unset "ztics"
        unset "border"
        res iter
  | _ -> Printf.sprintf "\nset title 'iter = %d'" iter

and make_splot () (res, nsbdx, nsbdy) =
  let s = ref "\nsplot [0:1][0:1][0:0.07]" in
  let to_do n =
    let resn = Printf.sprintf "%s%d" res n
    and resbn = Printf.sprintf "%sB%d" res n in
    if Sys.file_exists resn then Sys.rename resn resbn else
    (Printf.eprintf "File %s not found!\n" resn; flush stderr);
    let file = Printf.sprintf " '%s'" resbn in
    let style =
      Printf.sprintf " notitle with points pointtype 7%a," pointsize () in
    s := !s ^ file ^ style in
  Rdec.for_all_subdomains (nsbdx, nsbdy) to_do;
  !s

and make_reference_surface () output =
  let f, wl = "x*(1-x)*y*(1-y)", "with lines" in
  match output with
  | Postscript -> Printf.sprintf " %s notitle %s linetype 9 linewidth 4\n" f wl
  | _ -> Printf.sprintf " %s %s\n" f wl;;

let make_gnuplot_command output res iter nsbdx nsbdy =
  Printf.sprintf "%a%a%a%a"
    make_isosample (nsbdx, nsbdy)
    make_term_or_title (output, res, iter)
    make_splot (res, nsbdx, nsbdy)
    make_reference_surface output;;
