# -*- coding: utf-8 -*-
#
#  Copyright (C) 2004-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - きのことサーフェスの間に他のウインドウが入ることができてしまうのを直す.
# - NAYUKI/2.0
# - 透明度の設定

import os

import gtk
import glib
import cairo

import ninix.config
import ninix.seriko
import ninix.pix


class Menu:

    def __init__(self, callback, accelgroup):
        self.callback = callback
        ui_info = '''
        <ui>
          <popup name='popup'>
            <menuitem action='Settings'/>
            <menu action='Skin'>
            </menu>
            <separator/>
            <menuitem action='Exit'/>
          </popup>
        </ui>
        '''
        self.__menu_list = {
            'settings': [('Settings', None, ('Settings...(_O)'), None,
                          '', self.callback['edit_preferences']),
                         '/ui/popup/Settings'],
            'skin':     [('Skin', None, _('Skin(_K)'), None),
                         None, '/ui/popup/Skin'],
            'exit':     [('Exit', None, _('Exit(_Q)'), None,
                          '', self.callback['close']),
                         '/ui/popup/Exit'],
            }
        self.__skin_list = None
        actions = gtk.ActionGroup('Actions')
        entry = []
        for value in self.__menu_list.itervalues():
            entry.append(value[0])
        actions.add_actions(tuple(entry))
        ui_manager = gtk.UIManager()
        ui_manager.insert_action_group(actions, 0)
        ui_manager.add_ui_from_string(ui_info)
        self.__popup_menu = ui_manager.get_widget('/ui/popup')
        for key in self.__menu_list:
            path = self.__menu_list[key][-1]
            self.__menu_list[key][1] = ui_manager.get_widget(path)

    def popup(self, button):
        skin_list = self.callback['get_skin_list']()
        self.__set_skin_menu(skin_list)
        self.__popup_menu.popup(
            None, None, None, button, gtk.get_current_event_time())

    def __set_skin_menu(self, list): ## FIXME
        key = 'skin'
        if list:
            menu = gtk.Menu()
            for skin in list:
                item = gtk.MenuItem(skin['title'])
                item.connect('activate', self.callback['select_skin'], (skin))
                menu.add(item)
                item.show()
            self.__menu_list[key][1].set_submenu(menu)
            menu.show()
            self.__menu_list[key][1].show()
        else:
            self.__menu_list[key][1].hide()


class Nayuki:

    def __init__(self):
        pass


class Kinoko:

    def __init__(self, skin_list):
        self.skin_list = skin_list
        self.skin = None

    def edit_preferences(self, action):
        pass

    def finalize(self):
        self.__running = 0
        self.target.delete_observer(self)
        if self.skin is not None:
            self.skin.destroy()

    def notify_observer(self, event, args): ## FIXME
        if self.skin is None:
            return
        if event in ['set position', 'set surface']:
            self.skin.set_position()
            self.skin.show()
        elif event == 'set scale':
            scale = self.target.get_surface_scale()
            self.skin.set_scale(scale)
        elif event == 'hide':
            if not self.target.surface_is_shown(0):
                self.skin.hide()
        elif event == 'iconified':
            self.skin.hide()
        elif event == 'deiconified':
            self.skin.show()
        elif event == 'finalize':
            self.finalize()
        elif event == 'move surface':
            side, xoffset, yoffset = args
            if side == 0:
                self.skin.set_position(xoffset, yoffset)
        elif event == 'raise':
            side = args
            if side == 0:
                self.skin.set_position() ## FIXME
        else:
            ##print 'OBSERVER(kinoko): ignore -', event
            pass

    def load_skin(self):
        scale = self.target.get_surface_scale()
        skin_callback ={
            'edit_preferences': self.edit_preferences,
            'close': self.close,
            'get_skin_list': self.get_skin_list,
            'select_skin': self.select_skin,
            'get_target_window': lambda *a: self.target.surface.window[0].window, # XXX
            'get_kinoko_position': self.target.get_kinoko_position,
            }
        self.skin = Skin(self.data, self.accelgroup, skin_callback, scale)

    def load(self, data, target):
        self.data = data
        self.target = target
        self.target.set_observer(self)
        self.accelgroup = gtk.AccelGroup()
        self.load_skin()
        if self.skin is None:
            return 0
        else:
            self.send_event('OnKinokoObjectCreate')
        self.__running = 1
        glib.timeout_add(10, self.do_idle_tasks) # 10ms
        return 1

    def do_idle_tasks(self):
        return True if self.__running else False

    def close(self, action):
        self.finalize()
        self.send_event('OnKinokoObjectDestroy')

    def send_event(self, event):
        if event not in ['OnKinokoObjectCreate', 'OnKinokoObjectDestroy',
                         'OnKinokoObjectChanging', 'OnKinokoObjectChanged',
                         'OnKinokoObjectInstalled']:
                         ## 'OnBatteryLow', 'OnBatteryCritical',
                         ## 'OnSysResourceLow', 'OnSysResourceCritical'
            return
        args = (self.data['title'],
                self.data['ghost'],
                self.data['category'])
        self.target.notify_event(event, *args)

    def get_skin_list(self):
        return self.skin_list

    def select_skin(self, widget, args):
        self.send_event('OnKinokoObjectChanging')
        self.skin.destroy()
        self.data = args
        self.load_skin()
        if self.skin is None:
            return 0
        else:
            self.send_event('OnKinokoObjectChanged')
        return 1


class Skin:

    def __init__(self, data, accelgroup, callback, scale):
        self.frame_buffer = []
        self.data = data
        self.accelgroup = accelgroup
        self.callback = callback
        menu_callback ={}
        for key in ['edit_preferences', 'close',
                    'get_skin_list', 'select_skin']:
            menu_callback[key] = self.callback[key]
        self.__menu = Menu(menu_callback, self.accelgroup)
        self.__scale = scale
        self.__shown = 0
        self.surface_id = 0 # dummy
        self.window = ninix.pix.TransparentWindow()
        self.window.set_focus_on_map(False)
        ##self.window.set_title(''.join(('surface.', name)))
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.set_skip_taskbar_hint(True)
        self.window.connect('delete_event', self.delete)
        self.window.realize()
        self.window.window.set_back_pixmap(None, False)
        self.window.add_accel_group(self.accelgroup) ## FIXME
        if self.data['animation'] is not None:
            path = os.path.join(self.data['dir'], self.data['animation'])
            self.seriko = ninix.seriko.Controler(
                {'': ninix.seriko.get_actors(ninix.config.create_from_file(path))})
        else:
            base, ext = os.path.splitext(self.data['base'])
            path = os.path.join(self.data['dir'], ''.join((base, 'a.txt')))
            if os.path.exists(path):
                self.seriko = ninix.seriko.Controler(
                    {'': ninix.seriko.get_actors(ninix.config.create_from_file(path))})
            else:
                self.seriko = ninix.seriko.Controler({'': []})
        path = os.path.join(self.data['dir'], self.data['base'])
        try:
            self.pixbuf = ninix.pix.create_pixbuf_from_file(path)
            w = max(8, self.pixbuf.get_width() * self.__scale / 100)
            h = max(8, self.pixbuf.get_height() * self.__scale / 100)
            surface_pixbuf = self.pixbuf.scale_simple(
                w, h, gtk.gdk.INTERP_BILINEAR)
            mask_pixmap = gtk.gdk.Pixmap(None, w, h, 1)
            surface_pixbuf.render_threshold_alpha(
                mask_pixmap, 0, 0, 0, 0, w, h, 1)
        except: ## FIXME
            self.callback['close'](None)
            return
        self.w, self.h = w, h
        self.current_surface_pixbuf = surface_pixbuf
        self.current_mask = mask_pixmap
        self.set_position()
        if self.window.is_composited():
            self.window.input_shape_combine_mask(mask_pixmap, 0, 0)
        else:
            self.window.shape_combine_mask(mask_pixmap, 0, 0)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                               gtk.gdk.BUTTON_PRESS_MASK|
                               gtk.gdk.BUTTON_RELEASE_MASK|
                               gtk.gdk.POINTER_MOTION_MASK|
                               gtk.gdk.LEAVE_NOTIFY_MASK)
        self.darea.connect('button_press_event', self.button_press)
        self.darea.connect('button_release_event', self.button_release)
        self.darea.connect('motion_notify_event', self.motion_notify)
        self.darea.connect('leave_notify_event', self.leave_notify)
        self.darea.connect('expose_event', self.redraw)
        self.darea.set_size_request(self.w, self.h)
        self.darea.show()
        self.window.add(self.darea)
        target_window = self.callback['get_target_window']()
        if self.data['ontop']:
            self.window.set_transient_for(target_window)
        else:
            target_window.set_transient_for(self.window)
        self.show()
        self.seriko.reset(self, '') # XXX
        self.seriko.start(self)
        self.seriko.invoke_kinoko(self)

    def show(self):
        if not self.__shown:
            self.window.show()
            self.__shown = 1

    def hide(self):
        if self.__shown:
            self.window.hide()
            self.__shown = 0

    def append_actor(self, frame, actor):
        self.seriko.append_actor(frame, actor)

    def set_position(self, xoffset=0, yoffset=0):
        base_x, base_y = self.callback['get_kinoko_position'](
            self.data['baseposition'])
        a, b = [(0.5, 1), (0.5, 0), (0, 0.5), (1, 0.5), (0, 1),
                (1, 1), (0, 0), (1, 0), (0.5, 0.5)][self.data['baseadjust']]
        offsetx = self.data['offsetx'] * self.__scale / 100
        offsety = self.data['offsety'] * self.__scale / 100
        self.x = base_x - int(self.w * a) + offsetx + xoffset
        self.y = base_y - int(self.h * b) + offsety + yoffset
        self.window.move(self.x, self.y) ## FIXME: same as surfaces

    def set_scale(self, scale):
        self.__scale = scale
        self.set_position()

    def get_surface(self): ## FIXME
        return None

    def redraw(self, darea, event):
        cr = darea.window.cairo_create()
        cr.save()
        cr.set_operator(cairo.OPERATOR_CLEAR)
        cr.paint()
        cr.restore()
        cr.set_source_pixbuf(self.current_surface_pixbuf, 0, 0)
        ##cr.paint_with_alpha(self.__alpha_channel)
        cr.paint()
        del cr

    def get_pixbuf(self, surface_id):
        path = os.path.join(self.data['dir'], ''.join(('surface', str(surface_id), '.png')))
        if os.path.exists(path):
            pixbuf = ninix.pix.create_pixbuf_from_file(path)
        else:
            pixbuf = None
        return pixbuf

    def create_surface_pixbuf(self, surface_id=None):
        if surface_id is not None and surface_id != '':
            surface_pixbuf = self.get_pixbuf(surface_id)
        else:
            surface_pixbuf = self.pixbuf.copy()
        return surface_pixbuf

    def update_frame_buffer(self):
        surface_pixbuf = self.create_surface_pixbuf(self.seriko.base_id)
        assert surface_pixbuf is not None
        # draw overlays
        for pixbuf_id, x, y in self.seriko.iter_overlays():
            try:
                pixbuf = self.get_pixbuf(pixbuf_id)
                w = pixbuf.get_width()
                h = pixbuf.get_height()
            except:
                continue
            # overlay surface pixbuf
            sw = surface_pixbuf.get_width()
            sh = surface_pixbuf.get_height()
            if x + w > sw:
                w = sw - x
            if y + h > sh:
                h = sh - y
            if x < 0:
                dest_x = 0
                w += x
            else:
                dest_x = x
            if y < 0:
                dest_y = 0
                h += y
            else:
                dest_y = y
            pixbuf.composite(surface_pixbuf, dest_x, dest_y,
                             w, h, x, y, 1.0, 1.0,
                             gtk.gdk.INTERP_BILINEAR, 255)
        w = max(8, surface_pixbuf.get_width() * self.__scale / 100)
        h = max(8, surface_pixbuf.get_height() * self.__scale / 100)
        surface_pixbuf = surface_pixbuf.scale_simple(
            w, h, gtk.gdk.INTERP_BILINEAR)
        mask_pixmap = gtk.gdk.Pixmap(None, w, h, 1)
        surface_pixbuf.render_threshold_alpha(
            mask_pixmap, 0, 0, 0, 0, w, h, 1)
        if self.window.is_composited():
            self.window.input_shape_combine_mask(mask_pixmap, 0, 0)
        else:
            self.window.shape_combine_mask(mask_pixmap, 0, 0)
        #print 'DRAW:', dir(self.darea)
        #self.darea.queue_draw_area(0, 0, w, h)
        self.current_mask = mask_pixmap
        self.current_surface_pixbuf = surface_pixbuf
        self.darea.queue_draw()

    def terminate(self):
        self.seriko.terminate(self)

    def add_overlay(self, actor, surface_id, x, y):
        self.seriko.add_overlay(self, actor, surface_id, x, y)

    def remove_overlay(self, actor):
        self.seriko.remove_overlay(actor)

    def move_surface(self, xoffset, yoffset): ## FIXME
        self.window.move(self.x + xoffset, self.y + yoffset)

    def set_surface(self, surface_id, restart=1): ## FIXME
        self.pixbuf = self.get_pixbuf(surface_id)

    def invoke(self, actor_id, update=0):
        self.seriko.invoke(self, actor_id, update)

    def delete(self, widget, event):
        self.callback['close'](None)

    def destroy(self):
        self.seriko.destroy()
        self.window.destroy()

    def button_press(self, widget, event): ## FIXME
        self.x_root = event.x_root
        self.y_root = event.y_root
        if event.type == gtk.gdk.BUTTON_PRESS:
            click = 1
        else:
            click = 2
        button = event.button
        if button == 3 and click == 1:
            self.__menu.popup(button)
        return True

    def button_release(self,  widget, event): ## FIXME
        pass

    def motion_notify(self,  widget, event): ## FIXME
        pass

    def leave_notify(self,  widget, event): ## FIXME
        pass
