# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import logging
import os
import re
import sys

import ninix.config
import ninix.alias
import ninix.dll


def get_ninix_home():
    return os.path.join(os.path.expanduser('~'), '.ninix')

def get_archive_dir():
    return os.path.join(get_ninix_home(), 'archive')

def get_pango_fontrc():
    return os.path.join(get_ninix_home(), 'pango_fontrc')

def get_preferences():
    return os.path.join(get_ninix_home(), 'preferences')

def get_normalized_path(path, encode=1):
    path = path.replace('\\', '/')
    if encode:
        path = path.encode('utf-8')
    if not os.path.isabs(path): # XXX
        path = path.lower()
    return os.path.normpath(path)

def load_config():
    if not os.path.exists(get_ninix_home()):
        return None
    ghosts = search_ghosts()
    balloons = search_balloons()
    nekoninni = search_nekoninni()
    katochan = search_katochan()
    kinoko = search_kinoko()
    return ghosts, balloons, nekoninni, katochan, kinoko

def get_shiori(path):
    table = {}
    shiori_lib = ninix.dll.Library('shiori', path, saori_lib=None)
    for filename in os.listdir(path):
        if os.access(os.path.join(path, filename), os.R_OK):
            name = None
            basename, ext = os.path.splitext(filename)
            ext = ext.lower()
            if ext in ['.py', '.pyc']:
                name = basename
            if name and name not in table:
                shiori = shiori_lib.request(('', name))
                if shiori:
                    table[name] = shiori
    return table

def search_ghosts(target=None):
    home_dir = get_ninix_home()
    ghosts = []
    if target:
        dirlist = []
        dirlist.extend(target)
    else:
        try:
            dirlist = os.listdir(os.path.join(home_dir, 'ghost'))
        except OSError:
            dirlist = []
    shiori_table = get_shiori(os.path.join(sys.path[0], 'ninix/dll')) # XXX
    for subdir in dirlist:
        prefix = os.path.join(home_dir, 'ghost', subdir)
        ghost_dir = os.path.join(prefix, 'ghost', 'master')
        desc = read_descript_txt(ghost_dir)
        if desc is None:
            desc = ninix.config.null_config()
        shiori_dll = desc.get('shiori')
        # find a pseudo AI, shells, and a built-in balloon
        candidate = {'name': '', 'score': 0}
        # SHIORI compatible modules
        for name, shiori in shiori_table.iteritems():
            score = int(shiori.find(ghost_dir, shiori_dll))
            if score > candidate['score']:
                candidate['name'] = name
                candidate['score'] = score
        shell_name, surface_set = find_surface_set(prefix)
        if candidate['score'] == 0:
            continue
        shiori_name = candidate['name']
        pos = 0 if desc.get('name') == 'default' else len(ghosts)
        use_makoto = find_makoto_dll(ghost_dir)
        ## FIXME: check surface_set
        ghosts.insert(pos, (
            desc, ghost_dir, use_makoto, surface_set, prefix,
            shiori_dll, shiori_name))
    return ghosts

def search_balloons():
    home_dir = get_ninix_home()
    balloon_dir = os.path.join(home_dir, 'balloon')
    try:
        dirlist = os.listdir(balloon_dir)
    except OSError:
        dirlist = []
    buf = []
    for subdir in dirlist:
        path = os.path.join(balloon_dir, subdir)
        if not os.path.isdir(path):
            continue
        desc = read_descript_txt(path) # REQUIRED
        if not desc:
            continue
        balloon_info = read_balloon_info(path) # REQUIRED
        if not balloon_info:
            continue
        if 'balloon_dir' in balloon_info: # XXX
            logging.warninig('Oops: balloon id confliction')
            continue
        else:
            balloon_info['balloon_dir'] = (subdir, ninix.config.null_config())
        pos = 0 if desc.get('name') == 'default' else len(buf)
        buf.insert(pos, (desc, balloon_info))
    return buf

def search_nekoninni():
    home_dir = get_ninix_home()
    buf = []
    skin_dir = os.path.join(home_dir, 'nekodorif/skin')
    try:
        dirlist = os.listdir(skin_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        nekoninni = read_profile_txt(os.path.join(skin_dir, subdir))
        if nekoninni is None:
            continue
        buf.append(nekoninni)
    return buf

def search_katochan():
    home_dir = get_ninix_home()
    buf = []
    katochan_dir = os.path.join(home_dir, 'nekodorif/katochan')
    try:
        dirlist = os.listdir(katochan_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        katochan = read_katochan_txt(os.path.join(katochan_dir, subdir))
        if katochan is None:
            continue
        buf.append(katochan)
    return buf

def search_kinoko():
    home_dir = get_ninix_home()
    buf = []
    kinoko_dir = os.path.join(home_dir, 'kinoko')
    try:
        dirlist = os.listdir(kinoko_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        kinoko = read_kinoko_ini(os.path.join(kinoko_dir, subdir))
        if kinoko is None:
            continue
        buf.append(kinoko)
    return buf

def read_kinoko_ini(top_dir):
    path = os.path.join(top_dir, 'kinoko.ini')
    kinoko = {}
    kinoko['base'] = 'surface0.png'
    kinoko['animation'] = None
    kinoko['category'] = None
    kinoko['title'] = None
    kinoko['ghost'] = None
    kinoko['dir'] = top_dir
    kinoko['offsetx'] = 0
    kinoko['offsety'] = 0
    kinoko['ontop'] = 0
    kinoko['baseposition'] = 0
    kinoko['baseadjust'] = 0
    kinoko['extractpath'] = None
    kinoko['nayuki'] = None
    if os.access(path, os.R_OK):
        with open(path) as f:
            line = f.readline()
            if not line.strip() or line.strip() != '[KINOKO]':
                return None
            lineno = 0
            error = None
            for line in f:
                lineno += 1
                if line.endswith(chr(0)): # XXX
                    line = line[:-1]
                if not line.strip():
                    continue
                if '=' not in line:
                    error = 'line %d: syntax error' % lineno
                    break
                name, value = [x.strip() for x in line.split('=', 1)]
                if name in ['title', 'ghost', 'category']:
                    kinoko[name] = unicode(value, 'Shift_JIS', 'ignore')
                elif name in ['offsetx', 'offsety']:
                    kinoko[name] = int(value)
                elif name in ['base', 'animation', 'extractpath']:
                    kinoko[name] = value
                elif name in ['ontop', 'baseposition', 'baseadjust']:
                    kinoko[name] = int(value)
        if error:
            logging.error('Error: %s\n%s (skipped)' % (error, path))
            return None
    return kinoko if kinoko['title'] else None

def read_profile_txt(top_dir):
    path = os.path.join(top_dir, 'profile.txt')
    name = None
    if os.access(path, os.R_OK):
        with open(path) as f:
            line = f.readline()
        if line:
            name = unicode(line.strip(), 'Shift_JIS', 'ignore')
    if name:
        return (name, top_dir) ## FIXME
    else:
        return None

def read_katochan_txt(top_dir):
    path = os.path.join(top_dir, 'katochan.txt')
    katochan = {}
    katochan['dir'] = top_dir
    if os.access(path, os.R_OK):
        with open(path) as f:
            name = None
            lineno = 0
            error = None
            for line in f:
                lineno += 1
                if not line.strip():
                    continue
                if line.startswith('#'):
                    name = line[1:].strip()
                    continue
                elif not name:
                    error = 'line %d: syntax error' % lineno
                    break
                else:
                    value = line.strip()
                    if name in ['name', 'category']:
                        katochan[name] = unicode(value, 'Shift_JIS', 'ignore')
                    if name.startswith('before.script') or \
                            name.startswith('hit.script') or \
                            name.startswith('after.script') or \
                            name.startswith('end.script') or \
                            name.startswith('dodge.script'):
                        ## FIXME: should be array
                        katochan[name] = unicode(value, 'Shift_JIS', 'ignore')
                    elif name in ['before.fall.speed',
                                  'before.slide.magnitude',
                                  'before.slide.sinwave.degspeed',
                                  'before.appear.ofset.x',
                                  'before.appear.ofset.y',
                                  'hit.waittime', 'hit.ofset.x', 'hit.ofset.y',
                                  'after.fall.speed', 'after.slide.magnitude',
                                  'after.slide.sinwave.degspeed']:
                        katochan[name] = int(value)
                    elif name in ['target',
                                  'before.fall.type', 'before.slide.type',
                                  'before.wave', 'before.wave.loop',
                                  'before.appear.direction',
                                  'hit.wave', 'hit.wave.loop',
                                  'after.fall.type', 'after.slide.type',
                                  'after.wave', 'after.wave.loop',
                                  'end.wave', 'end.wave.loop',
                                  'end.leave.direction',
                                  'dodge.wave', 'dodge.wave.loop']:
                        katochan[name] = value
                    else:
                        name = None
        if error:
            logging.error('Error: %s\n%s (skipped)' % (error, path))
            return None
    return katochan if katochan['name'] else None

def read_descript_txt(top_dir):
    path = os.path.join(top_dir, 'descript.txt')
    if os.access(path, os.R_OK):
        return ninix.config.create_from_file(path)
    return None

def read_install_txt(top_dir):
    path = os.path.join(top_dir, 'install.txt')
    if os.access(path, os.R_OK):
        return ninix.config.create_from_file(path)
    return None

def read_alias_txt(top_dir):
    path = os.path.join(top_dir, 'alias.txt')
    if os.access(path, os.R_OK):
        return ninix.alias.create_from_file(path)
    return None

def find_makoto_dll(top_dir):
    return 1 if os.access(os.path.join(top_dir, 'makoto.dll'), os.R_OK) else 0

def find_surface_set(top_dir):
    desc = read_descript_txt(os.path.join(top_dir, 'ghost', 'master'))
    shell_name = desc.get('name') if desc else None
    if not shell_name:
        inst = read_install_txt(top_dir)
        if inst:
            shell_name = inst.get('name')
    surface_set = []
    shell_dir = os.path.join(top_dir, 'shell')
    for name, desc, subdir in find_surface_dir(shell_dir):
        surface_dir = os.path.join(shell_dir, subdir)
        surface_info, alias, tooltips = read_surface_info(surface_dir)
        if surface_info and \
           'surface0' in surface_info and 'surface10' in surface_info:
            if alias is None:
                alias = read_alias_txt(surface_dir)
            surface_set.append((name, surface_dir, desc, alias, surface_info,
                                tooltips))
    return shell_name, surface_set

def find_surface_dir(top_dir):
    buf = []
    path = os.path.join(top_dir, 'surface.txt')
    if os.path.exists(path):
        config = ninix.config.create_from_file(path)
        for name, subdir in config.items():
            subdir = subdir.lower()
            desc = read_descript_txt(os.path.join(top_dir, subdir))
            if desc is None:
                desc = ninix.config.null_config()
            buf.append((name, desc, subdir))
    else:
        try:
            dirlist = os.listdir(top_dir)
        except OSError:
            dirlist = []
        for subdir in dirlist:
            desc = read_descript_txt(os.path.join(top_dir, subdir))
            if desc is None:
                desc = ninix.config.null_config()
            name = desc.get('name', subdir)
            buf.append((name, desc, subdir))
    return buf

re_surface = re.compile('surface([0-9]+)\.(png|dgp|ddp)')

def read_surface_info(surface_dir):
    surface = {}
    try:
        filelist = os.listdir(surface_dir)
    except OSError:
        filelist = []
    filename_alias = {}
    path = os.path.join(surface_dir, 'alias.txt')
    if os.path.exists(path):
        dic = ninix.alias.create_from_file(path)
        for basename, alias in dic.iteritems():
            if basename.startswith('surface'):
                filename_alias[alias] = basename
    # find png image and associated configuration file
    for filename in filelist:
        basename, ext = os.path.splitext(filename)
        if basename in filename_alias:
            match = re_surface.match(
                ''.join((filename_alias[basename], ext)))
        else:
            match = re_surface.match(filename)
        if not match:
            continue
        img = os.path.join(surface_dir, filename)
        if not os.access(img, os.R_OK):
            continue
        key = ''.join(('surface', str(int(match.group(1)))))
        txt = os.path.join(surface_dir, ''.join((basename, 's.txt')))
        if os.access(txt, os.R_OK):
            config = ninix.config.create_from_file(txt)
        else:
            config = ninix.config.null_config()
        txt = os.path.join(surface_dir, ''.join((basename, 'a.txt')))
        if os.access(txt, os.R_OK):
            config.update(ninix.config.create_from_file(txt))
        surface[key] = (img, config)
    # find surfaces.txt
    alias = None
    tooltips = {}
    for key, config in read_surfaces_txt(surface_dir):
        if key == '__alias__':
            alias = config
        elif key == '__tooltips__':
            tooltips = config
        elif key.startswith('surface'):
            try:
                img, null_config = surface[key]
            except KeyError:
                img = None
            surface[key] = (img, config)
    # find surface elements
    for key, (img, config) in surface.items():
        for key, method, filename, x, y in list_surface_elements(config):
            filename = filename.lower()
            basename, ext = os.path.splitext(filename)
            if basename not in surface:
                surface[basename] = (os.path.join(surface_dir, filename),
                                     ninix.config.null_config())
    return surface, alias, tooltips

def read_surfaces_txt(surface_dir):
    config_list = []
    path = os.path.join(surface_dir, 'surfaces.txt')
    try:
        with open(path) as f:
            alias_buffer = []
            tooltips = {}
            charset = 'Shift_JIS'
            while 1:
                line = f.readline()
                if not line:
                    break
                if line.startswith('#') or line.startswith('//'):
                    continue
                key = line.strip()
                if key.startswith('charset'):
                    try:
                        charset = line.split(',', 1)[1].strip()
                    except:
                        pass
                    continue
                if not key:
                    continue
                try:
                    while 1:
                        line = f.readline()
                        if not line:
                            raise ValueError, 'unexpected end of file'
                        line = line.replace('\x81\x40', '').strip()
                        if not line:
                            continue
                        elif line == '{':
                            break
                        key = line # ignore the preceding key
                    buf = []
                    while 1:
                        line = f.readline()
                        if not line:
                            raise ValueError, 'unexpected end of file'
                        line = line.replace('\x81\x40', '').strip()
                        if not line:
                            continue
                        elif line == '}':
                            break
                        buf.append(line)
                    if key in ['sakura.surface.alias', 'kero.surface.alias']:
                        alias_buffer.append(key)
                        alias_buffer.append('{')
                        alias_buffer.extend(buf)
                        alias_buffer.append('}')
                    elif key.endswith('.tooltips'):
                        try:
                            key = key[:-9]
                        except:
                            pass
                        value = {}
                        for line in buf:
                            line = line.split(',', 1)
                            region, text = [unicode(s.strip(), charset, 'ignore') for s in line]
                            value[region] = text
                            tooltips[key] = value
                    elif key.startswith('surface'):
                        try:
                            key = ''.join((key[:7], str(int(key[7:]))))
                        except ValueError:
                            pass
                        config_list.append((key, ninix.config.create_from_buffer(buf)))
                except ValueError, error:
                    logging.error(
                        '%s: %s (parsing not completed)' % (path, error))
                    break
    except IOError:
        return config_list
    if alias_buffer:
        config_list.append(('__alias__', ninix.alias.create_from_buffer(alias_buffer)))
    config_list.append(('__tooltips__', tooltips))
    return config_list

def list_surface_elements(config):
    buf = []
    for n in range(256):
        key = ''.join(('element', str(n)))
        if key not in config:
            break
        spec = [value.strip() for value in config[key].split(',')]
        try:
            method, filename, x, y = spec
            x = int(x)
            y = int(y)
        except ValueError:
            logging.error(
                'invalid element spec for %s: %s' % (key, config[key]))
            continue
        buf.append((key, method, filename, x, y))
    return buf

re_balloon = re.compile('balloon([skc][0-9]+)\.(png)')
re_annex   = re.compile('(arrow[01]|sstp)\.(png)')

def read_balloon_info(balloon_dir):
    balloon = {}
    try:
        filelist = os.listdir(balloon_dir)
    except OSError:
        filelist = []
    for filename in filelist:
        match = re_balloon.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if match.group(2) != 'png' and \
           os.access(''.join((img[-3:], 'png')), os.R_OK):
                continue
        if not os.access(img, os.R_OK):
            continue
        key = match.group(1)
        txt = os.path.join(balloon_dir, 'balloon%ss.txt' % key)
        if os.access(txt, os.R_OK):
            config = ninix.config.create_from_file(txt)
        else:
            config = ninix.config.null_config()
        balloon[key] = (img, config)
    for filename in filelist:
        match = re_annex.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if not os.access(img, os.R_OK):
            continue
        key = match.group(1)
        config = ninix.config.null_config()
        balloon[key] = (img, config)
    return balloon


###   TEST   ###

def test():
    import locale
    locale.setlocale(locale.LC_ALL, '')
    config = load_config()
    if config is None:
        raise SystemExit, 'Home directory not found.\n'
    ghosts, balloons, nekoninni, katochan, kinoko = config
    # ghosts
    for desc, shiori_dir, use_makoto, surface_set, prefix, shiori_dll, shiori_name in ghosts:
        print 'GHOST', '=' * 50
        print prefix
        print str(desc).encode('utf-8', 'ignore')
        print shiori_dir
        print shiori_dll
        print shiori_name
        print 'use_makoto =', use_makoto
        if surface_set:
            for name, surface_dir, desc, alias, surface, tooltips in surface_set:
                print '-' * 50
                print 'surface:', name.encode('utf-8', 'ignore')
                print str(desc).encode('utf-8', 'ignore')
                for k, v in surface.iteritems():
                    print k, '=', v[0]
                    print str(v[1]).encode('utf-8', 'ignore')
                if alias:
                    buf = []
                    for k, v in alias.iteritems():
                        if k in ['sakura.surface.alias', 'kero.surface.alias']:
                            print ''.join((k, ':'))
                            for alias_id, alias_list in v.iteritems():
                                print alias_id, \
                                      ''.join(('= [', ', '.join(alias_list), ']'))
                            print
                        else:
                            buf.append((k, v))
                    if buf:
                        print 'filename alias:'
                        for k, v in buf:
                            print k, '=', v
                        print
    # balloons
    for desc, balloon in balloons:
        print 'BALLOON', '=' * 50
        print str(desc).encode('utf-8', 'ignore')
        for k, v in balloon.iteritems():
            print k, '=', v[0]
            print str(v[1]).encode('utf-8', 'ignore')
    ## FIXME
    # kinoko
    # nekoninni
    # katochan


if __name__ == '__main__':
    test()
