# -*- coding: utf-8 -*-
#
#  ninix-install.py - a command-line installer for ninix
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import distutils.dir_util
import getopt
import os
import shutil
import stat
import sys
import logging
import tempfile
import urllib
import zipfile

import gettext
gettext.install('ninix')

import ninix.home
import ninix.version


PROGRAM_NAME = os.path.basename(sys.argv[0])

USAGE = '''\
Usage: %s [options] [file ...]
Options:
  -S, --supplement NAME  the name of the supplement target
  -i, --interactive      prompt before overwrite
  -d, --download         download files into the archive directory
  -L, --lower            lower file names with upper case letters
  -r, --rebuild          rebuild ninix home directory (transition from the release of 4.1.3 or older)
  -h, --help             show this message
'''

def usage():
    raise SystemExit, USAGE % PROGRAM_NAME


class InstallError(Exception):
    pass


def fatal(error):
    raise InstallError, error

# mode masks
INTERACTIVE = 1
DOWNLOAD    = 2

def main():
    global mode, arcdir
    try:
        options, files = getopt.getopt(
            sys.argv[1:], 'S:idLrh',
            ['supplement=', 'interactive', 'download',
             'lower', 'rebuild', 'help'])
    except getopt.error, e:
        logging.error('%s' % str(e))
        usage()
    target = None
    mode = 0
    to_lower = 0
    rebuild = 0
    host = ''
    for opt, val in options:
        if opt in ['-h', '--help']:
            usage()
        elif opt in ['-S', '--supplement']:
            target = val
        elif opt in ['-i', '--interactive']:
            mode = mode | INTERACTIVE
        elif opt in ['-d', '--download']:
            mode = mode | DOWNLOAD
        elif opt in ['-L', '--lower']:
            to_lower = 1
        elif opt in ['-r', '--rebuild']:
            rebuild = 1
    arcdir = ninix.home.get_archive_dir()
    if to_lower:
        n = 0
        homedir = ninix.home.get_ninix_home()
        for data in ['ghost', 'balloon']:
            try:
                buf = os.listdir(os.path.join(homedir, data))
            except OSError:
                continue
            for subdir in buf:
                path = os.path.join(homedir, data, subdir)
                if os.path.isdir(path):
                    n += lower_files(path)
        print n, 'files renamed'
        raise SystemExit
    if rebuild:
        mode |= INTERACTIVE
        n = 0
        homedir = ninix.home.get_ninix_home()
        ghost_top = os.path.join(homedir, 'ghost')
        balloon_top = os.path.join(homedir, 'balloon')
        try:
            buf = os.listdir(ghost_top)
        except OSError:
            raise SystemExit
        for subdir in buf:
            ghost_dir = os.path.join(ghost_top, subdir)
            if os.path.isdir(ghost_dir):
                inst = ninix.home.read_install_txt(ghost_dir)
                if inst is None:
                    continue
                balloon_dir = inst.get('balloon.directory')
                if balloon_dir:
                    balloon_dir = ninix.home.get_normalized_path(balloon_dir)
                    path = os.path.join(ghost_dir, 'ghost/master', balloon_dir)
                    if os.path.exists(path):
                        balloon_dst = os.path.join(balloon_top, balloon_dir)
                        if os.path.exists(balloon_dst):
                            # uninstall older versions of the balloon
                            if confirm_removal(balloon_dst):
                                remove_files_and_dirs([], balloon_dst)
                            else:
                                continue
                        os.rename(path, balloon_dst)
                        # create SETTINGS
                        settings = os.path.join(ghost_dir, 'SETTINGS')
                        if not os.path.exists(settings):
                            try:
                                with open(settings, 'w') as f:
                                    f.write('balloon_directory, %s\n' % balloon_dir)
                            except IOError, (code, message):
                                logging.error('cannot write %s' % settings)
                        n += 1
                ##os.remove(os.path.join(path, 'install.txt'))
        print n, 'ghosts updated'
        raise SystemExit
    for filename in files:
        try:
            install(filename, target, ninix.home.get_ninix_home())
        except InstallError, error:
            logging.error('%s: %s (skipped)' % (filename, error))

def install(filename, target, homedir):
    # check archive format
    basename, ext = os.path.splitext(filename)
    ext = ext.lower()
    if ext in ['.nar', '.zip']:
        pass
    else:
        fatal('unknown archive format')
    # extract files from the archive
    tmpdir = tempfile.mkdtemp('ninix')
    shutil.rmtree(tmpdir) # XXX
    try:
        os.makedirs(tmpdir)
    except:
        fatal('cannot make temporary directory')
    url = None
    if filename.startswith('http:') or filename.startswith('ftp:'):
        url = filename
        filename = download(filename, tmpdir)
        if filename is None:
            shutil.rmtree(tmpdir)
            fatal('cannot download the archive file')
    zf = zipfile.ZipFile(filename)
    try:
        zf.extractall(tmpdir)
        zf.close()
    except:
        shutil.rmtree(tmpdir)
        fatal('cannot extract files from the archive')
    if url and not mode & DOWNLOAD:
        os.remove(filename)
    for (dirpath, dirnames, filenames) in os.walk(tmpdir):
        for name in dirnames:
            path = os.path.join(dirpath, name)
            st_mode = os.stat(path).st_mode
            os.chmod(path, st_mode | stat.S_IWUSR | stat.S_IRUSR | stat.S_IXUSR)
        for name in filenames:
            path = os.path.join(dirpath, name)
            st_mode = os.stat(path).st_mode
            os.chmod(path, st_mode | stat.S_IWUSR | stat.S_IRUSR)
    rename_files(tmpdir)
    # check the file type
    inst = ninix.home.read_install_txt(tmpdir)
    if inst:
        filetype = inst.get('type')
        if filetype == 'ghost':
            if os.path.exists(os.path.join(tmpdir, 'ghost', 'master')):
                filetype = 'ghost.redo'
            else:
                filetype = 'ghost.inverse'
        elif filetype == 'ghost with balloon':
            filetype = 'ghost.inverse'
        elif filetype in ['shell', 'shell with balloon']:
            if 'accept' in inst:
                filetype = 'supplement'
            else:
                filetype = 'shell.inverse'
        elif filetype == 'balloon':
            pass
        elif filetype == 'skin':
            filetype = 'nekoninni'
        elif filetype == 'katochan':
            pass
        else:
            fatal('unsupported file type(%s)' % filetype)
    else:
        fatal('cannot read install.txt from the archive')
    if filetype in ['shell.inverse', 'ghost.inverse']:
        fatal('unsupported file type(%s)' % filetype)
    try:
        if filetype == 'ghost.redo':
            install_redo_ghost(filename, tmpdir, homedir)
        elif filetype == 'supplement':
            install_redo_supplement(filename, tmpdir, homedir, target)
        elif filetype == 'balloon':
            install_balloon(filename, tmpdir, homedir)
        elif filetype == 'kinoko':
            install_kinoko(filename, tmpdir, homedir)
        elif filetype == 'nekoninni':
            install_nekoninni(filename, tmpdir, homedir)
        elif filetype == 'katochan':
            install_katochan(filename, tmpdir, homedir)
    finally:
        shutil.rmtree(tmpdir)


class URLopener(urllib.FancyURLopener):
    version = 'ninix-aya/%s' % ninix.version.VERSION


urllib._urlopener = URLopener()

def download(url, basedir): ## FIXME
    print 'downloading', url
    try:
        ifile = urllib.urlopen(url)
    except IOError:
        return None
    headers = ifile.info()
    if 'content-length' in headers:
        print '(size = %s bytes)' % headers.get('content-length')
    if mode & DOWNLOAD:
        if not os.path.exists(arcdir):
            os.makedirs(arcdir)
        basedir = arcdir
    filename = os.path.join(basedir, os.path.basename(url))
    try:
        with open(filename, 'wb') as ofile:
            while 1:
                data = ifile.read(4096)
                if not data:
                    break
                ofile.write(data)
    except IOError:
        return None
    ifile.close()
    # check the format of the downloaded file
    basename, ext = os.path.splitext(filename)
    ext = ext.lower()
    if ext in ['.nar', '.zip']:
        try:
            zf = zipfile.ZipFile(filename)
        except:
            return None
        test_zip = zf.testzip()
        zf.close()
        return None if test_zip is not None else filename
    else:
        fatal('unknown archive format')

def rename_files(basedir):
    if os.name == 'nt': # XXX
        return
    for filename in os.listdir(basedir):
        filename2 = filename.lower()
        path = os.path.join(basedir, filename2)
        if filename != filename2:
            os.rename(os.path.join(basedir, filename), path)
        if os.path.isdir(path):
            rename_files(path)

def list_all_directories(top, basedir):
    dirlist = []
    for path in os.listdir(os.path.join(top, basedir)):
        if os.path.isdir(os.path.join(top, basedir, path)):
            dirlist.extend(list_all_directories(top, os.path.join(basedir, path)))
            dirlist.append(os.path.join(basedir, path))
    return dirlist

def remove_files_and_dirs(mask, target_dir):
    path = os.path.abspath(target_dir)
    if not os.path.isdir(path):
        return
    os.path.walk(path, remove_files, mask)
    dirlist = list_all_directories(path, '')
    dirlist.sort()
    dirlist.reverse()
    for name in dirlist:
        current_path = os.path.join(path, name)
        if os.path.isdir(current_path):
            head, tail = os.path.split(current_path)
            if tail not in mask and not os.listdir(current_path):
                shutil.rmtree(current_path)

def remove_files(mask, top_dir, filelist):
    for name in filelist:
        path = os.path.join(top_dir, name)
        if os.path.isdir(path) or name in mask:
            pass
        else:
            os.remove(path)

def lower_files(top_dir):
    if os.name == 'nt': # XXX
        return    
    n = 0
    for filename in os.listdir(top_dir):
        filename2 = filename.lower()
        path = os.path.join(top_dir, filename2)
        if filename != filename2:
            os.rename(os.path.join(top_dir, filename), path)
            print 'renamed', os.path.join(top_dir, filename)
            n += 1
        if os.path.isdir(path):
            n += lower_files(path)
    return n

def confirm_overwrite(path):
    if not mode & INTERACTIVE:
        return 1
    print ''.join((PROGRAM_NAME, ': overwrite "%s"? (yes/no)' % path))
    try:
        answer = raw_input()
    except EOFError:
        answer = None
    except KeyboardInterrupt:
        raise SystemExit
    if not answer:
        return 0
    return answer.lower().startswith('y')

def confirm_removal(path):
    if not mode & INTERACTIVE:
        return 1
    print ''.join((PROGRAM_NAME, ': remove "%s"? (yes/no)' % path))
    try:
        answer = raw_input()
    except EOFError:
        answer = None
    except KeyboardInterrupt:
        raise SystemExit
    if not answer:
        return 0
    return answer.lower().startswith('y')

def install_redo_ghost(archive, tmpdir, homedir):
    global mode
    # find install.txt
    inst = ninix.home.read_install_txt(tmpdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    prefix = os.path.join(homedir, 'ghost', target_dir)
    ghost_src = os.path.join(tmpdir, 'ghost', 'master')
    shell_src = os.path.join(tmpdir, 'shell')
    ghost_dst = os.path.join(prefix, 'ghost', 'master')
    shell_dst = os.path.join(prefix, 'shell')
    filelist = []
    ##filelist.append((os.path.join(tmpdir, 'install.txt'),
    ##                 os.path.join(prefix, 'install.txt'))) # XXX
    readme_txt = os.path.join(tmpdir, 'readme.txt')
    if os.path.exists(readme_txt):
        filelist.append((readme_txt,
                         os.path.join(prefix, 'readme.txt')))
    thumbnail_png = os.path.join(tmpdir, 'thumbnail.png')
    thumbnail_pnr = os.path.join(tmpdir, 'thumbnail.pnr')
    if os.path.exists(thumbnail_png):
        filelist.append((thumbnail_png,
                         os.path.join(prefix, 'thumbnail.png')))
    elif os.path.exists(thumbnail_pnr):
        filelist.append((thumbnail_pnr,
                         os.path.join(prefix, 'thumbnail.pnr')))
    for path in list_all_files(ghost_src, ''):
        filelist.append((os.path.join(ghost_src, path),
                         os.path.join(ghost_dst, path)))
    # find shell
    for path in list_all_files(shell_src, ''):
        filelist.append((os.path.join(shell_src, path),
                         os.path.join(shell_dst, path)))
    # find balloon
    balloon_dir = inst and inst.get('balloon.directory')
    if balloon_dir:
        balloon_list = []
        balloon_dir = ninix.home.get_normalized_path(balloon_dir)
        balloon_dst = os.path.join(homedir, 'balloon', balloon_dir)
        if os.path.exists(balloon_dst) and not confirm_removal(balloon_dst):
            pass # don't install balloon
        else:
            if os.path.exists(balloon_dst):
                # uninstall older versions of the balloon
                remove_files_and_dirs([], balloon_dst)
            for path in list_all_files(tmpdir, balloon_dir):
                balloon_list.append((os.path.join(tmpdir, path),
                                     os.path.join(homedir, 'balloon', path)))
            install_files(balloon_list)
    if os.path.exists(prefix):
        inst_dst = ninix.home.read_install_txt(prefix)
        if not inst_dst or inst_dst.get('name') != inst.get('name'):
            mode |= INTERACTIVE
        if inst.get_with_type('refresh', int, 0):
            # uninstall older versions of the ghost
            if confirm_removal(prefix):
                mask = [
                    ninix.home.get_normalized_path(path) for path in \
                        inst.get('refreshundeletemask', '').split(':')]
                mask.append('HISTORY')
                remove_files_and_dirs(mask, prefix)
            else:
                return
        else:
            if not confirm_overwrite(prefix):
                return
    # install files
    print 'installing', archive, '(ghost)'
    install_files(filelist)
    # create SETTINGS
    path = os.path.join(prefix, 'SETTINGS')
    if not os.path.exists(path):
        try:
            with open(path, 'w') as f:
                if balloon_dir:
                    f.write('balloon_directory, %s\n' % balloon_dir)
        except IOError, (code, message):
            logging.error('cannot write %s' % path)

def install_redo_supplement(archive, tmpdir, homedir, target):
    inst = ninix.home.read_install_txt(tmpdir)
    if inst and 'accept' in inst:
        print 'searching supplement target ...',
        candidates = []
        try:
            dirlist = os.listdir(os.path.join(homedir, 'ghost'))
        except OSError:
            dirlist = []
        for dirname in dirlist:
            path = os.path.join(homedir, 'ghost', dirname)
            if os.path.exists(os.path.join(path, 'shell', 'surface.txt')):
                continue # ghost.inverse(obsolete)
            desc = ninix.home.read_descript_txt(
                os.path.join(path, 'ghost', 'master'))
            if desc and desc.get('sakura.name') == inst.get('accept'):
                candidates.append(dirname)
        if not candidates:
            print 'not found'
        elif target and target not in candidates:
            print "'%s' not found" % target
            return
        elif len(candidates) == 1 or target in candidates:
            if target:
                path = os.path.join(homedir, 'ghost', target)
            else:
                path = os.path.join(homedir, 'ghost', candidates[0])
            if 'directory' in inst:
                if inst.get('type') == 'shell':
                    path = os.path.join(path, 'shell', inst['directory'])
                else:
                    if 'type' not in inst:
                        print 'supplement type not specified'
                    else:
                        print 'unsupported supplement type:', inst['type']
                    return
            print 'found'
            if not os.path.exists(path):
                os.makedirs(path)
            os.remove(os.path.join(tmpdir, 'install.txt'))
            distutils.dir_util.copy_tree(tmpdir, path)
            return
        else:
            print 'multiple candidates found'
            for candidate in candidates:
                print '   ', candidate
            fatal('try -S option with a target ghost name')

def install_balloon(archive, srcdir, homedir):
    global mode
    filelist = []
    # find install.txt
    inst = ninix.home.read_install_txt(srcdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    dstdir = os.path.join(homedir, 'balloon', target_dir)
    for path in list_all_files(srcdir, ''):
        filelist.append((os.path.join(srcdir, path),
                         os.path.join(dstdir, path)))
    ##filelist.append((os.path.join(srcdir, 'install.txt'),
    ##                 os.path.join(dstdir, 'install.txt')))
    if os.path.exists(dstdir):
        inst_dst = ninix.home.read_install_txt(dstdir)
        if not inst_dst or inst_dst.get('name') != inst.get('name'):
            mode |= INTERACTIVE
        if inst.get_with_type('refresh', int, 0):
            # uninstall older versions of the balloon
            if confirm_removal(dstdir):
                mask = [
                    ninix.home.get_normalized_path(path) for path in \
                        inst.get('refreshundeletemask', '').split(':')]
                remove_files_and_dirs(mask, dstdir)
            else:
                return
        else:
            if not confirm_overwrite(dstdir):
                return
    # install files
    print 'installing', archive, '(balloon)'
    install_files(filelist)

def uninstall_kinoko(homedir, name):
    try:
        dirlist = os.listdir(os.path.join(homedir, 'kinoko'))
    except OSError:
        return
    for subdir in dirlist:
        path = os.path.join(homedir, 'kinoko', subdir)
        kinoko = ninix.home.read_kinoko_ini(path)
        if kinoko is None:
            continue
        kinoko_name = kinoko['title']
        if kinoko_name == name:
            kinoko_dir = os.path.join(homedir, 'kinoko', subdir)
            if confirm_removal(kinoko_dir):
                shutil.rmtree(kinoko_dir)

def install_kinoko(archive, srcdir, homedir):
    filelist = []
    # find kinoko.ini
    kinoko = ninix.home.read_kinoko_ini(srcdir)
    if kinoko is None:
        fatal('failed to read kinoko.ini')
    kinoko_name = kinoko['title']
    if kinoko['extractpath'] is not None:
        dstdir = os.path.join(
            homedir, 'kinoko', kinoko['extractpath'].encode('utf-8', 'ignore'))
    else:
        dstdir = os.path.join(
            homedir, 'kinoko', os.path.basename(archive)[:-4])
    # find files
    for filename in os.listdir(srcdir):
        path = os.path.join(srcdir, filename)
        if os.path.isfile(path):
            filelist.append((path, os.path.join(dstdir, filename)))
    # uninstall older versions of the kinoko
    uninstall_kinoko(homedir, kinoko_name)
    # install files
    print 'installing', archive, '(kinoko)'
    install_files(filelist)

def uninstall_nekoninni(homedir, dir):
    nekoninni_dir = os.path.join(homedir, 'nekodorif', 'skin', dir)
    if not os.path.exists(nekoninni_dir):
        return
    if confirm_removal(nekoninni_dir):
        shutil.rmtree(nekoninni_dir)

def install_nekoninni(archive, srcdir, homedir):
    global mode
    filelist = []
    # find install.txt
    inst = ninix.home.read_install_txt(srcdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    dstdir = os.path.join(homedir, 'nekodorif', 'skin', target_dir)
    # find files
    for filename in os.listdir(srcdir):
        path = os.path.join(srcdir, filename)
        if os.path.isfile(path):
            filelist.append((path, os.path.join(dstdir, filename)))
    # uninstall older versions of the skin
    uninstall_nekoninni(homedir, target_dir)
    # install files
    print 'installing', archive, '(nekodorif skin)'
    install_files(filelist)

def uninstall_katochan(homedir, target_dir):
    katochan_dir = os.path.join(homedir, 'nekodorif', 'katochan', target_dir)
    if not os.path.exists(katochan_dir):
        return
    if confirm_removal(katochan_dir):
        shutil.rmtree(katochan_dir)

def install_katochan(archive, srcdir, homedir):
    global mode
    filelist = []
    # find install.txt
    inst = ninix.home.read_install_txt(srcdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    dstdir = os.path.join(homedir, 'nekodorif', 'katochan', target_dir)
    # find files
    for filename in os.listdir(srcdir):
        path = os.path.join(srcdir, filename)
        if os.path.isfile(path):
            filelist.append((path, os.path.join(dstdir, filename)))
    # uninstall older versions of the skin
    uninstall_katochan(homedir, target_dir)
    # install files
    print 'installing', archive, '(nekodorif katochan)'
    install_files(filelist)

def list_all_files(top, target_dir):
    filelist = []
    for path in os.listdir(os.path.join(top, target_dir)):
        if os.path.isdir(os.path.join(top, target_dir, path)):
            filelist.extend(list_all_files(top, os.path.join(target_dir, path)))
        else:
            filelist.append(os.path.join(target_dir, path))
    return filelist

def install_files(filelist):
    for from_path, to_path in filelist:
        dirname, filename = os.path.split(to_path)
        if not os.path.exists(dirname):
            os.makedirs(dirname)
        shutil.copy(from_path, to_path)


if __name__ == '__main__':
    main()
