#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <math.h>

#include "mgwtypes.h"
#include "process.h"
#include "interface.h"
#include "image.h"
#include "utils.h"

typedef enum {
  FLIP_X,
  FLIP_Y,
  FLIP_XY
} Flip;

static MgwScaleImage msi = { 1.0, TRUE, FALSE };

static GdkPixbuf *
flip (GdkPixbuf *org, Flip direction)
{
  GdkPixbuf *new;
  gint width, height;
  gint i, j, k;
  gint org_rs, org_nc, new_rs, new_nc;
  guchar *new_p, *org_p;

  if (direction == FLIP_XY) {
    width  = gdk_pixbuf_get_height (org);
    height = gdk_pixbuf_get_width (org);
  } else {
    width  = gdk_pixbuf_get_width (org);
    height = gdk_pixbuf_get_height (org);
  }
  new = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, width, height);

  org_rs = gdk_pixbuf_get_rowstride (org);
  org_nc = gdk_pixbuf_get_n_channels (org);
  org_p  = gdk_pixbuf_get_pixels (org);
  new_rs = gdk_pixbuf_get_rowstride (new);
  new_nc = gdk_pixbuf_get_n_channels (new);
  new_p  = gdk_pixbuf_get_pixels (new);

  switch (direction) {
  case FLIP_X:
    for (i = 0; i < height; i++)
      for (j = 0; j < width; j++)
	for (k = 0; k < new_nc; k++)
	  *(new_p + i * new_rs + j * new_nc + k)
	    = *(org_p + i * org_rs + (width - j - 1) * org_nc + k);
    break;

  case FLIP_Y:
    for (i = 0; i < height; i++)
      for (j = 0; j < width; j++)
	for (k = 0; k < new_nc; k++)
	  *(new_p + i * new_rs + j * new_nc + k)
	    = *(org_p + (height - i - 1) * org_rs + j * org_nc + k);
    break;

  case FLIP_XY:
    for (i = 0; i < height; i++)
      for (j = 0; j < width; j++)
	for (k = 0; k < new_nc; k++)
	  *(new_p + i * new_rs + j * new_nc + k)
	    = *(org_p + j * org_rs + i * org_nc + k);
    break;

  default:
    g_warning ("flip: Internal error");
    break;
  }

  return new;
}

#ifdef USE_UNTIALIAS
static void
untialias (GdkPixbuf *new, GdkPixbuf *org)
{
  gdouble fx, fy, p, q;
  gint shift_max, shift;
  gint org_width, org_height, new_width, new_height;
  gint i, j, k;
  gint org_rs, org_nc, new_rs, new_nc;
  guchar *new_p, *org_p;

  org_width  = gdk_pixbuf_get_width (org);
  org_height = gdk_pixbuf_get_height (org);
  org_rs     = gdk_pixbuf_get_rowstride (org);
  org_nc     = gdk_pixbuf_get_n_channels (org);
  org_p      = gdk_pixbuf_get_pixels (org);

  new_width  = gdk_pixbuf_get_width (new);
  new_height = gdk_pixbuf_get_height (new);
  new_rs     = gdk_pixbuf_get_rowstride (new);
  new_nc     = gdk_pixbuf_get_n_channels (new);
  new_p      = gdk_pixbuf_get_pixels (new);

  fx = (gdouble) org_width / new_width;
  fy = (gdouble) org_height / new_height;

  shift_max = org_rs * org_height - org_rs - org_nc * 2;

  for (i = 0; i < new_height; i++) {
    for (j = 0; j < new_width; j++) {
      p = fx * j - (gint) (fx * j);
      q = fy * i - (gint) (fy * i);

      shift = 3 * (gint) (fx * j) + org_rs * (gint) (fy * i);

      if (shift_max < shift)
	shift = shift_max;

      for (k = 0; k < new_nc; k++) {
	*(new_p + i * new_rs + j * new_nc + k)
	  = (1 - p) * (1 - q) * *(org_p + shift + k)
	  +      p  * (1 - q) * *(org_p + shift + k + 3)
	  + (1 - p) *      q  * *(org_p + shift + k + org_rs)
	  +      p  *      q  * *(org_p + shift + k + org_rs + 3);
      }
    }
  }
}
#endif

static GdkPixbuf *
scale_image_to_screen_size (GdkPixbuf *pbuf)
{
  GdkPixbuf *p;
  gint sx, sy, ix, iy, xsize, ysize;

  sx = gdk_screen_width ();
  sy = gdk_screen_height ();
  ix = gdk_pixbuf_get_width (pbuf);
  iy = gdk_pixbuf_get_height (pbuf);

  xsize = sx - SCREENSIZE_MARGE_X;
  ysize = iy * xsize / ix;
  if (sy - SCREENSIZE_MARGE_Y < ysize) {
    ysize = sy - SCREENSIZE_MARGE_Y;
    xsize = ix * ysize / iy;
  }

  p = gdk_pixbuf_scale_simple (pbuf, xsize, ysize, GDK_INTERP_BILINEAR);

#ifdef USE_UNTIALIAS
  untialias (p, pbuf);
#endif
  
  return p;
}

static GdkPixbuf *
scale_image_size (gint w, gint h, GdkPixbuf *pbuf)
{
  GdkPixbuf *p;

  p = gdk_pixbuf_scale_simple (pbuf, w, h, GDK_INTERP_BILINEAR);

  return p;
}

static GdkPixbuf *
scale_image (gdouble factor, GdkPixbuf *pbuf)
{
  GdkPixbuf *p;

  p = gdk_pixbuf_scale_simple (pbuf,
			       gdk_pixbuf_get_width (pbuf) * factor + .5,
			       gdk_pixbuf_get_height (pbuf) * factor + .5,
			       GDK_INTERP_BILINEAR);
  return p;
}

static void
scale_factor_changed (GtkWidget *widget)
{
  GdkPixbuf *pbuf = get_pbuf ();
  gchar buff[32];

  msi.factor =
    gtk_spin_button_get_value (GTK_SPIN_BUTTON (widget));

  sprintf (buff, _("(W x H) = (%d x %d)"),
	   (gint) (gdk_pixbuf_get_width (pbuf)  * msi.factor + .5),
	   (gint) (gdk_pixbuf_get_height (pbuf) * msi.factor + .5));
  gtk_label_set_text (GTK_LABEL (msi.target_size_label), buff);
}

static void
on_cancel_button_clicked (GtkWidget *w, GtkWidget *dialog)
{
  gtk_widget_destroy (dialog);
}

static void
on_scale_factor_ok_button_clicked (GtkWidget *w, GtkWidget *dialog)
{
  GdkPixbuf *p, *pbuf = get_pbuf ();
  gchar buff[32];

  gtk_widget_destroy (dialog);

  if (0.99 < msi.factor && msi.factor < 1.01)
    return;

  p = scale_image (msi.factor, pbuf);

  update_pbuf (p);
  image_changed ();
  sprintf (buff, "Size changed: (%d x %d)",
	   gdk_pixbuf_get_width (p), gdk_pixbuf_get_height (p));
  put_string_to_appbar (buff);
}

static void
scale_xsize_changed (GtkWidget *widget)
{
  msi.xsize = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));

  if (msi.keep_aspect_ratio) {
    GdkPixbuf *p = get_pbuf ();
    gdouble aspect;
    gint s;

    aspect = ((gdouble) gdk_pixbuf_get_height (p)) / gdk_pixbuf_get_width (p);

    s = (gint) (aspect * msi.xsize + 0.5);
    if (msi.ysize != s) {
      gtk_adjustment_set_value (GTK_ADJUSTMENT (msi.spnbtn_adj_ysize), s);
      msi.ysize = s;
    }
  }
}

static void
scale_ysize_changed (GtkWidget *widget)
{
  msi.ysize = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));

  if (msi.keep_aspect_ratio) {
    GdkPixbuf *p = get_pbuf ();
    gdouble aspect;
    gint s;

    aspect = ((gdouble) gdk_pixbuf_get_width (p)) / gdk_pixbuf_get_height (p);

    s = (gint) (aspect * msi.ysize + 0.5);
    if (msi.xsize != s) {
      gtk_adjustment_set_value (GTK_ADJUSTMENT (msi.spnbtn_adj_xsize), s);
      msi.xsize = s;
    }
  }
}

static void
on_scale_size_ok_button_clicked (GtkWidget *w, GtkWidget *dialog)
{
  GdkPixbuf *p, *pbuf = get_pbuf ();
  gint width, height;
  gchar buff[32];

  gtk_widget_destroy (dialog);

  width = gdk_pixbuf_get_width (pbuf);
  height = gdk_pixbuf_get_height (pbuf);

  if (width != msi.xsize || height != msi.ysize) {
    p = scale_image_size (msi.xsize, msi.ysize, pbuf);
    update_pbuf (p);
    image_changed ();
    sprintf (buff, "Size changed: (%d x %d)",
	     gdk_pixbuf_get_width (p), gdk_pixbuf_get_height (p));
    put_string_to_appbar (buff);
  }
}

static void
on_button_keep_asp_clicked (void)
{
  msi.keep_aspect_ratio = !msi.keep_aspect_ratio;
}

static GtkWidget *
create_scale_size_dialog (void)
{
  GtkWidget *dialog;
  GtkWidget *dialog_vbox, *hbox, *label;
  GtkWidget *dialog_action_area;
  GtkWidget *btn_cancel;
  GtkWidget *btn_ok;
  GtkWidget *btn_keep_asp;
  GtkWidget *spnbtn_xsize;
  GtkWidget *spnbtn_ysize;
  GdkPixbuf *pbuf = get_pbuf ();

  /* GET SCALE ORIGINAL SIZE */
  msi.xsize = gdk_pixbuf_get_width (pbuf);
  msi.ysize = gdk_pixbuf_get_height (pbuf);

  /* DIALOG */
  dialog = gnome_dialog_new (NULL, NULL);
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_scale_factor", dialog);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Image size"));
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, FALSE);

  /* DIALOG VBOX */
  dialog_vbox = GNOME_DIALOG (dialog)->vbox;
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_vbox", dialog_vbox);
  gtk_widget_show (dialog_vbox);

  /* HBOX */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_ref (hbox);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "hbox_scale_factor", hbox,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox, TRUE, TRUE, 0);

  /* LABEL */
  label = gtk_label_new (_("W x H"));
  gtk_widget_ref (label);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "label_scale", label,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

  /* SPIN BUTTON (X-size) */
  msi.spnbtn_adj_xsize
    = gtk_adjustment_new (msi.xsize, 5, 10000, 1, 10, 100);
  spnbtn_xsize
    = gtk_spin_button_new (GTK_ADJUSTMENT (msi.spnbtn_adj_xsize), 1, 0);
  gtk_widget_ref (spnbtn_xsize);
  gtk_object_set_data_full (GTK_OBJECT (dialog),
			    "spnbtn_xsize", spnbtn_xsize,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_set_usize (spnbtn_xsize, 60, 20);
  gtk_widget_show (spnbtn_xsize);
  gtk_box_pack_start (GTK_BOX (hbox), spnbtn_xsize, TRUE, FALSE, 0);

  /* SPIN BUTTON (Y-size) */
  msi.spnbtn_adj_ysize = gtk_adjustment_new (msi.ysize, 5, 10000, 1, 10, 100);
  spnbtn_ysize =
    gtk_spin_button_new (GTK_ADJUSTMENT (msi.spnbtn_adj_ysize), 1, 0);
  gtk_widget_ref (spnbtn_ysize);
  gtk_object_set_data_full (GTK_OBJECT (dialog),
			    "spnbtn_ysize", spnbtn_ysize,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_set_usize (spnbtn_ysize, 60, 20);
  gtk_widget_show (spnbtn_ysize);
  gtk_box_pack_start (GTK_BOX (hbox), spnbtn_ysize, TRUE, FALSE, 0);

  /* keep aspect ratio */
  btn_keep_asp = gtk_check_button_new_with_label (_("Keep aspect ratio"));
  gtk_widget_ref (btn_keep_asp);
  gtk_object_set_data_full (GTK_OBJECT (dialog),
			    "btn_keep_asp", btn_keep_asp,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (btn_keep_asp);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), btn_keep_asp, TRUE, TRUE, 0);
  if (msi.keep_aspect_ratio)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn_keep_asp), TRUE);

  /* DIALOG ACTION AREA */
  dialog_action_area = GNOME_DIALOG (dialog)->action_area;
  gtk_object_set_data (GTK_OBJECT (dialog),
		       "dialog_action_area", dialog_action_area);
  gtk_widget_show (dialog_action_area);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area),
			     GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area), 8);

  gnome_dialog_append_button (GNOME_DIALOG (dialog),
			      GNOME_STOCK_BUTTON_CANCEL);
  btn_cancel =
    GTK_WIDGET (g_list_last (GNOME_DIALOG (dialog)->buttons)->data);
  gtk_widget_ref (btn_cancel);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "btn_cancel", btn_cancel,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (btn_cancel);

  gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_OK);
  btn_ok = GTK_WIDGET (g_list_last (GNOME_DIALOG (dialog)->buttons)->data);
  gtk_widget_ref (btn_ok);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "btn_ok", btn_ok,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (btn_ok);
  GTK_WIDGET_SET_FLAGS (btn_ok, GTK_CAN_DEFAULT);

  /* CALLBACKS */
  gtk_signal_connect (GTK_OBJECT (btn_cancel), "clicked",
		      GTK_SIGNAL_FUNC (on_cancel_button_clicked), dialog);
  gtk_signal_connect (GTK_OBJECT (btn_ok), "clicked",
		      GTK_SIGNAL_FUNC (on_scale_size_ok_button_clicked),
		      dialog);
  gtk_signal_connect (GTK_OBJECT (spnbtn_xsize), "value_changed",
		      GTK_SIGNAL_FUNC (scale_xsize_changed), NULL);
  gtk_signal_connect (GTK_OBJECT (spnbtn_ysize), "value_changed",
		      GTK_SIGNAL_FUNC (scale_ysize_changed), NULL);
  gtk_signal_connect (GTK_OBJECT (btn_keep_asp), "clicked",
		      GTK_SIGNAL_FUNC (on_button_keep_asp_clicked), NULL);
  return dialog;
}

static GtkWidget *
create_scale_factor_dialog (void)
{
  GtkWidget *dialog;
  GtkWidget *dialog_vbox, *hbox, *label;
  GtkWidget *dialog_action_area;
  GtkWidget *btn_cancel;
  GtkWidget *btn_ok;
  GtkWidget *spnbtn_scale_factor;
  GtkWidget *frame;
  GtkObject *spnbtn_adj_scale_factor;

  /* DIALOG */
  dialog = gnome_dialog_new (NULL, NULL);
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_scale_factor", dialog);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Scale factor"));
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, FALSE);

  /* DIALOG VBOX */
  dialog_vbox = GNOME_DIALOG (dialog)->vbox;
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_vbox", dialog_vbox);
  gtk_widget_show (dialog_vbox);

  /* HBOX */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_ref (hbox);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "hbox_scale_factor", hbox,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox, TRUE, TRUE, 0);

  /* LABEL */
  label = gtk_label_new (_("Factor"));
  gtk_widget_ref (label);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "label_scale", label,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

  /* SPIN BUTTON */
  spnbtn_adj_scale_factor =
    gtk_adjustment_new (msi.factor, 0.01, 10, 0.01, 0.01, 0.02);
  spnbtn_scale_factor =
    gtk_spin_button_new (GTK_ADJUSTMENT (spnbtn_adj_scale_factor), 1, 2);
  gtk_widget_ref (spnbtn_scale_factor);
  gtk_object_set_data_full (GTK_OBJECT (dialog),
			    "spnbtn_scale_factor", spnbtn_scale_factor,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (spnbtn_scale_factor);
  gtk_box_pack_start (GTK_BOX (hbox), spnbtn_scale_factor, TRUE, FALSE, 0);

  /* TARGET SIZE LABEL */
  frame = gtk_frame_new (_("Target size"));
  gtk_box_pack_start (GTK_BOX (dialog_vbox), frame, TRUE, TRUE, 2);
  gtk_widget_show (frame);
  msi.target_size_label = gtk_label_new ("");
  gtk_widget_show (msi.target_size_label);
  gtk_container_add (GTK_CONTAINER (frame), msi.target_size_label);

  /* DIALOG ACTION AREA */
  dialog_action_area = GNOME_DIALOG (dialog)->action_area;
  gtk_object_set_data (GTK_OBJECT (dialog),
		       "dialog_action_area", dialog_action_area);
  gtk_widget_show (dialog_action_area);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area),
			     GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area), 8);

  gnome_dialog_append_button (GNOME_DIALOG (dialog),
			      GNOME_STOCK_BUTTON_CANCEL);
  btn_cancel =
    GTK_WIDGET (g_list_last (GNOME_DIALOG (dialog)->buttons)->data);
  gtk_widget_ref (btn_cancel);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "btn_cancel", btn_cancel,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (btn_cancel);

  gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_OK);
  btn_ok = GTK_WIDGET (g_list_last (GNOME_DIALOG (dialog)->buttons)->data);
  gtk_widget_ref (btn_ok);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "btn_ok", btn_ok,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (btn_ok);
  GTK_WIDGET_SET_FLAGS (btn_ok, GTK_CAN_DEFAULT);

  /* CALLBACKS */
  gtk_signal_connect (GTK_OBJECT (btn_cancel), "clicked",
		      GTK_SIGNAL_FUNC (on_cancel_button_clicked), dialog);
  gtk_signal_connect (GTK_OBJECT (btn_ok), "clicked",
		      GTK_SIGNAL_FUNC (on_scale_factor_ok_button_clicked),
		      dialog);
  gtk_signal_connect (GTK_OBJECT (spnbtn_scale_factor), "changed",
		      GTK_SIGNAL_FUNC (scale_factor_changed), NULL);
  return dialog;
}

/* Entry point */
void
on_scale_image_size_activate (void)
{
  GtkWidget *scale_size_dialog;

  scale_size_dialog = create_scale_size_dialog ();
  gtk_widget_show (scale_size_dialog);
  gtk_grab_add (scale_size_dialog);
}

void
on_scale_image_factor_activate (void)
{
  GtkWidget *scale_factor_dialog;

  scale_factor_dialog = create_scale_factor_dialog ();
  gtk_widget_show (scale_factor_dialog);
  gtk_grab_add (scale_factor_dialog);
}

void
process_exec (GtkWidget *dummy, gchar *name)
{
  MgwImageInfo *mii;
  MgwClipRect c;
  GdkPixbuf *pbuf, *org, *p = NULL, *q = NULL;
  gchar buff[64];
  gboolean has_clip;

  if (!(org = get_pbuf ()))
    return;

  mii = get_image_info ();
  if (mii->handling)
    return;
  else
    mii->handling = TRUE;

  menubar_hilite_control (LOCK);
  set_cursor (GDK_WATCH);
  sprintf (buff, "%s: EXECUTING...", name);
  put_string_to_appbar (buff);

  if (has_clip = get_clip_rect_area (&c)) {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, c.x1, c.y1);
    gdk_pixbuf_copy_area (org, c.x0, c.y0, c.x1, c.y1, pbuf, 0, 0);
  } else {
    pbuf = org;
  }

  if (!strcmp (name, "Screen Size"))
    p = scale_image_to_screen_size (pbuf);
  else if (!strncmp ("Resize ", name, 7))
    p = scale_image ((gdouble) atoi (name + 7) / 100., pbuf);
  else if (!strcmp ("Rotate 90deg.", name)) {
    q = flip (pbuf, FLIP_XY);
    p = flip (q, FLIP_X);
  } else if (!strcmp ("Rotate 180deg.", name)) {
    q = flip (pbuf, FLIP_X);
    p = flip (q, FLIP_Y);
  } else if (!strcmp ("Rotate 270deg.", name)) {
    q = flip (pbuf, FLIP_XY);
    p = flip (q, FLIP_Y);
  } else if (!strcmp("Flip Horizontal", name))
    p = flip (pbuf, FLIP_X);
  else if (!strcmp("Flip Vertical", name))
    p = flip (pbuf, FLIP_Y);
  else
    g_warning ("process_exec: Internal error: %s", name);

  if (q)
    g_object_unref (q);

  if (has_clip) {
    gdk_pixbuf_copy_area (p, 0, 0,
			  MIN (gdk_pixbuf_get_width (p),
			       gdk_pixbuf_get_width (org) - c.x0),
			  MIN (gdk_pixbuf_get_height (p),
			       gdk_pixbuf_get_height (org) - c.y0),
			  org, c.x0, c.y0);
    redraw_image (org);
    draw_clip_rect_area ();
  } else {
    if (p)
      update_pbuf (p);
  }

  sprintf (buff, "%s: DONE", name);
  put_string_to_appbar (buff);

  set_cursor (GDK_TOP_LEFT_ARROW);
  menubar_hilite_control (UNLOCK);
  image_changed ();
  mii->handling = FALSE;
}
