
/*
 * LaTeX Symbols Selector
 *
 * Copyright (C) 2006 Tomasz Mąka <pasp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "gui.h"
#include "send.h"
#include "prefs.h"
#include "config.h"
#include "i18n.h"
#include "graphics.h"

/*------------------------------------------------------------------------------*/

const gchar *
get_group_name (gint index) {

const gchar *group_name[NUMBER_OF_SETS] = {
        N_("Accents"),          N_("Arrows"),       N_("Delimiters"),   N_("Greek Letters"),
        N_("Miscellaneous"),    N_("Operators"),    N_("Relations"),    N_("Text Mode")
};

	return gettext (group_name[index]);

}

/*------------------------------------------------------------------------------*/

gint
get_symbolset_len (gint index) {

const gint symbolset_len[NUMBER_OF_SETS] = {
        LSYMBOLS_accents,       LSYMBOLS_arrows,    LSYMBOLS_delimiters,
        LSYMBOLS_greek_letters, LSYMBOLS_misc_math, LSYMBOLS_operators,
        LSYMBOLS_relations,     LSYMBOLS_text_mode
};

	return symbolset_len[index];

}

/*------------------------------------------------------------------------------*/

const gchar *
get_ams_group_name (gint index) {

const gchar *ams_group_name[AMS_NUMBER_OF_SETS] = {
        N_("AMS Accents"),          N_("AMS Arrows"),       N_("AMS Delimiters"),
        N_("AMS Miscellaneous"),    N_("AMS Operators"),    N_("AMS Relations")
};

	return gettext (ams_group_name[index]);

}

/*------------------------------------------------------------------------------*/

gint
get_ams_symbolset_len (gint index) {

const gint ams_symbolset_len[AMS_NUMBER_OF_SETS] = {
        AMS_LSYMBOLS_accents,
        AMS_LSYMBOLS_arrows,
        AMS_LSYMBOLS_delimiters,
        AMS_LSYMBOLS_misc_math,
        AMS_LSYMBOLS_operators,
        AMS_LSYMBOLS_relations
};

	return ams_symbolset_len[index];

}

/*------------------------------------------------------------------------------*/

void
show_status (gchar *symbol_name, gint status, GUI *appGUI) {

    gchar message[BUFFER_SIZE], tmp[BUFFER_SIZE];

    if (status & SELECTED) {
		snprintf (message, BUFFER_SIZE, "%s (%s) ", _("Selected symbol"), symbol_name);
    }

    if ((status & SENT_TO_GVIM) && (status & SENT_TO_CLIPBOARD)) {

		snprintf (tmp, BUFFER_SIZE, _("sent to gVIM and clipboard"));
		strncat (message, tmp, BUFFER_SIZE);

    } else if (status & SENT_TO_GVIM) {

		snprintf (tmp, BUFFER_SIZE, _("sent to gVIM"));
		strncat (message, tmp, BUFFER_SIZE);

    } else if (status & SENT_TO_CLIPBOARD) {

		snprintf (tmp, BUFFER_SIZE, _("sent to clipboard"));
		strncat (message, tmp, BUFFER_SIZE);
    }

    gtk_statusbar_pop (GTK_STATUSBAR(appGUI->statusbar), appGUI->statusbar_context_id);
    gtk_statusbar_push (GTK_STATUSBAR(appGUI->statusbar), appGUI->statusbar_context_id, message);

}

/*------------------------------------------------------------------------------*/

void
group_selected_cb (GtkTreeSelection *selection, gpointer user_data) {

    GtkTreeIter iter;
    GtkTreeModel *model;
    gint i;
    gchar message[BUFFER_SIZE];

    GUI *appGUI = (GUI *)user_data;

    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

        gtk_tree_model_get (model, &iter, 1, &i, -1);

        config.current_group = i;

        if (i >= NUMBER_OF_SETS) {
            snprintf (message, BUFFER_SIZE, "%s: %s (%d %s)",
			          _("Selected group"), get_ams_group_name(i-NUMBER_OF_SETS),
			          get_ams_symbolset_len(i-NUMBER_OF_SETS),
			          ngettext ("symbol", "symbols", get_ams_symbolset_len(i-NUMBER_OF_SETS)));
        } else {
            snprintf (message, BUFFER_SIZE, "%s: %s (%d %s)",
			          _("Selected group"), get_group_name(i),
			          get_symbolset_len(i), ngettext ("symbol", "symbols", get_symbolset_len(i)));
        }

        gtk_statusbar_pop (GTK_STATUSBAR(appGUI->statusbar), appGUI->statusbar_context_id);
        gtk_statusbar_push (GTK_STATUSBAR(appGUI->statusbar), appGUI->statusbar_context_id, message);

        display_symbols_list(config.current_group, appGUI);

    }

}

/*------------------------------------------------------------------------------*/

void
show_statusbar_option_desc(gchar *message, gint state, GUI *appGUI) {

    gchar buffer[BUFFER_SIZE];

    if (state) {
		snprintf (buffer, BUFFER_SIZE, _("'%s' option activated"), message);
    } else {
		snprintf (buffer, BUFFER_SIZE, _("'%s' option deactivated"), message);
    }

    gtk_statusbar_pop (GTK_STATUSBAR(appGUI->statusbar), appGUI->statusbar_context_id);
    gtk_statusbar_push (GTK_STATUSBAR(appGUI->statusbar), appGUI->statusbar_context_id, buffer);
}

/*------------------------------------------------------------------------------*/

void
option_enable_ams_symbols_cb (GtkCheckMenuItem *check_menu_item, gpointer user_data) {

    GUI *appGUI = (GUI *)user_data;

    config.enable_ams_symbols = gtk_check_menu_item_get_active (check_menu_item);

    if (!config.enable_ams_symbols) {

        if (config.current_group >= NUMBER_OF_SETS) {

            config.current_group = NUMBER_OF_SETS - 1;
        }

    }

    display_names_list (appGUI);
    display_symbols_list(config.current_group, appGUI);
    set_cursor_on_group(config.current_group, appGUI);

    show_statusbar_option_desc(_("Enable AMS symbols"), config.enable_ams_symbols, appGUI);
}

/*------------------------------------------------------------------------------*/

void
option_display_symbol_name_cb (GtkCheckMenuItem *check_menu_item, gpointer user_data) {

    GUI *appGUI = (GUI *)user_data;

    config.display_symbol_name =  gtk_check_menu_item_get_active (check_menu_item);

    if(config.display_symbol_name) {

        gtk_icon_view_set_text_column (GTK_ICON_VIEW (appGUI->icons_view), COL_NAME);
        gtk_icon_view_set_item_width (GTK_ICON_VIEW (appGUI->icons_view), 120);

    } else {

        gtk_icon_view_set_text_column (GTK_ICON_VIEW (appGUI->icons_view), -1);
        gtk_icon_view_set_item_width (GTK_ICON_VIEW (appGUI->icons_view), -1);
    }

    show_statusbar_option_desc(_("Display symbol name"), config.display_symbol_name, appGUI);
}

/*------------------------------------------------------------------------------*/

void
option_single_click_cb (GtkCheckMenuItem *check_menu_item, gpointer user_data) {

    GUI *appGUI = (GUI *)user_data;

    config.single_click = gtk_check_menu_item_get_active (check_menu_item);

    show_statusbar_option_desc(_("Activate action on single click"), config.single_click, appGUI);
}

/*------------------------------------------------------------------------------*/

void
option_send_to_gvim_cb (GtkCheckMenuItem *check_menu_item, gpointer user_data) {

    GUI *appGUI = (GUI *)user_data;

    config.send_to_gvim = gtk_check_menu_item_get_active (check_menu_item);

    show_statusbar_option_desc(_("Send symbol to gVIM"), config.send_to_gvim, appGUI);
}

/*------------------------------------------------------------------------------*/

void
option_send_to_clipboard_cb (GtkCheckMenuItem *check_menu_item, gpointer user_data) {

    GUI *appGUI = (GUI *)user_data;

    config.send_to_clipboard =  gtk_check_menu_item_get_active (check_menu_item);

    show_statusbar_option_desc(_("Send symbol to clipboard"), config.send_to_clipboard, appGUI);
}

/*------------------------------------------------------------------------------*/

void
display_about_cb (GtkWidget *widget, gpointer data) {

GdkPixbuf *logo;

const gchar *authors[] = {
    "Tomasz Mąka <pasp@users.sourceforge.net>", NULL
};
const gchar *artists[] = {
    "Maja Kocoń <ironya@ll.pl> - (http://ironya.ll.pl)", NULL
};

const gchar *translators = _("translator_credits-PLEASE_ADD_YOURSELF_HERE");

const gchar license[] = {
  "This program is free software; you can redistribute it and/or modify "
  "it under the terms of the GNU General Public License as published by "
  "the Free Software Foundation; either version 2 of the License, or "
  "(at your option) any later version."
  "\n\n"
  "This program is distributed in the hope that it will be useful, "
  "but WITHOUT ANY WARRANTY; without even the implied warranty of "
  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
  "GNU General Public License for more details."
  "\n\n"
  "You should have received a copy of the GNU General Public License "
  "along with this program; if not, write to the Free Software "
  "Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA."
};

    GUI *appGUI = (GUI *)data;

    logo = gdk_pixbuf_new_from_inline(-1, lss_logo, FALSE, NULL);

    gtk_show_about_dialog(GTK_WINDOW(appGUI->main_window),
                          "name", "LSS",
                          "version", VERSION,
                          "website", "http://clay.ll.pl/lss",
                          "copyright", "Copyright © 2006 Tomasz Mąka",
                          "authors", authors,
                          "artists", artists,
                          "translator-credits", strcmp (translators, "translator_credits-PLEASE_ADD_YOURSELF_HERE") ? translators : NULL,
                          "license", license,
                          "wrap-license", TRUE,
                          "logo", logo,
                          NULL);

    g_object_unref(logo);

}

/*------------------------------------------------------------------------------*/

void
gui_window_closed_cb (GtkWidget *widget, GdkEvent *event, gpointer user_data) {

    GUI *appGUI = (GUI *)user_data;

    gtk_window_get_size (GTK_WINDOW(appGUI->main_window),
                        &config.window_size_x, &config.window_size_y);

    gdk_window_get_root_origin ((appGUI->main_window)->window,
                        &config.window_x, &config.window_y);

    gtk_main_quit ();
}

/*------------------------------------------------------------------------------*/

void
quit_selected_cb (GtkWidget *widget, gpointer user_data) {

    GUI *appGUI = (GUI *)user_data;
	gui_window_closed_cb (NULL, NULL, appGUI);

}

/*------------------------------------------------------------------------------*/

void
display_names_list(GUI *appGUI) {

    gint i;
    GtkTreeIter iter;

    gtk_list_store_clear (appGUI->groups_list_store);

    for(i = 0; i < NUMBER_OF_SETS; i++) {

        gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(appGUI->groups_list_store), &iter, NULL, i);
        gtk_list_store_append(appGUI->groups_list_store, &iter);
        gtk_list_store_set (appGUI->groups_list_store, &iter, 0, get_group_name(i), 1, i, -1);

    }

    if (config.enable_ams_symbols) {

        for(i = 0; i < AMS_NUMBER_OF_SETS; i++) {

            gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(appGUI->groups_list_store), &iter, NULL, NUMBER_OF_SETS+i);
            gtk_list_store_append(appGUI->groups_list_store, &iter);
            gtk_list_store_set (appGUI->groups_list_store, &iter, 0, get_ams_group_name(i), 1, NUMBER_OF_SETS+i, -1);

        }

    }
}

/*------------------------------------------------------------------------------*/

void
display_symbols_list(gint setnumber, GUI *appGUI) {

    gint i;
    GtkTreeIter iter;
    GdkPixbuf *icon;

    const guint8 **symbolset_icon[NUMBER_OF_SETS] = {
        symbols_accents_data,
        symbols_arrows_data,
        symbols_delimiters_data,
        symbols_greek_letters_data,
        symbols_misc_math_data,
        symbols_operators_data,
        symbols_relations_data,
        symbols_text_mode_data
    };

    const gchar **symbolset_name[NUMBER_OF_SETS] = {
        symbols_accents_name,
        symbols_arrows_name,
        symbols_delimiters_name,
        symbols_greek_letters_name,
        symbols_misc_math_name,
        symbols_operators_name,
        symbols_relations_name,
        symbols_text_mode_name
    };

    const guint8 **ams_symbolset_icon[AMS_NUMBER_OF_SETS] = {
        ams_symbols_accents_data,
        ams_symbols_arrows_data,
        ams_symbols_delimiters_data,
        ams_symbols_misc_math_data,
        ams_symbols_operators_data,
        ams_symbols_relations_data
    };

    const gchar **ams_symbolset_name[AMS_NUMBER_OF_SETS] = {
        ams_symbols_accents_name,
        ams_symbols_arrows_name,
        ams_symbols_delimiters_name,
        ams_symbols_misc_math_name,
        ams_symbols_operators_name,
        ams_symbols_relations_name
    };

    gtk_list_store_clear (appGUI->icons_store);

    if (setnumber < NUMBER_OF_SETS) {

        for (i = 0; i < get_symbolset_len(setnumber); i++) {

            gtk_list_store_append (appGUI->icons_store, &iter);

            icon = gdk_pixbuf_new_from_inline(-1, symbolset_icon[setnumber][i], FALSE, NULL);
            gtk_list_store_set (appGUI->icons_store, &iter, COL_PIXBUF, icon, COL_NAME, symbolset_name[setnumber][i], -1);

            g_object_unref(icon);

        }

    } else {

        setnumber -= NUMBER_OF_SETS;

        for (i = 0; i < get_ams_symbolset_len(setnumber); i++) {

            gtk_list_store_append (appGUI->icons_store, &iter);

            icon = gdk_pixbuf_new_from_inline(-1, ams_symbolset_icon[setnumber][i], FALSE, NULL);
            gtk_list_store_set (appGUI->icons_store, &iter, COL_PIXBUF, icon, COL_NAME, ams_symbolset_name[setnumber][i], -1);

            g_object_unref(icon);

        }

    }
}

/*------------------------------------------------------------------------------*/
/* double click callback */

void
icon_activated_dc_cb (GtkIconView *icons_view, GtkTreePath *tree_path, gpointer user_data) {

    GtkListStore *store;
    gchar *icon_name;
    GtkTreeIter iter;
    gint status;

    GUI *appGUI = (GUI *)user_data;

    if (!config.single_click) {

        store = GTK_LIST_STORE (appGUI->icons_store);

        gtk_tree_model_get_iter (GTK_TREE_MODEL (store), &iter, tree_path);
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, COL_NAME, &icon_name, -1);

        status = SELECTED;

        if (config.send_to_gvim) {
            if (send_symbol_to_gvim(icon_name) == TRUE)
                status |= SENT_TO_GVIM;
        }

        if (config.send_to_clipboard) {
            send_symbol_to_clipboard(icon_name);
            status |= SENT_TO_CLIPBOARD;
        }

        show_status (icon_name, status, appGUI);

        g_free (icon_name);
    }

}

/*------------------------------------------------------------------------------*/
/* single click callback */

void
icon_activated_sc_cb (GtkIconView *icon_view, gpointer user_data) {

    GtkTreePath *tree_path;
    GtkTreeIter iter;
    gchar *icon_name;
    GList *items_list, *current;
    gint status;

    GUI *appGUI = (GUI *)user_data;

    if (config.single_click) {

        items_list = gtk_icon_view_get_selected_items (icon_view);

        current = g_list_nth (items_list, 0);

        if(current != NULL) {

            tree_path = current->data;

            gtk_tree_model_get_iter (GTK_TREE_MODEL (appGUI->icons_store), &iter, tree_path);
            gtk_tree_model_get (GTK_TREE_MODEL (appGUI->icons_store), &iter, COL_NAME, &icon_name, -1);
            status = SELECTED;

            if (config.send_to_gvim) {
                if (send_symbol_to_gvim(icon_name) == TRUE) {
                    status |= SENT_TO_GVIM;
                }
            }

            if (config.send_to_clipboard) {
                send_symbol_to_clipboard(icon_name);
                status |= SENT_TO_CLIPBOARD;
            }

            show_status (icon_name, status, appGUI);

            strncpy(config.latest_symbol_name, icon_name, SYMBOL_NAME_LEN-1);

            g_free (icon_name);

        }

        g_list_foreach (items_list, (GFunc) gtk_tree_path_free, NULL);
        g_list_free (items_list);

    }
}

/*------------------------------------------------------------------------------*/

gboolean
mouse_button_press_cb (GtkWidget *user_data, GdkEventButton *event) {   /* trick */

    GUI *appGUI = (GUI *)user_data;

    if (event->type == GDK_BUTTON_PRESS && event->button == 3) {

        GdkEventButton *bevent = (GdkEventButton *) event;

        gtk_menu_popup (GTK_MENU (appGUI->popup_menu), NULL, NULL, NULL, NULL,
                        bevent->button, bevent->time);
        return TRUE;
    }

    return FALSE;
}

/*------------------------------------------------------------------------------*/
/* set cursor on last selected group */

void
set_cursor_on_group(gint group, GUI *appGUI) {

    gint i;
    GtkTreeIter iter;
    GtkTreePath *visible_path;

    if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(appGUI->groups_list_store), &iter)) {

        do {
            gtk_tree_model_get(GTK_TREE_MODEL(appGUI->groups_list_store), &iter, 1, &i, -1);

            if (i == group) {

                visible_path = gtk_tree_model_get_path (GTK_TREE_MODEL(appGUI->groups_list_store), &iter);
                gtk_tree_view_set_cursor (GTK_TREE_VIEW (appGUI->groups_list), visible_path, NULL, TRUE);
                gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (appGUI->groups_list), visible_path, NULL, TRUE, 0.5, 0.0);
                gtk_tree_path_free(visible_path);
                break;
            }

        } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(appGUI->groups_list_store), &iter));

    }
}

/*------------------------------------------------------------------------------*/

gint
key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer user_data) {

    GtkAction *action;

    GUI *appGUI = (GUI *)user_data;

    switch (event->keyval) {

        case GDK_Return:
            if (event->state & GDK_CONTROL_MASK) {  /* CTRL + Enter */
                if(strlen(config.latest_symbol_name)) {
                    if (config.send_to_gvim) {
                        send_symbol_to_gvim(config.latest_symbol_name);
                    }
                    if (config.send_to_clipboard) {
                        send_symbol_to_clipboard(config.latest_symbol_name);
                    }
                    gui_window_closed_cb(NULL, NULL, appGUI);
                }
            }
            return TRUE;

        case GDK_space:
            action = gtk_ui_manager_get_action(appGUI->uim_widget, "/popup_menu/display_symbol_name");
            config.display_symbol_name = !config.display_symbol_name;
            gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), config.display_symbol_name);
            return TRUE;

        case GDK_Escape:
            gui_window_closed_cb(NULL, NULL, appGUI);
            return TRUE;

        case GDK_q:
        case GDK_w:
            if (event->state & GDK_CONTROL_MASK) {  /* CTRL + Q, CTRL + W */
                gui_window_closed_cb(NULL, NULL, appGUI);
            }
            return TRUE;

        case GDK_F1:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F1 */
                set_cursor_on_group(GROUP_AMS_ACCENTS, appGUI);
            } else {
                set_cursor_on_group(GROUP_ACCENTS, appGUI);
            }
            return TRUE;

        case GDK_F2:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F2 */
                set_cursor_on_group(GROUP_AMS_ARROWS, appGUI);
            } else {
                set_cursor_on_group(GROUP_ARROWS, appGUI);
            }
            return TRUE;

        case GDK_F3:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F3 */
                set_cursor_on_group(GROUP_AMS_DELIMITERS, appGUI);
            } else {
                set_cursor_on_group(GROUP_DELIMITERS, appGUI);
            }
            return TRUE;

        case GDK_F4:
            set_cursor_on_group(GROUP_GREEK_LETTERS, appGUI);
            return TRUE;

        case GDK_F5:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F5 */
                set_cursor_on_group(GROUP_AMS_MISCELLANEOUS, appGUI);
            } else {
                set_cursor_on_group(GROUP_MISCELLANEOUS, appGUI);
            }
            return TRUE;

        case GDK_F6:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F6 */
                set_cursor_on_group(GROUP_AMS_OPERATORS, appGUI);
            } else {
                set_cursor_on_group(GROUP_OPERATORS, appGUI);
            }
            return TRUE;

        case GDK_F7:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F7 */
                set_cursor_on_group(GROUP_AMS_RELATIONS, appGUI);
            } else {
                set_cursor_on_group(GROUP_RELATIONS, appGUI);
            }
            return TRUE;

        case GDK_F8:
            set_cursor_on_group(GROUP_TEXT_MODE, appGUI);
            return TRUE;
    }

    return FALSE;
}

/*------------------------------------------------------------------------------*/

void 
gui_create_window(GUI *appGUI) {

GtkWidget           *hbox;
GtkWidget           *vbox1, *vbox2;
GtkWidget           *scrolledwindow_icons;
GtkWidget           *scrolledwindow_list;
GtkCellRenderer     *renderer;
GtkTreeViewColumn   *column;
GdkPixbuf           *icon;
GtkAction           *action;
GtkActionGroup      *action_group = NULL;

GtkActionEntry entries[] = {
  { "about",    GTK_STOCK_ABOUT,    NULL, NULL, NULL, NULL },
  { "quit",     GTK_STOCK_QUIT,     NULL, NULL, NULL, NULL }
};

guint n_entries = G_N_ELEMENTS (entries);

GtkToggleActionEntry toggle_entries[] = {
  { "send_to_gvim",        NULL, _("Send symbol to GVIM"),             NULL, NULL, NULL, FALSE },
  { "send_to_clipboard",   NULL, _("Send symbol to clipboard"),        NULL, NULL, NULL, FALSE },
  { "single_click",        NULL, _("Activate action on single click"), NULL, NULL, NULL, FALSE },
  { "display_symbol_name", NULL, _("Display symbol name"),             NULL, NULL, NULL, FALSE },
  { "enable_ams_symbols",  NULL, _("Enable AMS symbols"),              NULL, NULL, NULL, FALSE }
};

static guint n_toggle_entries = G_N_ELEMENTS (toggle_entries);

const gchar *ui_info =
"  <popup name='popup_menu'>\n"
"      <menuitem action='send_to_clipboard' />\n"
"      <menuitem action='send_to_gvim' />\n"
"      <separator />\n"
"      <menuitem action='single_click' />\n"
"      <menuitem action='enable_ams_symbols' />\n"
"      <menuitem action='display_symbol_name' />\n"
"      <separator />\n"
"      <menuitem action='about' />\n"
"      <separator />\n"
"      <menuitem action='quit' />\n"
"  </popup>\n";


    appGUI->main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (appGUI->main_window), "LaTeX Symbols Selector " VERSION);

    icon = gdk_pixbuf_new_from_inline(-1, lss_icon, FALSE, NULL);
    gtk_window_set_icon (GTK_WINDOW(appGUI->main_window), icon);
    g_object_unref(icon);

    gtk_window_move (GTK_WINDOW (appGUI->main_window), config.window_x, config.window_y);
    gtk_window_set_default_size (GTK_WINDOW(appGUI->main_window), config.window_size_x, config.window_size_y);
    gtk_window_set_resizable (GTK_WINDOW (appGUI->main_window), TRUE);

    g_signal_connect (G_OBJECT (appGUI->main_window), "delete_event",
                      G_CALLBACK(gui_window_closed_cb), appGUI);
    g_signal_connect (G_OBJECT (appGUI->main_window), "key_press_event",
                        G_CALLBACK (key_press_cb), appGUI);

    action_group = gtk_action_group_new ("lss_actions");
    gtk_action_group_add_actions (action_group, entries, n_entries, NULL);
    gtk_action_group_add_toggle_actions (action_group, toggle_entries, n_toggle_entries, NULL);
    gtk_action_group_set_sensitive(action_group, TRUE);

    appGUI->uim_widget = gtk_ui_manager_new ();

    gtk_ui_manager_insert_action_group (appGUI->uim_widget, action_group, 0);

    gtk_window_add_accel_group (GTK_WINDOW (appGUI->main_window),
                                gtk_ui_manager_get_accel_group (appGUI->uim_widget));

    if (!gtk_ui_manager_add_ui_from_string (appGUI->uim_widget, ui_info, -1, NULL)) {
        g_message ("building menus failed");
    }

    gtk_ui_manager_ensure_update (appGUI->uim_widget);

    g_signal_connect (G_OBJECT(gtk_ui_manager_get_widget (appGUI->uim_widget, "/popup_menu/about")), "activate",
                      G_CALLBACK(display_about_cb), appGUI);
    g_signal_connect (G_OBJECT(gtk_ui_manager_get_widget (appGUI->uim_widget, "/popup_menu/quit")), "activate",
                      G_CALLBACK(quit_selected_cb), appGUI);

    g_signal_connect (G_OBJECT(gtk_ui_manager_get_widget (appGUI->uim_widget, "/popup_menu/send_to_gvim")), "toggled",
                      G_CALLBACK(option_send_to_gvim_cb), appGUI);
    g_signal_connect (G_OBJECT(gtk_ui_manager_get_widget (appGUI->uim_widget, "/popup_menu/send_to_clipboard")), "toggled",
                      G_CALLBACK(option_send_to_clipboard_cb), appGUI);
    g_signal_connect (G_OBJECT(gtk_ui_manager_get_widget (appGUI->uim_widget, "/popup_menu/single_click")), "toggled",
                      G_CALLBACK(option_single_click_cb), appGUI);
    g_signal_connect (G_OBJECT(gtk_ui_manager_get_widget (appGUI->uim_widget, "/popup_menu/display_symbol_name")), "toggled",
                      G_CALLBACK(option_display_symbol_name_cb), appGUI);
    g_signal_connect (G_OBJECT(gtk_ui_manager_get_widget (appGUI->uim_widget, "/popup_menu/enable_ams_symbols")), "toggled",
                      G_CALLBACK(option_enable_ams_symbols_cb), appGUI);

   /*-------------------------------------------------------------------------------------*/

    vbox2 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox2);
    gtk_container_add (GTK_CONTAINER (appGUI->main_window), vbox2);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);
    gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);

    /* groups list */

    scrolledwindow_list = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow_list);
    gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow_list), 6);
    gtk_box_pack_start (GTK_BOX (hbox), scrolledwindow_list, FALSE, TRUE, 0);
    gtk_widget_set_size_request (GTK_WIDGET(scrolledwindow_list), 170, -1);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow_list), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow_list), GTK_SHADOW_IN);

    appGUI->groups_list_store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
    appGUI->groups_list = gtk_tree_view_new_with_model(GTK_TREE_MODEL(appGUI->groups_list_store));
    g_signal_connect_swapped (G_OBJECT (appGUI->groups_list), "button_press_event",
                              G_CALLBACK (mouse_button_press_cb), appGUI);
    gtk_widget_show (appGUI->groups_list);

    appGUI->groups_list_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (appGUI->groups_list));
    g_signal_connect(G_OBJECT(appGUI->groups_list_selection), "changed", G_CALLBACK(group_selected_cb), appGUI);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes (_("Symbol groups"), renderer, "text", 0, NULL);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN(column),
                                    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), FALSE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(appGUI->groups_list), column);
    gtk_tree_view_set_enable_search (GTK_TREE_VIEW(appGUI->groups_list), FALSE);

    gtk_container_add (GTK_CONTAINER (scrolledwindow_list), appGUI->groups_list);

    /* symbols list */

    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox1);
    gtk_box_pack_start (GTK_BOX (hbox), vbox1, TRUE, TRUE, 0);

    scrolledwindow_icons = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow_icons);
    gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow_icons), 6);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow_icons), GTK_SHADOW_IN);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow_icons), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    gtk_box_pack_start (GTK_BOX (vbox1), scrolledwindow_icons, TRUE, TRUE, 0);

    appGUI->icons_store = gtk_list_store_new (NUM_COLS, GDK_TYPE_PIXBUF, G_TYPE_STRING);

    appGUI->icons_view = gtk_icon_view_new_with_model (GTK_TREE_MODEL (appGUI->icons_store));
    g_signal_connect_swapped (G_OBJECT (appGUI->icons_view), "button_press_event",
                              G_CALLBACK (mouse_button_press_cb), appGUI);
    gtk_widget_show(appGUI->icons_view);
    gtk_icon_view_set_pixbuf_column (GTK_ICON_VIEW (appGUI->icons_view), COL_PIXBUF);

    g_signal_connect (G_OBJECT (appGUI->icons_view), "selection_changed",
                      G_CALLBACK (icon_activated_sc_cb), appGUI);
    g_signal_connect (G_OBJECT (appGUI->icons_view), "item_activated",
                      G_CALLBACK (icon_activated_dc_cb), appGUI);

    gtk_container_add (GTK_CONTAINER (scrolledwindow_icons), appGUI->icons_view);
    g_object_unref (appGUI->icons_store);

    drag_and_drop_init (appGUI->icons_view, appGUI);

    /* statusbar */

    appGUI->statusbar = gtk_statusbar_new ();
    gtk_widget_show (appGUI->statusbar);

    appGUI->statusbar_context_id = gtk_statusbar_get_context_id( GTK_STATUSBAR(appGUI->statusbar), "lss");

    gtk_box_pack_start (GTK_BOX (vbox2), appGUI->statusbar, FALSE, TRUE, 0);

    /* popup menu */

    appGUI->popup_menu = gtk_ui_manager_get_widget (appGUI->uim_widget, "/popup_menu");

    /* restore state of toggle items */

    action = gtk_ui_manager_get_action(appGUI->uim_widget, "/popup_menu/send_to_gvim");
    gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), config.send_to_gvim);
    action = gtk_ui_manager_get_action(appGUI->uim_widget, "/popup_menu/send_to_clipboard");
    gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), config.send_to_clipboard);
    action = gtk_ui_manager_get_action(appGUI->uim_widget, "/popup_menu/single_click");
    gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), config.single_click);
    action = gtk_ui_manager_get_action(appGUI->uim_widget, "/popup_menu/display_symbol_name");
    gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), config.display_symbol_name);
    action = gtk_ui_manager_get_action(appGUI->uim_widget, "/popup_menu/enable_ams_symbols");
    gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), config.enable_ams_symbols);

    display_names_list(appGUI);
    set_cursor_on_group(config.current_group, appGUI);
    gtk_statusbar_push (GTK_STATUSBAR(appGUI->statusbar), appGUI->statusbar_context_id, _("Please select symbol..."));

    gtk_widget_show(appGUI->main_window);

}

/*------------------------------------------------------------------------------*/

