/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#ifndef AutoPtr_H_Included
#define AutoPtr_H_Included


template <class DataT, bool bArray = false>

class RefCounter
  {
    private:
      DataT         * m_pData;
      unsigned long   m_count;

    public:
                      RefCounter (DataT * pData)
                        : m_pData (pData),
                          m_count (1L)
                      {
                      }

      virtual       ~ RefCounter ()
                      {
                        if ( bArray )
                          delete [] m_pData;
                        else
                          delete m_pData;
                      }

      DataT *         getPtr   ()       { return m_pData; }
      const DataT *   getPtr   () const { return m_pData; }

      unsigned long   getCount () const { return m_count; }
      void            inc      ()       { m_count++; }
      void            dec      ()       { if ( 0L < m_count ) m_count--; }
  };


/*****************************************************************************/
/**
 *
 * @brief Template class for managed pointers
 *
 * AutoPtr is an attempt to add garbage collection cababilities to C++. Any
 * pointer that is managed by an AutoPtr wrapper will automatically be
 * deleted when the last existing reference to it is removed.
 *
 * Be careful, however, not to mix raw pointers with managed pointers, as C++
 * itself (i.e. your compiler) does not know about the reference counting that
 * AutoPtr does. So, if you assign a managed data pointer to a raw pointer,
 * you can never be sure if that is still valid when you want to use it, if
 * you don't assert that the AutoPtr is "alive", too. It might well be that
 * your AutoPtr is the last reference to that data, when it gets destroyed,
 * which will cause the data itself to be deleted, too).
 *
 * But as long as there is at least one AutoPtr for it, you can be sure
 * that it is safe to use that data.
 *
 * You will never have to (AND MUST NOT!!!) delete the data yourself. Just
 * remove the reference by destroying the AutoPtr or assigning a different
 * pointer to it (which may also be NULL).
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray = false>

class AutoPtr
  {
    protected:
      typedef AutoPtr<DataT, bArray>    AutoPtrImp;


    private:
      RefCounter<DataT, bArray> * m_pRefCounter;


    protected:
      RefCounter<DataT, bArray> * getRefCounter () const;


    public:
                     AutoPtr ();
                     AutoPtr (DataT * pData);
                     AutoPtr (const AutoPtrImp & ptr);
      virtual       ~AutoPtr ();

      unsigned long  getRefCount () const;

      bool           isValid () const;
      bool           isNull () const;

      DataT *        getPtr ();
      const DataT *  getPtr () const;

      DataT *        operator -> ();
      const DataT *  operator -> () const;

      AutoPtrImp &   operator = (DataT * pData);
      AutoPtrImp &   operator = (const AutoPtrImp & ptr);

      bool           operator == (const DataT * pData) const;
      bool           operator == (const AutoPtrImp & ptr) const;
      bool           operator != (const DataT * pData) const;
      bool           operator != (const AutoPtrImp & ptr) const;

      operator DataT * ();
      operator const DataT * () const;
  };


/*****************************************************************************/
/**
 *
 * @brief Creates an invalid AutoPtr (i.e. that points to NULL)
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline AutoPtr<DataT, bArray>::AutoPtr()
{
    m_pRefCounter = 0;
}


/*****************************************************************************/
/**
 *
 * @brief Creates an AutoPtr wrapper for a data pointer
 *
 * @param	pData : The data pointer that should be wrapped
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline AutoPtr<DataT, bArray>::AutoPtr(DataT * pData)
{
    if ( pData )
        m_pRefCounter = new RefCounter<DataT, bArray>(pData);
    else
        m_pRefCounter = 0;
}


/*****************************************************************************/
/**
 *
 * @brief Creates another AutoPtr from an existing one
 *
 * The new AutoPtr points to the same data (if any) that ptr points to. The
 * internal reference count for this data pointer will be incremented by 1.
 *
 * @param	ptr	: Reference to an existing AutoPtr
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline AutoPtr<DataT, bArray>::AutoPtr(const AutoPtrImp & ptr)
{
    m_pRefCounter = ptr.getRefCounter();

    if ( m_pRefCounter )
        m_pRefCounter->inc();
}


/*****************************************************************************/
/**
 *
 * @brief Destructs this AutoPtr
 *
 * The internal data pointer reference count is decremented by 1. If no other
 * AutoPtr points to the managed data, the data pointer itself is deleted.
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline AutoPtr<DataT, bArray>::~AutoPtr()
{
    if ( m_pRefCounter )
    {
        m_pRefCounter->dec();

        if ( 0L == m_pRefCounter->getCount() )
            delete m_pRefCounter;
    }
}


/*****************************************************************************/
/**
 *
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline RefCounter<DataT, bArray> * AutoPtr<DataT, bArray>::getRefCounter() const
{
    return m_pRefCounter;
}


/*****************************************************************************/
/**
 *
 * @returns	The number of references for this data pointer
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline unsigned long  AutoPtr<DataT, bArray>::getRefCount() const
{
    return m_pRefCounter ? m_pRefCounter->getCount() : 0L;
}


/*****************************************************************************/
/**
 *
 * @brief Determines if this AutoPtr points to anything
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline bool AutoPtr<DataT, bArray>::isValid() const
{
    return !isNull();
}


/*****************************************************************************/
/**
 *
 * @brief Determines if this AutoPtr is a NULL-pointer
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline bool AutoPtr<DataT, bArray>::isNull() const
{
    return 0 == m_pRefCounter || 0 == m_pRefCounter->getPtr();
}


/*****************************************************************************/
/**
 *
 * @brief Gets the actual data pointer
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline DataT * AutoPtr<DataT, bArray>::getPtr()
{
    return m_pRefCounter ? m_pRefCounter->getPtr() : 0;
}


/*****************************************************************************/
/**
 *
 * @brief Gets the actual data pointer
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline const DataT * AutoPtr<DataT, bArray>::getPtr() const
{
    return m_pRefCounter ? m_pRefCounter->getPtr() : 0;
}


/*****************************************************************************/
/**
 *
 * @brief  Assigns a different data pointer to this AutoPtr instance.
 *
 * The reference count for the previous data pointer is automatically
 * decremented.
 *
 * BE CAREFUL: Do not assign a raw data pointer that is already wrapped by a
 *             AutoPtr. Assign the appropriate AutoPtr instance in that
 *             case!
 *
 * @param	pData		The new data pointer
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline AutoPtr<DataT, bArray> & AutoPtr<DataT, bArray>::operator = (DataT * pData)
{
    if ( !m_pRefCounter || m_pRefCounter->getPtr () != pData )
    {
        if ( m_pRefCounter )
        {
            m_pRefCounter->dec();

            if ( 0L == m_pRefCounter->getCount() )
                delete m_pRefCounter;
        }

        m_pRefCounter = new RefCounter<DataT, bArray>(pData);
    }

    return *this;
}


/*****************************************************************************/
/**
 *
 * @brief  Assigns a different pointer to this AutoPtr instance.
 *
 * The reference count for the previous data pointer is automatically
 * decremented, and the reference count for the data that ptr points to is
 * incremented by 1.
 *
 * @param	ptr		Our AutoPtr instance now points to the same data
 *                  as this one.
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline AutoPtr<DataT, bArray> & AutoPtr<DataT, bArray>::operator = (const AutoPtrImp & ptr)
{
    if ( m_pRefCounter != ptr.getRefCounter() )
    {
        if ( m_pRefCounter )
        {
            m_pRefCounter->dec();

            if ( 0L == m_pRefCounter->getCount() )
                delete m_pRefCounter;
        }

        m_pRefCounter = ptr.getRefCounter();

        if ( m_pRefCounter )
            m_pRefCounter->inc();
    }

    return *this;
}


/*****************************************************************************/
/**
 *
 * @brief Interpret this AutoPtr instance as pointer to DataT
 *
 * @returns	Pointer to the actual data
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline AutoPtr<DataT, bArray>::operator const DataT * () const
{
    return getPtr();
}


/*****************************************************************************/
/**
 *
 * @brief Interpret this AutoPtr instance as pointer to DataT
 *
 * @returns	Pointer to the actual data
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline AutoPtr<DataT, bArray>::operator DataT * ()
{
    return getPtr();
}


/*****************************************************************************/
/**
 *
 * @brief Overloads the "points to"-operator.
 *
 * Interprets this AutoPtr instance as pointer to DataT
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline DataT * AutoPtr<DataT, bArray>::operator -> ()
{
    return getPtr();
}


/*****************************************************************************/
/**
 *
 * @brief Overloads the "points to"-operator.
 *
 * Interprets this AutoPtr instance as pointer to DataT
 *
 * @returns	Pointer to the actual data
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline const DataT * AutoPtr<DataT, bArray>::operator -> () const
{
    return getPtr();
}


/*****************************************************************************/
/**
 *
 * @brief Compares this AutoPtr instance with another pointer to DataT
 *
 * @returns	true	This AutoPtr instance points to pData
 * @returns	false	This AutoPtr instance does not point to pData
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline bool AutoPtr<DataT, bArray>::operator == (const DataT * pData) const
{
    return getPtr() == pData;
}


/*****************************************************************************/
/**
 *
 * @brief Compares this AutoPtr instance with another AutoPtr instance
 *
 * @returns	true	This AutoPtr instance points to the same data as ptr
 * @returns	false	This AutoPtr instance points to different data than ptr
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline bool AutoPtr<DataT, bArray>::operator == (const AutoPtrImp & ptr) const
{
    return getPtr() == ptr.getPtr();
}


/*****************************************************************************/
/**
 *
 * @brief Tests if this instance does not point to pData
 *
 * @returns	true	This AutoPtr instance does not point to pData
 * @returns	false	This AutoPtr instance DOES point to pData
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline bool AutoPtr<DataT, bArray>::operator != (const DataT * pData) const
{
    return ! operator == (pData);
}


/*****************************************************************************/
/**
 *
 * @brief Tests if this instance does not point to the same data as ptr
 *
 * @returns	true	This AutoPtr instance does not point to the same data
 * @returns	false	This AutoPtr instance DOES point to the same data
 *
 */
/*****************************************************************************/

template <class DataT, bool bArray>

inline bool AutoPtr<DataT, bArray>::operator != (const AutoPtrImp & ptr) const
{
    return ! operator == (ptr);
}

#endif
