/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#ifndef ActionsView_H_Included
#define ActionsView_H_Included

#include <qwidget.h>
#include <qmutex.h>

#include "PackageInfo.h"


class QListViewItemIterator;
class QListView;
class QPushButton;
class QCheckBox;
class QMutex;
class QLabel;
class QLineEdit;

class Process;
class PkgListView;
class ListTitle;

class ActionsView : public QWidget
{
    Q_OBJECT

    private:
      enum
        {
          // Action types...
          c_fl_List     = 0x01,
          c_fl_Info     = 0x02,
          c_fl_Search   = 0x04,
          c_fl_Remove   = 0x08,
          c_fl_Install  = 0x10,
          c_fl_Clean    = 0x20,

          // List types...
          c_fl_Updates    = 0x0100,
          c_fl_Available  = 0x0200,
          c_fl_Installed  = 0x0400,
          c_fl_AllLists   = c_fl_Updates | c_fl_Available | c_fl_Installed
        };

      enum E_Action
        {
          c_NoAction        = 0,
          c_ListAvail       = c_fl_List | c_fl_Available,
          c_ListUpdates     = c_fl_List | c_fl_Updates,
          c_ListInstalled   = c_fl_List | c_fl_Installed,
          c_InfoAvail       = c_fl_Info | c_fl_Available,
          c_InfoUpdates     = c_fl_Info | c_fl_Updates,
          c_InfoInstalled   = c_fl_Info | c_fl_Installed,
          c_Search          = c_fl_Search | c_fl_Updates | c_fl_Available,
          c_Info            = c_fl_Info   | c_fl_AllLists,
          c_Remove          = c_fl_Remove,
          c_Install         = c_fl_Install,
          c_FindAndInstall  = c_fl_Search | c_fl_Install,
          c_InfoAndInstall  = c_fl_Info | c_fl_Install,
          c_Clean           = c_fl_Clean
        };

      enum E_List
        {
          c_NoList       = 0,
          c_Updates      = c_fl_Updates,
          c_Available    = c_fl_Available,
          c_Installed    = c_fl_Installed,
          c_SearchResult = c_fl_Installed | c_fl_Available
        };

      enum E_InfoTag
        {
          c_InfoTag_None,
          c_InfoTag_Name,
          c_InfoTag_Repo,
          c_InfoTag_Arch,
          c_InfoTag_Version,
          c_InfoTag_Release,
          c_InfoTag_Size,
          c_InfoTag_Summary,
          c_InfoTag_Description
        };

      typedef std::map<QString, PackageInfo::Ptr> PackageMap;
      typedef std::map<const QListViewItem *, PackageInfo::Ptr> PkgListMap;

    private:
      PackageMap        m_packageMap;
      PkgListMap        m_pkgListMap;
      QPushButton     * m_pButtonListUpdates,
                      * m_pButtonListAvail,
                      * m_pButtonListInstalled,
                      * m_pButtonSearch,
                      * m_pButtonInstall,
                      * m_pButtonUpdate,
                      * m_pButtonRemove,
                      * m_pButtonSelectAll,
                      * m_pButtonDeselectAll,
                      * m_pButtonFindNextInList,
                      * m_pButtonFindPrevInList;
      QCheckBox       * m_pCheckObsoletes,
                      * m_pCheckCache,
                      * m_pCheckDescriptions;
      QLabel          * m_pLabelFindInList,
                      * m_pSelectionInfo;
      QLineEdit       * m_pEditFindInList;
      PkgListView     * m_pListView;
      Process         * m_pProcess;
      E_Action          m_currentAction;
      E_List            m_currentList;
      QMutex          * m_pMutex;
      QString           m_strStdOutBuf,
                        m_strStdErrBuf,
                        m_strYumOutput;
      PackageInfo::Ptr  m_ptrCurPackage;  // For info mode: The package info currently gathered
      E_InfoTag         m_curInfoTag;
      QListViewItem   * m_pCurrentItem;
      bool              m_bActionsEnabled;
      int               m_numCheckedItems;
      double            m_totalPkgSize;
      ListTitle       * m_pListTitle;


    public:
                        ActionsView     (QWidget * pParent);
      virtual         ~ ActionsView     ();

      void              stopAction      ();
      void              enableActions   (bool bEnable = true);


    private:
      bool              startAction     ();

      void              findNextInList  (QListViewItemIterator & start);
      void              findPrevInList  (QListViewItemIterator & start);

      PackageInfo::Ptr  getPackageFromMap   (const QString & strKey) const;
      PackageInfo::Ptr  getPackageFromMap   (const PackageInfo & package) const;
      PackageInfo::Ptr  getPackageFromMap   (const QListViewItem * pItem) const;
      void              putPackageIntoMap   (PackageInfo::Ptr ptrPackage, QListViewItem * pItem);
      void              addPackageToList    (PackageInfo::Ptr ptrPackage);

      void              addYumLineInfo      (QString strLine);
      void              addYumInfo          (QString strLine);
      void              addYumError         (QString strLine);

      void              switchAffectedRepos ();

      void              showChangeLog       (PackageInfo::Ptr ptrPkgInfo);
      void              showDependencies    (PackageInfo::Ptr ptrPkgInfo);

      void              clearPkgList        ();
      void              updateSelectionInfo ();
      double            getPkgSize          (const QListViewItem * pItem) const;

    signals:
      void              yumInfo             (QString strLine);
      void              yumPkgInfo          (QString strLine);
      void              yumError            (QString strLine);
      void              actionStarted       (Process * pProcess);
      void              actionTerminated    (Process * pProcess);
      void              showPackageInfo     (const PackageInfo & info);


    private slots:
      void              slotListContextMenu (QListViewItem *, const QPoint &, int);

    public slots:
      void              slotLaunchFinished  ();
      void              slotProcessExited   ();

      void              slotReadyStdout ();
      void              slotReadyStderr ();
      void              listAvailable   ();
      void              listUpdates     ();
      void              listInstalled   ();
      void              slotSearch      ();
      void              slotSelectAll   ();
      void              slotDeselectAll ();
      void              slotInstall     ();
      void              slotUpdate      ();
      void              slotRemove      ();
      void              cleanCache      ();
      void              slotLocalUpdate ();
      void              slotLocalInstall();

      void              slotListItemSelected(QListViewItem * pItem);

      void              slotFindInList      (const QString & strText);
      void              slotFindPrevInList  ();
      void              slotFindNextInList  ();
};

#endif
