/* 
 * Copyright (C) 2002-2004 Benoit Poulot-Cazajous
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *   
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *   
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _POSIX_C_SOURCE 199506L
#define _XOPEN_SOURCE 500
#define _GNU_SOURCE

#include <sys/types.h>
#include <sys/param.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/select.h>
#include <alloca.h>
#include <signal.h>
#include <limits.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <poll.h>
#include <dlfcn.h>

#define NDEBUG_NO
#include <assert.h>

#include "cachecc1.h"

#define LOG(...) if (flog) do { fprintf(flog, "%ld %s:%d --> ", mypid, __FILE__, __LINE__); fprintf(flog, __VA_ARGS__); fprintf(flog, "\n"); fflush(flog); } while (0)

static int pip[2];

static int main_cachecc1(char *name, char *argv[], char *envp[])
{
    int i, j;
    unsigned char buf[65536];
    char fname[PATH_MAX];
    char fntmp[PATH_MAX];
    char compname[PATH_MAX];
    long child;
    int status, rc, lc, argc;
    int fdlog = 0;
    FILE *flog = 0;
    int f, g;
    struct stat st;
    int cacher, cachew;
    char *input;
    char *output;
    char *cppdefs;
    char *myself;
    long mypid;

        /* first, stat the "compiler", it may not exist */
    if (stat(name, &st)) goto direct;
    
    mypid = (long) getpid();
    compname[0] = 0;

        /* read the environnment */
    read_env(envp);
            
        /* sometimes, disable cache */
    for (i = 0; argv[i]; ++i) {
        if (argv[i][0] != '-') continue;
        if (!strcmp(argv[i], "-aux-info")) env_dir = 0;
        if (argv[i][1] == 'd' && argv[i][2] && !argv[i][3]) env_dir = 0;
    }
    argc = i;

        /* dont cache without a cache */
    if (!env_dir) goto direct;
    lc = strlen(env_dir);
    
        /* callee detection */
    myself = 0;
    i = strlen(name);
    cacher = cachew = 0;
    if (i >= 3 && !strcmp(&name[i-3], "cc1")) {
        myself = "cc1";
        cacher = ! OPTS_NO_R_C;
        cachew = ! OPTS_NO_W_C;
    }
    if (i >= 7 && !strcmp(&name[i-7], "cc1plus")) {
        myself = "cc+";
        cacher = ! OPTS_NO_R_CPP;
        cachew = ! OPTS_NO_W_CPP;
    }
    if (i >= 2 && !strcmp(&name[i-2], "as")) {
        myself = "as+";
        cacher = ! OPTS_NO_R_AS;
        cachew = ! OPTS_NO_W_AS;
    }
    if (!myself) goto direct;

        /* init crcs with the compiler signature */
    crc_init((unsigned int) st.st_mtime,
             (unsigned int) st.st_size,
             (unsigned int) st.st_ino,
             (unsigned int) st.st_dev,
             0);
    for (j = 0; name[j]; ++j) crc_put(name[j]);
    sprintf(fname, "%s/%s/%s", env_dir, myself, crc_get());
    i = readlink(fname, fntmp, sizeof(fntmp));
    if (i < 1) {
            /* unknown compiler, sum it */
        int f = open(name, O_RDONLY);
        if (f < 0) goto direct;
        crc_init(0, 0, 0, 0, 1);
        while (1) {
            int n = read(f, buf, sizeof(buf));
            if (n <= 0) break;
            for (j = 0; j < n; ++j) crc_put(buf[j]);
        }
        close(f);
        fname[lc] = 0;
        (void) mkdir(fname, 0777);
        fname[lc] = '/';
        fname[lc+4] = 0;
        (void) mkdir(fname, 0777);
        fname[lc+4] = '/';
        strcpy(fntmp, crc_get());
        i = strlen(fntmp);
        if (symlink(fntmp, fname)) goto direct;
    }
        /* known compiler */
    crc_init(0, 0, 0, 0, 2);
    for (j = 0; j < i; ++j) crc_put(((unsigned char*)fntmp)[j]);
    strcpy(compname, crc_get());

    if (env_log) {
        fdlog = open(env_log, O_RDWR | O_APPEND | O_CREAT, 0666);
        flog = fdopen(fdlog, "a");
        if (flog) fprintf(flog, "\n");
    }
    LOG("****** New run for %s", name);
    LOG("Log on fd=%d f=%p", fdlog, flog);

        /* parse parameters and init crcs */
    input = output = 0;
    cppdefs = 0;
    for (i = 1; argv[i]; ++i) {
        if (!strcmp(argv[i-1], "-iprefix")) continue;
        if (!strncmp(argv[i-1], "-D", 2)) cppdefs = argv[i-1];
        if (!strncmp(argv[i-1], "-I", 2)) cppdefs = argv[i-1];
        if (!strcmp(argv[i-1], "-o") && argv[i][0] != '-') {
            output = argv[i];
        } else if (argv[i][0] != '-' && !input) {
            input = argv[i];
        } else {
            for (j = 0; argv[i][j]; ++j) crc_put(argv[i][j]);
        }
    }
    if (!input) goto direct;
    if (!output) goto direct;
    if (cppdefs) {
            /* followcc1 asked if this is gcc-3.x. Answer */
        if (env_ifgcc3) mkdir(env_ifgcc3, 0777);
            /* next time, the integrated cpp will be turned off, hopefully */
        goto direct;
    }

        /* sum the input */
    g = open(input, O_RDONLY);
    if (g < 0) goto direct;
    do {
        unsigned char *p, *q, *l;
        int s;
        if (fstat(g, &st)) goto direct;
        p = mmap(NULL, st.st_size, PROT_READ, MAP_PRIVATE, g, 0);
        if (p == (unsigned char*) -1) goto direct;
        l = p+st.st_size;
        s = st.st_size / sizeof(unsigned int);
        crc_block((unsigned int *)p, s);
        q = p + (s * sizeof(unsigned int));
        while (q < l) crc_put(*q++);
        munmap(p, st.st_size);
    } while (0);
    
    sprintf(fname, "%s//%s", env_dir, crc_get());
    fname[lc+1] = fname[lc+3];
    fname[lc+3] = '/';

    do {
        if (!cacher) break;
        f = open(fname, O_RDONLY);
        LOG("open %s -> %d", fname, f);
        if (f < 0) break;
            /* cache hit */
        LOG("CACHE HIT");
            /* make a copy */
        unlink(output);
        g = open(output, O_RDWR | O_CREAT | O_TRUNC, 0666);
        if (g < 0) goto direct;
        copy(f, g, output);
        close(g);
        return 0;
    } while (0);

        /* cache miss */
    LOG("CACHE MISS");
    rc = pipe(pip);
    assert(rc == 0);
    if ((child = fork())) {
        int x, usestdout;
        assert(child > 0);
        close(pip[1]);
            /*
             * read the command output : if it writes anything,
             * then disable caching, unless CACHECC1IGNOREWARNINGS is set
             */
        usestdout = 0;
        while (1) {
            int n;
            buf[0] = 0;
            n = read(pip[0], buf, sizeof(buf));
            LOG("Got %d", n);
            if (n <= 0) break;
            send(2, buf, n);
            usestdout = 1;
        }
        if (env_ignorewarnings) usestdout = 0;
            
            /* wait for the command to finish */
        errno = 0;
        rc = waitpid(child, &status, 0);
        x = errno;
        if (rc == child) x = WEXITSTATUS(status);
        LOG("WAITPID(%ld) -> %d %d %d", child, rc, x, usestdout);
        do {
            int f, g;
            if (x || usestdout) break;
            if (!cachew) break;
                /* make a copy */
            sprintf(fntmp, "%s.%ld", fname, mypid);
            f = open(fntmp, O_RDWR | O_CREAT | O_EXCL, 0666);
            if (f < 0) {
                fntmp[lc+3] = 0;
                (void) mkdir(fntmp, 0777);
                fntmp[lc+3] = '/';
                f = open(fntmp, O_RDWR | O_CREAT | O_EXCL, 0666);
            }
            if (f < 0) break;
            g = open(output, O_RDONLY);
            if (g < 0) break;
            copy(g, f, fntmp);
            close(f);
            close(g);
            rc = rename(fntmp, fname);
            LOG("rename %s %s => %d", fntmp, fname, rc);
        } while (0);
        return x;
    } else {
        int rc;
        rc = dup2 (pip[1], 2);
        assert(rc == 2);
        rc = dup2 (pip[1], 1);
        assert(rc == 1);
        close(pip[0]);
        close(pip[1]);
            /* fall through */
    }
        /* cache miss, try distcc for C/C++ compilations */
    if (env_distccdir && env_distccdir[0] == '/' && myself[0] == 'c') {
        char **nargv;
        nargv = alloca((argc+5)*sizeof(char*));
        if (distcc1(name, argv, nargv, input, output, compname)) {
            argv = nargv;
            name = argv[0];
                /* cheap recursion prevention */
            env_distccdir[0] = ':';
        }
    }
  direct:
    if (envp) {
        rc = real_execve(name, argv, envp);
    } else {
        rc = real_execv(name, argv);
    }
    fprintf(stderr, "execv(%s) -> %d (errno = %d)\n",
            name, rc, errno);
    return rc;
}

int exec_cachecc1(char *name, char *argv[], char *envp[])
{
    int rc = main_cachecc1(name, argv, envp);
    exit(rc);
}

/* arch-tag: 91c20347-1b13-4003-9ccc-bd4b6b68332c */
