
/***************************************************************************
 * PacketStats.cc -- The PacketStats class handles packet statistics. It   *
 * is intended to keep track of the number of packets and bytes sent and   *
 * received, keep track of start and finish times, etc.                    *
 *                                                                         *
 ***********************IMPORTANT NMAP LICENSE TERMS************************
 *                                                                         *
 * The Nmap Security Scanner is (C) 1996-2010 Insecure.Com LLC. Nmap is    *
 * also a registered trademark of Insecure.Com LLC.  This program is free  *
 * software; you may redistribute and/or modify it under the terms of the  *
 * GNU General Public License as published by the Free Software            *
 * Foundation; Version 2 with the clarifications and exceptions described  *
 * below.  This guarantees your right to use, modify, and redistribute     *
 * this software under certain conditions.  If you wish to embed Nmap      *
 * technology into proprietary software, we sell alternative licenses      *
 * (contact sales@insecure.com).  Dozens of software vendors already       *
 * license Nmap technology such as host discovery, port scanning, OS       *
 * detection, and version detection.                                       *
 *                                                                         *
 * Note that the GPL places important restrictions on "derived works", yet *
 * it does not provide a detailed definition of that term.  To avoid       *
 * misunderstandings, we consider an application to constitute a           *
 * "derivative work" for the purpose of this license if it does any of the *
 * following:                                                              *
 * o Integrates source code from Nmap                                      *
 * o Reads or includes Nmap copyrighted data files, such as                *
 *   nmap-os-db or nmap-service-probes.                                    *
 * o Executes Nmap and parses the results (as opposed to typical shell or  *
 *   execution-menu apps, which simply display raw Nmap output and so are  *
 *   not derivative works.)                                                *
 * o Integrates/includes/aggregates Nmap into a proprietary executable     *
 *   installer, such as those produced by InstallShield.                   *
 * o Links to a library or executes a program that does any of the above   *
 *                                                                         *
 * The term "Nmap" should be taken to also include any portions or derived *
 * works of Nmap.  This list is not exclusive, but is meant to clarify our *
 * interpretation of derived works with some common examples.  Our         *
 * interpretation applies only to Nmap--we don't speak for other people's  *
 * GPL works.                                                              *
 *                                                                         *
 * If you have any questions about the GPL licensing restrictions on using *
 * Nmap in non-GPL works, we would be happy to help.  As mentioned above,  *
 * we also offer alternative license to integrate Nmap into proprietary    *
 * applications and appliances.  These contracts have been sold to dozens  *
 * of software vendors, and generally include a perpetual license as well  *
 * as providing for priority support and updates as well as helping to     *
 * fund the continued development of Nmap technology.  Please email        *
 * sales@insecure.com for further information.                             *
 *                                                                         *
 * As a special exception to the GPL terms, Insecure.Com LLC grants        *
 * permission to link the code of this program with any version of the     *
 * OpenSSL library which is distributed under a license identical to that  *
 * listed in the included COPYING.OpenSSL file, and distribute linked      *
 * combinations including the two. You must obey the GNU GPL in all        *
 * respects for all of the code used other than OpenSSL.  If you modify    *
 * this file, you may extend this exception to your version of the file,   *
 * but you are not obligated to do so.                                     *
 *                                                                         *
 * If you received these files with a written license agreement or         *
 * contract stating terms other than the terms above, then that            *
 * alternative license agreement takes precedence over these comments.     *
 *                                                                         *
 * Source is provided to this software because we believe users have a     *
 * right to know exactly what a program is going to do before they run it. *
 * This also allows you to audit the software for security holes (none     *
 * have been found so far).                                                *
 *                                                                         *
 * Source code also allows you to port Nmap to new platforms, fix bugs,    *
 * and add new features.  You are highly encouraged to send your changes   *
 * to nmap-dev@insecure.org for possible incorporation into the main       *
 * distribution.  By sending these changes to Fyodor or one of the         *
 * Insecure.Org development mailing lists, it is assumed that you are      *
 * offering the Nmap Project (Insecure.Com LLC) the unlimited,             *
 * non-exclusive right to reuse, modify, and relicense the code.  Nmap     *
 * will always be available Open Source, but this is important because the *
 * inability to relicense code has caused devastating problems for other   *
 * Free Software projects (such as KDE and NASM).  We also occasionally    *
 * relicense the code to third parties as discussed above.  If you wish to *
 * specify special license conditions of your contributions, just say so   *
 * when you send them.                                                     *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License v2.0 for more details at                         *
 * http://www.gnu.org/licenses/gpl-2.0.html , or in the COPYING file       *
 * included with Nmap.                                                     *
 *                                                                         *
 ***************************************************************************/

#include "stats.h"
#include "nping.h"
#include "NpingOps.h"
#include "output.h"






/*****************************************************************************
 * Copied from timing.cc
 *****************************************************************************/
/* current_rate_history defines how far back (in seconds) we look when
   calculating the current rate. */
RateMeter::RateMeter(double current_rate_history) {
  this->current_rate_history = current_rate_history;
  start_tv.tv_sec = 0;
  start_tv.tv_usec = 0;
  stop_tv.tv_sec = 0;
  stop_tv.tv_usec = 0;
  last_update_tv.tv_sec = 0;
  last_update_tv.tv_usec = 0;
  total = 0.0;
  current_rate = 0.0;
  assert(!isSet(&start_tv));
  assert(!isSet(&stop_tv));
}

void RateMeter::start(const struct timeval *now) {
  assert(!isSet(&start_tv));
  assert(!isSet(&stop_tv));
  if (now == NULL)
    gettimeofday(&start_tv, NULL);
  else
    start_tv = *now;
}

void RateMeter::stop(const struct timeval *now) {
  assert(isSet(&start_tv));

  if( isSet(&stop_tv) )
    return;
    
  if (now == NULL)
    gettimeofday(&stop_tv, NULL);
  else
    stop_tv = *now;
}

/* Update the rates to reflect the given amount added to the total at the time
   now. If now is NULL, get the current time with gettimeofday. */
void RateMeter::update(double amount, const struct timeval *now) {
  struct timeval tv;
  double diff;
  double interval;
  double count;

  assert(isSet(&start_tv));
  //assert(!isSet(&stop_tv)); /* TODO: think about the implications of commenting this line out  */
  /* The thing here is that we need to be able to update the timer even when it has
   * been stopped because of the after-probe-transmission wait-for-replies time, otherwise
   * we would always report 1 packet missing (the response to the last probe) */

  /* Update the total. */
  total += amount;

  if (now == NULL) {
    gettimeofday(&tv, NULL);
    now = &tv;
  }
  if (!isSet(&last_update_tv))
    last_update_tv = start_tv;

  /* Calculate the approximate moving average of how much was recorded in the
     last current_rate_history seconds. This average is what is returned as the
     "current" rate. */

  /* How long since the last update? */
  diff = TIMEVAL_SUBTRACT(*now, last_update_tv) / 1000000.0;

  if (diff < -current_rate_history)
    /* This happened farther in the past than we care about. */
    return;

  if (diff < 0.0) {
    /* If the event happened in the past, just add it into the total and don't
       change last_update_tv, as if it had happened at the same time as the most
       recent event. */
    now = &last_update_tv;
    diff = 0.0;
  }

  /* Find out how far back in time to look. We want to look back
     current_rate_history seconds, or to when the last update occurred,
     whichever is longer. However, we never look past the start. */
  struct timeval tmp;
  /* Find the time current_rate_history seconds after the start. That's our
     threshold for deciding how far back to look. */
  TIMEVAL_ADD(tmp, start_tv, (time_t) (current_rate_history * 1000000.0));
  if (TIMEVAL_AFTER(*now, tmp))
    interval = MAX(current_rate_history, diff);
  else
    interval = TIMEVAL_SUBTRACT(*now, start_tv) / 1000000.0;
  assert(diff <= interval);
  /* If we record an amount in the very same instant that the timer is started,
     there's no way to calculate meaningful rates. Ignore it. */
  if (interval == 0.0)
    return;

  /* To calculate the approximate average of the rate over the last
     interval seconds, we assume that the rate was constant over that interval.
     We calculate how much would have been received in that interval, ignoring
     the first diff seconds' worth:
       (interval - diff) * current_rate.
     Then we add how much was received in the most recent diff seconds. Divide
     by the width of the interval to get the average. */
  count = (interval - diff) * current_rate + amount;
  current_rate = count / interval;

  last_update_tv = *now;
}

double RateMeter::getOverallRate(const struct timeval *now) const {
  double elapsed;

  elapsed = elapsedTime(now);
  if (elapsed <= 0.0)
    return 0.0;
  else
    return total / elapsed;
}

/* Get the "current" rate (actually a moving average of the last
   current_rate_history seconds). If update is true (its default value), lower
   the rate to account for the time since the last record. */
double RateMeter::getCurrentRate(const struct timeval *now, bool update) {
  if (update)
    this->update(0.0, now);

  return current_rate;
}

double RateMeter::getTotal(void) const {
  return total;
}

/* Get the number of seconds the meter has been running: if it has been stopped,
   the amount of time between start and stop, or if it is still running, the
   amount of time between start and now. */
double RateMeter::elapsedTime(const struct timeval *now) const {
  struct timeval tv;
  const struct timeval *end_tv;

  assert(isSet(&start_tv));

  if (isSet(&stop_tv)) {
    end_tv = &stop_tv;
  } else if (now == NULL) {
    gettimeofday(&tv, NULL);
    end_tv = &tv;
  } else {
    end_tv = now;
  }

  return TIMEVAL_SUBTRACT(*end_tv, start_tv) / 1000000.0;
}

/* Returns true if tv has been initialized; i.e., its members are not all
   zero. */
bool RateMeter::isSet(const struct timeval *tv) {
  return tv->tv_sec != 0 || tv->tv_usec != 0;
}

PacketRateMeter::PacketRateMeter(double current_rate_history) {
  packet_rate_meter = RateMeter(current_rate_history);
  byte_rate_meter = RateMeter(current_rate_history);
}

void PacketRateMeter::start(const struct timeval *now) {
  packet_rate_meter.start(now);
  byte_rate_meter.start(now);
}

void PacketRateMeter::stop(const struct timeval *now) {
  packet_rate_meter.stop(now);
  byte_rate_meter.stop(now);
}

/* Record one packet of length len. */
void PacketRateMeter::update(u32 len, const struct timeval *now) {
  packet_rate_meter.update(1, now);
  byte_rate_meter.update(len, now);
}

double PacketRateMeter::getOverallPacketRate(const struct timeval *now) const {
  return packet_rate_meter.getOverallRate(now);
}

double PacketRateMeter::getCurrentPacketRate(const struct timeval *now, bool update) {
  return packet_rate_meter.getCurrentRate(now, update);
}

double PacketRateMeter::getOverallByteRate(const struct timeval *now) const {
  return byte_rate_meter.getOverallRate(now);
}

double PacketRateMeter::getCurrentByteRate(const struct timeval *now, bool update) {
  return byte_rate_meter.getCurrentRate(now, update);
}

unsigned long long PacketRateMeter::getNumPackets(void) const {
  return (unsigned long long) packet_rate_meter.getTotal();
}

unsigned long long PacketRateMeter::getNumBytes(void) const {
  return (unsigned long long) byte_rate_meter.getTotal();
}

/* Method added for nping */
double PacketRateMeter::elapsedTime(void) const {
  return byte_rate_meter.elapsedTime();
}

/*****************************************************************************
 * End of code copied from timing.cc
 *****************************************************************************/






PacketStats::PacketStats(){
    sendPackets=0;
    sendBytes=0;
    recvPackets=0;
    recvBytes=0;
    isstopped=false;
    isstopped_tx=false;
    isstopped_rx=false;

}

PacketStats::~PacketStats(){

}

int PacketStats::addSentPacket(unsigned len){

    this->sendPackets++;
    this->sendBytes+=len;

    this->timer_sent.update(len, NULL);
    
    return OP_SUCCESS;

} /* End of addSentPacket() */



/** Updates packet and byte count for received packets.  This method waits
 * for a mutex because it cannot be used after the clock has been stopped and
 * since addRecvPacket() is called by the thread that is in change of the
 * pcap packet capture, and clockStop() is called by the thread that handles
 * raw packet transmission, there is a need to use this kind of syncronization.
 * */
int PacketStats::addRecvPacket(unsigned len){


  /* Update time only if clock has not been stopped */
  //if ( this->isClockStopped() == false ){

    this->recvPackets++;
    this->recvBytes+=len;

    this->timer_recv.update(len, NULL);

  //}else{
  //  outError(DBG_4, "addRecvPacket(): Cannot update, clock is stopped\n");
  //} 

  return OP_SUCCESS;

} /* End of addRecvPacket() */


int PacketStats::startClock(){

    this->timer_sent.start();
    this->timer_recv.start();
    return OP_SUCCESS;

}




/** Stops the Tx clock */
int PacketStats::stopTxClock(){

  isstopped_tx=true;
  /* If Rx is also stopped already then we set the global clock stop flag */
  if(isstopped_rx==true)
    isstopped=true;
    
  this->timer_sent.stop();

  return OP_SUCCESS;

} /* End of stopTxClock() */


/** Stops the Rx clock */
int PacketStats::stopRxClock(){

  isstopped_rx=true;
  /* If Tx is also stopped already then we set the global clock stop flag */
  if(isstopped_tx==true)
    isstopped=true;
    
  this->timer_recv.stop();

  return OP_SUCCESS;

} /* End of stopTxClock() */




bool PacketStats::isClockStopped(){

  return this->isstopped;

} /* End of isClockStopped() */





u64_t PacketStats::getSentPackets(){

    return this->sendPackets;

} /* End of getSentPackets() */


u64_t PacketStats::getSentBytes(){

    return this->sendBytes;

} /* End of getSentBytes() */



u64_t PacketStats::getRecvPackets(){

    return this->recvPackets;

} /* End of getRecvPackets() */


u64_t PacketStats::getRecvBytes(){

    return this->recvBytes;

} /* End of getRecvBytes() */





u64_t PacketStats::getLostPackets(){

    return this->sendPackets - this->recvPackets;

} /* End of getLostPackets() */




double PacketStats::getLostPacketPercentage(){

  u64_t lost=this->sendPackets-this->recvPackets;
  u64_t sent=this->sendPackets;

  /* Only compute percentage if we actually sent packets, don't do divisions
   * by zero! (this could happen when user presses CTRL-C and we print the
   * stats */
  double percentlost=0.0;
  if( lost!=0 && sent!=0)
    percentlost=((double)lost)/((double)sent);

    return percentlost;

} /* End of getLostPacketPercentage() */


double PacketStats::getLostPacketPercentage100(){

    return getLostPacketPercentage()*100;

} /* End of getLostPacketPercentage100() */








