/***************************************************************************
 *   Copyright (C) 2004 by  ͤ                                            *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "kannadic.h"

#include "canna.h"
#include "anthy.h"
#include "pubdic.h"
#include "dixchange.h"
#include "dlgopen.h"
#include "appendcommand.h"
#include "editcommand.h"
#include "multieditcommand.h"
#include "deletecommand.h"
#include "pastecommand.h"
#include "dlgtypeinfo.h"
#include "importmsimecommand.h"
#include "importpubdiccommand.h"
#include "importdixchangecommand.h"
#include "dlgpreference.h"

#include <kapplication.h>
#include <kaccel.h>
#include <kaction.h>
#include <kactionclasses.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kstatusbar.h>
#include <kshortcut.h>
#include <qcursor.h>
#include <qheader.h>
#include <qptrlist.h>
#include <qstring.h>
#include <qstringlist.h>
#include <kxmlguifactory.h>
#include <kstandarddirs.h>
#include <kstdguiitem.h>
#include <kglobal.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kurl.h>
#include <kinputdialog.h>
#include <qprogressdialog.h>
#include <qheader.h>
#include <qclipboard.h>
#include <qregexp.h>
#include <kcmdlineargs.h>

KannaDic::KannaDic()
{
	m_blChangable = true;
	m_dic = NULL;
	m_cmdHistory = new KCommandHistory( actionCollection() );
	m_timer = new QTimer( this );
	connect( m_timer, SIGNAL(timeout()), SLOT(repaint()) );
	setupCentralWidget();
	setupAcitons();
	createGUI();
	setStandardToolBarMenuEnabled( true );

	setupStatusBar();
	setEditEnabled();

	if( KCmdLineArgs::parsedArgs()->count() != 0 ){
		KURL url( KCmdLineArgs::parsedArgs()->url( 0 ) );
		if( QRegExp( ".*\\.dxc$" ).search( url.path() ) != -1 ){
			actDictionaryType->setCurrentItem( Dictionary::name2type( "Dixchange" ) );
		}else if( QRegExp( ".*\\.t$" ).search( url.path() ) != -1 || QRegExp( ".*\\.ctd$" ).search( url.path() ) != -1 ){
			actDictionaryType->setCurrentItem( Dictionary::name2type( "Pubdic" ) );
		}else{
			KMessageBox::error( (QWidget*)0, i18n( "Can't open %1(unknown suffix)." ).arg( url.path() ) );
			readConfiguration( true );
			return;
		}
		QString dicname = url.path();
		if( dicname.isEmpty() ){
			kdError() << "???" << endl;
		}else{
			dicOpen( Dictionary::name2type( actDictionaryType->currentText() ), dicname );
		}
	}
	readConfiguration( KCmdLineArgs::parsedArgs()->count() != 0 );
}

KannaDic::~KannaDic()
{
	writeConfiguration();
}

void KannaDic::setupCentralWidget()
{
	lvwDictionary = new KListView( this );
	lvwDictionary->addColumn( i18n( "From" ) );
	lvwDictionary->addColumn( i18n( "To" ) );
	lvwDictionary->addColumn( i18n( "Type" ) );
//	lvwDictionary->addColumn( i18n( "Freq" ) );
	lvwDictionary->setAllColumnsShowFocus( true );
	lvwDictionary->setSelectionModeExt( KListView::Extended );
	for( int i = 0; i != lvwDictionary->header()->count(); i++ ){
		lvwDictionary->setColumnWidthMode( i, QListView::Manual );
	}

	connect( lvwDictionary, SIGNAL( executed( QListViewItem* ) ) , this, SLOT( slotEdit() ) );
	connect( lvwDictionary, SIGNAL( returnPressed( QListViewItem* ) ), this, SLOT( slotEdit() ) );
	connect( lvwDictionary, SIGNAL( contextMenu( KListView*, QListViewItem*, const QPoint& ) ), this, SLOT( slotPopup( KListView*, QListViewItem *, const QPoint& ) ) );
	setCentralWidget( lvwDictionary );
}

void KannaDic::readConfiguration( bool blOpened )
{
	KConfig *conf = KGlobal::config();
	conf->setGroup("KannaDic");

	if( !blOpened ){
		actDictionaryType->setCurrentItem( conf->readNumEntry( "dictype", 0 ) );
		QString dicname = conf->readEntry( "dicname", QString::null );
		if( dicname.isEmpty() ){
			setCaption( i18n( "untitled" ) );
		}else{
			Dictionary::DictionaryType dictype = Dictionary::name2type( actDictionaryType->currentText() );
			if( dictype == Dictionary::Anthy ) m_blChangable = false;
			dicOpen( dictype, dicname );
			if( dictype == Dictionary::Anthy ) m_blChangable = true;
		}
	}
	applyMainWindowSettings( conf, "KannaDic" );
	lvwDictionary->restoreLayout( conf, "lvwDictionary" );
}

void KannaDic::writeConfiguration()
{
	KConfig *conf = KGlobal::config();
	conf->setGroup("KannaDic");
	if( m_dic == NULL ){
		conf->writeEntry( "dictype", actDictionaryType->currentItem() );
		conf->writeEntry( "dicname", QString::null );
	}else{
		QStringList list = actDictionaryType->items();
		for ( unsigned int i = 0; i != list.size(); i++ ){
			if( Dictionary::name2type( list[i] ) == m_dic->type() ){
				conf->writeEntry( "dictype", i );
				break;
			}
		}
		conf->writeEntry( "dicname", m_dic->name() );
	}
	lvwDictionary->saveLayout( conf, "lvwDictionary" );
	saveMainWindowSettings( conf, "KannaDic" );
}

void KannaDic::setupAcitons()
{
	KAction *act;

	QString strDic( i18n( "&Dictionary" ) );

	actDictionaryType = new KSelectAction( i18n("Dictionary &Type"), KShortcut(), actionCollection(), "type" );
	QStringList lst;
	if( Canna::isAvailable() ){
		lst.push_back( "Canna" );
	}
	if( Anthy::isAvailable() ){
		lst.push_back( "Anthy" );
	}
	if( Pubdic::isAvailable() ){
		lst.push_back( "Pubdic" );
	}
	if( Dixchange::isAvailable() ){
		lst.push_back( "Dixchange" );
	}
	actDictionaryType->setItems( lst );

	connect( actDictionaryType, SIGNAL( activated( const QString& ) ), this, SLOT( setEditEnabled() ) );


	act = new KAction( i18n("&Create..."), "folder_new", KShortcut(CTRL+Qt::Key_N), this, SLOT( slotDicNew() ), actionCollection(), "createdic" );
	act->setWhatsThis( i18n("Create new user dictionary.") );

	act = KStdAction::open( this, SLOT( slotDicOpen() ), actionCollection(), "open" );
	act->setWhatsThis( i18n("Open dictionary for editing.") );

	act = new KAction( i18n("&Delete..."), "cancel", KShortcut(CTRL+Qt::Key_D), this, SLOT( slotDicDelete() ), actionCollection(), "deletedic" );
	act->setWhatsThis( i18n( "Delete current user dictionary." ) );
	actEdit.append( act );

	act = KStdAction::save( this, SLOT( slotDicSave() ), actionCollection(), "save" );
	act->setWhatsThis( i18n( "Save modified." ) );
	actEdit.append( act );

	act = KStdAction::saveAs( this, SLOT( slotDicSaveAs() ), actionCollection(), "saveas" );
	act->setWhatsThis( i18n( "Save as new." ) );
	actEdit.append( act );

	act = KStdAction::quit( this, SLOT( close() ), actionCollection(), "quit" );
	act->setWhatsThis( i18n( "Close this window." ) );


	act = new KAction( i18n("&Append..."), "filenew", KShortcut(Qt::Key_Insert), this, SLOT( slotAppend() ), actionCollection(), "append" );
	act->setWhatsThis( i18n( "Append new word." ) );
	actEdit.append( act );

	act = new KAction( i18n("&Edit..."), "edit", KShortcut(Qt::Key_Return), this, SLOT( slotEdit() ), actionCollection(), "edit" );
	act->setWhatsThis( i18n( "Edit selected word(s)." ) );
	actEdit.append( act );

	act = new KAction( i18n("&Delete"), "editdelete", KShortcut(Qt::Key_Delete), this, SLOT( slotDelete() ), actionCollection(), "delete" );
	act->setWhatsThis( i18n( "Delete selected word(s)." ) );
	actEdit.append( act );

	act = KStdAction::cut( this, SLOT( slotCut() ), actionCollection(), "edit_cut" );
	act->setWhatsThis( i18n( "Cut to Clipboard." ) );
	actEdit.append( act );

	act = KStdAction::copy( this, SLOT( slotCopy() ), actionCollection(), "edit_copy" );
	act->setWhatsThis( i18n( "Copy to Clipboard." ) );
	actEdit.append( act );

	act = KStdAction::paste( this, SLOT( slotPaste() ), actionCollection(), "edit_paste" );
	act->setWhatsThis( i18n( "Paste from Clipboard." ) );
	actEdit.append( act );

	act = KStdAction::selectAll( this, SLOT( slotSelectAll() ), actionCollection(), "selectall" );
	act->setWhatsThis( i18n( "Select all words." ) );
	actEdit.append( act );

	act = KStdAction::deselect( this, SLOT( slotDeselect() ), actionCollection(), "deselect" );
	act->setWhatsThis( i18n( "Deselect all words." ) );
	actEdit.append( act );

	KActionMenu* actImport = new KActionMenu( i18n("&Import"), "fileimport", actionCollection(), "import" );
	act = new KAction( i18n("&Pubdic text dictionary..."), "kannadic", KShortcut(), this, SLOT( slotImportPubdic() ), actionCollection() );
	act->setWhatsThis( i18n( "Import pubdic style dictionary." ) );
	actImport->insert( act );

	act = new KAction( i18n("&Dixchange text dictionary..."), "kannadic", KShortcut(), this, SLOT( slotImportDixchange() ), actionCollection() );
	act->setWhatsThis( i18n( "Import dixchange style dictionary." ) );
	actImport->insert( act );

	act = new KAction( i18n("&MS-IME text dictionary..."), "kannadic", KShortcut(), this, SLOT( slotImportMSIME() ), actionCollection() );
	act->setWhatsThis( i18n( "Import Microsoft IME style text dictionary." ) );
	actImport->insert( act );

	actShowMenubar = KStdAction::showMenubar( NULL, NULL, actionCollection(), "showmenubar" );
	actShowMenubar->setWhatsThis( i18n( "Toggle menubar's visible." ) );
	connect( actShowMenubar, SIGNAL( toggled( bool ) ), this, SLOT( slotShowMenubar( bool ) ) );

	actShowToolbar = KStdAction::showToolbar( NULL, NULL, actionCollection(), "showtoolbar" );
	actShowToolbar->setWhatsThis( i18n( "Toggle toolbar's visible." ) );
	connect( actShowToolbar, SIGNAL( toggled( bool ) ), this, SLOT( slotShowToolbar( bool ) ) );

	actShowStatusbar = KStdAction::showStatusbar( NULL, NULL, actionCollection(), "showstatusbar" );
	actShowStatusbar->setWhatsThis( i18n( "Toggle statusbar's visible." ) );
	connect( actShowStatusbar, SIGNAL( toggled( bool ) ), this, SLOT( slotShowStatusbar( bool ) ) );

	act = new KAction( i18n("&Type Info..."), "download", KShortcut(), this, SLOT( slotTypeInfo() ), actionCollection(), "typeinfo" );

	act = KStdAction::preferences( this, SLOT( slotPreferences() ), actionCollection(), "preferences" );
	act->setWhatsThis( i18n( "Configure kannadic." ) );
}

void KannaDic::setEditEnabled()
{
	Dictionary::DictionaryType dictype = Dictionary::name2type( actDictionaryType->currentText() );
	actionCollection()->action( "save" )->setEnabled( m_dic != NULL );
	actionCollection()->action( "saveas" )->setEnabled( dictype != Dictionary::Anthy );
	actionCollection()->action( "deletedic" )->setEnabled( m_dic == NULL ? false :( m_dic->type() != Dictionary::Anthy ) );
//	actionCollection()->action( "deletedic" )->setEnabled( false );	//ToDo
	if( m_dic == NULL ){
		switch( dictype ){
			case Dictionary::Canna:
			case Dictionary::Pubdic:
				columnChange( 3 );
				break;
			case Dictionary::Anthy:
			case Dictionary::Dixchange:
				columnChange( 4 );
		}
	}
	if( dictype == Dictionary::Anthy && m_blChangable )
	{
		int result = KMessageBox::questionYesNo( (QWidget*)0, i18n( "Auto open Anthy Dictionary?" ), i18n("Question"), KStdGuiItem::yes(), KStdGuiItem::no(), "AutoOpenAnthy" );
		if( result == KMessageBox::Yes )
		{
			m_blChangable = false;
			dicOpen( dictype, "user" );
			m_blChangable = true;
		}
	}
}

void KannaDic::setupStatusBar()
{
	statusBar()->setSizeGripEnabled( true );
	statusBar()->insertItem( "", 1, 0, true );
	statusBar()->setSizeGripEnabled( true );
	statusBar()->show();
}

void KannaDic::slotDicNew()
{
	if( !CheckChangedAndSaved() ) return;
	if( m_dic != NULL ){
		delete m_dic;
		m_dic = NULL;
	}
	Dictionary::DictionaryType dictype = Dictionary::name2type( actDictionaryType->currentText() );
	switch( dictype ){
		case Dictionary::Canna:
		case Dictionary::Pubdic:
			columnChange( 3 );
			break;
		case Dictionary::Anthy:
		case Dictionary::Dixchange:
			columnChange( 4 );
	}
	lvwDictionary->clear();
	setCaption( i18n( "untitled" ) );
}

void KannaDic::slotDicDelete()
{
	if( !CheckChangedAndSaved() ) return;
	int ans = KMessageBox::warningYesNo( this, i18n("Do you delete '%1' dictionary really?").arg( m_dic->name() ), i18n("Delete dictionary") );
	if( ans == KMessageBox::No ) return;
	connect( m_dic, SIGNAL( deleted( bool ) ), this, SLOT( slotDicDeleted( bool ) ) );
	m_dic->rmdic();
	setEditEnabled();
}

void KannaDic::slotDicDeleted( bool blSuccess )
{
	disconnect( this, SLOT( slotDicDeleted( bool ) ) );
	if( blSuccess ){
		slotDicNew();
	}
}

void KannaDic::slotDicOpen()
{
	if( !CheckChangedAndSaved() ) return;
	dlgOpen dlg( this );
	Dictionary::DictionaryType dictype = Dictionary::name2type( actDictionaryType->currentText() );
	KURL url;
	switch( dictype ){
		case Dictionary::Canna:
			dlg.setDictionaryType( dictype );
			if( dlg.exec() == QDialog::Accepted )
			{
				dicOpen( dictype, dlg.dicname() );
			}
			break;

		case Dictionary::Anthy:
			dicOpen( dictype, "user" );
			break;

		case Dictionary::Pubdic:
			url = KFileDialog::getOpenURL( QString::null, "*.ctd *.t|Canna text dictionary files(*.ctd *.t)\n*.txt|Text files(*.txt)\n*.*|All files" );
			if( url.isEmpty() ) return;

			if( !url.isLocalFile() ){
				KMessageBox::sorry( this, i18n("%1 can not handle remote files yet").arg( "KannaDic" ) );
				return;
			}
			dicOpen( dictype, url.path() );
			break;

		case Dictionary::Dixchange:
			url = KFileDialog::getOpenURL( QString::null, "*.dxc|Dixchange File (*.dxc)\n*|All files" );
			if( url.isEmpty() ) return;

			if( !url.isLocalFile() ){
				KMessageBox::sorry( this, i18n("%1 can not handle remote files yet").arg( "KannaDic" ) );
				return;
			}
			dicOpen( dictype, url.path() );
			break;
	}
}

void KannaDic::slotDicSave()
{
	if( !actionCollection()->action( "edit_undo" )->isEnabled() ) return;
	if( m_dic == NULL ){
		slotDicSaveAs();
		return;
	}
	QApplication::setOverrideCursor( waitCursor );
	if( m_dic->save( lvwDictionary ) ){
		m_cmdHistory->clear();
	}else{
		KMessageBox::sorry( this, i18n( "Save Error" ) );
	}
	QApplication::restoreOverrideCursor();
}

void KannaDic::slotDicSaveAs()
{
	KURL url;
	QString dicname;
	Dictionary::DictionaryType dictype = Dictionary::name2type( actDictionaryType->currentText() );
	switch( dictype ){
		case Dictionary::Canna:
			dicname = KInputDialog::getText( i18n( "Dictionary Name" ), i18n( "Please input dictionary name" ) );
			if( dicname != QString::null && !dicname.stripWhiteSpace().isEmpty() ){
				dicCreate( dictype, dicname.stripWhiteSpace() );
			}
			break;
		case Dictionary::Anthy:
			// never
			break;
		case Dictionary::Pubdic:
			url = KFileDialog::getSaveURL( QString::null, "*.ctd *.t|Dixchange File (*.ctd *.t)\n*|All files" );
			if( url.isEmpty() ) return;

			if( !url.isLocalFile() ){
				KMessageBox::sorry( this, i18n("%1 can not handle remote files yet").arg( "KannaDic" ) );
				return;
			}
			if( QFile::exists( url.path() ) ){
				int ans = KMessageBox::warningYesNo( this, i18n("%1 exists. Do you over write?").arg( url.path() ), i18n("Confirm") );
				if( ans == KMessageBox::No ) return;
			}
			dicCreate( dictype, url.path() );
			break;
		case Dictionary::Dixchange:
			url = KFileDialog::getSaveURL( QString::null, "*.dxc|Dixchange File (*.dxc)\n*|All files" );
			if( url.isEmpty() ) return;

			if( !url.isLocalFile() ){
				KMessageBox::sorry( this, i18n("%1 can not handle remote files yet").arg( "KannaDic" ) );
				return;
			}
			if( QFile::exists( url.path() ) ){
				int ans = KMessageBox::warningYesNo( this, i18n("%1 exists. Do you over write?").arg( url.path() ), i18n("Confirm") );
				if( ans == KMessageBox::No ) return;
			}
			dicCreate( dictype, url.path() );
			break;
	}
}

bool KannaDic::queryClose()
{
	return CheckChangedAndSaved();
}

void KannaDic::dicCreate( Dictionary::DictionaryType dictype, QString dicname )
{
	switch( dictype ){
		case Dictionary::Canna:
			if( !Canna::mkdic( dicname ) ) return;
			columnChange( 3 );
			m_dic = new Canna( dicname );
			break;
		case Dictionary::Anthy:
			//never
			break;
		case Dictionary::Pubdic:
			if( !Pubdic::mkdic( dicname ) ) return;
			columnChange( 3 );
			m_dic = new Pubdic( dicname );
			break;
		case Dictionary::Dixchange:
			if( !Dixchange::mkdic( dicname ) ) return;
			columnChange( 4 );
			m_dic = new Dixchange( dicname );
			break;
	}
	setCaption( QString( "%1(%2)" ).arg( m_dic->name() ).arg( actDictionaryType->currentText() ) );
	setEditEnabled();

	m_dic->save( lvwDictionary );
	actionCollection()->action("edit_redo")->setEnabled( false );
	actionCollection()->action("edit_undo")->setEnabled( false );

	statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
}

void KannaDic::dicOpen( Dictionary::DictionaryType dictype, QString dicname )
{
	QApplication::setOverrideCursor( waitCursor );
	QProgressDialog* progress = new QProgressDialog( this );
	progress->setCaption( i18n("Open" ) );
	progress->setLabelText( i18n( "Please wait..." ) );
	progress->setCursor( QCursor::WaitCursor );
	progress->setCancelButton( 0 );
	progress->setTotalSteps( 100 );
	m_timer->start( 0, FALSE );
	switch( dictype ){
		case Dictionary::Canna:
			columnChange( 3 );
			m_dic = new Canna( dicname );
			break;
		case Dictionary::Anthy:
			columnChange( 4 );
			m_dic = new Anthy( dicname );
			break;
		case Dictionary::Pubdic:
			columnChange( 3 );
			m_dic = new Pubdic( dicname );
			break;
		case Dictionary::Dixchange:
			columnChange( 4 );
			m_dic = new Dixchange( dicname );
			break;
	}
	setCaption( QString( "%1(%2)" ).arg( m_dic->name() ).arg( actDictionaryType->currentText() ) );
	setEditEnabled();

	connect( m_dic, SIGNAL( progress( int ) ), progress, SLOT( setProgress( int ) ) );
	progress->show();
	lvwDictionary->clear();
	m_dic->expand( lvwDictionary );
	m_timer->stop();
	actionCollection()->action("edit_redo")->setEnabled( false );
	actionCollection()->action("edit_undo")->setEnabled( false );

	statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
	disconnect( progress, SLOT(setProgress( int ) ) );
	delete progress;

	QApplication::restoreOverrideCursor();
}

void KannaDic::slotAppend()
{
	AppendCommand* cmd = new AppendCommand( lvwDictionary );
	if( cmd->succeed() ){
		m_cmdHistory->addCommand( cmd );
		statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
	}
}

void KannaDic::slotEdit()
{
	EditCommand* ec;
	MultiEditCommand* mec;
	switch( lvwDictionary->selectedItems().count() ){
		case 0:
			KMessageBox::sorry( this, i18n("Select line(s)") );
			break;
		case 1:
			ec = new EditCommand( lvwDictionary );
			if( ec->succeed() ){
				m_cmdHistory->addCommand( ec );
				statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
			}
			break;
		default:
			mec = new MultiEditCommand( lvwDictionary );
			if( mec->succeed() ){
				m_cmdHistory->addCommand( mec );
				statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
			}
	}
}

void KannaDic::slotCut()
{
	slotCopy();
	slotDelete();
}

void KannaDic::slotCopy()
{
	if( !lvwDictionary->selectedItems().count() ){
		KMessageBox::sorry( this, i18n("Select word(s)") );
		return;
	}
	QString str;
	QPtrList< QListViewItem > selectedItems = lvwDictionary->selectedItems();
	for( QListViewItem* item = selectedItems.first(); item; item = selectedItems.next() ){
		str += ((WordItem*)item)->word().toString() + "\n";
	}
	KApplication::clipboard()->setText( str, QClipboard::Clipboard );
}

void KannaDic::slotPaste()
{
	QRegExp regexp( "^(.*)\t(.*)\t(.*)\t(.*)\n", false, false );
	QString str = KApplication::clipboard()->text( QClipboard::Clipboard );
	if( regexp.search( str ) == -1 ){
		AppendCommand* cmd = new AppendCommand( lvwDictionary, str );
		if( cmd->succeed() ){
			m_cmdHistory->addCommand( cmd );
			statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
		}
	}else{
		PasteCommand* cmd = new PasteCommand( lvwDictionary, str );
		if( cmd->succeed() ){
			m_cmdHistory->addCommand( cmd );
			statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
		}
	}
}

void KannaDic::slotDelete()
{
	if( !lvwDictionary->selectedItems().count() ){
		KMessageBox::sorry( this, i18n("Select word(s)") );
		return;
	}
	QApplication::setOverrideCursor( waitCursor );
	DeleteCommand* cmd = new DeleteCommand( lvwDictionary );
	QApplication::restoreOverrideCursor();
	if( cmd->succeed() ){
		m_cmdHistory->addCommand( cmd );
		statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
	}
}

void KannaDic::slotImportMSIME()
{
	ImportMSIMECommand* cmd = new ImportMSIMECommand( lvwDictionary );
	if( cmd->succeed() ){
		m_cmdHistory->addCommand( cmd );
		statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
	}
}

void KannaDic::slotImportPubdic()
{
	ImportPubdicCommand* cmd = new ImportPubdicCommand( lvwDictionary );
	if( cmd->succeed() ){
		m_cmdHistory->addCommand( cmd );
		statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
	}
}

void KannaDic::slotImportDixchange()
{
	ImportDixchangeCommand* cmd = new ImportDixchangeCommand( lvwDictionary );
	if( cmd->succeed() ){
		m_cmdHistory->addCommand( cmd );
		statusBar()->changeItem( i18n("Count of word:%1").arg( lvwDictionary->childCount() ), 1 );
	}
}

void KannaDic::slotSelectAll()
{
	lvwDictionary->selectAll( true );
}

void KannaDic::slotDeselect()
{
	lvwDictionary->selectAll( false );
}

void KannaDic::slotShowMenubar( bool bl )
{
	actShowMenubar->setChecked( bl );
	if( bl ){
		menuBar()->show();
	}else{
		menuBar()->hide();
	}
}

void KannaDic::slotShowToolbar( bool bl )
{
	actShowToolbar->setChecked( bl );
	if( bl ){
		toolBar()->show();
	}else{
		toolBar()->hide();
	}
}

void KannaDic::slotShowStatusbar( bool bl )
{
	actShowStatusbar->setChecked( bl );
	if( bl ){
		statusBar()->show();
	}else{
		statusBar()->hide();
	}
}

void KannaDic::slotPopup( KListView* lvw, QListViewItem* item, const QPoint& pos )
{
	if( item ){
		( (KPopupMenu*)( factory()->container( "item_popup", this ) ) )->exec( pos );
	}else{
		( (KPopupMenu*)( factory()->container( "noitem_popup", this ) ) )->exec( pos );
	}
}

void KannaDic::columnChange( int i )
{
	switch( i )
	{
		case 3:
			if( lvwDictionary->header()->count() != i ){
				lvwDictionary->removeColumn( lvwDictionary->header()->count() - 1 );
			}
			break;
		case 4:
			if( lvwDictionary->header()->count() != i ){
				lvwDictionary->addColumn( i18n( "Freq" ) );
				lvwDictionary->setColumnAlignment( 4, Qt::AlignRight );
			}
			break;
	}
}

void KannaDic::slotTypeInfo()
{
	dlgTypeInfo* dlg = new dlgTypeInfo( this );
	dlg->show();
}

void KannaDic::slotPreferences()
{
	dlgPreference* dlg = new dlgPreference( this );
	dlg->show();
}

bool KannaDic::CheckChangedAndSaved()
{
	bool blRtn = true;
	if( actionCollection()->action( "edit_undo" )->isEnabled() ){
		int ans = KMessageBox::questionYesNoCancel( this, i18n("This dictionary is changed.\nDo you save change?"), i18n("Confirm") );
		switch( ans )
		{
			case KMessageBox::Yes:
				blRtn = m_dic->save( lvwDictionary );
				if( blRtn ){
					m_cmdHistory->clear();
				}
				break;
			case KMessageBox::No:
				m_cmdHistory->clear();
				break;
			case KMessageBox::Cancel:
				blRtn = false;
				break;
		}
	}
	return blRtn;
}
