/*
 * Worker thread for checkout
 *
 * This file contains all of the routines executed directly
 * by the worker thread
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define G_LOG_DOMAIN "worker-thread"

#include <gnubversion.h>
#include <assert.h>
#include <unistd.h>
#include <svn_client.h>

#include "gvn-checkout.h"

static void
progress_notify( apr_off_t progress,
                 apr_off_t total,
                 void *baton,
                 apr_pool_t *pool)
{
    worker_comms_t *wc = baton;

    gvn_debug("Progress: %"APR_OFF_T_FMT"/%"APR_OFF_T_FMT, progress, total);
    gdouble ratio = (1.0 * progress) / total;

    GnubVersionThreadMessage *msg;
    msg = gnubversion_threadmessage_new_progress(ratio);
    gnubversion_progresswindow_send_message( wc->progressWindow, msg);
}

static svn_error_t *
cancel_func( void *cancel_baton )
{
    worker_comms_t *wc = cancel_baton;

    if ( wc->cancel_requested )
    {
        gvn_debug("Flagging up cancellation");
        return svn_error_create(SVN_ERR_CANCELLED, NULL,
                                "Operation cancelled by user");
    }
    else
        return SVN_NO_ERROR;
}

static void 
debug_comms(worker_comms_t *wc)
{
    gvn_debug("Repository URL: %s", wc->repository_url);
    gvn_debug("Destination path: %s", wc->destination_path);
    gvn_debug("Working copy name: %s", wc->working_copy_name);
}

static void 
cancel_requested(GnubVersionProgressWindow *progress, gpointer user_data)
{
    gvn_debug("Cancel_requested signal received");
    worker_comms_t *wc = user_data;

    wc->cancel_requested = TRUE;

    GnubVersionThreadMessage *msg;
    msg = gnubversion_threadmessage_new_info(_("Cancelling Checkout..."));
    gnubversion_progresswindow_send_message(wc->progressWindow, msg);
}

gpointer 
checkout_thread_main (gpointer data)
{
    worker_comms_t *worker_comms = data;

    gvn_debug("checkout_thread_main");

    gvn_debug("wc address %p", &worker_comms);
    gnubversion_progresswindow_connect__cancel_requested(worker_comms->progressWindow, cancel_requested, worker_comms);

    debug_comms(worker_comms);

    gnubversion_auth_baton->progressWindow = worker_comms->progressWindow;

    svn_error_t *err = NULL;
    svn_opt_revision_t revision;

    revision.kind = svn_opt_revision_head;

    gnubversion_svn_client_ctx->notify_func2 = gnubversion_progresswindow_svn_notify2;
    gnubversion_svn_client_ctx->notify_baton2 = worker_comms->progressWindow;

    gnubversion_svn_client_ctx->progress_func = progress_notify;
    gnubversion_svn_client_ctx->progress_baton = worker_comms;

    gnubversion_svn_client_ctx->cancel_func = cancel_func;
    gnubversion_svn_client_ctx->cancel_baton = worker_comms;
    svn_revnum_t result_rev;

    GString *path = g_string_new(worker_comms->destination_path);
    g_string_append(path, "/");
    g_string_append(path, worker_comms->working_copy_name);

    gvn_debug("Checking out %s into %s...",
            worker_comms->repository_url, 
            path->str
            );

    err = svn_client_checkout2(
            &result_rev,
            worker_comms->repository_url, 
            path->str,
            &worker_comms->revision,
            &revision,
            worker_comms->recursive,
            worker_comms->ignore_externals,
            gnubversion_svn_client_ctx,
            gnubversion_apr_pool);
    g_string_free(path, TRUE);

    gnubversion_svn_client_ctx->cancel_func = NULL;
    gnubversion_svn_client_ctx->progress_func = NULL;
    gnubversion_svn_client_ctx->notify_func2 = NULL;

    GnubVersionThreadMessage *msg;

    if (err != NULL)
    {
        const GError *gerr = g_error_from_svn(err);
        gvn_debug("Checkout failed!: %s", gerr->message);
        msg = gnubversion_threadmessage_new_bye_error(gerr);
    }
    else
    {
        msg = gnubversion_threadmessage_new_bye_success(_("Checkout Complete"));
    }

    gvn_debug("Worker thread saying bye");
    gnubversion_progresswindow_send_message( worker_comms->progressWindow, msg);

    g_thread_exit (err);

    /*notreached*/
    return NULL;
}
