/*
 *  Copyright (c) 2008-2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <cstring>

// GTLCore Headers
#include <GTLCore/PixelDescription.h>
#include <GTLCore/CompilationMessages.h>
#include <GTLCore/Debug.h>
#include <GTLCore/Value.h>
#include <GTLCore/Region.h>
#include <GTLCore/Type.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/Utils_p.h>

// GTLImageIO Headers
#include <GTLImageIO/ImageDC.h>
#include <GTLImageIO/ImageDCRegistry.h>

// OpenShiva Headers
#include <OpenShiva/Debug.h>
#include <GTLCore/Image.h>
#include <OpenShiva/Kernel.h>
#include <OpenShiva/Version.h>
#include <OpenShiva/LibrariesManager.h>

void printVersion()
{
  std::cout << OpenShiva::LibraryShortName() << " - " << OpenShiva::LibraryName() << " - " << OpenShiva::LibraryVersionString() << std::endl;
  std::cout << OpenShiva::LibraryCopyright() << std::endl;
  std::cout << "Shiva Version : " << OpenShiva::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : shiva [option] fileName.shiva [input0.png input1.png ...] output.png" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -a --alpha              set the position of the alpha channel (by default -1)" << std::endl;
  std::cout << "  -S --asm-source         print the assembly source code generated after the execution of the kernel" << std::endl;
  std::cout << "  -L --module-dir   add a location where to find modules" << std::endl;
  std::cout << std::endl;
  std::cout << "  -w --width [w]          define the width of the output" << std::endl;
  std::cout << "  -h --height [h]         define the height of the output" << std::endl;
  std::cout << "  -c --channels [count]   set the number of channels of the output" << std::endl;
  std::cout << "  -t --channeltype [type] set the type of the output (uint8, uint16, half, float)" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}
#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)
#define max(a,b) ((a) > (b) ? (a) : (b))
int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  GTLCore::String outputFileName = "";
  GTLCore::String output = "";
  std::list< GTLCore::String > fileNames;
  bool showAssembly = false;
  int width = -1;
  int height = -1;
  int channelsCount = 3;
  int alphaPos = -1;
  // Select the type of the resulting image
  const GTLCore::Type* typeChannelImage = GTLCore::Type::UnsignedInteger8;
  bool userChannelType = false;
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-S","--asm-source")) {
      showAssembly = true;
    } else if(ARG_IS("-L", "--module-dir")) {
      if( ai == argc )
      {
        std::cerr << "Expected directory after -L --module-dir." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        OpenShiva::LibrariesManager::instance()->addDirectory(argv[ai]);
      }
    } else if(ARG_IS("-a", "--alpha")) {
      if( ai == argc )
      {
        std::cerr << "Expected alpha after -a --alpha." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        alphaPos = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-w", "--width")) {
      if( ai == argc )
      {
        std::cerr << "Expected width after -w --width." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        width = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-h", "--height")) {
      if( ai == argc )
      {
        std::cerr << "Expected height after -h --height." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        height = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-c", "--channels")) {
      if( ai == argc )
      {
        std::cerr << "Expected count after -c --channels." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        channelsCount = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-t", "--channeltype")) {
      if( ai == argc )
      {
        std::cerr << "Expected type after -t --channeltype." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        GTLCore::String channelType = argv[ai];
        userChannelType = true;
        if( channelType == "uint16" )
        {
          typeChannelImage = GTLCore::Type::UnsignedInteger16;
        } else if( channelType == "half" )
        {
          typeChannelImage = GTLCore::Type::Float16;
        } else if( channelType == "float" )
        {
          typeChannelImage = GTLCore::Type::Float32;
        } else if( channelType != "uint8" )
        {
          std::cerr << "Unknown channel type: '" << channelType << "'." << std::endl;
          return EXIT_FAILURE;
        }
      }
    } else {
      fileNames.push_back( argv[ai] );
    }
  }
  if( fileNames.size() > 1 )
  {
    fileName = *( fileNames.begin() );
    fileNames.pop_front();
    outputFileName = *( --fileNames.end() );
    fileNames.pop_back();
  }
  if( fileName == "")
  {
    std::cerr << "Invalid command line parameters." << std::endl;
    printHelp();
  } else {
    GTLCore::String errMsg;
    // Load source code
    GTLCore::String source;
    std::ifstream in;
    in.open(fileName.c_str() );
    if(not in)
    {
      std::cerr << "Impossible to open file " << fileName << std::endl;
      return EXIT_FAILURE;
    }
    std::string str;
    std::getline(in,str);
    while ( in ) {
      source += str;
      source += "\n";
      std::getline(in,str);
    }
    // Load Images
    std::list<const GTLCore::AbstractImage*> images;
    std::list<GTLCore::RegionI> inputDOD;
    int maxWidth = -1;
    int maxHeight = -1;
    for( std::list< GTLCore::String >::iterator it = fileNames.begin();
         it != fileNames.end(); ++it )
    {
      const GTLImageIO::ImageDC* decoder = GTLImageIO::ImageDCRegistry::instance()->decoder( *it );
      if( not decoder )
      {
        std::cerr << "Can't find decoder for " << *it << std::endl;
        return EXIT_FAILURE;
      }
      GTLCore::RegionI region;
      GTLCore::AbstractImage* image = decoder->decode( *it, &region, &errMsg );
      if( not image )
      {
        std::cerr << "Can't read image " << *it << " : " << errMsg << std::endl;
      }
      images.push_back( image );
      inputDOD.push_back( region );
      maxWidth = max(region.columns(), maxWidth);
      maxHeight = max(region.rows(), maxHeight);
      if(not userChannelType)
      {
        foreach( const GTLCore::Type* type, image->pixelDescription().channelTypes())
        {
          if( type->bitsSize() > typeChannelImage->bitsSize() )
          {
            typeChannelImage = type;
          }
        }
      }
    }
    
    // Set width/height
    if(width == -1)
    {
      if(maxWidth == -1)
      {
        width = 800;
      } else {
        width = maxWidth;
      }
    }
    if(height == -1)
    {
      if(maxHeight == -1)
      {
        height = 600;
      } else {
        height = maxHeight;
      }
    }
    
    GTLCore::PixelDescription pixel( typeChannelImage, channelsCount, alphaPos );
    OpenShiva::Kernel p(channelsCount);
    p.setSource( source );
    p.setParameter(OpenShiva::Kernel::IMAGE_HEIGHT, GTLCore::Value((float)width));
    p.setParameter(OpenShiva::Kernel::IMAGE_WIDTH, GTLCore::Value((float)height));
    p.compile();
    if(not p.isCompiled())
    {
      std::cerr << "Error: " << std::endl << p.compilationMessages().toString() << std::endl;
      while(true) {}
      return EXIT_FAILURE;
    }
    
    GTLCore::RegionF region(0,0,width,height);
    if( p.hasGeneratedFunction() )
    {
      region = p.generated();
    }
    if( p.hasChangedFunction() )
    {
      int i = 0;
      for( std::list<GTLCore::RegionI>::iterator it = inputDOD.begin(); it != inputDOD.end(); ++it, ++i)
      {
        region += p.changed( *it, i, inputDOD );
      }
    }
    GTLCore::Image image( region.columns(), region.rows(), pixel );
    p.evaluatePixels( region.toRegionI(), images, &image );
    if( showAssembly )
    {
      std::cout << p.asmSourceCode();
    }
    const GTLImageIO::ImageDC* encoder = GTLImageIO::ImageDCRegistry::instance()->encoder( outputFileName );
    if( not encoder )
    {
      std::cerr << "Can't find encoder for " << outputFileName << std::endl;
      return EXIT_FAILURE;
    }
    if( not encoder->encode( &image, image.boundingBox(), outputFileName, 0, &errMsg ) )
    {
      std::cerr << "Can't encode " << outputFileName << " : " << errMsg << std::endl;
      return EXIT_FAILURE;
    }
    GTLCore::deleteAll( images );
  }
  return EXIT_SUCCESS;
}
