/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Parser_p.h"

#include "Debug.h"

#include "Compiler_p.h"
#include "Lexer_p.h"

#include <GTLCore/AST/Expression.h>
#include <GTLCore/AST/Statement.h>
#include <GTLCore/AST/Tree.h>
#include <GTLCore/CompilerBase_p.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/TypesManager.h>
#include <GTLCore/Value.h>
#include <GTLCore/VariableNG_p.h>
#include <GTLCore/VariablesManager_p.h>

namespace AST = GTLCore::AST;
using namespace OpenShiva;

struct Parser::Private {
  GTLCore::String kernelName;
  GTLCore::AST::Tree* tree;
  Compiler* compiler;
  std::list<GTLCore::VariableNG*> dependents;
};

Parser::Parser( Compiler* _compiler , Lexer* _lexer) : ParserBase(_compiler, _lexer), d(new Private)
{
  d->tree = 0;
  d->compiler = _compiler;
}

Parser::~Parser()
{
  delete d;
}

GTLCore::AST::Tree* Parser::parse()
{
  SHIVA_DEBUG("Parsing with ParserNG");
  SHIVA_ASSERT( not d->tree );
  d->tree = new GTLCore::AST::Tree();
  variablesManager()->startContext();
  
  getNextToken();
  // Start by ingoring metadata if any
  if( currentToken().type == GTLCore::Token::INFERIOR )
  {
    int count = 0;
    do
    {
      switch( currentToken().type )
      {
        case GTLCore::Token::INFERIOR:
          ++count;
          break;
        case GTLCore::Token::SUPPERIOR:
          --count;
          break;
        default:
          break;
      }
      getNextToken();
    } while( count > 0 and currentToken().type != GTLCore::Token::END_OF_FILE  );
    if( isOfType( currentToken(), GTLCore::Token::SEMI ) )
    {
      getNextToken();
    }
  }
  
  // Parse "import"
  while( currentToken().type == GTLCore::Token::IMPORT )
  {
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::STRING_CONSTANT ) )
    {
      d->compiler->importModule( currentToken().string );
    }
    checkNextTokenIsSemi();
    getNextToken();
  }
  SHIVA_DEBUG("Set parameters as constants");
  // Set parameters as constants
  for( std::map< GTLCore::String, GTLCore::Value >::const_iterator it = d->compiler->parameters().begin();
       it != d->compiler->parameters().end(); ++it )
  {
    GTLCore::ScopedName scopedName( nameSpace(), it->first );
    const GTLCore::Type* declarationType = it->second.type();
    if (declarationType == GTLCore::Type::Color)
    {
      declarationType = GTLCore::TypesManager::getVector(GTLCore::Type::Float32, 4);
    }
    AST::GlobalConstantDeclaration* gcd = new AST::GlobalConstantDeclaration(
        scopedName, declarationType,
        GTLCore::AST::Expression::fromValue( it->second ), false );
    variablesManager()->declareConstant( scopedName, gcd->variable() );
    tree()->append( gcd );
  }
          
  SHIVA_DEBUG("Parse body");
  // Parse "kernel" or "library"
  if( (d->compiler->isKernel() and isOfType( currentToken(), GTLCore::Token::KERNEL ) )
      or isOfType( currentToken(), GTLCore::Token::LIBRARY ) )
  {
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
    {
      d->kernelName = currentToken().string;
      SHIVA_ASSERT( d->kernelName != "" );
      SHIVA_DEBUG( d->compiler->isStdLib() << " " << d->kernelName );
      if( not d->compiler->isStdLib() )
      {
        setNameSpace( d->kernelName );
        SHIVA_ASSERT( d->kernelName == nameSpace() );
      }
      getNextToken();
      if( isOfType( currentToken(), GTLCore::Token::STARTBRACE ) )
      {
        getNextToken();
        parseKernelBody();
        isOfType( currentToken(), GTLCore::Token::ENDBRACE );
        getNextToken();
      }
    }
  }
  
  return d->tree;
}

void Parser::parseKernelBody()
{
  while(true)
  {
    switch(currentToken().type)
    {
      case GTLCore::Token::END_OF_FILE:
      case GTLCore::Token::ENDBRACE:
        return;
      case GTLCore::Token::CONST:
        parseConstantDeclaration();
        break;
      case GTLCore::Token::STRUCT:
        parseStructDefinition();
        break;
      case GTLCore::Token::DEPENDENT:
        parseDependentDeclaration();
        break;
      default:
      {
        if( currentToken().isFunctionType() )
        {
          parseFunction();
        } else {
          SHIVA_DEBUG("unexpected " << GTLCore::Token::typeToString( currentToken().type ) );
          reportUnexpected( currentToken() );
          getNextToken();
        }
      }
    }
  }
}

GTLCore::AST::Tree* Parser::tree()
{
  return d->tree;
}

AST::Statement* Parser::parseStatement()
{
  if( isType(currentToken()) )
  {
    return parseVariableDeclaration();
  }
  switch( currentToken().type )
  {
    case GTLCore::Token::CONST:
      return parseVariableDeclaration();
    case GTLCore::Token::STRUCT:
    {
      parseStructDefinition();
      return new AST::DummyStatement();
    }
    case GTLCore::Token::STARTBRACKET:
    case GTLCore::Token::IDENTIFIER:
      return parseExpressionStatement();
    case GTLCore::Token::IF:
      return parseIfStatement();
    case GTLCore::Token::WHILE:
      return parseWhileStatement();
    case GTLCore::Token::FOR:
      return parseForStatement();
    case GTLCore::Token::RETURN:
      return parseReturnStatement();
    case GTLCore::Token::STARTBRACE:
    {
      variablesManager()->startContext();
      AST::Statement* statement = appendCurrentContextGarbageCollecting(parseStatementList());
      variablesManager()->endContext();
      return statement;
    }
    case GTLCore::Token::PRINT:
    {
      return parsePrintStatement();
    }
    default:
      if( currentToken().isUnaryOperator() or currentToken().isConstant() )
      {
        return parseExpressionStatement();
      }
      SHIVA_DEBUG("unexpected");
      reportUnexpected( currentToken() );
      getNextToken();
      return 0;
  }
}

const GTLCore::Type* Parser::parseType()
{
  switch( currentToken().type )
  {
    case GTLCore::Token::BOOL2:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Boolean, 2 );
    case GTLCore::Token::BOOL3:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Boolean, 3 );
    case GTLCore::Token::BOOL4:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Boolean, 4 );
    case GTLCore::Token::BOOLN:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Boolean, d->compiler->channelsNb() );
    case GTLCore::Token::INT2:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Integer32, 2 );
    case GTLCore::Token::INT3:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Integer32, 3 );
    case GTLCore::Token::INT4:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Integer32, 4 );
    case GTLCore::Token::INTN:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Integer32, d->compiler->channelsNb() );
    case GTLCore::Token::FLOAT2:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Float32, 2 );
    case GTLCore::Token::FLOAT3:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Float32, 3 );
    case GTLCore::Token::FLOAT4:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Float32, 4 );
    case GTLCore::Token::FLOATN:
      getNextToken();
      return typesManager()->getVector( GTLCore::Type::Float32, d->compiler->channelsNb() );
    default:
      return GTLCore::ParserBase::parseType();
  }
}

void Parser::parseDependentDeclaration()
{
  getNextToken();
  const GTLCore::Type* type = parseType();
  if( not type ) return;
  if( type->dataType() == GTLCore::Type::STRUCTURE 
      and ( type->structName().startWith("pixel")
            or type->structName().startWith("image") ) )
  {
    reportError( type->structName() + " can't be a dependent", currentToken());
  }
  while(true)
  {
    if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
    {
      GTLCore::String name = currentToken().string;
      getNextToken();
      GTL_DEBUG("Dependent: " << name);
      std::list<int> memberArraySize = expressionsListToIntegersList( parseArraySize(true) );
      type = d->compiler->typesManager()->getArray( type, memberArraySize.size() );
      GTLCore::ScopedName scopedName( nameSpace(), name );
      if( tree()->containsGlobalConstant( scopedName ) )
      {
        reportError("Constant '" + scopedName.toString() + "' has already been declared", currentToken());
      } else {
        AST::GlobalConstantDeclaration* gcd = new AST::GlobalConstantDeclaration( scopedName, type , memberArraySize, true );
        variablesManager()->declareConstant( scopedName, gcd->variable() );
        d->dependents.push_back( gcd->variable() );
        tree()->append( gcd );
      }
    } else {
      GTL_DEBUG("Unexpected");
      reportUnexpected( currentToken() );
      reachNextSemi();
    }
    if( currentToken().type == GTLCore::Token::COMA ) {
      getNextToken();
    } else {
      break;
    }
  }
  isOfType( currentToken(), GTLCore::Token::SEMI );
  getNextToken();
}

void Parser::startParsingFunction( const GTLCore::String& _name )
{
  if( _name == "evaluateDependents" )
  {
    foreach( GTLCore::VariableNG* var, d->dependents )
    {
      var->setConstant( false );
    }
  }
}

void Parser::endParsingFunction( const GTLCore::String& _name )
{
  if( _name == "evaluateDependents" )
  {
    foreach( GTLCore::VariableNG* var, d->dependents )
    {
      var->setConstant( true );
    }
  }
}
