/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

//Own
#include "kgrubeditor.h"

//Qt
#include <qtimer.h>

//KDE
#include <kaboutdata.h>
#include <kapplication.h>
#include <kconfigdialog.h>
#include <kdirwatch.h>
#include <kfiledialog.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knewstuff2/engine.h>
#include <kpluginfactory.h>
#include <kprocess.h>
#include <ksplashscreen.h>
#include <kstandarddirs.h>
#include <ktemporaryfile.h>
#include <threadweaver/JobCollection.h>
#include <threadweaver/ThreadWeaver.h>

//Ui
#include "backup.h"
#include "entry.h"
#include "install.h"
#include "map.h"
#include "password.h"
#include "passworddialog.h"
#include "quickeditor.h"
#include "viewdevices.h"
#include "viewfiles.h"

//KGRUBEditor
#include "core/backup.h"
#include "core/fileio.h"
#include "core/path.h"
#include "core/root.h"
#include "core/splashdelegate.h"
#include "core/splashpreviewjob.h"
#include "settings.h"

K_PLUGIN_FACTORY( GRUBFactory, registerPlugin<KGRUBEditor>(); )
K_EXPORT_PLUGIN( GRUBFactory( "kgrubeditor" ) )

KGRUBEditor::KGRUBEditor( QWidget *parent, const QVariantList &list ) : KCModule( GRUBFactory::componentData(), parent )
{
	Q_UNUSED( list )
	KAboutData *about = new KAboutData( "kgrubeditor", 0, ki18n( "KGRUBEditor" ), "0.8.5b", ki18n( "A KDE utility, that edits GRUB's configuration files through an inituitive user interface." ), KAboutData::License_GPL, ki18n( "Copyright (C) 2008 Konstantinos Smanis" ), KLocalizedString(), "http://sourceforge.net/projects/kgrubeditor", "kon.smanis@gmail.com" );
	about->addAuthor( ki18n( "Κonstantinos Smanis" ), ki18n( "Developer" ), "kon.smanis@gmail.com" );
	about->addCredit( ki18n( "Andreas Theodosiou" ), ki18n( "Application's Icon Designer" ), "andreasabu@gmail.com" );
	about->addCredit( ki18n( "Dimitris Palyvos-Giannas" ), ki18n( "Various help with the application" ), "jimaras@gmail.com" );
	about->addCredit( ki18n( "prts_1 (nickname)" ), ki18n( "Tips concerning usability" ), "prts_1@e-pcmag.gr" );
	setAboutData( about );

	setupUi( this );
	setButtons( Default | Apply );

	setupObjects();
	setupConnections();

	enableEntryActions( false );
}

//SETUP FUNCTIONS
void KGRUBEditor::setupObjects()
{
	timer_blink = new QTimer( this );

	m_splashDirWatchCreation = new KDirWatch( this );
	m_splashDirWatchDeletion = new KDirWatch( this );
	foreach( const QString &splashDir, Settings::splashDirs() )
		monitorSplashDirectory( splashDir );

	m_gfxmenuDirWatchDeletion = new KDirWatch( this );
	if ( !GrubSettings.gfxMenu().isEmpty() )
		m_gfxmenuDirWatchDeletion->addFile( Core::Path::convertToGenericPath( GrubSettings.gfxMenu() ) );

	m_splashModel = new SplashModel( this );
	ui_background.kcombobox_splashlist->setModel( m_splashModel );
	ui_background.kcombobox_splashlist->setItemDelegate( new SplashDelegate( this ) );
}
void KGRUBEditor::setupConnections()
{
	connect( timer_blink, SIGNAL( timeout() ), ui_color.colorPreview, SLOT( update() ) );
	connect( m_splashDirWatchCreation, SIGNAL( created( const QString & ) ), SLOT( splashImageCreated( const QString & ) ) );
	connect( m_splashDirWatchDeletion, SIGNAL( deleted( const QString & ) ), SLOT( splashImageDeleted( const QString & ) ) );
	connect( m_gfxmenuDirWatchDeletion, SIGNAL( deleted( const QString & ) ), SLOT( clearGfxmenu() ) );
//ENTRIES
	connect( ui_entries.entryWidget, SIGNAL( itemSelectionChanged() ), SLOT( selectionChanged() ) );
	connect( ui_entries.kpushbutton_up, SIGNAL( pressed() ), SLOT( moveUp() ) );
	connect( ui_entries.kpushbutton_down, SIGNAL( pressed() ), SLOT( moveDown() ) );
	connect( ui_entries.kpushbutton_add, SIGNAL( pressed() ), SLOT( add() ) );
	connect( action_quickEditor, SIGNAL( triggered( bool ) ), SLOT( quickEdit() ) );
	connect( action_fullEditor, SIGNAL( triggered( bool ) ), SLOT( fullEdit() ) );
	connect( ui_entries.kpushbutton_remove, SIGNAL( pressed() ), SLOT( remove() ) );
	connect( ui_entries.kpushbutton_details, SIGNAL( pressed() ), SLOT( showDetails() ) );

	connect( ui_entries.kpushbutton_back, SIGNAL( pressed() ), ui_entries.detailsWidget, SLOT( clear() ) );
	connect( ui_entries.kpushbutton_back, SIGNAL( pressed() ), SLOT( backToList() ) );
//SETTINGS
	//general
	connect( ui_general.checkBox_hiddenMenu, SIGNAL( clicked() ), SLOT( checkBox_hiddenMenu_clicked() ) );
	connect( ui_general.checkBox_timeout, SIGNAL( clicked() ), SLOT( checkBox_timeout_clicked() ) );
	connect( ui_general.kintspinbox_timeout, SIGNAL( editingFinished() ), SLOT( updateTimeout() ) );
	//password
	connect( ui_password.kpushbutton_create, SIGNAL( pressed() ), SLOT( createPassword() ) );
	connect( ui_password.kpushbutton_edit, SIGNAL( pressed() ), SLOT( editPassword() ) );
	connect( ui_password.kpushbutton_delete, SIGNAL( pressed() ), SLOT( deletePassword() ) );
	//background
	connect( ui_background.kcombobox_splashlist, SIGNAL( activated( int ) ), SLOT( updateSplashImage( int ) ) );
	connect( ui_background.kcombobox_splashlist->model(), SIGNAL( modelReset() ), SLOT( showSplashImage() ) );
	connect( ui_background.toolButton_browse, SIGNAL( clicked() ), SLOT( browseSplashImage() ) );
	connect( ui_background.kpushbutton_create, SIGNAL( pressed() ), SLOT( createSplashImage() ) );
	connect( ui_background.kpushbutton_getSplash, SIGNAL( pressed() ), SLOT( getNewSplashImages() ) );
	connect( ui_background.kpushbutton_preview, SIGNAL( pressed() ), SLOT( previewSplashImage() ) );

	connect( ui_background.gruburlrequester_gfxmenu, SIGNAL( pathChanged( const QString & ) ), SLOT( updateGfxmenu( const QString & ) ) );
	connect( ui_background.gruburlrequester_gfxmenu, SIGNAL( pathCleared() ), SLOT( clearGfxmenu() ) );
// 	connect( ui_background.kpushbutton_getGfxboot, SIGNAL( pressed() ), SLOT( getNewGfxboot() ) );
	//color
	connect( ui_color.checkBox_normal, SIGNAL( clicked( bool ) ), SLOT( enableNormal( bool ) ) );
	connect( ui_color.comboBox_normalBackground, SIGNAL( activated( int ) ), SLOT( updateColors() ) );
	connect( ui_color.comboBox_normalForeground, SIGNAL( activated( int ) ), SLOT( updateColors() ) );
	connect( ui_color.checkBox_normalBlink, SIGNAL( clicked() ), SLOT( blinkReset() ) );
	connect( ui_color.checkBox_highlighted, SIGNAL( clicked( bool ) ), SLOT( enableHighlight( bool ) ) );
	connect( ui_color.comboBox_highlightedBackground, SIGNAL( activated( int ) ), SLOT( updateColors() ) );
	connect( ui_color.comboBox_highlightedForeground, SIGNAL( activated( int ) ), SLOT( updateColors() ) );
	connect( ui_color.checkBox_highlightedBlink, SIGNAL( clicked() ), SLOT( blinkReset() ) );
	//maps
	connect( ui_map.kpushbutton_add, SIGNAL( pressed() ), SLOT( addMap() ) );
	connect( ui_map.kpushbutton_edit, SIGNAL( pressed() ), SLOT( editMap() ) );
	connect( ui_map.kpushbutton_remove, SIGNAL( pressed() ), SLOT( removeMap() ) );
//TOOLS
	connect( ui_tools.kpushbutton_backup, SIGNAL( pressed() ), SLOT( backup() ) );
	connect( ui_tools.kpushbutton_install, SIGNAL( pressed() ), SLOT( install() ) );
	connect( ui_tools.kpushbutton_devices, SIGNAL( pressed() ), SLOT( viewDevices() ) );
	connect( ui_tools.kpushbutton_files, SIGNAL( pressed() ), SLOT( viewFiles() ) );
	connect( ui_tools.kpushbutton_preferences, SIGNAL( pressed() ), SLOT( preferences() ) );
}

//SHOW FUNCTIONS
void KGRUBEditor::showEntries( const int entryRow )
{
	ui_entries.entryWidget->clear();
	for ( int i = 0; i < GrubEntries.size(); i++ )
	{
		ui_entries.entryWidget->addTopLevelItem( new QTreeWidgetItem( QStringList() << GrubEntries.at( i ).title() ) );
//FALLBACK
		if ( GrubSettings.fallback() == i )
			ui_entries.entryWidget->topLevelItem( i )->setText( 0, ui_entries.entryWidget->topLevelItem( i )->text( 0 ) + " " + i18nc( "@info:status", "[Fallback]" ) );
//AUTOMAGIC
		if ( GrubSettings.automagic().firstEntry() <= i && i <= GrubSettings.automagic().lastEntry() )
			ui_entries.entryWidget->topLevelItem( i )->setText( 0, ui_entries.entryWidget->topLevelItem( i )->text( 0 ) + " " + i18nc( "@info:status", "[AutoMagic]" ) );
//ICONS
		QDir iconsDir( KStandardDirs::locate( "data", "kgrubeditor/icons/" ) );
		foreach( const QString &icon, iconsDir.entryList( QStringList( "*.png" ) ) )
		{
			if ( ui_entries.entryWidget->topLevelItem( i )->text( 0 ).contains( QString( icon ).remove( ".png" ), Qt::CaseInsensitive ) )
			{
				ui_entries.entryWidget->topLevelItem( i )->setIcon( 0, KIcon( iconsDir.absoluteFilePath( icon ) ) );
				break;
			}
		}

		if ( ui_entries.entryWidget->topLevelItem( i )->icon( 0 ).isNull() )
		{
			if ( ui_entries.entryWidget->topLevelItem( i )->text( 0 ).contains( "Other Operating Systems", Qt::CaseInsensitive ) || ui_entries.entryWidget->topLevelItem( i )->text( 0 ).contains( QRegExp( "\\-+" ) ) )
			{
				ui_entries.entryWidget->topLevelItem( i )->setTextAlignment( 0, Qt::AlignCenter );
				ui_entries.entryWidget->setFirstItemColumnSpanned( ui_entries.entryWidget->topLevelItem( i ), true );
			}
			else
				ui_entries.entryWidget->topLevelItem( i )->setIcon( 0, KIcon( "unknown" ) );
		}
// DEFAULT
		if ( !ui_entries.entryWidget->topLevelItem( i )->icon( 0 ).isNull() )
		{
			QRadioButton *radio = new QRadioButton;
			ui_entries.entryWidget->setItemWidget( ui_entries.entryWidget->topLevelItem( i ), 1, radio );
			radio->setChecked( GrubSettings._default() == i );
			connect( radio, SIGNAL( clicked() ), SLOT( defaultUpdated() ) );
		}
	}
	if ( GrubSettings._default() == -1 )
	{
		QRadioButton *radio = static_cast<QRadioButton *>( ui_entries.entryWidget->itemWidget( ui_entries.entryWidget->topLevelItem( 0 ), 1 ) );
		if ( radio )
			radio->setChecked( true );
	}

	ui_entries.entryWidget->setCurrentItem( ui_entries.entryWidget->topLevelItem( entryRow ) );
	enableEntryActions( entryRow >= 0 && entryRow < GrubEntries.size() );
}
void KGRUBEditor::showHiddenMenu()
{
	ui_general.checkBox_hiddenMenu->setChecked( GrubSettings.hiddenMenu() );
}
void KGRUBEditor::showTimeout()
{
	const bool state = GrubSettings.timeout() != -1;
	ui_general.checkBox_timeout->setChecked( state );
	ui_general.kintspinbox_timeout->setEnabled( state );
	if ( state )
		ui_general.kintspinbox_timeout->setValue( GrubSettings.timeout() );
}

void KGRUBEditor::showPassword()
{
	ui_password.checkBox_md5->setChecked( GrubSettings.password().md5crypted() );
	ui_password.klineedit_password->setText( GrubSettings.password().password() );
	ui_password.klineedit_menu->setText( GrubSettings.password().configFile() );

	ui_password.kpushbutton_edit->setDisabled( GrubSettings.password().isEmpty() );
	ui_password.kpushbutton_delete->setDisabled( GrubSettings.password().isEmpty() );
}
void KGRUBEditor::showSplashImage()
{
	if ( !GrubSettings.splashImage().isEmpty() )
		ui_background.kcombobox_splashlist->setCurrentIndex( m_splashModel->indexOf( Core::Path::convertToGenericPath( GrubSettings.splashImage() ) ) );
	else
		ui_background.kcombobox_splashlist->setCurrentIndex( ui_background.kcombobox_splashlist->count() - 1 );
}
void KGRUBEditor::showGfxMenu()
{
	ui_background.gruburlrequester_gfxmenu->setPath( GrubSettings.gfxMenu() );
}
void KGRUBEditor::showColors()
{
	ui_color.colorPreview->setColor( GrubSettings.color() );
	ui_color.colorPreview->setEntries( GrubEntries );
	ui_color.colorPreview->update();
	if ( GrubSettings.color().blinkNormal() || GrubSettings.color().blinkHighlighted() )
		timer_blink->start( 500 );
	else
		if ( timer_blink->isActive() )
			timer_blink->stop();

	ui_color.checkBox_normal->setChecked( GrubSettings.color().normalIsEnabled() );
	ui_color.comboBox_normalBackground->setEnabled( GrubSettings.color().normalIsEnabled() );
	ui_color.comboBox_normalForeground->setEnabled( GrubSettings.color().normalIsEnabled() );
	ui_color.checkBox_normalBlink->setEnabled( GrubSettings.color().normalIsEnabled() );
	ui_color.checkBox_highlighted->setEnabled( GrubSettings.color().normalIsEnabled() );
	if ( GrubSettings.color().normalIsEnabled() )
	{
		ui_color.comboBox_normalBackground->setCurrentIndex( GRUB::ComplexCommand::Color::colorMap().key( GrubSettings.color().normalBackground() ) );
		ui_color.comboBox_normalForeground->setCurrentIndex( GRUB::ComplexCommand::Color::colorMap().key( GrubSettings.color().normalForeground() ) );
		ui_color.checkBox_normalBlink->setChecked( GrubSettings.color().blinkNormal() );
	}
	ui_color.checkBox_highlighted->setChecked( GrubSettings.color().highlightedIsEnabled() );
	ui_color.comboBox_highlightedBackground->setEnabled( GrubSettings.color().highlightedIsEnabled() );
	ui_color.comboBox_highlightedForeground->setEnabled( GrubSettings.color().highlightedIsEnabled() );
	ui_color.checkBox_highlightedBlink->setEnabled( GrubSettings.color().highlightedIsEnabled() );
	if ( GrubSettings.color().highlightedIsEnabled() )
	{
		ui_color.comboBox_highlightedBackground->setCurrentIndex( GRUB::ComplexCommand::Color::colorMap().key( GrubSettings.color().highlightedBackground() ) );
		ui_color.comboBox_highlightedForeground->setCurrentIndex( GRUB::ComplexCommand::Color::colorMap().key( GrubSettings.color().highlightedForeground() ) );
		ui_color.checkBox_highlightedBlink->setChecked( GrubSettings.color().blinkHighlighted() );
	}
}
void KGRUBEditor::showMaps( const int mapListRow )
{
	ui_map.klistwidget_maps->clear();
	if ( !GrubSettings.maps().isEmpty() )
	{
		foreach( const GRUB::ComplexCommand::Map map, GrubSettings.maps() )
		{
			ui_map.klistwidget_maps->addItem( map.result() );
		}
	}
	ui_map.kpushbutton_edit->setDisabled( GrubSettings.maps().isEmpty() );
	ui_map.kpushbutton_remove->setDisabled( GrubSettings.maps().isEmpty() );
	ui_map.klistwidget_maps->setCurrentRow( mapListRow );
}
void KGRUBEditor::showInput( const int entryRow, const int mapListRow, QWidget* focusWidget )
{
	showEntries( entryRow );

	showHiddenMenu();
	showTimeout();
	showPassword();
	showGfxMenu();
	showColors();
	showMaps( mapListRow );

	if ( focusWidget )
		focusWidget->setFocus();
}

//OTHER FUNCTIONS
void KGRUBEditor::enableEntryActions( const bool state )
{
	const int currentRow = ui_entries.entryWidget->indexOfTopLevelItem( ui_entries.entryWidget->currentItem() );
	const bool allowUp = ( currentRow != GrubSettings.automagic().firstEntry() && currentRow - 1 != GrubSettings.automagic().lastEntry() );
	const bool allowDown = ( currentRow != GrubSettings.automagic().lastEntry() && currentRow + 1 != GrubSettings.automagic().firstEntry() );
//RIGHT BUTTONS
	ui_entries.kpushbutton_up->setEnabled( state && currentRow != 0 && allowUp );
	ui_entries.kpushbutton_up->setToolTip( allowUp || currentRow == 0 ? QString() : i18nc( "@info:tooltip", "Moving this entry is not allowed, so as not to interefere with the AutoMagic tags." ) );

	ui_entries.kpushbutton_down->setEnabled( state && currentRow != GrubEntries.size() - 1 && allowDown );
	ui_entries.kpushbutton_down->setToolTip( allowDown || currentRow == GrubEntries.size() - 1 ? QString() : i18nc( "@info:tooltip", "Moving this entry is not allowed, so as not to interefere with the AutoMagic tags." ) );
//BOTTOM BUTTONS
	ui_entries.kpushbutton_edit->setEnabled( state );
	ui_entries.kpushbutton_remove->setEnabled( state );

	ui_entries.kpushbutton_details->setEnabled( state );
}

void KGRUBEditor::moveEntry( const int source, const int target )
{
	if ( source != target )
	{
		GRUB::ConfigFile::Entry tmp_entry = GrubEntries.at( source );
		GrubEntries.remove( source );
		GrubEntries.insert( target, tmp_entry );
		emit changed( true );
	}
}

void KGRUBEditor::monitorSplashDirectory( const QString &directory )
{
	const QFileInfo fi( Core::Path::convertToGenericPath( directory ) );
	if ( !fi.isDir() )
		return;

	m_splashDirWatchCreation->addDir( fi.absoluteFilePath(), KDirWatch::WatchFiles );
	foreach( const QString &splashImage, QDir( fi.absoluteFilePath() ).entryList( QStringList() << "*.xpm.gz", QDir::Files ) )
		m_splashDirWatchDeletion->addFile( fi.absoluteFilePath() + "/" + splashImage );
}
void KGRUBEditor::appendSplashDirectory( const QString &directory )
{
	const QFileInfo fi( Core::Path::convertToGenericPath( directory ) );
	if ( !fi.isDir() )
		return;

	if ( !Settings::splashDirs().contains( fi.absoluteFilePath() ) )
	{
		QStringList splashDirs = Settings::splashDirs();
		splashDirs.append( fi.absoluteFilePath() );
		Settings::setSplashDirs( splashDirs );
		Settings::self()->writeConfig();

		monitorSplashDirectory( fi.absoluteFilePath() );

		m_splashModel->reload();
	}
}
//SLOTS
void KGRUBEditor::defaultUpdated()
{
	for ( int i = 0; i < ui_entries.entryWidget->topLevelItemCount(); i++ )
	{
		QRadioButton *radio = static_cast<QRadioButton *>( ui_entries.entryWidget->itemWidget(  ui_entries.entryWidget->topLevelItem( i ), 1 ) );
		if ( radio && radio->isChecked() )
		{
			GrubSettings.setDefault( i );
			emit changed( true );
			return;
		}
	}
}

void KGRUBEditor::splashImageCreated( const QString &path )
{
	const QFileInfo fi( path );
	if ( !fi.exists() || !fi.isFile() || !fi.fileName().endsWith( ".xpm.gz" ) )
		return;

	kDebug() << "Splash Image" << path << "was created";
	m_splashDirWatchDeletion->addFile( path );

	SplashPreviewJob *splashPreviewJob = new SplashPreviewJob( path, this );
	connect( splashPreviewJob, SIGNAL( done( ThreadWeaver::Job * ) ), m_splashModel, SLOT( appendSplashImagePreview( ThreadWeaver::Job * ) ) );
	ThreadWeaver::Weaver::instance()->enqueue( splashPreviewJob );
}
void KGRUBEditor::splashImageDeleted( const QString &path )
{
	kDebug() << "Splash Image" << path << "was deleted";
	m_splashDirWatchDeletion->removeFile( path );

	if ( path == ui_background.kcombobox_splashlist->itemText( ui_background.kcombobox_splashlist->currentIndex() ) )
		clearSplashImage();

	m_splashModel->removeSplashImagePreview( path );
}
void KGRUBEditor::splashImagePreviewDone( ThreadWeaver::Job *job )
{
	SplashPreviewJob *splashJob = dynamic_cast< SplashPreviewJob * >( job );
	if ( splashJob )
	{
		KSplashScreen *splash = new KSplashScreen( QPixmap::fromImage( splashJob->splashImage() ) );
		splash->setAttribute( Qt::WA_DeleteOnClose );
		splash->show();
	}
}

void KGRUBEditor::selectionChanged()
{
	enableEntryActions( !ui_entries.entryWidget->selectedItems().isEmpty() );
}

//ENTRIES
void KGRUBEditor::moveUp()
{
	const int currentRow = ui_entries.entryWidget->indexOfTopLevelItem( ui_entries.entryWidget->currentItem() );
	if ( currentRow == -1 || currentRow == 0 )
		return;

	if ( currentRow == GrubSettings._default() )
		GrubSettings.setDefault( GrubSettings._default() - 1 );
	else if ( currentRow - 1 == GrubSettings._default() )
		GrubSettings.setDefault( GrubSettings._default() + 1 );

	if ( currentRow == GrubSettings.fallback() )
		GrubSettings.setFallback( GrubSettings.fallback() - 1 );
	else if ( currentRow - 1 == GrubSettings.fallback() )
		GrubSettings.setFallback( GrubSettings.fallback() + 1 );

	moveEntry( currentRow, currentRow - 1 );
	showEntries( currentRow - 1 );
}
void KGRUBEditor::moveDown()
{
	const int currentRow = ui_entries.entryWidget->indexOfTopLevelItem( ui_entries.entryWidget->currentItem() );
	if ( currentRow == -1 || currentRow == ui_entries.entryWidget->topLevelItemCount() - 1 )
		return;

	if ( currentRow == GrubSettings._default() )
		GrubSettings.setDefault( GrubSettings._default() + 1 );
	else if ( currentRow + 1 == GrubSettings._default() )
		GrubSettings.setDefault( GrubSettings._default() - 1 );

	if ( currentRow == GrubSettings.fallback() )
		GrubSettings.setFallback( GrubSettings.fallback() + 1 );
	else if ( currentRow + 1 == GrubSettings.fallback() )
		GrubSettings.setFallback( GrubSettings.fallback() - 1 );

	moveEntry( currentRow, currentRow + 1 );
	showEntries( currentRow + 1 );
}
void KGRUBEditor::add()
{
	GRUB::ConfigFile::Entry tmp_entry;
	EntryAssistant *entryAssistant = new EntryAssistant( &tmp_entry, this );
	if ( entryAssistant->exec() )
	{
		GrubEntries.append( tmp_entry );
		showEntries( ui_entries.entryWidget->topLevelItemCount() );
		emit changed( true );
	}
}
void KGRUBEditor::quickEdit()
{
	const int currentRow = ui_entries.entryWidget->indexOfTopLevelItem( ui_entries.entryWidget->currentItem() );
	if ( currentRow == -1 )
		return;

	GRUB::ConfigFile::Entry tmp_entry = GrubEntries.at( currentRow );
	QuickEditor *quickEditor = new QuickEditor( &tmp_entry, this );
	if ( quickEditor->exec() )
	{
		GrubEntries[ currentRow ] = tmp_entry;
		showEntries( currentRow );
		emit changed( true );
	}
}
void KGRUBEditor::fullEdit()
{
	const int currentRow = ui_entries.entryWidget->indexOfTopLevelItem( ui_entries.entryWidget->currentItem() );
	if ( currentRow == -1 )
		return;

	GRUB::ConfigFile::Entry tmp_entry = GrubEntries.at( currentRow );
	EntryAssistant *entryAssistant = new EntryAssistant( &tmp_entry, this );
	if ( entryAssistant->exec() )
	{
		GrubEntries[ currentRow ] = tmp_entry;
		showEntries( currentRow );
		emit changed( true );
	}
}
void KGRUBEditor::remove()
{
	const int currentRow = ui_entries.entryWidget->indexOfTopLevelItem( ui_entries.entryWidget->currentItem() );
	if ( currentRow == -1 )
		return;

	if ( KMessageBox::questionYesNo( this, i18nc( "@info", "Are you sure you want to remove the entry named '%1'?", GrubEntries.at( currentRow ).title() ) ) == KMessageBox::Yes )
	{
		if ( GrubSettings._default() > -1 )
		{
			if ( GrubEntries.size() == 1 )
				GrubSettings.clearDefault();
			else if ( currentRow == GrubSettings._default() )
				GrubSettings.setDefault( 0 );
			else if ( currentRow < GrubSettings._default() )
				GrubSettings.setDefault( GrubSettings._default() - 1 );
		}
		if ( GrubSettings.fallback() > -1 )
		{
			if ( GrubEntries.size() == 1 || currentRow == GrubSettings.fallback() )
				GrubSettings.clearFallback();
			else if ( currentRow < GrubSettings.fallback() )
				GrubSettings.setFallback( GrubSettings.fallback() - 1 );
		}
		if ( !GrubSettings.automagic().isEmpty() )
		{
			if ( currentRow < GrubSettings.automagic().firstEntry() )
			{
				GRUB::Misc::Automagic tmp_automagic( GrubSettings.automagic() );
				tmp_automagic.setFirstEntry( tmp_automagic.firstEntry() - 1 );
				tmp_automagic.setLastEntry( tmp_automagic.lastEntry() - 1 );

				GrubSettings.setAutomagic( tmp_automagic );
			}
			else if ( currentRow >= GrubSettings.automagic().firstEntry() && currentRow <= GrubSettings.automagic().lastEntry() )
			{
				if ( GrubSettings.automagic().firstEntry() != GrubSettings.automagic().lastEntry() )
				{
					GRUB::Misc::Automagic tmp_automagic( GrubSettings.automagic() );
					tmp_automagic.setLastEntry( tmp_automagic.lastEntry() - 1 );

					GrubSettings.setAutomagic( tmp_automagic );
				}
				else
					GrubSettings.clearAutomagic();
			}
		}

		GrubEntries.remove( currentRow );
		showEntries( currentRow != 0 ? currentRow - 1 : currentRow );
		emit changed( true );
	}
}
void KGRUBEditor::showDetails()
{
	const int currentRow = ui_entries.entryWidget->indexOfTopLevelItem( ui_entries.entryWidget->currentItem() );
	if ( currentRow == -1 )
		return;

	ui_entries.detailsWidget->previewEntry( GrubEntries.at( currentRow ) );
	ui_entries.stackedWidget->setCurrentIndex( 1 );
}
void KGRUBEditor::backToList()
{
	ui_entries.stackedWidget->setCurrentIndex( 0 );
	ui_entries.entryWidget->setFocus();
}

//SETINGS
void KGRUBEditor::checkBox_hiddenMenu_clicked()
{
	GrubSettings.setHiddenMenu( ui_general.checkBox_hiddenMenu->isChecked() );
	emit changed( true );
}

void KGRUBEditor::checkBox_timeout_clicked()
{
	ui_general.kintspinbox_timeout->setEnabled( ui_general.checkBox_timeout->isChecked() );
	updateTimeout();
}
void KGRUBEditor::updateTimeout()
{
	( ui_general.kintspinbox_timeout->isEnabled() ? GrubSettings.setTimeout( ui_general.kintspinbox_timeout->value() ) : GrubSettings.setTimeout( -1 ) );
	emit changed( true );
}

void KGRUBEditor::enableNormal( const bool state )
{
	ui_color.comboBox_normalBackground->setEnabled( state );
	ui_color.comboBox_normalForeground->setEnabled( state );
	ui_color.checkBox_normalBlink->setEnabled( state );
	ui_color.checkBox_highlighted->setEnabled( state );
	updateColors();
}
void KGRUBEditor::enableHighlight( const bool state )
{
	ui_color.comboBox_highlightedBackground->setEnabled( state );
	ui_color.comboBox_highlightedForeground->setEnabled( state );
	ui_color.checkBox_highlightedBlink->setEnabled( state );
	updateColors();
}
void KGRUBEditor::blinkReset()
{
	ui_color.colorPreview->resetBlinking();
	updateColors();
}
void KGRUBEditor::updateColors()
{
	GRUB::ComplexCommand::Color m_color;
	if ( ui_color.checkBox_normal->isChecked() )
	{
		m_color.setBlinkNormal( ui_color.checkBox_normalBlink->isChecked() );
		m_color.setNormalBackground( GRUB::ComplexCommand::Color::colorMap().value( ui_color.comboBox_normalBackground->currentIndex() ) );
		m_color.setNormalForeground( GRUB::ComplexCommand::Color::colorMap().value( ui_color.comboBox_normalForeground->currentIndex() ) );

		if ( ui_color.checkBox_highlighted->isChecked() )
		{
			m_color.setBlinkHighlighted( ui_color.checkBox_highlightedBlink->isChecked() );
			m_color.setHighlightedBackground( GRUB::ComplexCommand::Color::colorMap().value( ui_color.comboBox_highlightedBackground->currentIndex() ) );
			m_color.setHighlightedForeground( GRUB::ComplexCommand::Color::colorMap().value( ui_color.comboBox_highlightedForeground->currentIndex() ) );
		}
	}
	GrubSettings.setColor( m_color );
	showColors();
	emit changed( true );
}

void KGRUBEditor::createPassword()
{
	GRUB::ComplexCommand::Password tmp_password;
	PasswordAssistant *passwordAssistant = new PasswordAssistant( &tmp_password, this );
	if ( passwordAssistant->exec() )
	{
		GrubSettings.setPassword( tmp_password );
		showPassword();
		emit changed( true );
	}
}
void KGRUBEditor::editPassword()
{
	GRUB::ComplexCommand::Password tmp_password = GrubSettings.password();
	PasswordAssistant *passwordAssistant = new PasswordAssistant( &tmp_password, this );
	if ( passwordAssistant->exec() )
	{
		GrubSettings.setPassword( tmp_password );
		showPassword();
		emit changed( true );
	}
}
void KGRUBEditor::deletePassword()
{
	if ( KMessageBox::questionYesNo( this, i18nc( "@info", "Are you sure you want to delete the password?" ), i18nc( "@window:title", "Confirmation" ) ) == KMessageBox::Yes )
	{
		GrubSettings.clearPassword();
		showPassword();
		emit changed( true );
	}
}

void KGRUBEditor::updateSplashImage( const int previewRow )
{
	if ( previewRow == ui_background.kcombobox_splashlist->count() - 1 )
	{
		clearSplashImage();
		return;
	}

	QString grubPath = Core::Path::convertToGRUBPath( ui_background.kcombobox_splashlist->itemText( previewRow ) );
	GrubSettings.setSplashImage( grubPath );
	emit changed( true );
}
void KGRUBEditor::clearSplashImage()
{
	if ( !GrubSettings.splashImage().isEmpty() )
	{
		GrubSettings.clearSplashImage();
		showSplashImage();
		emit changed( true );
	}
}
void KGRUBEditor::browseSplashImage()
{
	KFileDialog dlgOpen( KUrl(), "*.xpm.gz|" + i18n( "GRUB Splash Image (*.xpm.gz)" ), this );
	dlgOpen.setCaption( i18nc( "@window:title", "Select GRUB Splash Image" ) );
	dlgOpen.setOperationMode( KFileDialog::Opening );
	if ( !dlgOpen.exec() )
		return;

	KUrl imageFilename = dlgOpen.selectedUrl();
	QString parentDir = QFileInfo( imageFilename.path() ).dir().path();

	GrubSettings.setSplashImage( Core::Path::convertToGRUBPath( imageFilename.path() ) );
	emit changed( true );

	if ( Settings::splashDirs().contains( parentDir ) )
		showSplashImage();
	else
		appendSplashDirectory( QFileInfo( imageFilename.path() ).dir().path() );
}
void KGRUBEditor::createSplashImage()
{
	KFileDialog dlgOpen( KUrl(), "*.bmp *.jpe *.jpg *.jpeg *.png *.ppm *.pgm *.pnm *.tiff *.tga *.xbm *.xpm|" + i18n( "Image Files" ), this );
	dlgOpen.setCaption( i18nc( "@window:title", "Open Image To Convert" ) );
	dlgOpen.setOperationMode( KFileDialog::Opening );
	if ( !dlgOpen.exec() )
		return;
	KUrl imageFilename = dlgOpen.selectedUrl();

	KFileDialog dlgSave( KUrl(), "*.xpm.gz|" + i18n( "GRUB Splash Image (*.xpm.gz)" ), this );
	dlgSave.setCaption( i18nc( "@window:title", "Save GRUB Splash Image" ) );
	dlgSave.setOperationMode( KFileDialog::Saving );
	if ( !dlgSave.exec() )
		return;
	KUrl splashFilename = dlgSave.selectedUrl();

	KProcess convert( this );
	convert.setProgram( "convert", QStringList() << imageFilename.path() << "-resize" << "640x480!" << "-colors" << "14" << "-depth" << "8" << splashFilename.path() );
	convert.start();
	convert.waitForFinished();

	appendSplashDirectory( QFileInfo( splashFilename.path() ).dir().path() );
}
void KGRUBEditor::getNewSplashImages()
{
	const QString downloadPath( "/boot/grub/splashimages" );
	Core::Root::makeDirWritableByUserGroup( downloadPath, this );

	//If directory doesn't exist we should start monitoring it.
	if ( !m_splashDirWatchCreation->contains( downloadPath ) )
		m_splashDirWatchCreation->addDir( downloadPath, KDirWatch::WatchFiles );

	KNS::Engine engine( this );
	if ( engine.init( "kgrubeditor_splash.knsrc" ) )
		engine.downloadDialogModal( this );
}
void KGRUBEditor::previewSplashImage()
{
	if ( GrubSettings.splashImage().isEmpty() )
	{
		KMessageBox::information( this, i18nc( "@info", "Please select a splash image." ) );
		return;
	}

	SplashPreviewJob *splashPreviewJob = new SplashPreviewJob( Core::Path::convertToGenericPath( GrubSettings.splashImage() ), this );
	connect( splashPreviewJob, SIGNAL( done( ThreadWeaver::Job * ) ), SLOT( splashImagePreviewDone( ThreadWeaver::Job * ) ) );
	ThreadWeaver::Weaver::instance()->enqueue( splashPreviewJob );
}
void KGRUBEditor::updateGfxmenu( const QString &path )
{
	if ( !GrubSettings.gfxMenu().isEmpty() )
		m_gfxmenuDirWatchDeletion->removeFile( Core::Path::convertToGenericPath( GrubSettings.gfxMenu() ) );
	m_gfxmenuDirWatchDeletion->addFile( Core::Path::convertToGenericPath( path ) );

	GrubSettings.setGfxMenu( path );
	showGfxMenu();
	emit changed( true );
}
void KGRUBEditor::clearGfxmenu()
{
	if ( !GrubSettings.gfxMenu().isEmpty() )
	{
		GrubSettings.clearGfxMenu();
		showGfxMenu();
		emit changed( true );
	}
}
// void KGRUBEditor::getNewGfxboot()
// {
// 	KNS::Engine *engine = new KNS::Engine( 0 );
// 	if ( !engine->init( "kgrubeditor_gfxboot.knsrc" ) )
// 	{
// 		delete engine;
// 		return;
// 	}
// 
// 	KNS::Entry::List entries = engine->downloadDialogModal( this );
// 
// 	delete engine;
// }

void KGRUBEditor::addMap()
{
	GRUB::ComplexCommand::Map tmp_map;
	MapEditor *mapEditor = new MapEditor( &tmp_map, this );
	if ( mapEditor->exec() )
	{
		GrubSettings.addMap( tmp_map );
		showMaps( ui_map.klistwidget_maps->count() );
		emit changed( true );
	}
}
void KGRUBEditor::editMap()
{
	if ( !ui_map.klistwidget_maps->currentItem() )
		return;

	GRUB::ComplexCommand::Map tmp_map = GrubSettings.maps().at( ui_map.klistwidget_maps->currentRow() );
	MapEditor *mapEditor = new MapEditor( &tmp_map, this );
	if ( mapEditor->exec() )
	{
		GrubSettings.replaceMap( ui_map.klistwidget_maps->currentRow(), tmp_map );
		showMaps( ui_map.klistwidget_maps->currentRow() );
		emit changed( true );
	}
}
void KGRUBEditor::removeMap()
{
	if ( !ui_map.klistwidget_maps->currentItem() )
		return;

	if ( KMessageBox::questionYesNo( this, i18nc( "@info", "Are you sure you want to remove this map?" ) ) )
	{
		GrubSettings.removeMap( ui_map.klistwidget_maps->currentRow() );
		showMaps( ui_map.klistwidget_maps->currentRow() - 1 );
		emit changed( true );
	}
}

//TOOLS
void KGRUBEditor::backup()
{
	BackupAssistant *backupAssistant = new BackupAssistant( this );
	backupAssistant->setMenuFile( Settings::menulst() );
	backupAssistant->exec();
}
void KGRUBEditor::install()
{
	InstallAssistant *installAssistant = new InstallAssistant( this );
	installAssistant->exec();
}
void KGRUBEditor::viewDevices()
{
	ViewDevices *viewDevices = new ViewDevices( this );
	viewDevices->exec();
}
void KGRUBEditor::viewFiles()
{
	ViewFiles *viewFiles = new ViewFiles( Settings::menulst().pathOrUrl(), Settings::devicemap().pathOrUrl(), this );
	viewFiles->exec();
}
void KGRUBEditor::preferences()
{
	if ( KConfigDialog::showDialog( "Settings" ) )
		return;

	QWidget *pathsSettings = new QWidget;
	ui_settings_paths.setupUi( pathsSettings );

	KConfigDialog* dialog = new KConfigDialog( this, "Settings", Settings::self() );
	dialog->addPage( pathsSettings, i18nc( "@item:inlist", "Paths" ) )->setIcon( KIcon( "folder-txt" ) );
	connect( dialog, SIGNAL( settingsChanged( QString ) ), SLOT( refresh() ) );
	dialog->setAttribute( Qt::WA_DeleteOnClose );
	dialog->setFaceType( KPageDialog::Plain );
	dialog->show();
}

//OTHER
void KGRUBEditor::refresh( int entryRow, int mapListRow, QWidget* focusWidget )
{
	Core::FileIO::readFileInput( Settings::menulst(), Settings::devicemap(), &GrubSettings, &GrubEntries, &Core::Devices::DeviceList, this );

	if ( entryRow < -1 || entryRow >= GrubEntries.size() )
		entryRow = -1;
	if ( mapListRow < 0 || mapListRow >= GrubSettings.maps().size() )
		mapListRow = 0;

	showInput( entryRow, mapListRow, focusWidget );

	if ( !KIO::NetAccess::exists( KUrl( Settings::menulst().path() + "_original" ), true, this ) )
		Core::Backup::backup( Settings::menulst(), KUrl( Settings::menulst().path() + "_original" ), this, false );

	m_splashModel->reload();
}
void KGRUBEditor::load()
{
	QTimer::singleShot( 0, this, SLOT( refresh() ) );
}
void KGRUBEditor::save()
{
	Core::FileIO::writeFileOutput( Settings::menulst(), GrubSettings, GrubEntries, this );
}
void KGRUBEditor::defaults()
{
	if ( KIO::NetAccess::exists( KUrl( Settings::menulst().path() + "_original" ), true, this ) && Core::Backup::backup( KUrl( Settings::menulst().path() + "_original" ), Settings::menulst(), this ) )
		refresh();
}

#include "kgrubeditor.moc"
