;;;
;;; Copyright 2016 Clozure Associates
;;;
;;; Licensed under the Apache License, Version 2.0 (the "License");
;;; you may not use this file except in compliance with the License.
;;; You may obtain a copy of the License at
;;;
;;;     http://www.apache.org/licenses/LICENSE-2.0
;;;
;;; Unless required by applicable law or agreed to in writing, software
;;; distributed under the License is distributed on an "AS IS" BASIS,
;;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;;; See the License for the specific language governing permissions and
;;; limitations under the License.

(eval-when (:compile-toplevel :load-toplevel :execute)
  (require "ARM64-ARCH"))

(in-package "ARM64")

(defun count-trailing-zeros-64 (u64)
  (do* ((i 0 (1+ i)))
       ((or (= i 64) (logbitp i u64))
        i)
    (declare (fixnum i))))

(defun count-leading-zeros-64 (u64)
  (do* ((count 0 (1+ count))
        (i 63 (1- i)))
       ((or (= count 64) (logbitp i u64))
        count)
    (declare (fixnum count i))))

(defun count-leading-zeros-32 (u32)
  (do* ((count 0 (1+ count))
        (i 31 (1- i)))
       ((or (= count 32) (logbitp i u32))
        count)
    (declare (fixnum count i))))

#|
(defun test-ctz ()
  (let* ((n #xffffffffffffffff))
    (loop for i from 0 to 64 do
          (format t "~16,'0x: " n)
          (format t "~d~%" (count-trailing-zeros-64 n))
          (setq n (ldb (byte 64 0) (ash n 1))))))

(defun test-clz ()
  (let* ((all-ones #xffffffffffffffff)
         (n all-ones))
    (loop for i from 0 to 64 do
          (format t "~16,'0x: " n)
          (format t "~2d~%" (count-leading-zeros-64 n))
          (setq n (ldb (byte 64 0) (ash n -1))))))
|#

(defun clear-trailing-ones-64 (u64)
  (ldb (byte 64 0) (logand u64 (1+ u64))))

(defun rotate-right-64 (u64 n)
  (let* ((right (logand n 63))
         (left (logand (- n) 63)))
    (logior (ldb (byte 64 0) (ash u64 (- right)))
            (ldb (byte 64 0) (ash u64 left)))))

;;; Adapted from https://dougallj.wordpress.com/2021/10/30/bit-twiddling-optimising-aarch64-logical-immediate-encoding-and-decoding/

(defun %encode-logical-immediate (u64)
  ;; Consider an ARM64 logical immediate as a pattern of "o" ones preceded
  ;; by "z" more-significant zeroes, repeated to fill a 64-bit integer.
  ;; o > 0, z > 0, and the size (o + z) is a power of two in [2,64]. This
  ;; part of the pattern is encoded in the fields "imms" and "N".
  ;;
  ;; "immr" encodes a further right rotate of the repeated pattern, allowing
  ;; a wide range of useful bitwise constants to be represented.
  ;;
  ;; (The spec describes the "immr" rotate as rotating the "o + z" bit
  ;; pattern before repeating it to fill 64-bits, but, as it's a repeating
  ;; pattern, rotating afterwards is equivalent.)
  ;;
  ;; This encoding is not allowed to represent all-zero or all-one values,
  ;; which must have been excluded prior to calling this function,
  ;;
  ;; To detect an immediate that may be encoded in this scheme, we first
  ;; remove the right-rotate, by rotating such that the least significant
  ;; bit is a one and the most significant bit is a zero.
  ;;
  ;; We do this by clearing any trailing one bits, then counting the
  ;; trailing zeroes. This finds an "edge", where zero goes to one.
  ;; We then rotate the original value right by that amount, moving
  ;; the first one to the least significant bit.
  (let* ((rotation (count-trailing-zeros-64 (clear-trailing-ones-64 u64)))
         (normalized (rotate-right-64 u64 (logand rotation 63)))
         ;; Now we have normalized the value, and determined the
         ;; rotation, we can determine "z" by counting the leading
         ;; zeroes, and "o" by counting the trailing ones. (These will
         ;; both be positive, as we already rejected 0 and ~0, and
         ;; rotated the value to start with a zero and end with a
         ;; one.)
         (zeros (count-leading-zeros-64 normalized))
         (ones (count-trailing-zeros-64 (ldb (byte 64 0) (lognot normalized))))
         (size (+ zeros ones)))
    ;; Detect the repeating pattern (by comparing every repetition to the
    ;; one next to it, using rotate).
    (if (/= (rotate-right-64 u64 (logand size 63)) u64)
      nil
      ;; We do not need to further validate size to ensure it is a
      ;; power of two between 2 and 64. The only "minimal" patterns
      ;; that can repeat to fill a 64-bit value must have a length
      ;; that is a factor of 64 (i.e. it is a power of two in the
      ;; range [1,64]). And our pattern cannot be of length one (as we
      ;; already rejected 0 and ~0).
      ;;
      ;; By "minimal" patterns I refer to patterns which do not
      ;; themselves contain repetitions. For example, '010101' is a
      ;; non-minimal pattern of a non-power-of-two length that can
      ;; pass the above rotational test. It consists of the minimal
      ;; pattern '01'. All our patterns are minimal, as they contain
      ;; only one contiguous run of ones separated by at least one
      ;; zero.
      ;;
      ;; Finally, we encode the values. "rotation" is the amount we
      ;; rotated right by to "undo" the right-rotate encoded in immr,
      ;; so must be negated.
      ;;
      ;; size 2:  N=0 immr=00000r imms=11110s
      ;; size 4:  N=0 immr=0000rr imms=1110ss
      ;; size 8:  N=0 immr=000rrr imms=110sss
      ;; size 16: N=0 immr=00rrrr imms=10ssss
      ;; size 32: N=0 immr=0rrrrr imms=0sssss
      ;; size 64: N=1 immr=rrrrrr imms=ssssss
      (let* ((immr (logand (- rotation) (1- size)))
             (imms (logior (- (ash size 1))
                           (1- ones)))
             (n (ash size (- 6))))
        (logior (ash n 12) (ash immr 6) (ldb (byte 6 0) imms))))))

(defun encode-logical-immediate (n)
  "Return a 13 bit encoding of n, or NIL if it can't be encoded."
  (let* ((u64 (ldb (byte 64 0) n))
         (u64-inverted (ldb (byte 64 0) (lognot u64))))
    (if (or (/= n u64)                  ;n too big
            (zerop u64)                 ;can't encode all zeros...
            (zerop u64-inverted))       ;...or all ones
      nil
      (%encode-logical-immediate u64))))

;;; Form of an encoded logical immediate:
;;;
;;;      1
;;;  2 1 0 9 8 7 6 5 4 3 2 1 0
;;; +-+-+-+-+-+-+-+-+-+-+-+-+-+
;;; |N|   immr    |    imms   |
;;; +-+-+-+-+-+-+-+-+-+-+-+-+-+


(defconstant mask-lookup
  #(#xffffffffffffffff                  ;size = 64
    #x00000000ffffffff                  ;size = 32
    #x0000ffff0000ffff                  ;size = 16
    #x00ff00ff00ff00ff                  ;size = 8
    #x0f0f0f0f0f0f0f0f                  ;size = 4
    #x3333333333333333))                ;size = 2

(defun decode-logical-immediate (imm)
  (let* ((n (ldb (byte 1 12) imm))
         (immr (ldb (byte 6 6) imm))
         (imms (ldb (byte 6 0) imm))
         (pattern (logior (ash n 6) (logand (lognot imms) #x3f))))
    (if (zerop (logand pattern (1- pattern)))
      nil
      (let* ((leading-zeros (count-leading-zeros-32 pattern))
             (imms-mask (ash #x7fffffff (- leading-zeros)))
             (mask (aref mask-lookup (- leading-zeros 25)))
             (s (logand (1+ imms) imms-mask)))
        (rotate-right-64 (logxor mask (ash mask s)) immr)))))

#|
(defun all-logical-immediates ()
  "Return a list of all possible encoded logical immediates."
  ;; https://gist.github.com/dinfuehr/9e1c2f28d0f912eae5e595207cb835c2
  (flet ((encode-imms (size length)
           (logior length (ecase size
                            (2  #b111100)
                            (4  #b111000)
                            (8  #b110000)
                            (16 #b100000)
                            ((32 64) #b000000)))))
    (let ((results nil))
      (dolist (size '(2 4 8 16 32 64))
        (loop for length from 0 below (1- size) do
              (loop for rotation from 0 below size do
                    (let ((n (if (= size 64) 1 0))
                          (immr rotation)
                          (imms (encode-imms size length)))
                      (push (logior (ash n 12)
                                    (ash immr 6)
                                    (ldb (byte 6 0) imms))
                            results)))))
      (nreverse results))))

(defun test-logical-immediate-encode-decode (&optional show-values)
  (let ((values (all-logical-immediates)))
    (assert (= (length values) 5334))
    (dolist (val values t)
      (let ((decoded (decode-logical-immediate val)))
        (assert (not (null decoded)))
        (assert (= val (encode-logical-immediate decoded)))
        (when show-values
          (let ((n (ldb (byte 1 12) val))
                (immr (ldb (byte 6 6) val))
                (imms (ldb (byte 6 0) val)))
            (format t "~&~(~16,'0x~) ~64,'0b N=~b immr=~6,'0b imms=~6,'0b" decoded decoded
                    n immr imms)))))))
|#

(defparameter *arm64-operand-qualifiers*
  '(
    nil
    :w	
    :x
    :wsp	
    :sp	
    :s_b
    :s_h
    :s_s
    :s_d
    :s_q
    :v_8b
    :v_16b
    :v_4h
    :v_8h
    :v_2s
    :v_4s
    :v_1d
    :v_2d
    :v_1q
    :imm_0_7
    :imm_0_15
    :imm_0_31
    :imm_0_63
    :imm_1_32
    :imm_1_64
    :lsl
    :msl
    :w-ext                              ;word reg, maybe extended
    :x-ext                              ;x reg, maybe extended
    :w-shift                            ;word reg, maybe shifted
    :x-shift                            ;x reg, maybe shifted
    :aimm                               ;12-bit constant, maybe shifted left 12 bits
    :retrieve
    ))

(defun %encode-arm64-operand-qualifier (q)
  (or (position q *arm64-operand-qualifiers*)
      (error "Unknown arm64 operand qualifier: ~s" q)))

(defmacro encode-arm64-operand-qualifier (q)
  (%encode-arm64-operand-qualifier q))

(defmacro encode-arm64-operand-qualifiers (list)
  (mapcar #'%encode-arm64-operand-qualifier list))

(defparameter *arm64-condition-names*
  '(("eq" . 0)                          ;equal
    ("ne" . 1)                          ;not equal
    ("cs" . 2) ("hs" . 2)               ;carry set, unsigned higher or same
    ("cc" . 3) ("lo" . 3)               ;carry clear, unsigned lower
    ("mi" . 4)                          ;minus, negative
    ("pl" . 5)                          ;plus, positive or zero
    ("vs" . 6)                          ;overflow
    ("vc" . 7)                          ;no overflow
    ("hi" . 8)                          ;unsigned higher
    ("ls" . 9)                          ;unsigned lower or same
    ("ge" . 10)                         ;signed >=
    ("lt" . 11)                         ;signed <
    ("gt" . 12)                         ;signed >
    ("le" . 13)                         ;signed <=
    ("al" . 14)                         ;always
    ("nv" . 15)))                       ;identical to always

(defun lookup-arm64-condition-name (name)
  (cdr (assoc name *arm64-condition-names* :test #'string-equal)))

(defun lookup-arm64-condition-value (val)
  (car (rassoc val *arm64-condition-names* :test #'eq)))

(defun need-arm64-condition-name (name)
  (or (lookup-arm64-condition-name name)
      (error "Unknown ARM64 condition name ~s." name)))


(defstruct arm64-opcode
  name
  value
  mask
  class
  ??
  features
  operands
  qualifiers
  flags)



'(
  ("adc" #x1a000000 #xffe0fc00 :addsub-carry 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) 0)
  ("adc" #x9a000000 #xffe0fc00 :addsub-carry 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) 0) 
  ("adcs" #x3a000000 #xffe0fc00 :addsub-carry 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w)0)
  ("adcs" #xba000000 #xffe0fc00 :addsub-carry 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x)0) 
  ("sbc" #x5a000000 #xffe0fc00 :addsub-carry 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-HAS-ALIAS)
  ("sbc" #xda000000 #xffe0fc00 :addsub-carry 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-HAS-ALIAS)
  ("ngc" #x5a0003e0 #xffe0ffe0 :addsub-carry 0 :CORE '(:Rd :Rm) '(:w :w) F-ALIAS) 
  ("ngc" #xda0003e0 #xffe0ffe0 :addsub-carry 0 :CORE '(:Rd :Rm) '(:x :x) F-ALIAS)
  ("sbcs" #x7a000000 #xffe0fc00 :addsub-carry 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-HAS-ALIAS)
  ("sbcs" #xfa000000 #xffe0fc00 :addsub-carry 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-HAS-ALIAS)
 
  ("ngcs" #x7a0003e0 #xffe0ffe0 :addsub-carry 0 :CORE '(:Rd :Rm) '(:w :w) F-ALIAS)
  ("ngcs" #xfa0003e0 #xffe0ffe0 :addsub-carry 0 :CORE '(:Rd :Rm) '(:x :x) F-ALIAS) 
  ("add" #x0b200000 #x7fe00000 :addsub-ext 0 :CORE '(:Rd-SP :Rn-SP :Rm-EXT) QL-I3-EXT F-SF) 
  ("adds" #x2b200000 #x7fe00000 :addsub-ext 0 :CORE '(:Rd :Rn-SP :Rm-EXT) QL-I3-EXT (F-HAS-ALIAS  F-SF)) 
  ("cmn" #x2b20001f #x7fe0001f :addsub-ext 0 :CORE '(:Rn-SP :Rm-EXT) QL-I2-EXT (F-ALIAS  F-SF)) 
  ("sub" #x4b200000 #x7fe00000 :addsub-ext 0 :CORE '(:Rd-SP :Rn-SP :Rm-EXT) QL-I3-EXT F-SF) 
  ("subs" #x6b200000 #x7fe00000 :addsub-ext 0 :CORE '(:Rd :Rn-SP :Rm-EXT) QL-I3-EXT (F-HAS-ALIAS  F-SF)) 
  ("cmp" #x6b20001f #x7fe0001f :addsub-ext 0 :CORE '(:Rn-SP :Rm-EXT) QL-I2-EXT (F-ALIAS  F-SF)) 
  ("add" #x11000000 #xff000000 :addsub-imm OP-ADD :CORE '(:Rd-SP :Rn-SP :AIMM) '(:w :w :aimm) F-HAS-ALIAS) 
  ("add" #x91000000 #xff000000 :addsub-imm OP-ADD :CORE '(:Rd-SP :Rn-SP :AIMM) '(:x :x :aimm) F-HAS-ALIAS)
  ("mov" #x11000000 #x7ffffc00 :addsub-imm 0 :CORE '(:Rd-SP :Rn-SP) QL-I2SP (F-ALIAS  F-SF)) 
  ("adds" #x31000000 #xff000000 :addsub-imm 0 :CORE '(:Rd :Rn-SP :AIMM) '(:w :w :aimm) F-HAS-ALIAS) 
  ("adds" #xb1000000 #xff000000 :addsub-imm 0 :CORE '(:Rd :Rn-SP :AIMM) '(:x :x :aimm) F-HAS-ALIAS)
  ("cmn" #x3100001f #x7f00001f :addsub-imm 0 :CORE '(:Rn-SP :AIMM) QL-R1NIL (F-ALIAS  F-SF)) 
  ("sub" #x51000000 #xff000000 :addsub-imm 0 :CORE '(:Rd-SP :Rn-SP :AIMM) '(:w :w :aimm) 0)
  ("sub" #xd1000000 #xff000000 :addsub-imm 0 :CORE '(:Rd-SP :Rn-SP :AIMM) '(:x :x :aimm) 0)  
  ("subs" #x71000000 #xff000000 :addsub-imm 0 :CORE '(:Rd :Rn-SP :AIMM) '(:w :w :aimm) F-HAS-ALIAS)
  ("subs" #xf1000000 #xff000000 :addsub-imm 0 :CORE '(:Rd :Rn-SP :AIMM) '(:x :x :aimm) -HAS-ALIAS)  
  ("cmp" #x7100001f #x7f00001f :addsub-imm 0 :CORE '(:Rn-SP :AIMM) QL-R1NIL (F-ALIAS  F-SF)) 
  ("add" #xb000000 #xff200000 :addsub-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) 0)
  ("add" #x8b000000 #xff200000 :addsub-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) 0) 
  ("adds" #x2b000000 #xff200000 :addsub-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) F-HAS-ALIAS) 
  ("adds" #xab000000 #xff200000 :addsub-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) F-HAS-ALIAS)
  ("cmn" #x2b00001f #x7f20001f :addsub-shift 0 :CORE '(:Rn :Rm-SFT) QL-I2SAME (F-ALIAS  F-SF)) 
  ("sub" #x4b000000 #xff200000 :addsub-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) F-HAS-ALIAS)
  ("sub" #xcb000000 #xff200000 :addsub-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) F-HAS-ALIAS)

  ("neg" #x4b0003e0 #x7f2003e0 :addsub-shift 0 :CORE '(:Rd :Rm-SFT) QL-I2SAME (F-ALIAS  F-SF)) 
  ("subs" #x6b000000 #xff200000 :addsub-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) F-HAS-ALIAS)
  ("subs" #xeb000000 #xff200000 :addsub-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift F-HAS-ALIAS)
   ("cmp" #x6b00001f #x7f20001f :addsub-shift 0 :CORE '(:Rn :Rm-SFT) QL-I2SAME (F-ALIAS  F-SF)) 
   ("negs" #x6b0003e0 #x7f2003e0 :addsub-shift 0 :CORE '(:Rd :Rm-SFT) QL-I2SAME (F-ALIAS  F-SF)) 
   ("saddlv" #xe303800 #xbf3ffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES-L F-SIZEQ) 
   ("smaxv" #xe30a800 #xbf3ffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES F-SIZEQ) 
   ("sminv" #xe31a800 #xbf3ffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES F-SIZEQ) 
   ("addv" #xe31b800 #xbf3ffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES F-SIZEQ) 
   ("uaddlv" #x2e303800 #xbf3ffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES-L F-SIZEQ) 
   ("umaxv" #x2e30a800 #xbf3ffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES F-SIZEQ) 
   ("uminv" #x2e31a800 #xbf3ffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES F-SIZEQ) 
   ("fmaxnmv" #x2e30c800 #xbfbffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES-FP F-SIZEQ) 
   ("fmaxv" #x2e30f800 #xbfbffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES-FP F-SIZEQ) 
   ("fminnmv" #x2eb0c800 #xbfbffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES-FP F-SIZEQ) 
   ("fminv" #x2eb0f800 #xbfbffc00 :asimdall 0 SIMD '(:Fd :Vn) QL-XLANES-FP F-SIZEQ) 
   ("saddl" #x0e200000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("saddl2" #x4e200000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("saddw" #x0e201000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3WIDEBHS F-SIZEQ) 
   ("saddw2" #x4e201000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3WIDEBHS2 F-SIZEQ) 
   ("ssubl" #x0e202000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("ssubl2" #x4e202000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("ssubw" #x0e203000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3WIDEBHS F-SIZEQ) 
   ("ssubw2" #x4e203000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3WIDEBHS2 F-SIZEQ) 
   ("addhn" #x0e204000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3NARRBHS F-SIZEQ) 
   ("addhn2" #x4e204000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3NARRBHS2 F-SIZEQ) 
   ("sabal" #x0e205000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("sabal2" #x4e205000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("subhn" #x0e206000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3NARRBHS F-SIZEQ) 
   ("subhn2" #x4e206000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3NARRBHS2 F-SIZEQ) 
   ("sabdl" #x0e207000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("sabdl2" #x4e207000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("smlal" #x0e208000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("smlal2" #x4e208000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("sqdmlal" #x0e209000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGHS F-SIZEQ) 
   ("sqdmlal2" #x4e209000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGHS2 F-SIZEQ) 
   ("smlsl" #x0e20a000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("smlsl2" #x4e20a000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("sqdmlsl" #x0e20b000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGHS F-SIZEQ) 
   ("sqdmlsl2" #x4e20b000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGHS2 F-SIZEQ) 
   ("smull" #x0e20c000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("smull2" #x4e20c000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("sqdmull" #x0e20d000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGHS F-SIZEQ) 
   ("sqdmull2" #x4e20d000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGHS2 F-SIZEQ) 
   ("pmull" #x0e20e000 #xffe0fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGB 0) 
   ("pmull" #x0ee0e000 #xffe0fc00 :asimddiff 0 :CRYPTO '(:Vd :Vn :Vm) QL-V3LONGD 0) 
   ("pmull2" #x4e20e000 #xffe0fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGB2 0) 
   ("pmull2" #x4ee0e000 #xffe0fc00 :asimddiff 0 :CRYPTO '(:Vd :Vn :Vm) QL-V3LONGD2 0) 
   ("uaddl" #x2e200000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("uaddl2" #x6e200000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("uaddw" #x2e201000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3WIDEBHS F-SIZEQ) 
   ("uaddw2" #x6e201000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3WIDEBHS2 F-SIZEQ) 
   ("usubl" #x2e202000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("usubl2" #x6e202000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("usubw" #x2e203000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3WIDEBHS F-SIZEQ) 
   ("usubw2" #x6e203000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3WIDEBHS2 F-SIZEQ) 
   ("raddhn" #x2e204000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3NARRBHS F-SIZEQ) 
   ("raddhn2" #x6e204000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3NARRBHS2 F-SIZEQ) 
   ("uabal" #x2e205000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("uabal2" #x6e205000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("rsubhn" #x2e206000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3NARRBHS F-SIZEQ) 
   ("rsubhn2" #x6e206000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3NARRBHS2 F-SIZEQ) 
   ("uabdl" #x2e207000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("uabdl2" #x6e207000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("umlal" #x2e208000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("umlal2" #x6e208000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("umlsl" #x2e20a000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("umlsl2" #x6e20a000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("umull" #x2e20c000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS F-SIZEQ) 
   ("umull2" #x6e20c000 #xff20fc00 :asimddiff 0 SIMD '(:Vd :Vn :Vm) QL-V3LONGBHS2 F-SIZEQ) 
   ("smlal" #x0f002000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("smlal2" #x4f002000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("sqdmlal" #x0f003000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("sqdmlal2" #x4f003000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("smlsl" #x0f006000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("smlsl2" #x4f006000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("sqdmlsl" #x0f007000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("sqdmlsl2" #x4f007000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("mul" #xf008000 #xbf00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT F-SIZEQ) 
   ("smull" #x0f00a000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("smull2" #x4f00a000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("sqdmull" #x0f00b000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("sqdmull2" #x4f00b000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("sqdmulh" #xf00c000 #xbf00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT F-SIZEQ) 
   ("sqrdmulh" #xf00d000 #xbf00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT F-SIZEQ) 
   ("fmla" #xf801000 #xbf80f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-FP F-SIZEQ) 
   ("fmls" #xf805000 #xbf80f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-FP F-SIZEQ) 
   ("fmul" #xf809000 #xbf80f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-FP F-SIZEQ) 
   ("mla" #x2f000000 #xbf00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT F-SIZEQ) 
   ("umlal" #x2f002000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("umlal2" #x6f002000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("mls" #x2f004000 #xbf00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT F-SIZEQ) 
   ("umlsl" #x2f006000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("umlsl2" #x6f006000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("umull" #x2f00a000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L F-SIZEQ) 
   ("umull2" #x6f00a000 #xff00f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-L2 F-SIZEQ) 
   ("fmulx" #x2f809000 #xbf80f400 :asimdelem 0 SIMD '(:Vd :Vn :Em) QL-ELEMENT-FP F-SIZEQ) 
   ("ext" #x2e000000 #xbfe0c400 :asimdext 0 SIMD '(:Vd :Vn :Vm :IDX) QL-VEXT F-SIZEQ) 
   ("movi" #xf000400 #xbff89c00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S0W F-SIZEQ) 
   ("orr" #xf001400 #xbff89c00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S0W F-SIZEQ) 
   ("movi" #xf008400 #xbff8dc00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S0H F-SIZEQ) 
   ("orr" #xf009400 #xbff8dc00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S0H F-SIZEQ) 
   ("movi" #xf00c400 #xbff8ec00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S1W F-SIZEQ) 
   ("movi" #xf00e400 #xbff8fc00 :asimdimm OP-V-MOVI-B SIMD '(:Vd :SIMD-IMM) QL-SIMD-IMM-B F-SIZEQ) 
   ("fmov" #xf00f400 #xbff8fc00 :asimdimm 0 SIMD '(:Vd :SIMD-FPIMM) QL-SIMD-IMM-S F-SIZEQ) 
   ("mvni" #x2f000400 #xbff89c00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S0W F-SIZEQ) 
   ("bic" #x2f001400 #xbff89c00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S0W F-SIZEQ) 
   ("mvni" #x2f008400 #xbff8dc00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S0H F-SIZEQ) 
   ("bic" #x2f009400 #xbff8dc00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S0H F-SIZEQ) 
   ("mvni" #x2f00c400 #xbff8ec00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM-SFT) QL-SIMD-IMM-S1W F-SIZEQ) 
   ("movi" #x2f00e400 #xfff8fc00 :asimdimm 0 SIMD '(:Sd :SIMD-IMM) QL-SIMD-IMM-D F-SIZEQ) 
   ("movi" #x6f00e400 #xfff8fc00 :asimdimm 0 SIMD '(:Vd :SIMD-IMM) QL-SIMD-IMM-V2D F-SIZEQ) 
   ("fmov" #x6f00f400 #xfff8fc00 :asimdimm 0 SIMD '(:Vd :SIMD-FPIMM) QL-SIMD-IMM-V2D F-SIZEQ) 
   ("dup" #xe000400 #xbfe0fc00 :asimdins 0 SIMD '(:Vd :En) QL-DUP-VX F-T) 
   ("dup" #xe000c00 #xbfe0fc00 :asimdins 0 SIMD '(:Vd :Rn) QL-DUP-VR F-T) 
   ("smov" #xe002c00 #xbfe0fc00 :asimdins 0 SIMD '(:Rd :En) QL-SMOV F-GPRSIZE-IN-Q) 
   ("umov" #xe003c00 #xbfe0fc00 :asimdins 0 SIMD '(:Rd :En) QL-UMOV (F-HAS-ALIAS  F-GPRSIZE-IN-Q)) 
   ("mov" #xe003c00 #xbfe0fc00 :asimdins 0 SIMD '(:Rd :En) QL-MOV (F-ALIAS  F-GPRSIZE-IN-Q)) 
   ("ins" #x4e001c00 #xffe0fc00 :asimdins 0 SIMD '(:Ed :Rn) QL-INS-XR F-HAS-ALIAS) 
   ("mov" #x4e001c00 #xffe0fc00 :asimdins 0 SIMD '(:Ed :Rn) QL-INS-XR F-ALIAS) 
   ("ins" #x6e000400 #xffe08400 :asimdins 0 SIMD '(:Ed :En) QL-S-2SAME F-HAS-ALIAS) 
   ("mov" #x6e000400 #xffe08400 :asimdins 0 SIMD '(:Ed :En) QL-S-2SAME F-ALIAS) 
   ("rev64" #xe200800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEBHS F-SIZEQ) 
   ("rev16" #xe201800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEB F-SIZEQ) 
   ("saddlp" #xe202800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2PAIRWISELONGBHS F-SIZEQ) 
   ("suqadd" #xe203800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAME F-SIZEQ) 
   ("cls" #xe204800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEBHS F-SIZEQ) 
   ("cnt" #xe205800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEB F-SIZEQ) 
   ("sadalp" #xe206800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2PAIRWISELONGBHS F-SIZEQ) 
   ("sqabs" #xe207800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAME F-SIZEQ) 
   ("cmgt" #xe208800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAME F-SIZEQ) 
   ("cmeq" #xe209800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAME F-SIZEQ) 
   ("cmlt" #xe20a800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAME F-SIZEQ) 
   ("abs" #xe20b800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAME F-SIZEQ) 
   ("xtn" #xe212800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRBHS F-SIZEQ) 
   ("xtn2" #x4e212800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRBHS2 F-SIZEQ) 
   ("sqxtn" #xe214800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRBHS F-SIZEQ) 
   ("sqxtn2" #x4e214800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRBHS2 F-SIZEQ) 
   ("fcvtn" #xe216800 #xffbffc00 :asimdmisc OP-FCVTN SIMD '(:Vd :Vn) QL-V2NARRHS F-MISC) 
   ("fcvtn2" #x4e216800 #xffbffc00 :asimdmisc OP-FCVTN2 SIMD '(:Vd :Vn) QL-V2NARRHS2 F-MISC) 
   ("fcvtl" #xe217800 #xffbffc00 :asimdmisc OP-FCVTL SIMD '(:Vd :Vn) QL-V2LONGHS F-MISC) 
   ("fcvtl2" #x4e217800 #xffbffc00 :asimdmisc OP-FCVTL2 SIMD '(:Vd :Vn) QL-V2LONGHS2 F-MISC) 
   ("frintn" #xe218800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("frintm" #xe219800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtns" #xe21a800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtms" #xe21b800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtas" #xe21c800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("scvtf" #xe21d800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcmgt" #xea0c800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAMESD F-SIZEQ) 
   ("fcmeq" #xea0d800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAMESD F-SIZEQ) 
   ("fcmlt" #xea0e800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAMESD F-SIZEQ) 
   ("fabs" #xea0f800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("frintp" #xea18800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("frintz" #xea19800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtps" #xea1a800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtzs" #xea1b800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("urecpe" #xea1c800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMES F-SIZEQ) 
   ("frecpe" #xea1d800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("rev32" #x2e200800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEBH F-SIZEQ) 
   ("uaddlp" #x2e202800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2PAIRWISELONGBHS F-SIZEQ) 
   ("usqadd" #x2e203800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAME F-SIZEQ) 
   ("clz" #x2e204800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEBHS F-SIZEQ) 
   ("uadalp" #x2e206800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2PAIRWISELONGBHS F-SIZEQ) 
   ("sqneg" #x2e207800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAME F-SIZEQ) 
   ("cmge" #x2e208800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAME F-SIZEQ) 
   ("cmle" #x2e209800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAME F-SIZEQ) 
   ("neg" #x2e20b800 #xbf3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAME F-SIZEQ) 
   ("sqxtun" #x2e212800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRBHS F-SIZEQ) 
   ("sqxtun2" #x6e212800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRBHS2 F-SIZEQ) 
   ("shll" #x2e213800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn :SHLL-IMM) QL-V2LONGBHS F-SIZEQ) 
   ("shll2" #x6e213800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn :SHLL-IMM) QL-V2LONGBHS2 F-SIZEQ) 
   ("uqxtn" #x2e214800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRBHS F-SIZEQ) 
   ("uqxtn2" #x6e214800 #xff3ffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRBHS2 F-SIZEQ) 
   ("fcvtxn" #x2e616800 #xfffffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRS 0) 
   ("fcvtxn2" #x6e616800 #xfffffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2NARRS2 0) 
   ("frinta" #x2e218800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("frintx" #x2e219800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtnu" #x2e21a800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtmu" #x2e21b800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtau" #x2e21c800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("ucvtf" #x2e21d800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("not" #x2e205800 #xbffffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEB (F-SIZEQ  F-HAS-ALIAS)) 
   ("mvn" #x2e205800 #xbffffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEB (F-SIZEQ  F-ALIAS)) 
   ("rbit" #x2e605800 #xbffffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMEB F-SIZEQ) 
   ("fcmge" #x2ea0c800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAMESD F-SIZEQ) 
   ("fcmle" #x2ea0d800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn :IMM0) QL-V2SAMESD F-SIZEQ) 
   ("fneg" #x2ea0f800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("frinti" #x2ea19800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtpu" #x2ea1a800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fcvtzu" #x2ea1b800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("ursqrte" #x2ea1c800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMES F-SIZEQ) 
   ("frsqrte" #x2ea1d800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("fsqrt" #x2ea1f800 #xbfbffc00 :asimdmisc 0 SIMD '(:Vd :Vn) QL-V2SAMESD F-SIZEQ) 
   ("uzp1" #xe001800 #xbf20fc00 :asimdperm 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("trn1" #xe002800 #xbf20fc00 :asimdperm 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("zip1" #xe003800 #xbf20fc00 :asimdperm 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("uzp2" #xe005800 #xbf20fc00 :asimdperm 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("trn2" #xe006800 #xbf20fc00 :asimdperm 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("zip2" #xe007800 #xbf20fc00 :asimdperm 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("shadd" #xe200400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("sqadd" #xe200c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("srhadd" #xe201400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("shsub" #xe202400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("sqsub" #xe202c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("cmgt" #xe203400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("cmge" #xe203c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("sshl" #xe204400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("sqshl" #xe204c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("srshl" #xe205400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("sqrshl" #xe205c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("smax" #xe206400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("smin" #xe206c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("sabd" #xe207400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("saba" #xe207c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("add" #xe208400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("cmtst" #xe208c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("mla" #xe209400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("mul" #xe209c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("smaxp" #xe20a400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("sminp" #xe20ac00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("sqdmulh" #xe20b400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEHS F-SIZEQ) 
   ("addp" #xe20bc00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("fmaxnm" #xe20c400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fmla" #xe20cc00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fadd" #xe20d400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fmulx" #xe20dc00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fcmeq" #xe20e400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fmax" #xe20f400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("frecps" #xe20fc00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("and" #xe201c00 #xbfe0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB F-SIZEQ) 
   ("bic" #xe601c00 #xbfe0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB F-SIZEQ) 
   ("fminnm" #xea0c400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fmls" #xea0cc00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fsub" #xea0d400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fmin" #xea0f400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("frsqrts" #xea0fc00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("orr" #xea01c00 #xbfe0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB (F-HAS-ALIAS  F-SIZEQ)) 
   ("mov" #xea01c00 #xbfe0fc00 :asimdsame OP-MOV-V SIMD '(:Vd :Vn) QL-V2SAMEB (F-ALIAS  F-CONV)) 
   ("orn" #xee01c00 #xbfe0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB F-SIZEQ) 
   ("uhadd" #x2e200400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("uqadd" #x2e200c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("urhadd" #x2e201400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("uhsub" #x2e202400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("uqsub" #x2e202c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("cmhi" #x2e203400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("cmhs" #x2e203c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("ushl" #x2e204400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("uqshl" #x2e204c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("urshl" #x2e205400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("uqrshl" #x2e205c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("umax" #x2e206400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("umin" #x2e206c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("uabd" #x2e207400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("uaba" #x2e207c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("sub" #x2e208400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("cmeq" #x2e208c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAME F-SIZEQ) 
   ("mls" #x2e209400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("pmul" #x2e209c00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB F-SIZEQ) 
   ("umaxp" #x2e20a400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("uminp" #x2e20ac00 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEBHS F-SIZEQ) 
   ("sqrdmulh" #x2e20b400 #xbf20fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEHS F-SIZEQ) 
   ("fmaxnmp" #x2e20c400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("faddp" #x2e20d400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fmul" #x2e20dc00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fcmge" #x2e20e400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("facge" #x2e20ec00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fmaxp" #x2e20f400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fdiv" #x2e20fc00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("eor" #x2e201c00 #xbfe0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB F-SIZEQ) 
   ("bsl" #x2e601c00 #xbfe0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB F-SIZEQ) 
   ("fminnmp" #x2ea0c400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fabd" #x2ea0d400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fcmgt" #x2ea0e400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("facgt" #x2ea0ec00 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("fminp" #x2ea0f400 #xbfa0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMESD F-SIZEQ) 
   ("bit" #x2ea01c00 #xbfe0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB F-SIZEQ) 
   ("bif" #x2ee01c00 #xbfe0fc00 :asimdsame 0 SIMD '(:Vd :Vn :Vm) QL-V3SAMEB F-SIZEQ) 
   ("sshr" #xf000400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("ssra" #xf001400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("srshr" #xf002400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("srsra" #xf003400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("shl" #xf005400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFT 0) 
   ("sqshl" #xf007400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFT 0) 
   ("shrn" #xf008400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN 0) 
   ("shrn2" #x4f008400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN2 0) 
   ("rshrn" #xf008c00 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN 0) 
   ("rshrn2" #x4f008c00 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN2 0) 
   ("sqshrn" #xf009400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN 0) 
   ("sqshrn2" #x4f009400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN2 0) 
   ("sqrshrn" #xf009c00 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN 0) 
   ("sqrshrn2" #x4f009c00 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN2 0) 
   ("sshll" #xf00a400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFTL F-HAS-ALIAS) 
   ("sxtl" #xf00a400 #xff87fc00 :asimdshf OP-SXTL SIMD '(:Vd :Vn) QL-V2LONGBHS (F-ALIAS  F-CONV)) 
   ("sshll2" #x4f00a400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFTL2 F-HAS-ALIAS) 
   ("sxtl2" #x4f00a400 #xff87fc00 :asimdshf OP-SXTL2 SIMD '(:Vd :Vn) QL-V2LONGBHS2 (F-ALIAS  F-CONV)) 
   ("scvtf" #xf00e400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT-SD 0) 
   ("fcvtzs" #xf00fc00 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT-SD 0) 
   ("ushr" #x2f000400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("usra" #x2f001400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("urshr" #x2f002400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("ursra" #x2f003400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("sri" #x2f004400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT 0) 
   ("sli" #x2f005400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFT 0) 
   ("sqshlu" #x2f006400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFT 0) 
   ("uqshl" #x2f007400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFT 0) 
   ("sqshrun" #x2f008400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN 0) 
   ("sqshrun2" #x6f008400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN2 0) 
   ("sqrshrun" #x2f008c00 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN 0) 
   ("sqrshrun2" #x6f008c00 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN2 0) 
   ("uqshrn" #x2f009400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN 0) 
   ("uqshrn2" #x6f009400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN2 0) 
   ("uqrshrn" #x2f009c00 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN 0) 
   ("uqrshrn2" #x6f009c00 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFTN2 0) 
   ("ushll" #x2f00a400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFTL F-HAS-ALIAS) 
   ("uxtl" #x2f00a400 #xff87fc00 :asimdshf OP-UXTL SIMD '(:Vd :Vn) QL-V2LONGBHS (F-ALIAS  F-CONV)) 
   ("ushll2" #x6f00a400 #xff80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSL) QL-VSHIFTL2 F-HAS-ALIAS) 
   ("uxtl2" #x6f00a400 #xff87fc00 :asimdshf OP-UXTL2 SIMD '(:Vd :Vn) QL-V2LONGBHS2 (F-ALIAS  F-CONV)) 
   ("ucvtf" #x2f00e400 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT-SD 0) 
   ("fcvtzu" #x2f00fc00 #xbf80fc00 :asimdshf 0 SIMD '(:Vd :Vn :IMM-VLSR) QL-VSHIFT-SD 0) 
   ("tbl" #xe000000 #xbfe09c00 :asimdtbl 0 SIMD '(:Vd :LVn :Vm) QL-TABLE F-SIZEQ) 
   ("tbx" #xe001000 #xbfe09c00 :asimdtbl 0 SIMD '(:Vd :LVn :Vm) QL-TABLE F-SIZEQ) 
   ("sqdmlal" #x5e209000 #xff20fc00 :asisddiff 0 SIMD '(:Sd :Sn :Sm) QL-SISDL-HS F-SSIZE) 
   ("sqdmlsl" #x5e20b000 #xff20fc00 :asisddiff 0 SIMD '(:Sd :Sn :Sm) QL-SISDL-HS F-SSIZE) 
   ("sqdmull" #x5e20d000 #xff20fc00 :asisddiff 0 SIMD '(:Sd :Sn :Sm) QL-SISDL-HS F-SSIZE) 
   ("sqdmlal" #x5f003000 #xff00f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-SISDL-HS F-SSIZE) 
   ("sqdmlsl" #x5f007000 #xff00f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-SISDL-HS F-SSIZE) 
   ("sqdmull" #x5f00b000 #xff00f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-SISDL-HS F-SSIZE) 
   ("sqdmulh" #x5f00c000 #xff00f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-SISD-HS F-SSIZE) 
   ("sqrdmulh" #x5f00d000 #xff00f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-SISD-HS F-SSIZE) 
   ("fmla" #x5f801000 #xff80f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-FP3 F-SSIZE) 
   ("fmls" #x5f805000 #xff80f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-FP3 F-SSIZE) 
   ("fmul" #x5f809000 #xff80f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-FP3 F-SSIZE) 
   ("fmulx" #x7f809000 #xff80f400 :asisdelem 0 SIMD '(:Sd :Sn :Em) QL-FP3 F-SSIZE) 
   ("st4" #xc000000 #xbfff0000 :asisdlse 0 SIMD '(:LVt :SIMD-ADDR-SIMPLE) QL-SIMD-LDST (F-SIZEQ  F-OD)(4)) 
   ("st1" #xc000000 #xbfff0000 :asisdlse 0 SIMD '(:LVt :SIMD-ADDR-SIMPLE) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(1)) 
   ("st2" #xc000000 #xbfff0000 :asisdlse 0 SIMD '(:LVt :SIMD-ADDR-SIMPLE) QL-SIMD-LDST (F-SIZEQ  F-OD)(2)) 
   ("st3" #xc000000 #xbfff0000 :asisdlse 0 SIMD '(:LVt :SIMD-ADDR-SIMPLE) QL-SIMD-LDST (F-SIZEQ  F-OD)(3)) 
   ("ld4" #xc400000 #xbfff0000 :asisdlse 0 SIMD '(:LVt :SIMD-ADDR-SIMPLE) QL-SIMD-LDST (F-SIZEQ  F-OD)(4)) 
   ("ld1" #xc400000 #xbfff0000 :asisdlse 0 SIMD '(:LVt :SIMD-ADDR-SIMPLE) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(1)) 
   ("ld2" #xc400000 #xbfff0000 :asisdlse 0 SIMD '(:LVt :SIMD-ADDR-SIMPLE) QL-SIMD-LDST (F-SIZEQ  F-OD)(2)) 
   ("ld3" #xc400000 #xbfff0000 :asisdlse 0 SIMD '(:LVt :SIMD-ADDR-SIMPLE) QL-SIMD-LDST (F-SIZEQ  F-OD)(3)) 
   ("st4" #xc800000 #xbfe00000 :asisdlsep 0 SIMD '(:LVt :SIMD-ADDR-POST) QL-SIMD-LDST (F-SIZEQ  F-OD)(4)) 
   ("st1" #xc800000 #xbfe00000 :asisdlsep 0 SIMD '(:LVt :SIMD-ADDR-POST) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(1)) 
   ("st2" #xc800000 #xbfe00000 :asisdlsep 0 SIMD '(:LVt :SIMD-ADDR-POST) QL-SIMD-LDST (F-SIZEQ  F-OD)(2)) 
   ("st3" #xc800000 #xbfe00000 :asisdlsep 0 SIMD '(:LVt :SIMD-ADDR-POST) QL-SIMD-LDST (F-SIZEQ  F-OD)(3)) 
   ("ld4" #xcc00000 #xbfe00000 :asisdlsep 0 SIMD '(:LVt :SIMD-ADDR-POST) QL-SIMD-LDST (F-SIZEQ  F-OD)(4)) 
   ("ld1" #xcc00000 #xbfe00000 :asisdlsep 0 SIMD '(:LVt :SIMD-ADDR-POST) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(1)) 
   ("ld2" #xcc00000 #xbfe00000 :asisdlsep 0 SIMD '(:LVt :SIMD-ADDR-POST) QL-SIMD-LDST (F-SIZEQ  F-OD)(2)) 
   ("ld3" #xcc00000 #xbfe00000 :asisdlsep 0 SIMD '(:LVt :SIMD-ADDR-POST) QL-SIMD-LDST (F-SIZEQ  F-OD)(3)) 
   ("st1" #xd000000 #xbfff2000 :asisdlso 0 SIMD '(:LEt :SIMD-ADDR-SIMPLE) QL-SIMD-LDSTONE F-OD(1)) 
   ("st3" #xd002000 #xbfff2000 :asisdlso 0 SIMD '(:LEt :SIMD-ADDR-SIMPLE) QL-SIMD-LDSTONE F-OD(3)) 
   ("st2" #xd200000 #xbfff2000 :asisdlso 0 SIMD '(:LEt :SIMD-ADDR-SIMPLE) QL-SIMD-LDSTONE F-OD(2)) 
   ("st4" #xd202000 #xbfff2000 :asisdlso 0 SIMD '(:LEt :SIMD-ADDR-SIMPLE) QL-SIMD-LDSTONE F-OD(4)) 
   ("ld1" #xd400000 #xbfff2000 :asisdlso 0 SIMD '(:LEt :SIMD-ADDR-SIMPLE) QL-SIMD-LDSTONE F-OD(1)) 
   ("ld3" #xd402000 #xbfff2000 :asisdlso 0 SIMD '(:LEt :SIMD-ADDR-SIMPLE) QL-SIMD-LDSTONE F-OD(3)) 
   ("ld1r" #xd40c000 #xbfffe000 :asisdlso 0 SIMD '(:LVt-AL :SIMD-ADDR-SIMPLE) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(1)) 
   ("ld3r" #xd40e000 #xbfffe000 :asisdlso 0 SIMD '(:LVt-AL :SIMD-ADDR-SIMPLE) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(3)) 
   ("ld2" #xd600000 #xbfff2000 :asisdlso 0 SIMD '(:LEt :SIMD-ADDR-SIMPLE) QL-SIMD-LDSTONE F-OD(2)) 
   ("ld4" #xd602000 #xbfff2000 :asisdlso 0 SIMD '(:LEt :SIMD-ADDR-SIMPLE) QL-SIMD-LDSTONE F-OD(4)) 
   ("ld2r" #xd60c000 #xbfffe000 :asisdlso 0 SIMD '(:LVt-AL :SIMD-ADDR-SIMPLE) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(2)) 
   ("ld4r" #xd60e000 #xbfffe000 :asisdlso 0 SIMD '(:LVt-AL :SIMD-ADDR-SIMPLE) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(4)) 
   ("st1" #xd800000 #xbfe02000 :asisdlsop 0 SIMD '(:LEt :SIMD-ADDR-POST) QL-SIMD-LDSTONE F-OD(1)) 
   ("st3" #xd802000 #xbfe02000 :asisdlsop 0 SIMD '(:LEt :SIMD-ADDR-POST) QL-SIMD-LDSTONE F-OD(3)) 
   ("st2" #xda00000 #xbfe02000 :asisdlsop 0 SIMD '(:LEt :SIMD-ADDR-POST) QL-SIMD-LDSTONE F-OD(2)) 
   ("st4" #xda02000 #xbfe02000 :asisdlsop 0 SIMD '(:LEt :SIMD-ADDR-POST) QL-SIMD-LDSTONE F-OD(4)) 
   ("ld1" #xdc00000 #xbfe02000 :asisdlsop 0 SIMD '(:LEt :SIMD-ADDR-POST) QL-SIMD-LDSTONE F-OD(1)) 
   ("ld3" #xdc02000 #xbfe02000 :asisdlsop 0 SIMD '(:LEt :SIMD-ADDR-POST) QL-SIMD-LDSTONE F-OD(3)) 
   ("ld1r" #xdc0c000 #xbfe0e000 :asisdlsop 0 SIMD '(:LVt-AL :SIMD-ADDR-POST) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(1)) 
   ("ld3r" #xdc0e000 #xbfe0e000 :asisdlsop 0 SIMD '(:LVt-AL :SIMD-ADDR-POST) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(3)) 
   ("ld2" #xde00000 #xbfe02000 :asisdlsop 0 SIMD '(:LEt :SIMD-ADDR-POST) QL-SIMD-LDSTONE F-OD(2)) 
   ("ld4" #xde02000 #xbfe02000 :asisdlsop 0 SIMD '(:LEt :SIMD-ADDR-POST) QL-SIMD-LDSTONE F-OD(4)) 
   ("ld2r" #xde0c000 #xbfe0e000 :asisdlsop 0 SIMD '(:LVt-AL :SIMD-ADDR-POST) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(2)) 
   ("ld4r" #xde0e000 #xbfe0e000 :asisdlsop 0 SIMD '(:LVt-AL :SIMD-ADDR-POST) QL-SIMD-LDST-ANY (F-SIZEQ  F-OD)(4)) 
   ("suqadd" #x5e203800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAME F-SSIZE) 
   ("sqabs" #x5e207800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAME F-SSIZE) 
   ("cmgt" #x5e208800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-CMP-0 F-SSIZE) 
   ("cmeq" #x5e209800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-CMP-0 F-SSIZE) 
   ("cmlt" #x5e20a800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-CMP-0 F-SSIZE) 
   ("abs" #x5e20b800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-2SAMED F-SSIZE) 
   ("sqxtn" #x5e214800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-SISD-NARROW F-SSIZE) 
   ("fcvtns" #x5e21a800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("fcvtms" #x5e21b800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("fcvtas" #x5e21c800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("scvtf" #x5e21d800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("fcmgt" #x5ea0c800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-FCMP-0 F-SSIZE) 
   ("fcmeq" #x5ea0d800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-FCMP-0 F-SSIZE) 
   ("fcmlt" #x5ea0e800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-FCMP-0 F-SSIZE) 
   ("fcvtps" #x5ea1a800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("fcvtzs" #x5ea1b800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("frecpe" #x5ea1d800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("frecpx" #x5ea1f800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("usqadd" #x7e203800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAME F-SSIZE) 
   ("sqneg" #x7e207800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAME F-SSIZE) 
   ("cmge" #x7e208800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-CMP-0 F-SSIZE) 
   ("cmle" #x7e209800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-CMP-0 F-SSIZE) 
   ("neg" #x7e20b800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-2SAMED F-SSIZE) 
   ("sqxtun" #x7e212800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-SISD-NARROW F-SSIZE) 
   ("uqxtn" #x7e214800 #xff3ffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-SISD-NARROW F-SSIZE) 
   ("fcvtxn" #x7e216800 #xffbffc00 :asisdmisc OP-FCVTXN-S SIMD '(:Sd :Sn) QL-SISD-NARROW-S F-MISC) 
   ("fcvtnu" #x7e21a800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("fcvtmu" #x7e21b800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("fcvtau" #x7e21c800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("ucvtf" #x7e21d800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("fcmge" #x7ea0c800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-FCMP-0 F-SSIZE) 
   ("fcmle" #x7ea0d800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn :IMM0) QL-SISD-FCMP-0 F-SSIZE) 
   ("fcvtpu" #x7ea1a800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("fcvtzu" #x7ea1b800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("frsqrte" #x7ea1d800 #xffbffc00 :asisdmisc 0 SIMD '(:Sd :Sn) QL-S-2SAMESD F-SSIZE) 
   ("dup" #x5e000400 #xffe0fc00 :asisdone 0 SIMD '(:Sd :En) QL-S-2SAME F-HAS-ALIAS) 
   ("mov" #x5e000400 #xffe0fc00 :asisdone 0 SIMD '(:Sd :En) QL-S-2SAME F-ALIAS) 
   ("addp" #x5e31b800 #xff3ffc00 :asisdpair 0 SIMD '(:Sd :Vn) QL-SISD-PAIR-D F-SIZEQ) 
   ("fmaxnmp" #x7e30c800 #xffbffc00 :asisdpair 0 SIMD '(:Sd :Vn) QL-SISD-PAIR F-SIZEQ) 
   ("faddp" #x7e30d800 #xffbffc00 :asisdpair 0 SIMD '(:Sd :Vn) QL-SISD-PAIR F-SIZEQ) 
   ("fmaxp" #x7e30f800 #xffbffc00 :asisdpair 0 SIMD '(:Sd :Vn) QL-SISD-PAIR F-SIZEQ) 
   ("fminnmp" #x7eb0c800 #xffbffc00 :asisdpair 0 SIMD '(:Sd :Vn) QL-SISD-PAIR F-SIZEQ) 
   ("fminp" #x7eb0f800 #xffbffc00 :asisdpair 0 SIMD '(:Sd :Vn) QL-SISD-PAIR F-SIZEQ) 
   ("sqadd" #x5e200c00 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAME F-SSIZE) 
   ("sqsub" #x5e202c00 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAME F-SSIZE) 
   ("sqshl" #x5e204c00 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAME F-SSIZE) 
   ("sqrshl" #x5e205c00 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAME F-SSIZE) 
   ("sqdmulh" #x5e20b400 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-SISD-HS F-SSIZE) 
   ("fmulx" #x5e20dc00 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("fcmeq" #x5e20e400 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("frecps" #x5e20fc00 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("frsqrts" #x5ea0fc00 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("cmgt" #x5ee03400 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("cmge" #x5ee03c00 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("sshl" #x5ee04400 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("srshl" #x5ee05400 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("add" #x5ee08400 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("cmtst" #x5ee08c00 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("uqadd" #x7e200c00 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAME F-SSIZE) 
   ("uqsub" #x7e202c00 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAME F-SSIZE) 
   ("uqshl" #x7e204c00 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAME F-SSIZE) 
   ("uqrshl" #x7e205c00 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAME F-SSIZE) 
   ("sqrdmulh" #x7e20b400 #xff20fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-SISD-HS F-SSIZE) 
   ("fcmge" #x7e20e400 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("facge" #x7e20ec00 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("fabd" #x7ea0d400 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("fcmgt" #x7ea0e400 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("facgt" #x7ea0ec00 #xffa0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-FP3 F-SSIZE) 
   ("cmhi" #x7ee03400 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("cmhs" #x7ee03c00 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("ushl" #x7ee04400 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("urshl" #x7ee05400 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("sub" #x7ee08400 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("cmeq" #x7ee08c00 #xffe0fc00 :asisdsame 0 SIMD '(:Sd :Sn :Sm) QL-S-3SAMED F-SSIZE) 
   ("sshr" #x5f000400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("ssra" #x5f001400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("srshr" #x5f002400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("srsra" #x5f003400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("shl" #x5f005400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSL) QL-SSHIFT-D 0) 
   ("sqshl" #x5f007400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSL) QL-SSHIFT 0) 
   ("sqshrn" #x5f009400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFTN 0) 
   ("sqrshrn" #x5f009c00 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFTN 0) 
   ("scvtf" #x5f00e400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-SD 0) 
   ("fcvtzs" #x5f00fc00 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-SD 0) 
   ("ushr" #x7f000400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("usra" #x7f001400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("urshr" #x7f002400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("ursra" #x7f003400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("sri" #x7f004400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-D 0) 
   ("sli" #x7f005400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSL) QL-SSHIFT-D 0) 
   ("sqshlu" #x7f006400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSL) QL-SSHIFT 0) 
   ("uqshl" #x7f007400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSL) QL-SSHIFT 0) 
   ("sqshrun" #x7f008400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFTN 0) 
   ("sqrshrun" #x7f008c00 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFTN 0) 
   ("uqshrn" #x7f009400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFTN 0) 
   ("uqrshrn" #x7f009c00 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFTN 0) 
   ("ucvtf" #x7f00e400 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-SD 0) 
   ("fcvtzu" #x7f00fc00 #xff80fc00 :asisdshf 0 SIMD '(:Sd :Sn :IMM-VLSR) QL-SSHIFT-SD 0) 
   ("sbfm" #x13000000 #x7f800000 :bitfield 0 :CORE '(:Rd :Rn :IMMR :IMMS) QL-BF ((F-HAS-ALIAS  F-SF)  F-N)) 
   ("sbfiz" #x13000000 #x7f800000 :bitfield OP-SBFIZ :CORE '(:Rd :Rn :IMM :WIDTH) QL-BF2 ((F-ALIAS  F-P1)  F-CONV)) 
   ("sbfx" #x13000000 #x7f800000 :bitfield OP-SBFX :CORE '(:Rd :Rn :IMM :WIDTH) QL-BF2 ((F-ALIAS  F-P1)  F-CONV)) 
   ("sxtb" #x13001c00 #x7fbffc00 :bitfield 0 :CORE '(:Rd :Rn) QL-EXT (((F-ALIAS  F-P3)  F-SF)  F-N)) 
   ("sxth" #x13003c00 #x7fbffc00 :bitfield 0 :CORE '(:Rd :Rn) QL-EXT (((F-ALIAS  F-P3)  F-SF)  F-N)) 
   ("sxtw" #x93407c00 #xfffffc00 :bitfield 0 :CORE '(:Rd :Rn) QL-EXT-W (F-ALIAS  F-P3)) 
   ("asr" #x13000000 #x7f800000 :bitfield OP-ASR-IMM :CORE '(:Rd :Rn :IMM) QL-SHIFT ((F-ALIAS  F-P2)  F-CONV)) 
   ("bfm" #x33000000 #x7f800000 :bitfield 0 :CORE '(:Rd :Rn :IMMR :IMMS) QL-BF ((F-HAS-ALIAS  F-SF)  F-N)) 
   ("bfi" #x33000000 #x7f800000 :bitfield OP-BFI :CORE '(:Rd :Rn :IMM :WIDTH) QL-BF2 ((F-ALIAS  F-P1)  F-CONV)) 
   ("bfxil" #x33000000 #x7f800000 :bitfield OP-BFXIL :CORE '(:Rd :Rn :IMM :WIDTH) QL-BF2 ((F-ALIAS  F-P1)  F-CONV)) 
   ("ubfm" #x53000000 #x7f800000 :bitfield 0 :CORE '(:Rd :Rn :IMMR :IMMS) QL-BF ((F-HAS-ALIAS  F-SF)  F-N)) 
   ("ubfiz" #x53000000 #x7f800000 :bitfield OP-UBFIZ :CORE '(:Rd :Rn :IMM :WIDTH) QL-BF2 ((F-ALIAS  F-P1)  F-CONV)) 
   ("ubfx" #x53000000 #x7f800000 :bitfield OP-UBFX :CORE '(:Rd :Rn :IMM :WIDTH) QL-BF2 ((F-ALIAS  F-P1)  F-CONV)) 
   ("uxtb" #x53001c00 #xfffffc00 :bitfield OP-UXTB :CORE '(:Rd :Rn) QL-I2SAMEW (F-ALIAS  F-P3)) 
   ("uxth" #x53003c00 #xfffffc00 :bitfield OP-UXTH :CORE '(:Rd :Rn) QL-I2SAMEW (F-ALIAS  F-P3)) 
   ("lsl" #x53000000 #x7f800000 :bitfield OP-LSL-IMM :CORE '(:Rd :Rn :IMM) QL-SHIFT ((F-ALIAS  F-P2)  F-CONV)) 
   ("lsr" #x53000000 #x7f800000 :bitfield OP-LSR-IMM :CORE '(:Rd :Rn :IMM) QL-SHIFT ((F-ALIAS  F-P2)  F-CONV)) 
   ("b" #x14000000 #xfc000000 :branch-imm OP-B :CORE '(:ADDR-PCREL26) QL-PCREL-26 0) 
   ("bl" #x94000000 #xfc000000 :branch-imm OP-BL :CORE '(:ADDR-PCREL26) QL-PCREL-26 0) 
   ("br" #xd61f0000 #xfffffc1f :branch-reg 0 :CORE '(:Rn) QL-I1X 0) 
   ("blr" #xd63f0000 #xfffffc1f :branch-reg 0 :CORE '(:Rn) QL-I1X 0) 
   ("ret" #xd65f0000 #xfffffc1f :branch-reg 0 :CORE '(:Rn) QL-I1X (F-OPD0-OPT  F-DEFAULT) (30)) 
   ("eret" #xd69f03e0 #xffffffff :branch-reg 0 :CORE '() () 0) 
   ("drps" #xd6bf03e0 #xffffffff :branch-reg 0 :CORE '() () 0) 
   ("cbz" #x34000000 #x7f000000 :compbranch 0 :CORE '(:Rt :ADDR-PCREL19) QL-R-PCREL F-SF) 
   ("cbnz" #x35000000 #x7f000000 :compbranch 0 :CORE '(:Rt :ADDR-PCREL19) QL-R-PCREL F-SF) 
   ("b.c" #x54000000 #xff000010 :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL F-COND) 
   ("ccmn" #x3a400800 #x7fe00c10 :condcmp-imm 0 :CORE '(:Rn :CCMP-IMM :NZCV :COND) QL-CCMP-IMM F-SF) 
   ("ccmp" #x7a400800 #x7fe00c10 :condcmp-imm 0 :CORE '(:Rn :CCMP-IMM :NZCV :COND) QL-CCMP-IMM F-SF) 
   ("ccmn" #x3a400000 #x7fe00c10 :condcmp-reg 0 :CORE '(:Rn :Rm :NZCV :COND) QL-CCMP F-SF) 
   ("ccmp" #x7a400000 #x7fe00c10 :condcmp-reg 0 :CORE '(:Rn :Rm :NZCV :COND) QL-CCMP F-SF) 
   ("csel" #x1a800000 #x7fe00c00 :condsel 0 :CORE '(:Rd :Rn :Rm :COND) QL-CSEL F-SF) 
   ("csinc" #x1a800400 #x7fe00c00 :condsel 0 :CORE '(:Rd :Rn :Rm :COND) QL-CSEL (F-HAS-ALIAS  F-SF)) 
   ("cinc" #x1a800400 #x7fe00c00 :condsel OP-CINC :CORE '(:Rd :Rn :COND) QL-CSEL ((F-ALIAS  F-SF)  F-CONV)) 
   ("cset" #x1a9f07e0 #x7fff0fe0 :condsel OP-CSET :CORE '(:Rd :COND) QL-DST-R (((F-ALIAS  F-P1)  F-SF)  F-CONV)) 
   ("csinv" #x5a800000 #x7fe00c00 :condsel 0 :CORE '(:Rd :Rn :Rm :COND) QL-CSEL (F-HAS-ALIAS  F-SF)) 
   ("cinv" #x5a800000 #x7fe00c00 :condsel OP-CINV :CORE '(:Rd :Rn :COND) QL-CSEL ((F-ALIAS  F-SF)  F-CONV)) 
   ("csetm" #x5a9f03e0 #x7fff0fe0 :condsel OP-CSETM :CORE '(:Rd :COND) QL-DST-R (((F-ALIAS  F-P1)  F-SF)  F-CONV)) 
   ("csneg" #x5a800400 #x7fe00c00 :condsel 0 :CORE '(:Rd :Rn :Rm :COND) QL-CSEL (F-HAS-ALIAS  F-SF)) 
   ("cneg" #x5a800400 #x7fe00c00 :condsel OP-CNEG :CORE '(:Rd :Rn :COND) QL-CSEL ((F-ALIAS  F-SF)  F-CONV)) 
   ("aese" #x4e284800 #xfffffc00 :cryptoaes 0 :CRYPTO '(:Vd :Vn) QL-V2SAME16B 0) 
   ("aesd" #x4e285800 #xfffffc00 :cryptoaes 0 :CRYPTO '(:Vd :Vn) QL-V2SAME16B 0) 
   ("aesmc" #x4e286800 #xfffffc00 :cryptoaes 0 :CRYPTO '(:Vd :Vn) QL-V2SAME16B 0) 
   ("aesimc" #x4e287800 #xfffffc00 :cryptoaes 0 :CRYPTO '(:Vd :Vn) QL-V2SAME16B 0) 
   ("sha1h" #x5e280800 #xfffffc00 :cryptosha2 0 :CRYPTO '(:Fd :Fn) QL-2SAMES 0) 
   ("sha1su1" #x5e281800 #xfffffc00 :cryptosha2 0 :CRYPTO '(:Vd :Vn) QL-V2SAME4S 0) 
   ("sha256su0" #x5e282800 #xfffffc00 :cryptosha2 0 :CRYPTO '(:Vd :Vn) QL-V2SAME4S 0) 
   ("sha1c" #x5e000000 #xffe0fc00 :cryptosha3 0 :CRYPTO '(:Fd :Fn :Vm) QL-SHAUPT 0) 
   ("sha1p" #x5e001000 #xffe0fc00 :cryptosha3 0 :CRYPTO '(:Fd :Fn :Vm) QL-SHAUPT 0)
 
   ("sha1m" #x5e002000 #xffe0fc00 :cryptosha3 0 :CRYPTO '(:Fd :Fn :Vm) QL-SHAUPT 0) 
   ("sha1su0" #x5e003000 #xffe0fc00 :cryptosha3 0 :CRYPTO '(:Vd :Vn :Vm) QL-V3SAME4S 0) 
   ("sha256h" #x5e004000 #xffe0fc00 :cryptosha3 0 :CRYPTO '(:Fd :Fn :Vm) QL-SHA256UPT 0) 
   ("sha256h2" #x5e005000 #xffe0fc00 :cryptosha3 0 :CRYPTO '(:Fd :Fn :Vm) QL-SHA256UPT 0) 
   ("sha256su1" #x5e006000 #xffe0fc00 :cryptosha3 0 :CRYPTO '(:Vd :Vn :Vm) QL-V3SAME4S 0) 
   ("rbit" #x5ac00000 #x7ffffc00 :dp-1src 0 :CORE '(:Rd :Rn) QL-I2SAME F-SF) 
   ("rev16" #x5ac00400 #x7ffffc00 :dp-1src 0 :CORE '(:Rd :Rn) QL-I2SAME F-SF) 
   ("rev" #x5ac00800 #xfffffc00 :dp-1src 0 :CORE '(:Rd :Rn) QL-I2SAMEW 0) 
   ("rev" #xdac00c00 #x7ffffc00 :dp-1src 0 :CORE '(:Rd :Rn) QL-I2SAMEX 0) 
   ("clz" #x5ac01000 #x7ffffc00 :dp-1src 0 :CORE '(:Rd :Rn) QL-I2SAME F-SF) 
   ("cls" #x5ac01400 #x7ffffc00 :dp-1src 0 :CORE '(:Rd :Rn) QL-I2SAME F-SF) 
   ("rev32" #xdac00800 #xfffffc00 :dp-1src 0 :CORE '(:Rd :Rn) QL-I2SAMEX 0) 

   ("udiv" #x1ac00800 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) 0)
   ("udiv" #x9ac00800 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) 0) 
   ("sdiv" #x1ac00c00 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) 0) 
   ("sdiv" #x9ac00c00 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) 0) 

   ("lslv" #x1ac02000 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-HAS-ALIAS)
   ("lslv" #x9ac02000 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-HAS-ALIAS) 
   ("lsl" #x1ac02000 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-ALIAS)
   ("lsl" #x9ac02000 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-ALIAS) 
   ("lsrv" #x1ac02400 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-HAS-ALIAS) 
   ("lsrv" #x9ac02400 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-HAS-ALIAS))
  ("lsr" #x1ac02400 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-ALIAS) 
  ("lsr" #x9ac02400 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-ALIAS)
  ("asrv" #x1ac02800 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-HAS-ALIAS)
  ("asrv" #x9ac02800 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-HAS-ALIAS)
  ("asr" #x1ac02800 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-ALIAS)
  ("asr" #x9ac02800 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-ALIAS)
  ("rorv" #x1ac02c00 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-HAS-ALIAS)
  ("rorv" #x9ac02c00 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-HAS-ALIAS)
  ("ror" #x1ac02c00 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-ALIAS)
  ("ror" #x9ac02c00 #xffe0fc00 :dp-2src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-ALIAS)

  ("madd" #x1b000000 #x7fe08000 :dp-3src 0 :CORE '(:Rd :Rn :Rm :Ra) QL-I4SAMER (F-HAS-ALIAS  F-SF)) 
  ("mul" #x1b007c00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-ALIAS) 
  ("mul" #x9b007c00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-ALIAS) 
  ("msub" #x1b008000 #x7fe08000 :dp-3src 0 :CORE '(:Rd :Rn :Rm :Ra) QL-I4SAMER (F-HAS-ALIAS  F-SF)) 
  ("mneg" #x1b00fc00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:w :w :w) F-ALIAS) 
  ("mneg" #x9b00fc00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:x :x :x) F-ALIAS)
  ("smaddl" #x9b200000 #xffe08000 :dp-3src 0 :CORE '(:Rd :Rn :Rm :Ra) '(:x :w :w :x) F-HAS-ALIAS) 
  ("smull" #x9b207c00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:X :W :W) F-ALIAS) 
  ("smsubl" #x9b208000 #xffe08000 :dp-3src 0 :CORE '(:Rd :Rn :Rm :Ra) '(:X :W :W :X) F-HAS-ALIAS) 
  ("smnegl" #x9b20fc00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:X :W :W) F-ALIAS) 
  ("smulh" #x9b407c00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:X :X :X) 0) 
  ("umaddl" #x9ba00000 #xffe08000 :dp-3src 0 :CORE '(:Rd :Rn :Rm :Ra) '(:X :W :W :X) F-HAS-ALIAS) 
  ("umull" #x9ba07c00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:X :W :W) F-ALIAS) 
  ("umsubl" #x9ba08000 #xffe08000 :dp-3src 0 :CORE '(:Rd :Rn :Rm :Ra) '(:X :W :W :X) F-HAS-ALIAS) 
  ("umnegl" #x9ba0fc00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:X :W :W) F-ALIAS) 
  ("umulh" #x9bc07c00 #xffe0fc00 :dp-3src 0 :CORE '(:Rd :Rn :Rm) '(:X :X :X) 0) 
  ("svc" #xd4000001 #xffe0001f :exception 0 :CORE '(:EXCEPTION) () 0) 
  ("hvc" #xd4000002 #xffe0001f :exception 0 :CORE '(:EXCEPTION) () 0) 
  ("smc" #xd4000003 #xffe0001f :exception 0 :CORE '(:EXCEPTION) () 0) 
  ("brk" #xd4200000 #xffe0001f :exception 0 :CORE '(:EXCEPTION) () 0) 
  ("hlt" #xd4400000 #xffe0001f :exception 0 :CORE '(:EXCEPTION) () 0) 
  ("dcps1" #xd4a00001 #xffe0001f :exception 0 :CORE '(:EXCEPTION) () (F-OPD0-OPT  F-DEFAULT) (0)) 
  ("dcps2" #xd4a00002 #xffe0001f :exception 0 :CORE '(:EXCEPTION) () (F-OPD0-OPT  F-DEFAULT) (0)) 
  ("dcps3" #xd4a00003 #xffe0001f :exception 0 :CORE '(:EXCEPTION) () (F-OPD0-OPT  F-DEFAULT) (0)) 
  ("extr" #x13800000 #x7fa00000 :extract 0 :CORE '(:Rd :Rn :Rm :IMMS) QL-EXTR ((F-HAS-ALIAS  F-SF)  F-N)) 
  ("ror" #x13800000 #x7fa00000 :extract OP-ROR-IMM :CORE '(:Rd :Rm :IMMS) QL-SHIFT (F-ALIAS  F-CONV)) 
  ("scvtf" #x1e020000 #x7f3f0000 :float2fix 0 FP '(:Fd :Rn :FBITS) QL-FIX2FP (F-FPTYPE  F-SF)) 
  ("ucvtf" #x1e030000 #x7f3f0000 :float2fix 0 FP '(:Fd :Rn :FBITS) QL-FIX2FP (F-FPTYPE  F-SF)) 
  ("fcvtzs" #x1e180000 #x7f3f0000 :float2fix 0 FP '(:Rd :Fn :FBITS) QL-FP2FIX (F-FPTYPE  F-SF)) 
  ("fcvtzu" #x1e190000 #x7f3f0000 :float2fix 0 FP '(:Rd :Fn :FBITS) QL-FP2FIX (F-FPTYPE  F-SF)) 
  ("fcvtns" #x1e200000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fcvtnu" #x1e210000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("scvtf" #x1e220000 #x7f3ffc00 :float2int 0 FP '(:Fd :Rn) QL-INT2FP (F-FPTYPE  F-SF)) 
  ("ucvtf" #x1e230000 #x7f3ffc00 :float2int 0 FP '(:Fd :Rn) QL-INT2FP (F-FPTYPE  F-SF)) 
  ("fcvtas" #x1e240000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fcvtau" #x1e250000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fmov" #x1e260000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fmov" #x1e270000 #x7f3ffc00 :float2int 0 FP '(:Fd :Rn) QL-INT2FP (F-FPTYPE  F-SF)) 
  ("fcvtps" #x1e280000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fcvtpu" #x1e290000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fcvtms" #x1e300000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fcvtmu" #x1e310000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fcvtzs" #x1e380000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fcvtzu" #x1e390000 #x7f3ffc00 :float2int 0 FP '(:Rd :Fn) QL-FP2INT (F-FPTYPE  F-SF)) 
  ("fmov" #x9eae0000 #xfffffc00 :float2int 0 FP '(:Rd :VnD1) QL-XVD1 0) 
  ("fmov" #x9eaf0000 #xfffffc00 :float2int 0 FP '(:VdD1 :Rn) QL-VD1X 0) 
  ("fccmp" #x1e200400 #xff200c10 :floatccmp 0 FP '(:Fn :Fm :NZCV :COND) QL-FCCMP F-FPTYPE) 
  ("fccmpe" #x1e200410 #xff200c10 :floatccmp 0 FP '(:Fn :Fm :NZCV :COND) QL-FCCMP F-FPTYPE) 
  ("fcmp" #x1e202000 #xff20fc1f :floatcmp 0 FP '(:Fn :Fm) QL-FP2 F-FPTYPE) 
  ("fcmpe" #x1e202010 #xff20fc1f :floatcmp 0 FP '(:Fn :Fm) QL-FP2 F-FPTYPE) 
  ("fcmp" #x1e202008 #xff20fc1f :floatcmp 0 FP '(:Fn :FPIMM0) QL-DST-SD F-FPTYPE) 
  ("fcmpe" #x1e202018 #xff20fc1f :floatcmp 0 FP '(:Fn :FPIMM0) QL-DST-SD F-FPTYPE) 
  ("fmov" #x1e204000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("fabs" #x1e20c000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("fneg" #x1e214000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("fsqrt" #x1e21c000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("fcvt" #x1e224000 #xff3e7c00 :floatdp1 OP-FCVT FP '(:Fd :Fn) QL-FCVT (F-FPTYPE  F-MISC)) 
  ("frintn" #x1e244000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("frintp" #x1e24c000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("frintm" #x1e254000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("frintz" #x1e25c000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("frinta" #x1e264000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("frintx" #x1e274000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("frinti" #x1e27c000 #xff3ffc00 :floatdp1 0 FP '(:Fd :Fn) QL-FP2 F-FPTYPE) 
  ("fmul" #x1e200800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fdiv" #x1e201800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fadd" #x1e202800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fsub" #x1e203800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fmax" #x1e204800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fmin" #x1e205800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fmaxnm" #x1e206800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fminnm" #x1e207800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fnmul" #x1e208800 #xff20fc00 :floatdp2 0 FP '(:Fd :Fn :Fm) QL-FP3 F-FPTYPE) 
  ("fmadd" #x1f000000 #xff208000 :floatdp3 0 FP '(:Fd :Fn :Fm :Fa) QL-FP4 F-FPTYPE) 
  ("fmsub" #x1f008000 #xff208000 :floatdp3 0 FP '(:Fd :Fn :Fm :Fa) QL-FP4 F-FPTYPE) 
  ("fnmadd" #x1f200000 #xff208000 :floatdp3 0 FP '(:Fd :Fn :Fm :Fa) QL-FP4 F-FPTYPE) 
  ("fnmsub" #x1f208000 #xff208000 :floatdp3 0 FP '(:Fd :Fn :Fm :Fa) QL-FP4 F-FPTYPE) 
  ("fmov" #x1e201000 #xff201fe0 :floatimm 0 FP '(:Fd :FPIMM) QL-DST-SD F-FPTYPE) 
  ("fcsel" #x1e200c00 #xff200c00 :floatsel 0 FP '(:Fd :Fn :Fm :COND) QL-FP-COND F-FPTYPE) 
  ("strb" #x38000400 #xffe00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W8 0) 
  ("ldrb" #x38400400 #xffe00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W8 0) 
  ("ldrsb" #x38800400 #xffa00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R8 F-LDS-SIZE) 
  ("str" #x3c000400 #x3f600400 :ldst-imm9 0 :CORE '(:Ft :ADDR-SIMM9) QL-LDST-FP 0) 
  ("ldr" #x3c400400 #x3f600400 :ldst-imm9 0 :CORE '(:Ft :ADDR-SIMM9) QL-LDST-FP 0) 
  ("strh" #x78000400 #xffe00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W16 0) 
  ("ldrh" #x78400400 #xffe00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W16 0) 
  ("ldrsh" #x78800400 #xffa00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R16 F-LDS-SIZE) 
  ("str" #xb8000400 #xbfe00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R F-GPRSIZE-IN-Q) 
  ("ldr" #xb8400400 #xbfe00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R F-GPRSIZE-IN-Q) 
  ("ldrsw" #xb8800400 #xffe00400 :ldst-imm9 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-X32 0) 
  ("strb" #x39000000 #xffc00000 :ldst-pos OP-STRB-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-W8 0) 
  ("ldrb" #x39400000 #xffc00000 :ldst-pos OP-LDRB-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-W8 0) 
  ("ldrsb" #x39800000 #xff800000 :ldst-pos OP-LDRSB-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-R8 F-LDS-SIZE) 
  ("str" #x3d000000 #x3f400000 :ldst-pos OP-STRF-POS :CORE '(:Ft :ADDR-UIMM12) QL-LDST-FP 0) 
  ("ldr" #x3d400000 #x3f400000 :ldst-pos OP-LDRF-POS :CORE '(:Ft :ADDR-UIMM12) QL-LDST-FP 0) 
  ("strh" #x79000000 #xffc00000 :ldst-pos OP-STRH-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-W16 0) 
  ("ldrh" #x79400000 #xffc00000 :ldst-pos OP-LDRH-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-W16 0) 
  ("ldrsh" #x79800000 #xff800000 :ldst-pos OP-LDRSH-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-R16 F-LDS-SIZE) 
  ("str" #xb9000000 #xbfc00000 :ldst-pos OP-STR-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-R F-GPRSIZE-IN-Q) 
  ("ldr" #xb9400000 #xbfc00000 :ldst-pos OP-LDR-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-R F-GPRSIZE-IN-Q) 
  ("ldrsw" #xb9800000 #xffc00000 :ldst-pos OP-LDRSW-POS :CORE '(:Rt :ADDR-UIMM12) QL-LDST-X32 0) 
  ("prfm" #xf9800000 #xffc00000 :ldst-pos OP-PRFM-POS :CORE '(:PRFOP :ADDR-UIMM12) QL-LDST-PRFM 0) 
  ("strb" #x38200800 #xffe00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-W8 0) 
  ("ldrb" #x38600800 #xffe00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-W8 0) 
  ("ldrsb" #x38a00800 #xffa00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-R8 F-LDS-SIZE) 
  ("str" #x3c200800 #x3f600c00 :ldst-regoff 0 :CORE '(:Ft :ADDR-REGOFF) QL-LDST-FP 0) 
  ("ldr" #x3c600800 #x3f600c00 :ldst-regoff 0 :CORE '(:Ft :ADDR-REGOFF) QL-LDST-FP 0) 
  ("strh" #x78200800 #xffe00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-W16 0) 
  ("ldrh" #x78600800 #xffe00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-W16 0) 
  ("ldrsh" #x78a00800 #xffa00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-R16 F-LDS-SIZE) 
  ("str" #xb8200800 #xbfe00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-R F-GPRSIZE-IN-Q) 
  ("ldr" #xb8600800 #xbfe00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-R F-GPRSIZE-IN-Q) 
  ("ldrsw" #xb8a00800 #xffe00c00 :ldst-regoff 0 :CORE '(:Rt :ADDR-REGOFF) QL-LDST-X32 0) 
  ("prfm" #xf8a00800 #xffe00c00 :ldst-regoff 0 :CORE '(:PRFOP :ADDR-REGOFF) QL-LDST-PRFM 0) 
  ("sttrb" #x38000800 #xffe00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W8 0) 
  ("ldtrb" #x38400800 #xffe00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W8 0) 
  ("ldtrsb" #x38800800 #xffa00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R8 F-LDS-SIZE) 
  ("sttrh" #x78000800 #xffe00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W16 0) 
  ("ldtrh" #x78400800 #xffe00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W16 0) 
  ("ldtrsh" #x78800800 #xffa00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R16 F-LDS-SIZE) 
  ("sttr" #xb8000800 #xbfe00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R F-GPRSIZE-IN-Q) 
  ("ldtr" #xb8400800 #xbfe00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R F-GPRSIZE-IN-Q) 
  ("ldtrsw" #xb8800800 #xffe00c00 :ldst-unpriv 0 :CORE '(:Rt :ADDR-SIMM9) QL-LDST-X32 0) 
  ("sturb" #x38000000 #xffe00c00 :ldst-unscaled OP-STURB :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W8 F-HAS-ALIAS) 
  ("ldurb" #x38400000 #xffe00c00 :ldst-unscaled OP-LDURB :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W8 F-HAS-ALIAS) 
  ("strb" #x38000000 #xffe00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-W8 F-ALIAS) 
  ("ldrb" #x38400000 #xffe00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-W8 F-ALIAS) 
  ("ldursb" #x38800000 #xffa00c00 :ldst-unscaled OP-LDURSB :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R8 (F-HAS-ALIAS  F-LDS-SIZE)) 
  ("ldrsb" #x38800000 #xffa00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-R8 (F-ALIAS  F-LDS-SIZE)) 
  ("stur" #x3c000000 #x3f600c00 :ldst-unscaled OP-STURV :CORE '(:Ft :ADDR-SIMM9) QL-LDST-FP F-HAS-ALIAS) 
  ("ldur" #x3c400000 #x3f600c00 :ldst-unscaled OP-LDURV :CORE '(:Ft :ADDR-SIMM9) QL-LDST-FP F-HAS-ALIAS) 
  ("str" #x3c000000 #x3f600c00 :ldst-unscaled 0 :CORE '(:Ft :ADDR-SIMM9-2) QL-LDST-FP F-ALIAS) 
  ("ldr" #x3c400000 #x3f600c00 :ldst-unscaled 0 :CORE '(:Ft :ADDR-SIMM9-2) QL-LDST-FP F-ALIAS) 
  ("sturh" #x78000000 #xffe00c00 :ldst-unscaled OP-STURH :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W16 F-HAS-ALIAS) 
  ("ldurh" #x78400000 #xffe00c00 :ldst-unscaled OP-LDURH :CORE '(:Rt :ADDR-SIMM9) QL-LDST-W16 F-HAS-ALIAS) 
  ("strh" #x78000000 #xffe00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-W16 F-ALIAS) 
  ("ldrh" #x78400000 #xffe00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-W16 F-ALIAS) 
  ("ldursh" #x78800000 #xffa00c00 :ldst-unscaled OP-LDURSH :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R16 (F-HAS-ALIAS  F-LDS-SIZE)) 
  ("ldrsh" #x78800000 #xffa00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-R16 (F-ALIAS  F-LDS-SIZE)) 
  ("stur" #xb8000000 #xbfe00c00 :ldst-unscaled OP-STUR :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R (F-HAS-ALIAS  F-GPRSIZE-IN-Q)) 
  ("ldur" #xb8400000 #xbfe00c00 :ldst-unscaled OP-LDUR :CORE '(:Rt :ADDR-SIMM9) QL-LDST-R (F-HAS-ALIAS  F-GPRSIZE-IN-Q)) 
  ("str" #xb8000000 #xbfe00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-R (F-ALIAS  F-GPRSIZE-IN-Q)) 
  ("ldr" #xb8400000 #xbfe00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-R (F-ALIAS  F-GPRSIZE-IN-Q)) 
  ("ldursw" #xb8800000 #xffe00c00 :ldst-unscaled OP-LDURSW :CORE '(:Rt :ADDR-SIMM9) QL-LDST-X32 F-HAS-ALIAS) 
  ("ldrsw" #xb8800000 #xffe00c00 :ldst-unscaled 0 :CORE '(:Rt :ADDR-SIMM9-2) QL-LDST-X32 F-ALIAS) 
  ("prfum" #xf8800000 #xffe00c00 :ldst-unscaled OP-PRFUM :CORE '(:PRFOP :ADDR-SIMM9) QL-LDST-PRFM F-HAS-ALIAS) 
  ("prfm" #xf8800000 #xffe00c00 :ldst-unscaled 0 :CORE '(:PRFOP :ADDR-SIMM9-2) QL-LDST-PRFM F-ALIAS) 
  ("stxrb" #x8007c00 #xffe0fc00 :ldstexcl 0 :CORE '(:Rs :Rt :ADDR-SIMPLE) QL-W2-LDST-EXC 0) 
  ("stlxrb" #x800fc00 #xffe0fc00 :ldstexcl 0 :CORE '(:Rs :Rt :ADDR-SIMPLE) QL-W2-LDST-EXC 0) 
  ("ldxrb" #x85f7c00 #xfffffc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-W1-LDST-EXC 0) 
  ("ldaxrb" #x85ffc00 #xfffffc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-W1-LDST-EXC 0) 
  ("stlrb" #x89ffc00 #xfffffc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-W1-LDST-EXC 0) 
  ("ldarb" #x8dffc00 #xfffffc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-W1-LDST-EXC 0) 
  ("stxrh" #x48007c00 #xfffffc00 :ldstexcl 0 :CORE '(:Rs :Rt :ADDR-SIMPLE) QL-W2-LDST-EXC 0) 
  ("stlxrh" #x4800fc00 #xfffffc00 :ldstexcl 0 :CORE '(:Rs :Rt :ADDR-SIMPLE) QL-W2-LDST-EXC 0) 
  ("ldxrh" #x485f7c00 #xfffffc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-W1-LDST-EXC 0) 
  ("ldaxrh" #x485ffc00 #xfffffc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-W1-LDST-EXC 0) 
  ("stlrh" #x489ffc00 #xfffffc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-W1-LDST-EXC 0) 
  ("ldarh" #x48dffc00 #xfffffc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-W1-LDST-EXC 0) 
  ("stxr" #x88007c00 #xbfe0fc00 :ldstexcl 0 :CORE '(:Rs :Rt :ADDR-SIMPLE) QL-R2-LDST-EXC F-GPRSIZE-IN-Q) 
  ("stlxr" #x8800fc00 #xbfe0fc00 :ldstexcl 0 :CORE '(:Rs :Rt :ADDR-SIMPLE) QL-R2-LDST-EXC F-GPRSIZE-IN-Q) 
  ("stxp" #x88200000 #xbfe0fc00 :ldstexcl 0 :CORE '(:Rs :Rt :Rt2 :ADDR-SIMPLE) QL-R3-LDST-EXC F-GPRSIZE-IN-Q) 
  ("stlxp" #x88208000 #xbfe08000 :ldstexcl 0 :CORE '(:Rs :Rt :Rt2 :ADDR-SIMPLE) QL-R3-LDST-EXC F-GPRSIZE-IN-Q) 
  ("ldxr" #x885f7c00 #xbfe08000 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-R1NIL F-GPRSIZE-IN-Q) 
  ("ldaxr" #x885ffc00 #xbfe0fc00 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-R1NIL F-GPRSIZE-IN-Q) 
  ("ldxp" #x887f0000 #xbfe08000 :ldstexcl 0 :CORE '(:Rt :Rt2 :ADDR-SIMPLE) QL-R2NIL F-GPRSIZE-IN-Q) 
  ("ldaxp" #x887f8000 #xbfe08000 :ldstexcl 0 :CORE '(:Rt :Rt2 :ADDR-SIMPLE) QL-R2NIL F-GPRSIZE-IN-Q) 
  ("stlr" #x889ffc00 #xbfe08000 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-R1NIL F-GPRSIZE-IN-Q) 
  ("ldar" #x88dffc00 #xbfe08000 :ldstexcl 0 :CORE '(:Rt :ADDR-SIMPLE) QL-R1NIL F-GPRSIZE-IN-Q) 
  ("stnp" #x28000000 #x7fc00000 :ldstnapair-offs 0 :CORE '(:Rt :Rt2 :ADDR-SIMM7) QL-LDST-PAIR-R F-SF) 
  ("ldnp" #x28400000 #x7fc00000 :ldstnapair-offs 0 :CORE '(:Rt :Rt2 :ADDR-SIMM7) QL-LDST-PAIR-R F-SF) 
  ("stnp" #x2c000000 #x3fc00000 :ldstnapair-offs 0 :CORE '(:Ft :Ft2 :ADDR-SIMM7) QL-LDST-PAIR-FP 0) 
  ("ldnp" #x2c400000 #x3fc00000 :ldstnapair-offs 0 :CORE '(:Ft :Ft2 :ADDR-SIMM7) QL-LDST-PAIR-FP 0) 
  ("stp" #x29000000 #x7ec00000 :ldstpair-off 0 :CORE '(:Rt :Rt2 :ADDR-SIMM7) QL-LDST-PAIR-R F-SF) 
  ("ldp" #x29400000 #x7ec00000 :ldstpair-off 0 :CORE '(:Rt :Rt2 :ADDR-SIMM7) QL-LDST-PAIR-R F-SF) 
  ("stp" #x2d000000 #x3fc00000 :ldstpair-off 0 :CORE '(:Ft :Ft2 :ADDR-SIMM7) QL-LDST-PAIR-FP 0) 
  ("ldp" #x2d400000 #x3fc00000 :ldstpair-off 0 :CORE '(:Ft :Ft2 :ADDR-SIMM7) QL-LDST-PAIR-FP 0) 
  ("ldpsw" #x69400000 #xffc00000 :ldstpair-off 0 :CORE '(:Rt :Rt2 :ADDR-SIMM7) QL-LDST-PAIR-X32 0) 
  ("stp" #x28800000 #x7ec00000 :ldstpair-indexed 0 :CORE '(:Rt :Rt2 :ADDR-SIMM7) QL-LDST-PAIR-R F-SF) 
  ("ldp" #x28c00000 #x7ec00000 :ldstpair-indexed 0 :CORE '(:Rt :Rt2 :ADDR-SIMM7) QL-LDST-PAIR-R F-SF) 
  ("stp" #x2c800000 #x3ec00000 :ldstpair-indexed 0 :CORE '(:Ft :Ft2 :ADDR-SIMM7) QL-LDST-PAIR-FP 0) 
  ("ldp" #x2cc00000 #x3ec00000 :ldstpair-indexed 0 :CORE '(:Ft :Ft2 :ADDR-SIMM7) QL-LDST-PAIR-FP 0) 
  ("ldpsw" #x68c00000 #xfec00000 :ldstpair-indexed 0 :CORE '(:Rt :Rt2 :ADDR-SIMM7) QL-LDST-PAIR-X32 0) 
  ("ldr" #x18000000 #xbf000000 :loadlit OP-LDR-LIT :CORE '(:Rt :ADDR-PCREL19) QL-R-PCREL F-GPRSIZE-IN-Q) 
  ("ldr" #x1c000000 #x3f000000 :loadlit OP-LDRV-LIT :CORE '(:Ft :ADDR-PCREL19) QL-FP-PCREL 0) 
  ("ldrsw" #x98000000 #xff000000 :loadlit OP-LDRSW-LIT :CORE '(:Rt :ADDR-PCREL19) QL-X-PCREL 0) 
  ("prfm" #xd8000000 #xff000000 :loadlit OP-PRFM-LIT :CORE '(:PRFOP :ADDR-PCREL19) QL-PRFM-PCREL 0) 
  ("and" #x12000000 #x7f800000 :log-imm 0 :CORE '(:Rd-SP :Rn :LIMM) QL-R2NIL (F-HAS-ALIAS  F-SF)) 
  ("bic" #x12000000 #x7f800000 :log-imm OP-BIC :CORE '(:Rd-SP :Rn :LIMM) QL-R2NIL ((F-ALIAS  F-PSEUDO)  F-SF)) 
  ("orr" #x32000000 #x7f800000 :log-imm 0 :CORE '(:Rd-SP :Rn :LIMM) QL-R2NIL (F-HAS-ALIAS  F-SF)) 
  ("mov" #x320003e0 #x7f8003e0 :log-imm OP-MOV-IMM-LOG :CORE '(:Rd-SP :IMM-MOV) QL-R1NIL (((F-ALIAS  F-P1)  F-SF)  F-CONV)) 
  ("eor" #x52000000 #x7f800000 :log-imm 0 :CORE '(:Rd-SP :Rn :LIMM) QL-R2NIL F-SF) 
  ("ands" #x72000000 #x7f800000 :log-imm 0 :CORE '(:Rd :Rn :LIMM) QL-R2NIL (F-HAS-ALIAS  F-SF)) 
  ("tst" #x7200001f #x7f80001f :log-imm 0 :CORE '(:Rn :LIMM) QL-R1NIL (F-ALIAS  F-SF)) 
  ("and" #xa000000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) 0)
  ("and" #x8a000000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) 0)
  ("bic" #xa200000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) 0)
  ("bic" #x8a200000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) 0)
  ("orr" #x2a000000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) F-HAS-ALIAS)
  ("orr" #xaa000000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) F-HAS-ALIAS) 

  ("mov" #x2a0003e0 #xff2003e0 :log-shift 0 :CORE '(:Rd :Rm) '(:w :w) F-ALIAS)
  ("mov" #xaa0003e0 #xff2003e0 :log-shift 0 :CORE '(:Rd :Rm) '(:x :x) F-ALIAS)
  ("uxtw" #x2a0003e0 #x7f2003e0 :log-shift OP-UXTW :CORE '(:Rd :Rm) QL-I2SAMEW (F-ALIAS  F-PSEUDO)) 
  ("orn" #x2a200000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) F-HAS-ALIAS) 
  ("orn" #xaa200000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) F-HAS-ALIAS)
 
  ("mvn" #x2a2003e0 #x7f2003e0 :log-shift 0 :CORE '(:Rd :Rm-SFT) QL-I2SAMER (F-ALIAS  F-SF)) 
  ("eor" #x4a000000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) 0)
  ("eor" #xca000000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) 0)
  ("eon" #x4a200000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) 0)
  ("eon" #xca200000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT)
   (:x :x :x-shift) 0) 
  ("ands" #x6a000000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) F-HAS-ALIAS) 
  ("ands" #xea000000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) F-HAS-ALIAS)
 
  ("tst" #x6a00001f #x7f20001f :log-shift 0 :CORE '(:Rn :Rm-SFT) QL-I2SAMER (F-ALIAS  F-SF)) 
  ("bics" #x6a200000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:w :w :w-shift) 0) 
  ("bics" #xea200000 #xff200000 :log-shift 0 :CORE '(:Rd :Rn :Rm-SFT) '(:x :x :x-shift) 0) 
  ("movn" #x12800000 #x7f800000 :movewide OP-MOVN :CORE '(:Rd :HALF) QL-DST-R (F-SF  F-HAS-ALIAS)) 
  ("mov" #x12800000 #x7f800000 :movewide OP-MOV-IMM-WIDEN :CORE '(:Rd :IMM-MOV) QL-DST-R ((F-SF  F-ALIAS)  F-CONV)) 
  ("movz" #x52800000 #x7f800000 :movewide OP-MOVZ :CORE '(:Rd :HALF) QL-DST-R (F-SF  F-HAS-ALIAS)) 
  ("mov" #x52800000 #x7f800000 :movewide OP-MOV-IMM-WIDE :CORE '(:Rd :IMM-MOV) QL-DST-R ((F-SF  F-ALIAS)  F-CONV)) 
  ("movk" #x72800000 #x7f800000 :movewide OP-MOVK :CORE '(:Rd :HALF) QL-DST-R F-SF) 
  ("adr" #x10000000 #x9f000000 :pcreladdr 0 :CORE '(:Rd :ADDR-PCREL21) QL-ADRP 0) 
  ("adrp" #x90000000 #x9f000000 :pcreladdr 0 :CORE '(:Rd :ADDR-ADRP) QL-ADRP 0) 
  ("msr" #xd500401f #xfff8f01f :ic-system 0 :CORE '(:PSTATEFIELD :UIMM4) () 0) 
  ("hint" #xd503201f #xfffff01f :ic-system 0 :CORE '(:UIMM7) () F-HAS-ALIAS) 
  ("nop" #xd503201f #xffffffff :ic-system 0 :CORE '() () F-ALIAS) 
  ("yield" #xd503203f #xffffffff :ic-system 0 :CORE '() () F-ALIAS) 
  ("wfe" #xd503205f #xffffffff :ic-system 0 :CORE '() () F-ALIAS) 
  ("wfi" #xd503207f #xffffffff :ic-system 0 :CORE '() () F-ALIAS) 
  ("sev" #xd503209f #xffffffff :ic-system 0 :CORE '() () F-ALIAS) 
  ("sevl" #xd50320bf #xffffffff :ic-system 0 :CORE '() () F-ALIAS) 
  ("clrex" #xd503305f #xfffff0ff :ic-system 0 :CORE '(:UIMM4) () (F-OPD0-OPT  F-DEFAULT) (#xF)) 
  ("dsb" #xd503309f #xfffff0ff :ic-system 0 :CORE '(:BARRIER) () 0) 
  ("dmb" #xd50330bf #xfffff0ff :ic-system 0 :CORE '(:BARRIER) () 0) 
  ("isb" #xd50330df #xfffff0ff :ic-system 0 :CORE '(:BARRIER-ISB) () (F-OPD0-OPT  F-DEFAULT) (#xF)) 
  ("sys" #xd5080000 #xfff80000 :ic-system 0 :CORE OP5 (UIMM3-OP1 :Cn Cm UIMM3-OP2 Rt) QL-SYS ((F-HAS-ALIAS  F-OPD4-OPT)  F-DEFAULT) (#x1F)) 
  ("at" #xd5080000 #xfff80000 :ic-system 0 :CORE '(:SYSREG-AT :Rt) QL-SRC-X F-ALIAS) 
  ("dc" #xd5080000 #xfff80000 :ic-system 0 :CORE '(:SYSREG-DC :Rt) QL-SRC-X F-ALIAS) 
  ("ic" #xd5080000 #xfff80000 :ic-system 0 :CORE '(:SYSREG-IC :Rt-SYS) QL-SRC-X ((F-ALIAS  F-OPD1-OPT)  F-DEFAULT) (#x1F)) 
  ("tlbi" #xd5080000 #xfff80000 :ic-system 0 :CORE '(:SYSREG-TLBI :Rt-SYS) QL-SRC-X ((F-ALIAS  F-OPD1-OPT)  F-DEFAULT) (#x1F)) 
  ("msr" #xd5100000 #xfff00000 :ic-system 0 :CORE '(:SYSREG :Rt) QL-SRC-X 0) 
  ("sysl" #xd5280000 #xfff80000 :ic-system 0 :CORE '(:Rt :UIMM3-OP1 :Cn :Cm :UIMM3-OP2) QL-SYSL 0) 
  ("mrs" #xd5300000 #xfff00000 :ic-system 0 :CORE '(:Rt :SYSREG) QL-DST-X 0) 
  ("tbz" #x36000000 #x7f000000 :testbranch 0 :CORE '(:Rt :BIT-NUM :ADDR-PCREL14) QL-PCREL-14 0) 
  ("tbnz" #x37000000 #x7f000000 :testbranch 0 :CORE '(:Rt :BIT-NUM :ADDR-PCREL14) QL-PCREL-14 0) 
  ("beq" #x54000000 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bne" #x54000001 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bcs" #x54000002 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bhs" #x54000002 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bcc" #x54000003 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("blo" #x54000003 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bmi" #x54000004 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bpl" #x54000005 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bvs" #x54000006 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bvc" #x54000007 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bhi" #x54000008 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bls" #x54000009 #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bge" #x5400000a #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("blt" #x5400000b #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("bgt" #x5400000c #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO)) 
  ("ble" #x5400000d #xff00001f :condbranch 0 :CORE '(:ADDR-PCREL19) QL-PCREL-NIL (F-ALIAS  F-PSEUDO))
 

  )

(provide "ARM64-ASM")
