const char crypto_lua[] =
"-- crypto.lua (internal file)\n"
"\n"
"local ffi = require('ffi')\n"
"local buffer = require('buffer')\n"
"local cord_ibuf_take = buffer.internal.cord_ibuf_take\n"
"local cord_ibuf_put = buffer.internal.cord_ibuf_put\n"
"\n"
"ffi.cdef[[\n"
"    /* from openssl/err.h */\n"
"    unsigned long crypto_ERR_get_error(void);\n"
"    char *crypto_ERR_error_string(unsigned long e, char *buf);\n"
"\n"
"    /* from openssl/evp.h */\n"
"    typedef void ENGINE;\n"
"\n"
"    typedef struct {} EVP_MD_CTX;\n"
"    typedef struct {} EVP_MD;\n"
"    EVP_MD_CTX *crypto_EVP_MD_CTX_new(void);\n"
"    void crypto_EVP_MD_CTX_free(EVP_MD_CTX *ctx);\n"
"    int crypto_EVP_DigestInit_ex(EVP_MD_CTX *ctx, const EVP_MD *type,\n"
"                                 ENGINE *impl);\n"
"    int crypto_EVP_DigestUpdate(EVP_MD_CTX *ctx, const void *d, size_t cnt);\n"
"    int crypto_EVP_DigestFinal_ex(EVP_MD_CTX *ctx, unsigned char *md,\n"
"                                  unsigned int *s);\n"
"    const EVP_MD *crypto_EVP_get_digestbyname(const char *name);\n"
"\n"
"    typedef struct {} HMAC_CTX;\n"
"    HMAC_CTX *crypto_HMAC_CTX_new(void);\n"
"    void crypto_HMAC_CTX_free(HMAC_CTX *ctx);\n"
"    int crypto_HMAC_Init_ex(HMAC_CTX *ctx, const void *key, int len,\n"
"                            const char *digest, const EVP_MD *md, ENGINE *impl);\n"
"    int crypto_HMAC_Update(HMAC_CTX *ctx, const unsigned char *data,\n"
"                           size_t len);\n"
"    int crypto_HMAC_Final(HMAC_CTX *ctx, unsigned char *md,\n"
"                          unsigned int *len, unsigned int size);\n"
"\n"
"    enum crypto_algo {\n"
"        CRYPTO_ALGO_NONE,\n"
"        CRYPTO_ALGO_AES128,\n"
"        CRYPTO_ALGO_AES192,\n"
"        CRYPTO_ALGO_AES256,\n"
"        CRYPTO_ALGO_DES,\n"
"    };\n"
"\n"
"    enum crypto_mode {\n"
"        CRYPTO_MODE_ECB,\n"
"        CRYPTO_MODE_CBC,\n"
"        CRYPTO_MODE_CFB,\n"
"        CRYPTO_MODE_OFB,\n"
"    };\n"
"\n"
"    enum crypto_direction {\n"
"        CRYPTO_DIR_DECRYPT = 0,\n"
"        CRYPTO_DIR_ENCRYPT = 1,\n"
"    };\n"
"\n"
"    struct crypto_stream;\n"
"\n"
"    struct crypto_stream *\n"
"    crypto_stream_new(enum crypto_algo algo, enum crypto_mode mode,\n"
"                      enum crypto_direction dir);\n"
"\n"
"    int\n"
"    crypto_stream_begin(struct crypto_stream *s, const char *key, int key_size,\n"
"                        const char *iv, int iv_size);\n"
"\n"
"    int\n"
"    crypto_stream_append(struct crypto_stream *s, const char *in, int in_size,\n"
"                         char *out, int out_size);\n"
"\n"
"    int\n"
"    crypto_stream_commit(struct crypto_stream *s, char *out, int out_size);\n"
"\n"
"    void\n"
"    crypto_stream_delete(struct crypto_stream *s);\n"
"]]\n"
"\n"
"local function openssl_err_str()\n"
"  return ffi.string(ffi.C.crypto_ERR_error_string(ffi.C.crypto_ERR_get_error(), nil))\n"
"end\n"
"\n"
"local digests = {}\n"
"for class, _ in pairs({\n"
"    md2 = 'MD2', md4 = 'MD4', md5 = 'MD5',\n"
"    sha1 = 'SHA1', sha224 = 'SHA224',\n"
"    sha256 = 'SHA256', sha384 = 'SHA384', sha512 = 'SHA512',\n"
"    dss = 'DSS', dss1 = 'DSS1', mdc2 = 'MDC2', ripemd160 = 'RIPEMD160'}) do\n"
"    local digest = ffi.C.crypto_EVP_get_digestbyname(class)\n"
"    if digest ~= nil then\n"
"        digests[class] = digest\n"
"    end\n"
"end\n"
"\n"
"local digest_mt = {}\n"
"\n"
"local function digest_gc(ctx)\n"
"    ffi.C.crypto_EVP_MD_CTX_free(ctx)\n"
"end\n"
"\n"
"local function digest_new(digest)\n"
"    local ctx = ffi.C.crypto_EVP_MD_CTX_new()\n"
"    if ctx == nil then\n"
"        return error('Can\\'t create digest ctx: ' .. openssl_err_str())\n"
"    end\n"
"    ffi.gc(ctx, digest_gc)\n"
"    local self = setmetatable({\n"
"        ctx = ctx,\n"
"        digest = digest,\n"
"        buf = buffer.ibuf(64),\n"
"        initialized = false,\n"
"    }, digest_mt)\n"
"    self:init()\n"
"    return self\n"
"end\n"
"\n"
"local function digest_init(self)\n"
"    if self.ctx == nil then\n"
"        return error('Digest context isn\\'t usable')\n"
"    end\n"
"    if ffi.C.crypto_EVP_DigestInit_ex(self.ctx, self.digest, nil) ~= 1 then\n"
"        return error('Can\\'t init digest: ' .. openssl_err_str())\n"
"    end\n"
"    self.initialized = true\n"
"end\n"
"\n"
"local function digest_update(self, input)\n"
"    if not self.initialized then\n"
"        return error('Digest not initialized')\n"
"    end\n"
"    if ffi.C.crypto_EVP_DigestUpdate(self.ctx, input, input:len()) ~= 1 then\n"
"        return error('Can\\'t update digest: ' .. openssl_err_str())\n"
"    end\n"
"end\n"
"\n"
"local function digest_final(self)\n"
"    if not self.initialized then\n"
"        return error('Digest not initialized')\n"
"    end\n"
"    self.initialized = false\n"
"    local ai = ffi.new('int[1]')\n"
"    if ffi.C.crypto_EVP_DigestFinal_ex(self.ctx, self.buf.wpos, ai) ~= 1 then\n"
"        return error('Can\\'t finalize digest: ' .. openssl_err_str())\n"
"    end\n"
"    return ffi.string(self.buf.wpos, ai[0])\n"
"end\n"
"\n"
"local function digest_free(self)\n"
"    ffi.C.crypto_EVP_MD_CTX_free(self.ctx)\n"
"    ffi.gc(self.ctx, nil)\n"
"    self.ctx = nil\n"
"    self.initialized = false\n"
"end\n"
"\n"
"digest_mt = {\n"
"    __index = {\n"
"          init = digest_init,\n"
"          update = digest_update,\n"
"          result = digest_final,\n"
"          free = digest_free\n"
"    }\n"
"}\n"
"\n"
"local hmacs = digests\n"
"\n"
"local hmac_mt = {}\n"
"\n"
"local function hmac_gc(ctx)\n"
"    ffi.C.crypto_HMAC_CTX_free(ctx)\n"
"end\n"
"\n"
"local function hmac_new(class, digest, key)\n"
"    if key == nil then\n"
"        return error('Key should be specified for HMAC operations')\n"
"    end\n"
"    local ctx = ffi.C.crypto_HMAC_CTX_new()\n"
"    if ctx == nil then\n"
"        return error('Can\\'t create HMAC ctx: ' .. openssl_err_str())\n"
"    end\n"
"    ffi.gc(ctx, hmac_gc)\n"
"    local self = setmetatable({\n"
"        ctx = ctx,\n"
"        class = class,\n"
"        digest = digest,\n"
"        initialized = false,\n"
"    }, hmac_mt)\n"
"    self:init(key)\n"
"    return self\n"
"end\n"
"\n"
"local function hmac_init(self, key)\n"
"    if self.ctx == nil then\n"
"        return error('HMAC context isn\\'t usable')\n"
"    end\n"
"    if ffi.C.crypto_HMAC_Init_ex(self.ctx, key, key:len(), self.class,\n"
"                                 self.digest, nil) ~= 1 then\n"
"        return error('Can\\'t init HMAC: ' .. openssl_err_str())\n"
"    end\n"
"    self.initialized = true\n"
"end\n"
"\n"
"local function hmac_update(self, input)\n"
"    if not self.initialized then\n"
"        return error('HMAC not initialized')\n"
"    end\n"
"    if ffi.C.crypto_HMAC_Update(self.ctx, input, input:len()) ~= 1 then\n"
"        return error('Can\\'t update HMAC: ' .. openssl_err_str())\n"
"    end\n"
"end\n"
"\n"
"local function hmac_final(self)\n"
"    if not self.initialized then\n"
"        return error('HMAC not initialized')\n"
"    end\n"
"    self.initialized = false\n"
"    local ibuf = cord_ibuf_take()\n"
"    local buf_size = 64\n"
"    local buf = ibuf:alloc(buf_size)\n"
"    local ai = ffi.new('int[1]')\n"
"    if ffi.C.crypto_HMAC_Final(self.ctx, buf, ai, buf_size) ~= 1 then\n"
"        cord_ibuf_put(ibuf)\n"
"        return error('Can\\'t finalize HMAC: ' .. openssl_err_str())\n"
"    end\n"
"    buf = ffi.string(buf, ai[0])\n"
"    cord_ibuf_put(ibuf)\n"
"    return buf\n"
"end\n"
"\n"
"local function hmac_free(self)\n"
"    ffi.C.crypto_HMAC_CTX_free(self.ctx)\n"
"    ffi.gc(self.ctx, nil)\n"
"    self.ctx = nil\n"
"    self.initialized = false\n"
"end\n"
"\n"
"hmac_mt = {\n"
"    __index = {\n"
"          init = hmac_init,\n"
"          update = hmac_update,\n"
"          result = hmac_final,\n"
"          free = hmac_free\n"
"    }\n"
"}\n"
"\n"
"local crypto_stream_mt = {}\n"
"\n"
"local function crypto_stream_gc(ctx)\n"
"    ffi.C.crypto_stream_delete(ctx)\n"
"end\n"
"\n"
"local function crypto_stream_new(algo, mode, key, iv, direction)\n"
"    local ctx = ffi.C.crypto_stream_new(algo, mode, direction)\n"
"    if ctx == nil then\n"
"        box.error()\n"
"    end\n"
"    ffi.gc(ctx, crypto_stream_gc)\n"
"    local self = setmetatable({\n"
"        ctx = ctx,\n"
"        buf = buffer.ibuf(),\n"
"        is_initialized = false,\n"
"    }, crypto_stream_mt)\n"
"    self:init(key, iv)\n"
"    return self\n"
"end\n"
"\n"
"local function crypto_stream_begin(self, key, iv)\n"
"    local ctx = self.ctx\n"
"    if not ctx then\n"
"        return error('Cipher context isn\\'t usable')\n"
"    end\n"
"    self.key = key or self.key\n"
"    self.iv = iv or self.iv\n"
"    if self.key and self.iv then\n"
"        if ffi.C.crypto_stream_begin(ctx, self.key, self.key:len(),\n"
"                                     self.iv, self.iv:len()) ~= 0 then\n"
"            box.error()\n"
"        end\n"
"        self.is_initialized = true\n"
"    end\n"
"end\n"
"\n"
"local function crypto_stream_append(self, input)\n"
"    if not self.is_initialized then\n"
"        return error('Cipher not initialized')\n"
"    end\n"
"    if type(input) ~= 'string' then\n"
"        error(\"Usage: cipher:update(string)\")\n"
"    end\n"
"    local append = ffi.C.crypto_stream_append\n"
"    local out_size = append(self.ctx, input, input:len(), nil, 0)\n"
"    local wpos = self.buf:reserve(out_size)\n"
"    out_size = append(self.ctx, input, input:len(), wpos, out_size)\n"
"    if out_size < 0 then\n"
"        box.error()\n"
"    end\n"
"    return ffi.string(wpos, out_size)\n"
"end\n"
"\n"
"local function crypto_stream_commit(self)\n"
"    if not self.is_initialized then\n"
"        return error('Cipher not initialized')\n"
"    end\n"
"    local commit = ffi.C.crypto_stream_commit\n"
"    local out_size = commit(self.ctx, nil, 0)\n"
"    local wpos = self.buf:reserve(out_size)\n"
"    out_size = commit(self.ctx, wpos, out_size)\n"
"    if out_size < 0 then\n"
"        box.error()\n"
"    end\n"
"    self.is_initialized = false\n"
"    return ffi.string(wpos, out_size)\n"
"end\n"
"\n"
"local function crypto_stream_free(self)\n"
"    crypto_stream_gc(ffi.gc(self.ctx, nil))\n"
"    self.ctx = nil\n"
"    self.key = nil\n"
"    self.iv = nil\n"
"    self.is_initialized = false\n"
"end\n"
"\n"
"crypto_stream_mt = {\n"
"    __index = {\n"
"          init = crypto_stream_begin,\n"
"          update = crypto_stream_append,\n"
"          result = crypto_stream_commit,\n"
"          free = crypto_stream_free\n"
"    }\n"
"}\n"
"\n"
"local digest_api = {}\n"
"for class, digest in pairs(digests) do\n"
"    digest_api[class] = setmetatable({\n"
"        new = function () return digest_new(digest) end\n"
"    }, {\n"
"        __call = function (self, str)\n"
"            if type(str) ~= 'string' then\n"
"                error(\"Usage: digest.\"..class..\"(string)\")\n"
"            end\n"
"            local ctx = digest_new(digest)\n"
"            ctx:update(str)\n"
"            local res = ctx:result()\n"
"            ctx:free()\n"
"            return res\n"
"        end\n"
"    })\n"
"end\n"
"\n"
"digest_api = setmetatable(digest_api,\n"
"    {__index = function(self, digest)\n"
"        return error('Digest method \"' .. digest .. '\" is not supported')\n"
"    end })\n"
"\n"
"local hmac_api = {}\n"
"for class, digest in pairs(hmacs) do\n"
"    hmac_api[class] = setmetatable({\n"
"        new = function (key) return hmac_new(class, digest, key) end\n"
"    }, {\n"
"        __call = function (self, key, str)\n"
"            if type(str) ~= 'string' then\n"
"                error(\"Usage: hmac.\"..class..\"(key, string)\")\n"
"            end\n"
"            local ctx = hmac_new(class, digest, key)\n"
"            ctx:update(str)\n"
"            local res = ctx:result()\n"
"            ctx:free()\n"
"            return res\n"
"        end\n"
"    })\n"
"    hmac_api[class .. '_hex'] = function (key, str)\n"
"        if type(str) ~= 'string' then\n"
"            error(\"Usage: hmac.\"..class..\"_hex(key, string)\")\n"
"        end\n"
"        return string.hex(hmac_api[class](key, str))\n"
"    end\n"
"end\n"
"\n"
"hmac_api = setmetatable(hmac_api,\n"
"    {__index = function(self, digest)\n"
"        return error('HMAC method \"' .. digest .. '\" is not supported')\n"
"    end })\n"
"\n"
"local crypto_algos = {\n"
"    none = ffi.C.CRYPTO_ALGO_NONE,\n"
"    aes128 = ffi.C.CRYPTO_ALGO_AES128,\n"
"    aes192 = ffi.C.CRYPTO_ALGO_AES192,\n"
"    aes256 = ffi.C.CRYPTO_ALGO_AES256,\n"
"    des = ffi.C.CRYPTO_ALGO_DES\n"
"}\n"
"local crypto_modes = {\n"
"    ecb = ffi.C.CRYPTO_MODE_ECB,\n"
"    cbc = ffi.C.CRYPTO_MODE_CBC,\n"
"    cfb = ffi.C.CRYPTO_MODE_CFB,\n"
"    ofb = ffi.C.CRYPTO_MODE_OFB\n"
"}\n"
"local crypto_dirs = {\n"
"    encrypt = ffi.C.CRYPTO_DIR_ENCRYPT,\n"
"    decrypt = ffi.C.CRYPTO_DIR_DECRYPT\n"
"}\n"
"\n"
"local algo_api_mt = {\n"
"    __index = function(self, mode)\n"
"        error('Cipher mode ' .. mode .. ' is not supported')\n"
"    end\n"
"}\n"
"local crypto_api_mt = {\n"
"    __index = function(self, cipher)\n"
"        return error('Cipher method \"' .. cipher .. '\" is not supported')\n"
"    end\n"
"}\n"
"\n"
"local crypto_api = setmetatable({}, crypto_api_mt)\n"
"for algo_name, algo_value in pairs(crypto_algos) do\n"
"    local algo_api = setmetatable({}, algo_api_mt)\n"
"    crypto_api[algo_name] = algo_api\n"
"    for mode_name, mode_value in pairs(crypto_modes) do\n"
"        local mode_api = {}\n"
"        algo_api[mode_name] = mode_api\n"
"        for dir_name, dir_value in pairs(crypto_dirs) do\n"
"            mode_api[dir_name] = setmetatable({\n"
"                new = function(key, iv)\n"
"                    return crypto_stream_new(algo_value, mode_value, key, iv,\n"
"                                             dir_value)\n"
"                end\n"
"            }, {\n"
"                __call = function(self, str, key, iv)\n"
"                    local ctx = crypto_stream_new(algo_value, mode_value, key,\n"
"                                                  iv, dir_value)\n"
"                    local res = ctx:update(str)\n"
"                    res = res .. ctx:result()\n"
"                    ctx:free()\n"
"                    return res\n"
"                end\n"
"            })\n"
"        end\n"
"    end\n"
"end\n"
"\n"
"local public_methods = {\n"
"    digest = digest_api,\n"
"    hmac   = hmac_api,\n"
"    cipher = crypto_api,\n"
"}\n"
"\n"
"local module_mt = {\n"
"    __serialize = function(self)\n"
"        return public_methods\n"
"    end,\n"
"    __index = public_methods\n"
"}\n"
"\n"
"return setmetatable({\n"
"    cipher_algo = crypto_algos,\n"
"    cipher_mode = crypto_modes,\n"
"}, module_mt)\n"
""
;
