// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A4__F282

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_96_3_F_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1200)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x3e442787u, 0x606c3e5au, 0, 126 }, // 3e442787606c3e5a = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0x0fac8e8bu, 0x6827258eu, 0, 6 }, // 0fac8e8b6827258e = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0xfc431c69u, 0xaff77c79u, 0, 46 }, // fc431c69aff77c79 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1200
    { 0x9d3fa1a3u, 0x620c6a22u, 0, 23 }, // 9d3fa1a3620c6a22 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x0abf96d7u, 0xa5341806u, 40, 126 }, // 0abf96d7a5341806 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0xa85787f5u, 0xe0579110u, 40, 6 }, // a85787f5e0579110 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0x28519786u, 0xba8bb708u, 40, 46 }, // 28519786ba8bb708 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1200
    { 0x4b701d94u, 0x299d9bf9u, 40, 23 }, // 4b701d94299d9bf9 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x4ebebc97u, 0x7eedc6f2u, 80, 63 }, // 4ebebc977eedc6f2 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0x45dd1098u, 0x8c5bab5du, 120, 155 }, // 45dd10988c5bab5d = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0xc0e15ba6u, 0x004982cau, 120, 86 }, // c0e15ba6004982ca = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0xbe918696u, 0x75113a7au, 120, 103 }, // be91869675113a7a = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0xf8e35437u, 0xf6c14e3cu, 120, 63 }, // f8e35437f6c14e3c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_3_F_T_0___gfx120x__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1200
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1200_mod0
{{ 0, 2, 3, 1, 3, 3, 3, 3, 3, 3},
 { 5, 3, 2, 2, 3, 3, 3, 7, 7, 7},
 { 7, 4, 6, 6, 7, 4, 6, 7, 7, 7},
 { 9,10,12,10,11, 9, 4, 9, 9,10},
 {11,10, 4, 4,11, 9,11, 9,10, 8},
 { 9, 7, 4,10,11,10,11,11, 8, 8},
 {10, 7, 4, 9,10,11, 9, 7, 7, 5},
 { 4, 5,11,10,11, 9, 9,10, 9,11},
 { 6, 7,11,11,11, 7, 4,11, 9, 9},
 {12, 7,11,10,11,10,10, 9,11, 9}}
// End of GPU gfx1200_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A4__F282 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 4;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 4;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = "i32"
// BLOCK_DMODEL = 96
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 0

// vim: set fileencoding=utf-8

