/*
 *  Copyright (C) 2022 - This file is part of libecc project
 *
 *  Authors:
 *      Arnaud EBALARD <arnaud.ebalard@ssi.gouv.fr>
 *      Ryad BENADJILA <ryadbenadjila@gmail.com>
 *
 *  This software is licensed under a dual BSD and GPL v2 license.
 *  See LICENSE file at the root folder of the project.
 */
#ifndef __BIGN_TEST_VECTORS_H__
#define __BIGN_TEST_VECTORS_H__

#if defined(WITH_HASH_BELT_HASH) && defined(WITH_CURVE_BIGN256V1)
/************************************************/
static const u8 bign_1_test_vectors_priv_key[] = {
	0x69, 0xe2, 0x73, 0xc2, 0x5f, 0x23, 0x79, 0x0c, 0x9e, 0x42, 0x32, 0x07, 0xed, 0x1f, 0x28, 0x34, 0x18, 0xf2, 0x74, 0x9c, 0x32, 0xf0, 0x33, 0x45, 0x67, 0x39, 0x73, 0x4b, 0xb8, 0xb5, 0x66, 0x1f,
 };
static const u8 bign_1_test_vectors_expected_sig[] = {
	0xE3, 0x6B, 0x7F, 0x03, 0x77, 0xAE, 0x4C, 0x52, 0x40, 0x27, 0xC3, 0x87, 0xFA, 0xDF, 0x1B, 0x20, 0xCE, 0x72, 0xF1, 0x53, 0x0B, 0x71, 0xF2, 0xB5, 0xFD, 0x3A, 0x8C, 0x58, 0x4F, 0xE2, 0xE1, 0xAE, 0xD2, 0x00, 0x82, 0xE3, 0x0C, 0x8A, 0xF6, 0x50, 0x11, 0xF4, 0xFB, 0x54, 0x64, 0x9D, 0xFD, 0x3D,
 };
static int bign_1_nn_random_belt_hash_test_vector(nn_t out, nn_src_t q)
{
        int ret, cmp;

        /*
         * Fixed ephemeral private key for bign signature
         * test vectors from STB 34.101.45-2013
         */
        const u8 k_buf[] = {
		0xd2, 0xb7, 0x08, 0xa3, 0x7a, 0xa7, 0x33, 0x5c, 0xe1, 0x1c, 0x46, 0x3c, 0x48, 0xec, 0xd6, 0x3e, 0x2c, 0x74, 0xfa, 0xe0, 0xe7, 0x3d, 0xf2, 0x21, 0xad, 0x11, 0x58, 0xcd, 0xb2, 0x74, 0x0e, 0x4c,
        };

        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 bign_1_test_vectors_adata[] = {
	0x00, 0x0b, 0x00, 0x00,
	0x06, 0x09, 0x2A, 0x70, 0x00, 0x02, 0x00, 0x22, 0x65, 0x1F, 0x51,
};

static const ec_test_case bign_1_test_case = {
        .name = "BIGN-BELT-HASH/bign256v1 1",
        .ec_str_p = &bign256v1_str_params,
        .priv_key = bign_1_test_vectors_priv_key,
        .priv_key_len = sizeof(bign_1_test_vectors_priv_key),
        .nn_random = bign_1_nn_random_belt_hash_test_vector,
        .hash_type = BELT_HASH,
        .msg = "\xB1\x94\xBA\xC8\x0A\x08\xF5\x3B\x36\x6D\x00\x8E\x58",
        .msglen = 13,
        .sig_type = BIGN,
        .exp_sig = bign_1_test_vectors_expected_sig,
        .exp_siglen = sizeof(bign_1_test_vectors_expected_sig),
	.adata = bign_1_test_vectors_adata,
	.adata_len = sizeof(bign_1_test_vectors_adata)
};

/********************************************************************/
static const u8 bign_2_test_vectors_priv_key[] = {
	0x69, 0xe2, 0x73, 0xc2, 0x5f, 0x23, 0x79, 0x0c, 0x9e, 0x42, 0x32, 0x07, 0xed, 0x1f, 0x28, 0x34, 0x18, 0xf2, 0x74, 0x9c, 0x32, 0xf0, 0x33, 0x45, 0x67, 0x39, 0x73, 0x4b, 0xb8, 0xb5, 0x66, 0x1f,
 };
static const u8 bign_2_test_vectors_expected_sig[] = {
	0x47, 0xA6, 0x3C, 0x8B, 0x9C, 0x93, 0x6E, 0x94, 0xB5, 0xFA, 0xB3, 0xD9, 0xCB, 0xD7, 0x83, 0x66,
	0x29, 0x0F, 0x32, 0x10, 0xE1, 0x63, 0xEE, 0xC8, 0xDB, 0x4E, 0x92, 0x1E, 0x84, 0x79, 0xD4, 0x13, 0x8F, 0x11, 0x2C, 0xC2, 0x3E, 0x6D, 0xCE, 0x65, 0xEC, 0x5F, 0xF2, 0x1D, 0xF4, 0x23, 0x1C, 0x28,
 };
static int bign_2_nn_random_belt_hash_test_vector(nn_t out, nn_src_t q)
{
        int ret, cmp;

        /*
         * Fixed ephemeral private key for bign signature
         * test vectors from STB 34.101.45-2013
	 * NOTE: the ephemeral private key has been recomputed using the private key as
	 * it is not explicitly provided by the example.
         */
        const u8 k_buf[] = {
		0xec, 0xe9, 0xbc, 0x3b, 0xd9, 0x90, 0x03, 0x0f, 0x4f, 0x28, 0x39, 0x0d, 0xdd, 0x51, 0x22, 0x57, 0x15, 0x44, 0x7f, 0x63, 0x9f, 0x40, 0x19, 0x6f, 0xe4, 0x39, 0x08, 0xbe, 0x98, 0x82, 0x4e, 0x0a,
        };

        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 bign_2_test_vectors_adata[] = {
	0x00, 0x0b, 0x00, 0x00,
	0x06, 0x09, 0x2A, 0x70, 0x00, 0x02, 0x00, 0x22, 0x65, 0x1F, 0x51,
};

static const ec_test_case bign_2_test_case = {
        .name = "BIGN-BELT-HASH/bign256v1 2",
        .ec_str_p = &bign256v1_str_params,
        .priv_key = bign_2_test_vectors_priv_key,
        .priv_key_len = sizeof(bign_2_test_vectors_priv_key),
        .nn_random = bign_2_nn_random_belt_hash_test_vector,
        .hash_type = BELT_HASH,
        .msg = "\xB1\x94\xBA\xC8\x0A\x08\xF5\x3B\x36\x6D\x00\x8E\x58\x4A\x5D\xE4\x85\x04\xFA\x9D\x1B\xB6\xC7\xAC\x25\x2E\x72\xC2\x02\xFD\xCE\x0D\x5B\xE3\xD6\x12\x17\xB9\x61\x81\xFE\x67\x86\xAD\x71\x6B\x89\x0B",
        .msglen = 48,
        .sig_type = BIGN,
        .exp_sig = bign_2_test_vectors_expected_sig,
        .exp_siglen = sizeof(bign_2_test_vectors_expected_sig),
	.adata = bign_2_test_vectors_adata,
	.adata_len = sizeof(bign_2_test_vectors_adata)
};

/************************************************/
static const u8 bign_3_test_vectors_priv_key[] = {
	0x69, 0xe2, 0x73, 0xc2, 0x5f, 0x23, 0x79, 0x0c, 0x9e, 0x42, 0x32, 0x07, 0xed, 0x1f, 0x28, 0x34, 0x18, 0xf2, 0x74, 0x9c, 0x32, 0xf0, 0x33, 0x45, 0x67, 0x39, 0x73, 0x4b, 0xb8, 0xb5, 0x66, 0x1f,
 };
static const u8 bign_3_test_vectors_expected_sig[] = {
	0x19, 0xD3, 0x2B, 0x7E, 0x01, 0xE2, 0x5B, 0xAE, 0x4A, 0x70, 0xEB, 0x6B, 0xCA, 0x42, 0x60, 0x2C,
	0xCA, 0x6A, 0x13, 0x94, 0x44, 0x51, 0xBC, 0xC5, 0xD4, 0xC5, 0x4C, 0xFD, 0x87, 0x37, 0x61, 0x9C, 0x32, 0x8B, 0x8A, 0x58, 0xFB, 0x9C, 0x68, 0xFD, 0x17, 0xD5, 0x69, 0xF7, 0xD0, 0x64, 0x95, 0xFB,
 };
static int bign_3_nn_random_belt_hash_test_vector(nn_t out, nn_src_t q)
{
        int ret, cmp;

        /*
         * Fixed ephemeral private key for bign signature
         */
        const u8 k_buf[] = {
		0x1e, 0x11, 0x97, 0xae, 0x17, 0xe4, 0x5c, 0xa4, 0xa1, 0xb6, 0xfa, 0x53, 0x95, 0x8c, 0xfd, 0x40, 0x64, 0x58, 0x04, 0x40, 0x1a, 0x47, 0xf2, 0xe1, 0xc4, 0xbb, 0x1d, 0x41, 0xd8, 0x14, 0x96, 0x82,
        };

        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 bign_3_test_vectors_adata[] = {
	0x00, 0x0b, 0x00, 0x00,
	0x06, 0x09, 0x2A, 0x70, 0x00, 0x02, 0x00, 0x22, 0x65, 0x1F, 0x51,
};

static const ec_test_case bign_3_test_case = {
        .name = "BIGN-BELT-HASH/bign256v1 3",
        .ec_str_p = &bign256v1_str_params,
        .priv_key = bign_3_test_vectors_priv_key,
        .priv_key_len = sizeof(bign_3_test_vectors_priv_key),
        .nn_random = bign_3_nn_random_belt_hash_test_vector,
        .hash_type = BELT_HASH,
        .msg = "\xB1\x94\xBA\xC8\x0A\x08\xF5\x3B\x36\x6D\x00\x8E\x58",
        .msglen = 13,
        .sig_type = BIGN,
        .exp_sig = bign_3_test_vectors_expected_sig,
        .exp_siglen = sizeof(bign_3_test_vectors_expected_sig),
	.adata = bign_3_test_vectors_adata,
	.adata_len = sizeof(bign_3_test_vectors_adata)
};

/********************************************************************/
static const u8 bign_4_test_vectors_priv_key[] = {
	0x69, 0xe2, 0x73, 0xc2, 0x5f, 0x23, 0x79, 0x0c, 0x9e, 0x42, 0x32, 0x07, 0xed, 0x1f, 0x28, 0x34, 0x18, 0xf2, 0x74, 0x9c, 0x32, 0xf0, 0x33, 0x45, 0x67, 0x39, 0x73, 0x4b, 0xb8, 0xb5, 0x66, 0x1f,
 };
static const u8 bign_4_test_vectors_expected_sig[] = {
	0x47, 0xA6, 0x3C, 0x8B, 0x9C, 0x93, 0x6E, 0x94, 0xB5, 0xFA, 0xB3, 0xD9, 0xCB, 0xD7, 0x83, 0x66,
	0x29, 0x0F, 0x32, 0x10, 0xE1, 0x63, 0xEE, 0xC8, 0xDB, 0x4E, 0x92, 0x1E, 0x84, 0x79, 0xD4, 0x13, 0x8F, 0x11, 0x2C, 0xC2, 0x3E, 0x6D, 0xCE, 0x65, 0xEC, 0x5F, 0xF2, 0x1D, 0xF4, 0x23, 0x1C, 0x28,
 };
static int bign_4_nn_random_belt_hash_test_vector(nn_t out, nn_src_t q)
{
        int ret, cmp;

        /*
         * Fixed ephemeral private key for bign signature
         * test vectors from STB 34.101.45-2013
	 * NOTE: the ephemeral private key has been recomputed using the private key as
	 * it is not explicitly provided by the example.
         */
        const u8 k_buf[] = {
		0xec, 0xe9, 0xbc, 0x3b, 0xd9, 0x90, 0x03, 0x0f, 0x4f, 0x28, 0x39, 0x0d, 0xdd, 0x51, 0x22, 0x57, 0x15, 0x44, 0x7f, 0x63, 0x9f, 0x40, 0x19, 0x6f, 0xe4, 0x39, 0x08, 0xbe, 0x98, 0x82, 0x4e, 0x0a,
        };

        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 bign_4_test_vectors_adata[] = {
	0x00, 0x0b, 0x00, 0x00,
	0x06, 0x09, 0x2A, 0x70, 0x00, 0x02, 0x00, 0x22, 0x65, 0x1F, 0x51,
};

static const ec_test_case bign_4_test_case = {
        .name = "BIGN-BELT-HASH/bign256v1 4",
        .ec_str_p = &bign256v1_str_params,
        .priv_key = bign_4_test_vectors_priv_key,
        .priv_key_len = sizeof(bign_4_test_vectors_priv_key),
        .nn_random = bign_4_nn_random_belt_hash_test_vector,
        .hash_type = BELT_HASH,
        .msg = "\xB1\x94\xBA\xC8\x0A\x08\xF5\x3B\x36\x6D\x00\x8E\x58\x4A\x5D\xE4\x85\x04\xFA\x9D\x1B\xB6\xC7\xAC\x25\x2E\x72\xC2\x02\xFD\xCE\x0D\x5B\xE3\xD6\x12\x17\xB9\x61\x81\xFE\x67\x86\xAD\x71\x6B\x89\x0B",
        .msglen = 48,
        .sig_type = BIGN,
        .exp_sig = bign_4_test_vectors_expected_sig,
        .exp_siglen = sizeof(bign_4_test_vectors_expected_sig),
	.adata = bign_4_test_vectors_adata,
	.adata_len = sizeof(bign_4_test_vectors_adata)
};
#endif

#if defined(WITH_HASH_BASH384) && defined(WITH_CURVE_BIGN384V1)
/********************************************************************/
static const u8 bign_5_test_vectors_priv_key[] = {
	0x0e, 0xb5, 0x93, 0x2d, 0x23, 0x1a, 0xe1, 0x37, 0x61, 0x41, 0x80, 0xec, 0x3b, 0xbe, 0x08, 0x0f, 0x93, 0xba, 0x80, 0x27, 0x8f, 0x05, 0xd8, 0x2f, 0x87, 0x6a, 0xf9, 0xf9, 0xea, 0x0a, 0xf1, 0x9e, 0x6c, 0xa1, 0x6f, 0x21, 0x6c, 0x38, 0x21, 0xdc, 0x72, 0x23, 0x3c, 0x7b, 0xbf, 0x1d, 0xc2, 0x84,
 };
static const u8 bign_5_test_vectors_expected_sig[] = {
	0xA7, 0xFC, 0x9D, 0x62,  0xB6, 0x85, 0x9E, 0xBB,  0x0A, 0x98, 0xAA, 0xE3,  0x6B, 0xE4, 0x79, 0x69,
	0xC3, 0x62, 0xB6, 0x66,  0x95, 0x75, 0x0D, 0xDD,  0x2C, 0xA1, 0x75, 0x32,  0x18, 0x26, 0xD4, 0xBC,
	0x8F, 0x78, 0xEB, 0xF5,  0x55, 0xA8, 0x71, 0x21,  0x3F, 0xEC, 0x6B, 0x50,  0xA6, 0x3D, 0x30, 0xC5,
	0x89, 0x73, 0x3B, 0x0A,  0x56, 0xF6, 0xC0, 0xC2,  0xBC, 0x03, 0xA3, 0x53,  0x29, 0x69, 0xCD, 0xF0,
	0x11, 0xDC, 0x28, 0xD4,  0x84, 0x4E, 0x79, 0xEC,
 };
static int bign_5_nn_random_belt_hash_test_vector(nn_t out, nn_src_t q)
{
        int ret, cmp;

        /*
         * Fixed ephemeral private key for bign signature
         */
        const u8 k_buf[] = {
		0x15, 0xf4, 0x26, 0x4c, 0xa3, 0x51, 0x9a, 0xa6, 0x0a, 0xd8, 0xae, 0x14, 0x79, 0x31, 0x73,
		0x19, 0x80, 0xa8, 0xe1, 0xc3, 0x61, 0x0d, 0xa5, 0xdd, 0xa3, 0x17, 0xea, 0x54, 0xca, 0x73,
		0x3a, 0xa1, 0xe7, 0x1d, 0x1a, 0x5b, 0x0a, 0xc1, 0xae, 0x49, 0xbc, 0xd0, 0x90, 0x02, 0xc1,
		0x9d, 0x3c, 0x19,
        };

        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 bign_5_test_vectors_adata[] = {
	0x00, 0x0b, 0x00, 0x00,
	0x06, 0x09, 0x2A, 0x70, 0x00, 0x02, 0x00, 0x22, 0x65, 0x4d, 0x0c,
};

static const ec_test_case bign_5_test_case = {
        .name = "BIGN-BASH384/bign384v1 5",
        .ec_str_p = &bign384v1_str_params,
        .priv_key = bign_5_test_vectors_priv_key,
        .priv_key_len = sizeof(bign_5_test_vectors_priv_key),
        .nn_random = bign_5_nn_random_belt_hash_test_vector,
        .hash_type = BASH384,
        .msg = "\xb1\x94\xba\xc8\x0a\x08\xf5\x3b\x36\x6d\x00\x8e\x58\x4a\x5d\xe4\x85\x04\xfa\x9d\x1b\xb6\xc7\xac\x25\x2e\x72\xc2\x02\xfd\xce\x0d\x5b\xe3\xd6\x12\x17\xb9\x61\x81\xfe\x67\x86\xad\x71\x6b\x89\x0b\x5c\xb0\xc0\xff\x33\xc3\x56\xb8\x35\xc4\x05\xae\xd8\xe0\x7f\x99\xe1\x2b\xdc\x1a\xe2\x82\x57\xec\x70\x3f\xcc\xf0\x95\xee\x8d\xf1\xc1\xab\x76\x38\x9f\xe6\x78\xca\xf7\xc6\xf8\x60\xd5\xbb\x9c",
        .msglen = 95,
        .sig_type = BIGN,
        .exp_sig = bign_5_test_vectors_expected_sig,
        .exp_siglen = sizeof(bign_5_test_vectors_expected_sig),
	.adata = bign_5_test_vectors_adata,
	.adata_len = sizeof(bign_5_test_vectors_adata)
};
/********************************************************************/
static const u8 bign_6_test_vectors_priv_key[] = {
	0x0e, 0xb5, 0x93, 0x2d, 0x23, 0x1a, 0xe1, 0x37, 0x61, 0x41, 0x80, 0xec, 0x3b, 0xbe, 0x08, 0x0f, 0x93, 0xba, 0x80, 0x27, 0x8f, 0x05, 0xd8, 0x2f, 0x87, 0x6a, 0xf9, 0xf9, 0xea, 0x0a, 0xf1, 0x9e, 0x6c, 0xa1, 0x6f, 0x21, 0x6c, 0x38, 0x21, 0xdc, 0x72, 0x23, 0x3c, 0x7b, 0xbf, 0x1d, 0xc2, 0x84,
 };
static const u8 bign_6_test_vectors_expected_sig[] = {
	0x51, 0xD1, 0x1A, 0xBB,  0x63, 0x92, 0xD9, 0x04,  0x06, 0x85, 0xC4, 0xCC,  0x3A, 0x87, 0x55, 0x3B,
	0xAF, 0x47, 0x44, 0x81,  0x19, 0x86, 0x02, 0xFC,  0xF1, 0x80, 0xDD, 0x0E,  0x0F, 0x00, 0x76, 0xB7,
	0x5A, 0x9B, 0x87, 0x52,  0x69, 0x56, 0x09, 0x30,  0x80, 0xDA, 0x21, 0xAC,  0xFE, 0x73, 0xA7, 0x0E,
	0xEF, 0x4E, 0x5C, 0xEA,  0xE8, 0xC0, 0x7C, 0xDB,  0xA5, 0x26, 0xCF, 0xA3,  0xF6, 0xC5, 0x0D, 0xFD,
	0x4E, 0x8E, 0x88, 0x17,  0xC1, 0xAE, 0x62, 0x4B,
 };
static int bign_6_nn_random_belt_hash_test_vector(nn_t out, nn_src_t q)
{
        int ret, cmp;

        /*
         * Fixed ephemeral private key for bign signature
         */
        const u8 k_buf[] = {
		0x15, 0xf4, 0x26, 0x4c, 0xa3, 0x51, 0x9a, 0xa6, 0x0a, 0xd8, 0xae, 0x14, 0x79, 0x31, 0x73,
		0x19, 0x80, 0xa8, 0xe1, 0xc3, 0x61, 0x0d, 0xa5, 0xdd, 0xa3, 0x17, 0xea, 0x54, 0xca, 0x73,
		0x3a, 0xa1, 0xe7, 0x1d, 0x1a, 0x5b, 0x0a, 0xc1, 0xae, 0x49, 0xbc, 0xd0, 0x90, 0x02, 0xc1,
		0x9d, 0x3c, 0x19,

        };

        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 bign_6_test_vectors_adata[] = {
	0x00, 0x0b, 0x00, 0x00,
	0x06, 0x09, 0x2A, 0x70, 0x00, 0x02, 0x00, 0x22, 0x65, 0x4d, 0x0c,
};

static const ec_test_case bign_6_test_case = {
        .name = "BIGN-BASH384/bign384v1 6",
        .ec_str_p = &bign384v1_str_params,
        .priv_key = bign_6_test_vectors_priv_key,
        .priv_key_len = sizeof(bign_6_test_vectors_priv_key),
        .nn_random = bign_6_nn_random_belt_hash_test_vector,
        .hash_type = BASH384,
        .msg = "\xb1\x94\xba\xc8\x0a\x08\xf5\x3b\x36\x6d\x00\x8e\x58\x4a\x5d\xe4\x85\x04\xfa\x9d\x1b\xb6\xc7\xac\x25\x2e\x72\xc2\x02\xfd\xce\x0d\x5b\xe3\xd6\x12\x17\xb9\x61\x81\xfe\x67\x86\xad\x71\x6b\x89\x0b\x5c\xb0\xc0\xff\x33\xc3\x56\xb8\x35\xc4\x05\xae\xd8\xe0\x7f\x99\xe1\x2b\xdc\x1a\xe2\x82\x57\xec\x70\x3f\xcc\xf0\x95\xee\x8d\xf1\xc1\xab\x76\x38\x9f\xe6\x78\xca\xf7\xc6\xf8\x60\xd5\xbb\x9c\x4f",
        .msglen = 96,
        .sig_type = BIGN,
        .exp_sig = bign_6_test_vectors_expected_sig,
        .exp_siglen = sizeof(bign_6_test_vectors_expected_sig),
	.adata = bign_6_test_vectors_adata,
	.adata_len = sizeof(bign_6_test_vectors_adata)
};

#endif


#if defined(WITH_HASH_BASH512) && defined(WITH_CURVE_BIGN512V1)
/********************************************************************/
static const u8 bign_7_test_vectors_priv_key[] = {
	0xb8, 0xb9, 0x4a, 0xd2, 0xe5, 0x05, 0x02, 0xc5, 0x21, 0xe8, 0x8f, 0x1b, 0xcf, 0x97, 0x0b, 0x1b, 0x47, 0x78, 0x02, 0x31, 0x5b, 0xfe, 0x1f, 0x71, 0xd8, 0x55, 0x71, 0xa2, 0xd1, 0xc1, 0x90, 0x60, 0xba, 0xa2, 0xf1, 0xe0, 0x7d, 0x73, 0xd5, 0x2f, 0x37, 0x31, 0x0b, 0x04, 0x02, 0x3a, 0xbf, 0x33, 0x63, 0xb5, 0xf2, 0x95, 0x2a, 0x62, 0x17, 0xa4, 0x8a, 0x56, 0xf4, 0x3e, 0x35, 0x96, 0xc0, 0xbe,
 };
static const u8 bign_7_test_vectors_expected_sig[] = {
	0xF9, 0x22, 0xD7, 0x01, 0x48, 0xC5, 0x5A, 0xD5, 0x47, 0xEB, 0x0A, 0x19, 0x30, 0xE5, 0x2C, 0xC5, 0x52, 0xD9, 0x28, 0xDC, 0x76, 0xE4, 0x03, 0x0C, 0x9E, 0xE5, 0x7D, 0xE8,
	0xDE, 0xD1, 0xB5, 0xF9, 0x5D, 0x4E, 0x33, 0xF4, 0x7C, 0xD3, 0x61, 0xDA, 0xCB, 0x1F, 0x26, 0xB2, 0x01, 0x32, 0xE4, 0x5D, 0x49, 0x88, 0xFB, 0x86, 0xAF, 0x1B, 0x54, 0x65,
	0x8F, 0xE2, 0xFE, 0x97, 0x11, 0x61, 0x77, 0xF5, 0xD3, 0x11, 0x7C, 0xE0, 0x63, 0x2D, 0xDF, 0xAA, 0x8E, 0xDE, 0x10, 0xDA, 0x7D, 0xF8, 0x35, 0xD2, 0x72, 0x5E, 0x2B, 0x83, 0x4C, 0xA9, 0x6C, 0x8C, 0x06, 0x1B, 0xC8, 0x31, 0xDF, 0xA5, 0xB3, 0x16,
 };
static int bign_7_nn_random_belt_hash_test_vector(nn_t out, nn_src_t q)
{
        int ret, cmp;

        /*
         * Fixed ephemeral private key for bign signature
         */
        const u8 k_buf[] = {
		0x62, 0x64, 0x5d, 0x68, 0xca, 0x6f, 0xb3, 0x52, 0x10, 0xf6, 0xdb, 0x92, 0xe2,
		0xe7, 0x86, 0xee, 0x34, 0x65, 0xdd, 0xc9, 0x62, 0xbe, 0x99, 0x70, 0xfe, 0x1c,
		0x12, 0xe6, 0xe7, 0xca, 0x95, 0xe9, 0x1a, 0xe0, 0x7b, 0x65, 0x9a, 0x8b, 0x7c,
		0x03, 0xf9, 0xb7, 0x2e, 0x64, 0x1f, 0xef, 0xdc, 0x5b, 0x7a, 0xb6, 0xcd, 0x8c,
		0x03, 0x95, 0xe3, 0x80, 0xb9, 0x2c, 0x2f, 0xfd, 0x19, 0xe7, 0x83, 0x2c,
        };

        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 bign_7_test_vectors_adata[] = {
	0x00, 0x0b, 0x00, 0x00,
	0x06, 0x09, 0x2A, 0x70, 0x00, 0x02, 0x00, 0x22, 0x65, 0x4d, 0x0d,
};

static const ec_test_case bign_7_test_case = {
        .name = "BIGN-BASH512/bign512v1 7",
        .ec_str_p = &bign512v1_str_params,
        .priv_key = bign_7_test_vectors_priv_key,
        .priv_key_len = sizeof(bign_7_test_vectors_priv_key),
        .nn_random = bign_7_nn_random_belt_hash_test_vector,
        .hash_type = BASH512,
        .msg = "\xb1\x94\xba\xc8\x0a\x08\xf5\x3b\x36\x6d\x00\x8e\x58\x4a\x5d\xe4\x85\x04\xfa\x9d\x1b\xb6\xc7\xac\x25\x2e\x72\xc2\x02\xfd\xce\x0d\x5b\xe3\xd6\x12\x17\xb9\x61\x81\xfe\x67\x86\xad\x71\x6b\x89\x0b\x5c\xb0\xc0\xff\x33\xc3\x56\xb8\x35\xc4\x05\xae\xd8\xe0\x7f",
        .msglen = 63,
        .sig_type = BIGN,
        .exp_sig = bign_7_test_vectors_expected_sig,
        .exp_siglen = sizeof(bign_7_test_vectors_expected_sig),
	.adata = bign_7_test_vectors_adata,
	.adata_len = sizeof(bign_7_test_vectors_adata)
};

/********************************************************************/
static const u8 bign_8_test_vectors_priv_key[] = {
	0xb8, 0xb9, 0x4a, 0xd2, 0xe5, 0x05, 0x02, 0xc5, 0x21, 0xe8, 0x8f, 0x1b, 0xcf, 0x97, 0x0b, 0x1b, 0x47, 0x78, 0x02, 0x31, 0x5b, 0xfe, 0x1f, 0x71, 0xd8, 0x55, 0x71, 0xa2, 0xd1, 0xc1, 0x90, 0x60, 0xba, 0xa2, 0xf1, 0xe0, 0x7d, 0x73, 0xd5, 0x2f, 0x37, 0x31, 0x0b, 0x04, 0x02, 0x3a, 0xbf, 0x33, 0x63, 0xb5, 0xf2, 0x95, 0x2a, 0x62, 0x17, 0xa4, 0x8a, 0x56, 0xf4, 0x3e, 0x35, 0x96, 0xc0, 0xbe,
 };
static const u8 bign_8_test_vectors_expected_sig[] = {
	0x4B, 0x47, 0x8B, 0x2B,  0x28, 0x79, 0x5A, 0x43,  0x8C, 0x3F, 0x4A, 0x70,  0xD7, 0xF3, 0x02, 0xD3,
	0xD1, 0xB6, 0x15, 0xE9,  0x85, 0xCE, 0x22, 0xDA,  0x71, 0x22, 0xAE, 0x1E,  0xAB, 0x0D, 0xD9, 0x87,
	0x92, 0x39, 0x94, 0x96,  0x8A, 0x24, 0xBF, 0x15,  0xC2, 0xE6, 0x59, 0xB4,  0x54, 0x6F, 0x83, 0xCF,
	0x16, 0x49, 0x33, 0x38,  0x79, 0xD4, 0x79, 0x54,  0xC4, 0xAB, 0x7E, 0x41,  0x04, 0x6E, 0xB3, 0xD9,
	0x27, 0x87, 0xF7, 0x85,  0xC9, 0x12, 0x30, 0xCD,  0x7E, 0x65, 0xE4, 0x55,  0x26, 0xD4, 0x56, 0x50,
	0x92, 0x1D, 0x77, 0x2D,  0xD4, 0x2B, 0xD3, 0x52,  0x2C, 0xF7, 0xBD, 0x5F,  0x7D, 0x79, 0xAB, 0x65,
 };
static int bign_8_nn_random_belt_hash_test_vector(nn_t out, nn_src_t q)
{
        int ret, cmp;

        /*
         * Fixed ephemeral private key for bign signature
         */
        const u8 k_buf[] = {
		0x62, 0x64, 0x5d, 0x68, 0xca, 0x6f, 0xb3, 0x52, 0x10, 0xf6, 0xdb, 0x92, 0xe2,
		0xe7, 0x86, 0xee, 0x34, 0x65, 0xdd, 0xc9, 0x62, 0xbe, 0x99, 0x70, 0xfe, 0x1c,
		0x12, 0xe6, 0xe7, 0xca, 0x95, 0xe9, 0x1a, 0xe0, 0x7b, 0x65, 0x9a, 0x8b, 0x7c,
		0x03, 0xf9, 0xb7, 0x2e, 0x64, 0x1f, 0xef, 0xdc, 0x5b, 0x7a, 0xb6, 0xcd, 0x8c,
		0x03, 0x95, 0xe3, 0x80, 0xb9, 0x2c, 0x2f, 0xfd, 0x19, 0xe7, 0x83, 0x2c,
        };

        ret = nn_init_from_buf(out, k_buf, sizeof(k_buf)); EG(ret, err);
        ret = nn_cmp(out, q, &cmp); EG(ret, err);

        ret = (cmp >= 0) ? -1 : 0;

err:
        return ret;
}
static const u8 bign_8_test_vectors_adata[] = {
	0x00, 0x0b, 0x00, 0x00,
	0x06, 0x09, 0x2A, 0x70, 0x00, 0x02, 0x00, 0x22, 0x65, 0x4d, 0x0d,
};

static const ec_test_case bign_8_test_case = {
        .name = "BIGN-BASH512/bign512v1 8",
        .ec_str_p = &bign512v1_str_params,
        .priv_key = bign_8_test_vectors_priv_key,
        .priv_key_len = sizeof(bign_8_test_vectors_priv_key),
        .nn_random = bign_8_nn_random_belt_hash_test_vector,
        .hash_type = BASH512,
        .msg = "\xb1\x94\xba\xc8\x0a\x08\xf5\x3b\x36\x6d\x00\x8e\x58\x4a\x5d\xe4\x85\x04\xfa\x9d\x1b\xb6\xc7\xac\x25\x2e\x72\xc2\x02\xfd\xce\x0d\x5b\xe3\xd6\x12\x17\xb9\x61\x81\xfe\x67\x86\xad\x71\x6b\x89\x0b\x5c\xb0\xc0\xff\x33\xc3\x56\xb8\x35\xc4\x05\xae\xd8\xe0\x7f\x99",
        .msglen = 64,
        .sig_type = BIGN,
        .exp_sig = bign_8_test_vectors_expected_sig,
        .exp_siglen = sizeof(bign_8_test_vectors_expected_sig),
	.adata = bign_8_test_vectors_adata,
	.adata_len = sizeof(bign_8_test_vectors_adata)
};
#endif

/************************************************/
#define BIGN_ALL_TESTS() \
        &bign_1_test_case, \
        &bign_2_test_case, \
        &bign_3_test_case, \
	&bign_4_test_case, \
	&bign_5_test_case, \
	&bign_6_test_case, \
	&bign_7_test_case, \
	&bign_8_test_case,

#endif /* __BIGN_TEST_VECTORS_H__ */
