/*	$NetBSD: t_cgd_blowfish.c,v 1.2 2017/01/13 21:30:39 christos Exp $	*/
/*-
 * Copyright (c) 2016 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Alexander Nasonov.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/sysctl.h>

#include <atf-c.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <util.h>

#include <dev/cgdvar.h>

#include <rump/rump.h>
#include <rump/rump_syscalls.h>

#include "h_macros.h"

#define SECSIZE 512

struct testvec {
	unsigned int blkno;
	const uint8_t *ptxt;	/* PlainText  */
	const uint8_t *ctxt;	/* CipherText */
};

/*
 * 128 bits Blowfish key, NUL terminated.
 */
static const char bf_cbc_128_key[17] = {
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, /* 01234567 */
	0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, /* 89ABCDEF */
	0
};

/*
 * 256 bits Blowfish key, NUL terminated.
 */
static const char bf_cbc_256_key[33] = {
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, /* 01234567 */
	0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, /* 89ABCDEF */
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, /* 01234567 */
	0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, /* 89abcdef */
	0
};

/*
 * 448 bits Blowfish key, NUL terminated.
 */
static const char bf_cbc_448_key[57] = {
	0x3a, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, /* :ABCDEFG */
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, /* HIJKLMNO */
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, /* PQRSTUVW */
	0x58, 0x59, 0x5a, 0x7e, 0x3a, 0x61, 0x62, 0x63, /* XYZ~:abc */
	0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, /* defghijk */
	0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, /* lmnopqrs */
	0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x23, /* tuvwxyz# */
	0
};

static const uint8_t bf_cbc_ptxt[SECSIZE] =
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop";

/*
 * IV method encblkno1, blkno 0.
 */
static const uint8_t bf_cbc_128_encblkno1_vec0_ctxt[SECSIZE] = {
	0x78, 0x53, 0x43, 0x2a, 0x08, 0xe7, 0x84, 0x3f,
	0xb7, 0x61, 0x9c, 0x17, 0x81, 0xbe, 0x38, 0xb9,
	0x65, 0x51, 0x68, 0xa2, 0x29, 0xd7, 0x45, 0xc9,
	0xee, 0x0e, 0x9d, 0xe1, 0x69, 0xc6, 0x81, 0x81,
	0xf3, 0x93, 0xa6, 0x62, 0xc9, 0x05, 0x2c, 0x1b,
	0x0e, 0x05, 0xca, 0xbe, 0x12, 0x25, 0x37, 0xd8,
	0x98, 0x66, 0xa2, 0xd3, 0xd0, 0x8a, 0x89, 0x57,
	0x44, 0x91, 0x1e, 0xe9, 0x07, 0x03, 0x5c, 0xa6,
	0xb8, 0x30, 0xf1, 0xc7, 0x8c, 0x66, 0x05, 0xb0,
	0x2d, 0xc3, 0xc3, 0xd7, 0x60, 0xef, 0x62, 0xd3,
	0x34, 0x9c, 0xa9, 0xd2, 0x0c, 0x1a, 0x9c, 0xfe,
	0x74, 0x92, 0xcb, 0x90, 0x80, 0xfa, 0x71, 0x5c,
	0xaa, 0x29, 0x39, 0xdd, 0x3b, 0x62, 0xa1, 0xfc,
	0xa5, 0x35, 0xcd, 0xa3, 0x29, 0x41, 0x1a, 0x03,
	0xf7, 0xe1, 0x36, 0xb2, 0xdc, 0x1a, 0xb3, 0x9f,
	0x46, 0xa3, 0xf7, 0xc3, 0xd1, 0x29, 0x83, 0xcf,
	0x0d, 0x88, 0x0b, 0xd1, 0xb7, 0xc7, 0x87, 0x21,
	0xb7, 0x1f, 0xe7, 0xa2, 0x8e, 0x5f, 0xac, 0x6b,
	0x49, 0x9c, 0x93, 0x6b, 0x6b, 0x05, 0x8e, 0x4c,
	0xbd, 0x31, 0x13, 0x5f, 0x4a, 0xd0, 0x35, 0x0c,
	0x67, 0x8f, 0xd0, 0x7a, 0xc9, 0xe3, 0x52, 0x50,
	0x4f, 0x85, 0x09, 0xf1, 0x27, 0xb9, 0xb1, 0x1e,
	0xe4, 0x6a, 0x40, 0xf6, 0x5a, 0x4f, 0x5f, 0xbe,
	0xab, 0xe8, 0xb9, 0xfe, 0xc7, 0x59, 0x6b, 0x0c,
	0xcd, 0x46, 0x4e, 0x90, 0x99, 0xde, 0xf7, 0x43,
	0xee, 0x6e, 0xb6, 0xae, 0xc2, 0x5e, 0x08, 0xbb,
	0xe9, 0x30, 0x2d, 0xb2, 0x91, 0xcc, 0xb9, 0xc7,
	0x58, 0xea, 0x35, 0xae, 0xa2, 0xd8, 0x00, 0xf7,
	0xc0, 0x01, 0xc4, 0x34, 0x2b, 0x34, 0x43, 0xae,
	0xeb, 0x27, 0xbc, 0x5c, 0x91, 0x5f, 0x5f, 0xc1,
	0x61, 0x42, 0x45, 0x68, 0x31, 0xbc, 0xce, 0xb4,
	0x5c, 0xd3, 0x07, 0xdf, 0x4e, 0x65, 0x65, 0x9d,
	0x2e, 0x26, 0x28, 0xfa, 0xcd, 0x53, 0x77, 0x6a,
	0x77, 0xad, 0x96, 0x0b, 0x1f, 0xea, 0x03, 0xc1,
	0xdd, 0xca, 0xe8, 0xfe, 0xe8, 0x36, 0x01, 0x61,
	0x72, 0xbb, 0xed, 0xfd, 0x8d, 0xa3, 0xc2, 0x15,
	0x25, 0x4f, 0xa6, 0x1a, 0x73, 0xbd, 0xcd, 0x45,
	0xdb, 0x08, 0x74, 0x7b, 0xa8, 0x23, 0xf5, 0x74,
	0x3a, 0x18, 0x6d, 0x90, 0xe0, 0xee, 0xae, 0xfe,
	0xc8, 0xac, 0x00, 0x57, 0xa0, 0xe1, 0xfe, 0x10,
	0xd4, 0xf3, 0xa8, 0x00, 0x21, 0x3e, 0x2d, 0xf9,
	0x63, 0xb8, 0xe9, 0xa4, 0x2e, 0xf4, 0x6b, 0xd7,
	0x5c, 0xfd, 0x32, 0x6c, 0x98, 0x05, 0x38, 0x0d,
	0x29, 0xb5, 0x5a, 0x5b, 0xbb, 0xad, 0xfd, 0x46,
	0x9b, 0x6a, 0x97, 0x4c, 0x24, 0xcc, 0x7d, 0x13,
	0x25, 0xe8, 0x2c, 0xb9, 0x13, 0x54, 0xb2, 0x8a,
	0x28, 0xa0, 0x8a, 0x3a, 0x4d, 0x7e, 0xf4, 0x29,
	0xff, 0xfb, 0x4f, 0xd6, 0x3d, 0xf7, 0xca, 0x89,
	0x2a, 0x58, 0x9e, 0x42, 0x00, 0x84, 0x61, 0x58,
	0x7c, 0x94, 0xf6, 0x50, 0x48, 0x2f, 0x34, 0x88,
	0xec, 0x97, 0xef, 0x8b, 0x2f, 0x84, 0xca, 0x23,
	0xe1, 0xb7, 0x63, 0x99, 0xdd, 0x4a, 0x76, 0xdd,
	0x20, 0xc1, 0xc2, 0x56, 0x45, 0xbe, 0x75, 0x9a,
	0x40, 0x72, 0xc8, 0xfb, 0x7e, 0x40, 0x6f, 0x38,
	0xfd, 0x76, 0xa4, 0x78, 0xf5, 0xde, 0x5f, 0xb7,
	0x4a, 0xa9, 0xaf, 0xad, 0xa1, 0x8b, 0x25, 0x8f,
	0xea, 0xb3, 0xeb, 0x54, 0x39, 0x5a, 0x91, 0xfe,
	0x86, 0x18, 0xea, 0x8c, 0xd6, 0x66, 0xd5, 0x85,
	0x02, 0x2b, 0x00, 0x5d, 0x7e, 0x13, 0xa0, 0x1f,
	0x73, 0x46, 0x6d, 0x5e, 0xcd, 0xe0, 0x82, 0x02,
	0x28, 0x88, 0xbf, 0x17, 0xfd, 0x9b, 0x83, 0x2c,
	0xa2, 0xf7, 0xde, 0x51, 0x98, 0x3f, 0xe2, 0x80,
	0x66, 0x14, 0x17, 0xce, 0x8e, 0x30, 0x2d, 0xe2,
	0x24, 0x68, 0x4b, 0xe5, 0xd1, 0x09, 0xfb, 0x6e,
};

/*
 * IV method encblkno1, blkno 1.
 */
static const uint8_t bf_cbc_128_encblkno1_vec1_ctxt[SECSIZE] = {
	0x87, 0xae, 0x01, 0x52, 0xe8, 0xe9, 0xd9, 0xba,
	0xa9, 0x18, 0x31, 0x2c, 0x1b, 0xab, 0x57, 0xad,
	0x45, 0x0e, 0x25, 0x5e, 0x0e, 0x04, 0xfa, 0xdd,
	0xf1, 0x59, 0xe6, 0xea, 0x78, 0x4b, 0x83, 0x07,
	0x8b, 0x46, 0x91, 0x09, 0x58, 0x5a, 0x11, 0x2e,
	0x54, 0x43, 0xa4, 0xc1, 0x04, 0x35, 0xd3, 0x3e,
	0xc9, 0xc8, 0xf5, 0xff, 0x69, 0x1e, 0x57, 0x85,
	0x6d, 0x91, 0x03, 0xeb, 0x8c, 0xa7, 0xe8, 0xcc,
	0x3f, 0xac, 0xf9, 0x14, 0x1e, 0x88, 0x50, 0xa5,
	0x38, 0x66, 0xa9, 0xf4, 0xf5, 0xc6, 0x30, 0x95,
	0xd6, 0x84, 0x0b, 0x81, 0xf9, 0x51, 0x05, 0x80,
	0x9a, 0x89, 0xbf, 0xd4, 0x7d, 0x6a, 0x26, 0x59,
	0x29, 0x44, 0xe7, 0x1d, 0x0e, 0xad, 0x8c, 0xa0,
	0x93, 0xe9, 0x4b, 0x4b, 0x51, 0x46, 0xa5, 0x07,
	0xe8, 0xcb, 0x59, 0xf4, 0x63, 0xb5, 0x36, 0xdb,
	0xbc, 0x54, 0x2d, 0xec, 0xf0, 0x90, 0x3a, 0xa5,
	0xed, 0xc8, 0x28, 0x0e, 0xd3, 0x79, 0xb8, 0x57,
	0xc6, 0x7f, 0x02, 0x22, 0x5e, 0x80, 0xe8, 0x7f,
	0xdf, 0xa0, 0x0f, 0xbc, 0x98, 0x79, 0x6f, 0xd2,
	0xb0, 0xb2, 0x4f, 0x9b, 0x1a, 0x21, 0x8f, 0x63,
	0xce, 0x54, 0x41, 0x64, 0xbf, 0xb9, 0xa7, 0x93,
	0xd6, 0x5b, 0x89, 0x86, 0xda, 0x90, 0x23, 0x2e,
	0x25, 0x35, 0x1a, 0x9a, 0xf5, 0x1e, 0x8f, 0xb4,
	0xe6, 0x66, 0x8e, 0x0e, 0x2d, 0x91, 0x06, 0x4b,
	0x3d, 0x4a, 0x97, 0xab, 0x9b, 0x92, 0x09, 0xaa,
	0x07, 0xbf, 0xc1, 0x7d, 0xe2, 0xbc, 0xd2, 0xf1,
	0x38, 0x8d, 0x02, 0x45, 0xc8, 0x13, 0x12, 0xda,
	0xaa, 0x53, 0xa2, 0x2c, 0x96, 0x69, 0x64, 0xce,
	0x86, 0xe4, 0x84, 0x56, 0xd0, 0xe5, 0x81, 0x99,
	0x27, 0xad, 0x86, 0x47, 0x5f, 0xaf, 0xa2, 0xa2,
	0x90, 0x7f, 0xe7, 0x86, 0xec, 0x7f, 0xf4, 0xa3,
	0xcd, 0x4f, 0x60, 0xcc, 0x1e, 0x35, 0x44, 0xe8,
	0xe9, 0x06, 0xaf, 0x5e, 0x53, 0x3d, 0x30, 0x91,
	0xfe, 0x44, 0x59, 0x66, 0x82, 0xc2, 0xea, 0x9f,
	0xc8, 0x3b, 0xe2, 0xe5, 0x58, 0xf7, 0x34, 0xd5,
	0x9e, 0xfc, 0x20, 0x84, 0x34, 0xaa, 0x4f, 0xe1,
	0xd4, 0x95, 0x76, 0x11, 0x59, 0x90, 0x90, 0xfd,
	0x4d, 0xf9, 0xb8, 0x41, 0xe1, 0xdb, 0x36, 0x05,
	0xe3, 0x0f, 0xa4, 0x4f, 0x0c, 0x61, 0x70, 0xa5,
	0x1b, 0xbf, 0xab, 0x65, 0x67, 0x75, 0x5c, 0x7d,
	0x46, 0x3b, 0x29, 0xd0, 0x3d, 0x06, 0x40, 0x25,
	0x47, 0x3e, 0x8d, 0x62, 0xf8, 0xd8, 0x08, 0xc1,
	0x03, 0x04, 0x4b, 0x5a, 0x40, 0x65, 0x84, 0x52,
	0x34, 0xa2, 0x4a, 0xcc, 0x3a, 0x9c, 0x1e, 0xbf,
	0x2d, 0xed, 0x08, 0x8b, 0xc3, 0x8f, 0x48, 0xba,
	0x06, 0x03, 0xea, 0x5b, 0xba, 0x6a, 0xac, 0x23,
	0x5a, 0x5e, 0x31, 0x08, 0x29, 0x69, 0x64, 0x44,
	0x1c, 0x31, 0xae, 0xb1, 0x86, 0x7b, 0x26, 0x89,
	0xa6, 0xbe, 0xef, 0x69, 0x81, 0xf7, 0x77, 0xd5,
	0x8e, 0x78, 0xa5, 0x11, 0x51, 0xca, 0xec, 0xd0,
	0x86, 0xa5, 0x33, 0xf3, 0x65, 0x5d, 0x04, 0xc5,
	0xd2, 0x17, 0x2a, 0xfe, 0x4a, 0x58, 0x0f, 0x98,
	0x61, 0xad, 0xc3, 0xb8, 0x5b, 0x45, 0xcc, 0x28,
	0x3d, 0x4d, 0x00, 0xf5, 0x4a, 0xe2, 0xbc, 0x6c,
	0x1b, 0x80, 0x7a, 0x2b, 0x40, 0xb8, 0x34, 0x0e,
	0x44, 0x53, 0x16, 0xda, 0x7c, 0x46, 0x8b, 0x42,
	0x5e, 0xa8, 0xe1, 0xb8, 0xf8, 0xcf, 0xff, 0x48,
	0xcf, 0x2c, 0x4c, 0x98, 0xdb, 0xe5, 0x55, 0xfe,
	0x45, 0xfa, 0xf8, 0xde, 0x72, 0xf9, 0x84, 0x3c,
	0xc0, 0x0c, 0x1f, 0x86, 0x97, 0x86, 0xb8, 0xfe,
	0x7d, 0xff, 0xa3, 0xaf, 0x68, 0x00, 0x66, 0x90,
	0xac, 0xb5, 0xd8, 0xde, 0x35, 0x01, 0xf7, 0xab,
	0xab, 0xe3, 0xe9, 0x85, 0x4c, 0x6f, 0xe6, 0xbc,
	0xce, 0x67, 0x4a, 0xbd, 0xad, 0x7b, 0xec, 0xa1,
};

/*
 * IV method encblkno1, blkno 2.
 */
static const uint8_t bf_cbc_128_encblkno1_vec2_ctxt[SECSIZE] = {
	0x17, 0xdd, 0x0f, 0x4b, 0x28, 0x33, 0x03, 0x89,
	0x21, 0x7b, 0x67, 0x15, 0x15, 0x65, 0x08, 0x4f,
	0x65, 0x18, 0xa6, 0x4b, 0x62, 0xdb, 0x1e, 0xc2,
	0xaa, 0x82, 0xb6, 0x1d, 0xf7, 0x12, 0x9e, 0x73,
	0xfe, 0xac, 0x2f, 0x1e, 0x2b, 0xea, 0x3a, 0x4f,
	0xc3, 0x0a, 0x59, 0x80, 0x0d, 0x3d, 0xbc, 0x62,
	0x8d, 0x70, 0xef, 0x1b, 0xfb, 0xdc, 0x4e, 0xc4,
	0x97, 0xf4, 0x77, 0xb7, 0x25, 0x94, 0x13, 0x48,
	0xf2, 0x3d, 0x4c, 0xa7, 0xb8, 0x8c, 0xf5, 0x26,
	0xa4, 0x35, 0xeb, 0xa0, 0xe7, 0x68, 0xb0, 0x69,
	0xf4, 0xf6, 0x13, 0x3a, 0x57, 0xa3, 0xd2, 0x26,
	0xe6, 0x70, 0xd8, 0xd4, 0x05, 0xb5, 0x01, 0xda,
	0xc7, 0x4a, 0x79, 0x1a, 0x6d, 0xb6, 0xf6, 0xb5,
	0x7d, 0x9a, 0x5c, 0xf1, 0x6a, 0xf8, 0xd1, 0x0a,
	0xbc, 0xe7, 0xea, 0xb4, 0x99, 0x72, 0x19, 0x97,
	0x41, 0x4f, 0x14, 0x5f, 0xa3, 0xb3, 0x9b, 0x36,
	0x00, 0x08, 0x88, 0x8c, 0xce, 0x7f, 0x3a, 0x9b,
	0xb0, 0x24, 0x17, 0x95, 0xc4, 0x59, 0x30, 0x5d,
	0xc6, 0x92, 0x19, 0x12, 0x99, 0xb0, 0x08, 0xa6,
	0x04, 0xdb, 0xc6, 0xd5, 0x61, 0xe4, 0xe1, 0x68,
	0xa8, 0xd7, 0x07, 0xfe, 0x2f, 0x47, 0xea, 0x14,
	0xe5, 0xf7, 0x61, 0x9b, 0xbb, 0x98, 0xcb, 0x3b,
	0x8c, 0x41, 0xd1, 0x55, 0x59, 0xb2, 0x41, 0x61,
	0x8e, 0x60, 0x17, 0xcd, 0xe8, 0xf7, 0x1d, 0xbd,
	0x28, 0x5d, 0x1e, 0x15, 0x28, 0x80, 0x8c, 0x29,
	0x34, 0x96, 0x31, 0xda, 0xe1, 0x19, 0x88, 0xd5,
	0xe0, 0xc8, 0xb4, 0xaa, 0x04, 0x21, 0xf5, 0xef,
	0xfa, 0x0e, 0xc9, 0xa5, 0x88, 0x77, 0x49, 0xf4,
	0x02, 0x22, 0x0b, 0x8b, 0x5e, 0xe1, 0xab, 0xd4,
	0xb1, 0xb6, 0x48, 0x54, 0x96, 0x08, 0xaf, 0xa1,
	0x0b, 0xc0, 0xfe, 0x2a, 0x12, 0x36, 0x56, 0x85,
	0x6a, 0xf7, 0x3d, 0x82, 0xe6, 0xda, 0x5d, 0xfe,
	0x4f, 0x4f, 0xc9, 0x43, 0xdc, 0x0f, 0x53, 0x05,
	0x09, 0xd4, 0x9c, 0x2e, 0x6e, 0xf3, 0x52, 0x6a,
	0x10, 0xc6, 0x48, 0xb1, 0x54, 0x70, 0xab, 0x7c,
	0x31, 0xf6, 0x47, 0xef, 0x64, 0x5f, 0xff, 0x45,
	0x8c, 0x3f, 0x87, 0x3a, 0x2d, 0xa6, 0xaf, 0xb2,
	0x44, 0xdf, 0x80, 0x2e, 0x89, 0x4c, 0x94, 0x67,
	0xfc, 0x20, 0x98, 0xb4, 0xcf, 0x58, 0x1e, 0x33,
	0x55, 0x6a, 0x7c, 0x67, 0x5c, 0x28, 0x2f, 0x19,
	0x02, 0x14, 0x06, 0x93, 0x8c, 0x84, 0xae, 0x62,
	0x14, 0xf9, 0x87, 0xae, 0x85, 0xa3, 0x60, 0x26,
	0xfc, 0x8d, 0x04, 0x92, 0x27, 0xfe, 0x35, 0x7b,
	0x45, 0x9d, 0x4a, 0x86, 0x75, 0xa6, 0xb3, 0xa1,
	0x59, 0xe4, 0x4b, 0x1c, 0xd2, 0x71, 0x36, 0xfe,
	0x73, 0xed, 0x54, 0x0d, 0x9d, 0xde, 0x63, 0xb2,
	0xc0, 0x7c, 0xf2, 0xb3, 0x36, 0x62, 0x06, 0x1f,
	0xcd, 0x41, 0x92, 0x73, 0xbc, 0x11, 0x68, 0xc9,
	0x69, 0x20, 0xf9, 0xbb, 0x9a, 0xe9, 0x6c, 0x05,
	0xcf, 0x01, 0x57, 0xc4, 0x1d, 0x95, 0x5e, 0xe3,
	0xb7, 0x15, 0xde, 0xa7, 0xb5, 0x1a, 0x4e, 0x78,
	0x44, 0x5b, 0x9a, 0xee, 0x29, 0xe2, 0x22, 0x8b,
	0xe9, 0xe3, 0xe6, 0x70, 0x3e, 0xcb, 0x9f, 0x7f,
	0xc3, 0xd0, 0x2c, 0xdc, 0x55, 0xb4, 0x0d, 0x67,
	0xf5, 0xd8, 0xff, 0xbb, 0xb1, 0x02, 0xbf, 0xf6,
	0x33, 0x4e, 0x7a, 0x3a, 0x50, 0xb1, 0x01, 0x77,
	0x51, 0xef, 0xb5, 0x75, 0xb3, 0x66, 0xe8, 0xe6,
	0xd6, 0x53, 0x7d, 0x33, 0x51, 0x62, 0x5d, 0xf2,
	0x77, 0x02, 0x34, 0x42, 0xda, 0xee, 0xd9, 0xee,
	0x0b, 0x4d, 0x71, 0x5c, 0xc0, 0xec, 0xdd, 0xc0,
	0x34, 0x6f, 0xf4, 0x65, 0x32, 0xde, 0xc5, 0xb2,
	0x97, 0x60, 0x89, 0x4e, 0x3b, 0x0c, 0xf2, 0xa7,
	0x74, 0x61, 0xd7, 0xe4, 0xa6, 0x80, 0x78, 0x76,
	0xe5, 0x7d, 0xab, 0x96, 0x04, 0x00, 0x76, 0x22,
};

/*
 * IV method encblkno1, blkno 3.
 */
static const uint8_t bf_cbc_128_encblkno1_vec3_ctxt[SECSIZE] = {
	0xdd, 0x8e, 0xce, 0x5b, 0xb8, 0x2a, 0xc8, 0x0e,
	0xd2, 0xbe, 0xcf, 0xa7, 0x2a, 0x5b, 0x0c, 0x1a,
	0xb2, 0x68, 0x5b, 0xe7, 0x53, 0xaf, 0xce, 0x56,
	0xfd, 0xbd, 0x73, 0x3c, 0x44, 0x02, 0x96, 0x57,
	0xaa, 0x47, 0x8d, 0xf1, 0x28, 0x59, 0xb6, 0xce,
	0xba, 0x1e, 0xc9, 0x78, 0x76, 0xdd, 0x43, 0x3a,
	0xbc, 0x43, 0x4c, 0x17, 0xd2, 0xba, 0xb1, 0xda,
	0xa8, 0xbf, 0x32, 0x25, 0xaa, 0xc0, 0xf7, 0xb6,
	0x72, 0x65, 0xe2, 0x67, 0xdb, 0xf0, 0xa8, 0x60,
	0xda, 0x9b, 0x70, 0xad, 0x8f, 0x1d, 0x34, 0x24,
	0x1a, 0xfd, 0x77, 0x2e, 0x1c, 0xb6, 0xc0, 0x6b,
	0xa0, 0x4b, 0x4a, 0xa0, 0xd5, 0x8b, 0xbb, 0xd4,
	0xcc, 0x7b, 0x4e, 0x4c, 0x71, 0x9a, 0x50, 0x12,
	0x36, 0xd4, 0xfd, 0x1f, 0xf1, 0xfc, 0x19, 0x31,
	0xec, 0x54, 0x24, 0xb4, 0x9f, 0xa9, 0xea, 0xd2,
	0x87, 0x11, 0x03, 0x29, 0xbb, 0x20, 0x20, 0x37,
	0xa0, 0xeb, 0x93, 0xa1, 0x60, 0x5f, 0x83, 0x9f,
	0x00, 0x09, 0xe4, 0x9c, 0x79, 0xcb, 0xfc, 0x4f,
	0x9e, 0xd2, 0x76, 0x9f, 0x56, 0x3b, 0x88, 0x1d,
	0x29, 0x8f, 0x36, 0x07, 0xf7, 0x7e, 0xf1, 0xa1,
	0xa4, 0x25, 0xfb, 0xa0, 0xbe, 0xc6, 0xa2, 0x76,
	0xd3, 0x59, 0x2a, 0x7f, 0xb7, 0x9b, 0xb8, 0x75,
	0xc7, 0xc1, 0xc0, 0xe9, 0x9b, 0x83, 0x16, 0x00,
	0xc8, 0x9c, 0x25, 0x2a, 0x8b, 0xd1, 0x8d, 0x16,
	0x9f, 0xd6, 0xd3, 0x03, 0x5b, 0xc7, 0x40, 0xac,
	0xb6, 0xf3, 0xbb, 0x22, 0xa3, 0x3e, 0x56, 0x55,
	0xdf, 0x06, 0x76, 0xe0, 0x7b, 0xd0, 0x52, 0x54,
	0x38, 0xb0, 0xaa, 0xab, 0x62, 0x31, 0xd1, 0x79,
	0x19, 0xec, 0x82, 0x36, 0x58, 0x31, 0xf9, 0x01,
	0xf9, 0x5e, 0xaf, 0x24, 0xb3, 0xc9, 0xb2, 0x30,
	0x3d, 0xbc, 0xf1, 0xbe, 0x17, 0xeb, 0xa0, 0x31,
	0x43, 0xed, 0xd7, 0x50, 0xcc, 0xc2, 0xe2, 0xaa,
	0x68, 0xc8, 0xf0, 0xd3, 0x89, 0xbd, 0xf5, 0x69,
	0x56, 0xe3, 0x88, 0x92, 0x32, 0x56, 0x85, 0x6f,
	0x25, 0x30, 0x28, 0x37, 0xd5, 0xe2, 0xa6, 0xf7,
	0x6e, 0xa9, 0x71, 0xda, 0x4a, 0x25, 0x94, 0x0b,
	0x84, 0x7f, 0x1f, 0x6b, 0x89, 0x2a, 0xf8, 0x30,
	0xcb, 0x60, 0x75, 0x21, 0xbd, 0xe2, 0x34, 0xf7,
	0x8f, 0x30, 0xd5, 0xd5, 0x1f, 0x17, 0x0d, 0x00,
	0x6c, 0x50, 0xde, 0x56, 0x15, 0x33, 0x1b, 0x83,
	0x68, 0x7b, 0x24, 0xe3, 0xa0, 0xda, 0xd5, 0x7a,
	0x3e, 0x93, 0x6d, 0xe0, 0x02, 0x79, 0x62, 0x5d,
	0x71, 0xe3, 0x7b, 0xa9, 0x0b, 0x7a, 0xcd, 0xb3,
	0xb2, 0x6f, 0x96, 0x19, 0x8f, 0xf8, 0x8b, 0x26,
	0x7a, 0x40, 0xc8, 0xae, 0xfe, 0x0d, 0x6f, 0x67,
	0xce, 0x5e, 0xa0, 0x04, 0x7e, 0x93, 0x1d, 0x17,
	0x1c, 0x32, 0x82, 0xf4, 0x54, 0xb9, 0x80, 0xdd,
	0x82, 0xae, 0xf5, 0xc5, 0x1e, 0x15, 0xab, 0xc2,
	0x5c, 0x60, 0xd2, 0x08, 0xc2, 0xa1, 0x1f, 0x89,
	0x0b, 0x59, 0x36, 0x07, 0xdc, 0x57, 0xd3, 0xa0,
	0x32, 0x42, 0xac, 0xa6, 0x90, 0x0b, 0xc0, 0xe4,
	0x91, 0x45, 0x85, 0x27, 0xb9, 0x48, 0x2a, 0x88,
	0x0a, 0xbf, 0xf6, 0x2d, 0xef, 0x4d, 0x1b, 0x64,
	0x49, 0x23, 0x47, 0x30, 0x29, 0x25, 0xb2, 0xc9,
	0xaf, 0xcd, 0xae, 0x56, 0x43, 0x28, 0xcf, 0x81,
	0x95, 0xa7, 0x3e, 0x51, 0x5b, 0x3b, 0xf7, 0x87,
	0x13, 0xc6, 0xee, 0x50, 0x2f, 0x78, 0xdd, 0xcf,
	0x63, 0xef, 0x15, 0xb9, 0x4f, 0x21, 0x27, 0x5e,
	0x94, 0x78, 0xad, 0xcd, 0x9b, 0x3d, 0xf2, 0xdb,
	0xed, 0xf2, 0xa2, 0x39, 0xca, 0xa3, 0xa8, 0x2e,
	0x68, 0xd5, 0xc3, 0xcf, 0x71, 0xec, 0x92, 0xdc,
	0xce, 0xe7, 0x7d, 0x2b, 0xf7, 0xbc, 0xe9, 0x2b,
	0x2e, 0xae, 0xaf, 0x0b, 0x92, 0x72, 0xac, 0x6e,
	0x49, 0xe1, 0xb3, 0x1f, 0xe5, 0x43, 0x2f, 0xa7,
};

const struct testvec bf_cbc_128_1_vectors[] = {
	{
		.blkno = 0,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_128_encblkno1_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_128_encblkno1_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_128_encblkno1_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_128_encblkno1_vec3_ctxt,
	},
};

/*
 * IV method encblkno8, blkno 0.
 */
static const uint8_t bf_cbc_128_encblkno8_vec0_ctxt[SECSIZE] = {
	0xb8, 0x65, 0x67, 0x8e, 0xe4, 0xd8, 0xb4, 0x93,
	0xa5, 0xbb, 0x13, 0x92, 0x27, 0x4b, 0xdd, 0xeb,
	0x0d, 0xad, 0x80, 0x6a, 0x57, 0x37, 0xc0, 0x23,
	0x23, 0xbf, 0xed, 0x86, 0x0c, 0x18, 0x48, 0x19,
	0xcd, 0x84, 0x66, 0xa7, 0xd6, 0xa0, 0x44, 0xd3,
	0x05, 0x4e, 0xf4, 0xfe, 0x6a, 0x57, 0x69, 0x01,
	0xaa, 0x91, 0x9c, 0x6e, 0x4f, 0x79, 0xc9, 0x8f,
	0x4c, 0xdf, 0x5b, 0x9c, 0xc4, 0xf7, 0x63, 0x16,
	0x20, 0x09, 0x07, 0x3f, 0x5e, 0x31, 0xcc, 0x81,
	0x71, 0xe3, 0x7b, 0xb5, 0xea, 0x2c, 0xb5, 0x14,
	0x1e, 0xf9, 0x0d, 0xe0, 0x45, 0xbc, 0x9f, 0x92,
	0x6c, 0xc9, 0x0a, 0x85, 0x62, 0x42, 0xf1, 0x4b,
	0xac, 0xe2, 0xfa, 0xad, 0x97, 0x7a, 0x43, 0x3d,
	0xb6, 0x5f, 0xcb, 0xe7, 0x17, 0x23, 0x28, 0xde,
	0x4e, 0xf8, 0xa1, 0x3c, 0x22, 0x63, 0x49, 0x31,
	0xa7, 0xbe, 0xbf, 0xfe, 0xee, 0xd9, 0x1f, 0xa0,
	0x2a, 0x0e, 0xf2, 0x4f, 0x3e, 0xf8, 0xbb, 0xae,
	0x9e, 0x0d, 0x2c, 0xaa, 0x2a, 0x2c, 0xf0, 0x6c,
	0x37, 0x2a, 0x5d, 0x96, 0x70, 0x9c, 0x87, 0xcc,
	0x2b, 0xca, 0x95, 0x37, 0xf4, 0x4d, 0x78, 0xae,
	0x4f, 0xb5, 0xe6, 0xad, 0xb1, 0xc1, 0x31, 0xd3,
	0x2d, 0xa6, 0xaf, 0xc1, 0x8c, 0xe4, 0x72, 0x05,
	0xb0, 0xfc, 0xb0, 0xf7, 0xfe, 0xf9, 0x3e, 0xa3,
	0xb9, 0xea, 0xc8, 0x69, 0xe3, 0x4e, 0x6d, 0xd1,
	0x8b, 0x2b, 0xf9, 0x2f, 0xd9, 0x40, 0x69, 0xff,
	0x90, 0x98, 0x7a, 0x82, 0xe3, 0x0d, 0x4e, 0x19,
	0x2f, 0x77, 0xf9, 0xab, 0x36, 0xa9, 0x4e, 0xbc,
	0x25, 0x32, 0xbd, 0x44, 0xea, 0x5a, 0x18, 0x31,
	0x37, 0xcd, 0x6c, 0x98, 0xdd, 0x1d, 0xf9, 0xf7,
	0x8f, 0x0b, 0x79, 0xbc, 0xe6, 0xf5, 0xf1, 0xa3,
	0x13, 0xe9, 0x39, 0xaf, 0xa4, 0x8a, 0x74, 0xae,
	0x60, 0x30, 0x63, 0x6e, 0xee, 0x97, 0x83, 0xee,
	0xc0, 0xdd, 0xde, 0xad, 0x92, 0x83, 0xc9, 0x3c,
	0xd8, 0x58, 0x6c, 0xcb, 0xe4, 0x29, 0x04, 0x69,
	0x4f, 0x45, 0xc2, 0x59, 0x98, 0x20, 0x91, 0x6e,
	0x95, 0x82, 0xb3, 0x47, 0x2c, 0xef, 0xdb, 0x96,
	0x38, 0xba, 0x01, 0x89, 0x84, 0x96, 0x71, 0xf9,
	0x2b, 0x23, 0xe0, 0x89, 0xb8, 0xb9, 0x80, 0xbf,
	0x0c, 0xdc, 0xf0, 0x5c, 0xd6, 0x4f, 0x18, 0x19,
	0xfe, 0x23, 0x5a, 0x1e, 0x20, 0x9a, 0x05, 0xf2,
	0x62, 0xd4, 0x04, 0x92, 0x24, 0xfc, 0xc0, 0x48,
	0xf0, 0x00, 0xb4, 0xbe, 0x2e, 0xea, 0x25, 0x17,
	0x5d, 0xab, 0x73, 0x26, 0x79, 0x77, 0xc5, 0x96,
	0xd3, 0xbf, 0x38, 0xda, 0x0f, 0xe1, 0x26, 0x9a,
	0x38, 0xfc, 0x43, 0x82, 0xd1, 0x4d, 0xf2, 0xae,
	0x98, 0x1e, 0xb0, 0x0d, 0xec, 0x7b, 0x56, 0x66,
	0xcb, 0x30, 0x57, 0x4f, 0xe7, 0x03, 0xe3, 0xa6,
	0x4a, 0x4a, 0xf9, 0xa3, 0xbf, 0x44, 0xac, 0x1a,
	0xe7, 0x4b, 0xc1, 0x5b, 0x03, 0x25, 0x4e, 0xc6,
	0x1f, 0x96, 0x4d, 0xf7, 0xbe, 0xa7, 0x5d, 0x60,
	0x20, 0x62, 0x10, 0xd7, 0xab, 0x64, 0xce, 0x22,
	0x8b, 0x52, 0x76, 0xa1, 0xa1, 0x8b, 0x1e, 0xb2,
	0x18, 0x29, 0x8f, 0xc5, 0x24, 0x39, 0xd4, 0xf8,
	0x75, 0x1e, 0x30, 0x57, 0x12, 0x01, 0x04, 0x78,
	0x68, 0x97, 0xa8, 0x65, 0x8c, 0xac, 0xb4, 0x3b,
	0x37, 0x45, 0x41, 0xbc, 0x7d, 0x4b, 0x09, 0xd7,
	0x46, 0x40, 0x99, 0x59, 0xa1, 0xb5, 0x9e, 0x84,
	0x24, 0x6d, 0xfb, 0x74, 0x22, 0xac, 0x4e, 0x5f,
	0x11, 0xd3, 0xa7, 0x9f, 0xa5, 0xca, 0x38, 0x54,
	0xe2, 0x65, 0x52, 0x02, 0x69, 0xe9, 0xa8, 0xf1,
	0xd7, 0x9d, 0x9a, 0x17, 0x54, 0xa0, 0xda, 0xbb,
	0x37, 0xb4, 0x0c, 0xb6, 0x00, 0xad, 0x6f, 0x88,
	0x84, 0xa7, 0x69, 0xd7, 0x0b, 0xbe, 0xb4, 0xbe,
	0x96, 0xbc, 0xcd, 0x08, 0xf1, 0x28, 0xe0, 0x6f,
};

/*
 * IV method encblkno8, blkno 1.
 */
static const uint8_t bf_cbc_128_encblkno8_vec1_ctxt[SECSIZE] = {
	0x7d, 0x95, 0x5a, 0x66, 0x23, 0x98, 0xa8, 0xbe,
	0x53, 0x63, 0x0d, 0x46, 0x4e, 0x38, 0x1b, 0x1d,
	0x36, 0xdd, 0x2a, 0x57, 0x3c, 0x17, 0x01, 0xba,
	0x4e, 0xf8, 0xaa, 0x22, 0x74, 0x05, 0xa2, 0x40,
	0xd0, 0x30, 0x61, 0x6c, 0x65, 0x5e, 0xfb, 0x21,
	0x63, 0xef, 0x62, 0x01, 0x74, 0x15, 0xf6, 0x87,
	0x92, 0xc1, 0x4e, 0x46, 0xdd, 0x76, 0xdb, 0x8b,
	0x20, 0x44, 0xc4, 0xfa, 0x7c, 0xd2, 0x07, 0x32,
	0x11, 0xeb, 0x5b, 0x38, 0x44, 0x32, 0xa1, 0xe7,
	0xcb, 0xa6, 0x1a, 0x12, 0xb9, 0x53, 0x13, 0x6f,
	0xc2, 0x0e, 0x6a, 0x77, 0x8b, 0x96, 0x14, 0x0a,
	0x23, 0x48, 0x65, 0xa5, 0xcd, 0x39, 0x38, 0x03,
	0xc8, 0x3a, 0x98, 0x69, 0x3d, 0x14, 0xae, 0xad,
	0x54, 0x57, 0xf6, 0x5a, 0xdd, 0x46, 0x4c, 0x3a,
	0x68, 0xa8, 0xb7, 0x57, 0xdd, 0x1e, 0x66, 0x0e,
	0xc2, 0x4c, 0x17, 0xba, 0xa4, 0x7e, 0x83, 0x45,
	0xc6, 0xf2, 0x34, 0x3b, 0x4e, 0xab, 0x67, 0x0c,
	0x73, 0xbf, 0x87, 0x7e, 0x93, 0x2b, 0x14, 0x33,
	0xd6, 0x24, 0x8d, 0xc7, 0x90, 0x11, 0xd2, 0x38,
	0xe6, 0xe0, 0x39, 0x1f, 0x00, 0x74, 0x40, 0xab,
	0xdc, 0xdd, 0x46, 0xe8, 0x85, 0x14, 0xb1, 0x78,
	0x34, 0x24, 0x04, 0x97, 0xde, 0xf9, 0x04, 0x69,
	0x0b, 0x15, 0x72, 0x37, 0xf4, 0x0d, 0xf4, 0x76,
	0x6f, 0xd8, 0x05, 0x75, 0x8f, 0x7e, 0x6b, 0xca,
	0x55, 0x20, 0x4a, 0x29, 0x16, 0xc1, 0x6e, 0x91,
	0x22, 0x01, 0x0d, 0x39, 0x5d, 0xb9, 0x09, 0xa4,
	0xe8, 0xc7, 0xff, 0x60, 0x39, 0xc6, 0xe4, 0x2a,
	0x1c, 0xf2, 0x3c, 0xf7, 0xf9, 0xd7, 0xde, 0x0b,
	0x0e, 0x30, 0xf1, 0x20, 0x7c, 0x93, 0x2f, 0x74,
	0x72, 0x40, 0x47, 0x2d, 0xeb, 0x8a, 0x5f, 0x69,
	0x60, 0xdf, 0xe9, 0x4d, 0x06, 0x24, 0x9c, 0x79,
	0xe7, 0x61, 0xd3, 0xa7, 0x57, 0x44, 0x49, 0x97,
	0x3a, 0xa3, 0x11, 0xc4, 0x70, 0xf4, 0x3d, 0xb5,
	0x4b, 0xb7, 0xae, 0x77, 0x36, 0xcf, 0x65, 0x3e,
	0xb6, 0x51, 0x83, 0xcb, 0x43, 0x5f, 0xd0, 0xfb,
	0x69, 0xc4, 0x1b, 0x77, 0x71, 0xcc, 0x72, 0xf4,
	0x5f, 0xc2, 0xda, 0xea, 0xa4, 0x33, 0xec, 0x8e,
	0x92, 0x22, 0x6a, 0x55, 0x34, 0x6a, 0x10, 0xb8,
	0x62, 0x66, 0xc1, 0x6f, 0x65, 0xdd, 0x9a, 0x40,
	0xa0, 0xbf, 0x88, 0xbb, 0x79, 0x1c, 0xa4, 0xaa,
	0xdf, 0xe8, 0xe7, 0x40, 0x88, 0xc6, 0x0f, 0xa2,
	0x2c, 0xee, 0xe7, 0x41, 0x32, 0x46, 0xa2, 0x46,
	0x85, 0xbf, 0x4c, 0xca, 0x4d, 0xd3, 0x9b, 0x49,
	0x43, 0x98, 0xae, 0xfc, 0x93, 0xa7, 0x94, 0x98,
	0x86, 0xa1, 0x0a, 0x85, 0x77, 0x67, 0xa6, 0x16,
	0x94, 0x76, 0xe5, 0x2f, 0x88, 0x5f, 0x24, 0x16,
	0xe5, 0x84, 0x4c, 0xd2, 0x58, 0x59, 0x82, 0x59,
	0x2c, 0xe2, 0x8d, 0xba, 0x08, 0x01, 0x67, 0x1f,
	0x2a, 0x9e, 0x4d, 0x53, 0x57, 0x2d, 0x6e, 0x35,
	0x38, 0xd5, 0x50, 0xa7, 0x0c, 0xe6, 0x77, 0x71,
	0xbe, 0x45, 0x2e, 0xf4, 0x7a, 0x3a, 0x51, 0x03,
	0x04, 0x2b, 0xd7, 0x42, 0x6c, 0x5b, 0x82, 0xba,
	0xb4, 0x09, 0xee, 0x9d, 0xea, 0x8f, 0xf0, 0xb3,
	0xb2, 0x9d, 0x0e, 0x09, 0x72, 0x8c, 0xd9, 0x1e,
	0x6d, 0x78, 0x57, 0x10, 0x1f, 0xeb, 0x4e, 0x53,
	0x57, 0x65, 0xe5, 0x43, 0xe8, 0xb4, 0xb6, 0xb8,
	0x25, 0x8a, 0xe2, 0xb3, 0x99, 0x95, 0x2c, 0xd0,
	0xc7, 0x89, 0xad, 0xdb, 0x72, 0xf0, 0x83, 0xe3,
	0x2f, 0x30, 0x33, 0xf4, 0x03, 0x14, 0x86, 0xa0,
	0xe0, 0x57, 0x15, 0x53, 0x26, 0xd0, 0x6d, 0x12,
	0x51, 0x96, 0x9b, 0x00, 0x8e, 0x41, 0xea, 0x05,
	0x75, 0x5d, 0xb3, 0x8d, 0x44, 0x7f, 0x41, 0x7f,
	0xd1, 0xed, 0x7c, 0xf7, 0xac, 0x6b, 0x21, 0xc7,
	0x0c, 0x49, 0xa1, 0x2e, 0x57, 0xa1, 0x21, 0xe2,
};

/*
 * IV method encblkno8, blkno 2.
 */
static const uint8_t bf_cbc_128_encblkno8_vec2_ctxt[SECSIZE] = {
	0x28, 0x0c, 0x49, 0x08, 0x84, 0xcb, 0xba, 0x4a,
	0xa5, 0xb6, 0x12, 0x4c, 0x14, 0x11, 0x1f, 0x88,
	0x57, 0x78, 0x1e, 0x51, 0x7c, 0x9d, 0xba, 0x31,
	0x80, 0x14, 0xeb, 0x4a, 0x98, 0x8a, 0xb5, 0x4d,
	0xc5, 0xbd, 0xd2, 0x48, 0x1e, 0x19, 0x43, 0x54,
	0x48, 0x1d, 0x24, 0x76, 0x7d, 0xd8, 0xcc, 0xe9,
	0xd9, 0x7b, 0xa9, 0xdf, 0xe3, 0x65, 0x85, 0x10,
	0xb8, 0x11, 0xce, 0xa3, 0x07, 0x4d, 0x87, 0x3f,
	0x95, 0xfb, 0xa5, 0x06, 0xa4, 0x37, 0xb0, 0x89,
	0x03, 0xa9, 0xef, 0x62, 0x98, 0xd3, 0x85, 0xe8,
	0xb3, 0x21, 0xab, 0xe9, 0xdc, 0x03, 0x2a, 0x20,
	0xf7, 0xb1, 0xa4, 0x7a, 0xd5, 0xdc, 0x61, 0x2e,
	0x15, 0x81, 0x3e, 0xcf, 0x8d, 0x8d, 0x54, 0x19,
	0x70, 0xde, 0xa9, 0x57, 0x93, 0x87, 0xc7, 0x16,
	0x06, 0x25, 0xf3, 0x93, 0x8f, 0x73, 0x92, 0x29,
	0x1e, 0xcd, 0x5a, 0x9d, 0x8f, 0x8d, 0x44, 0x15,
	0x8d, 0x92, 0x44, 0x95, 0x7a, 0x5e, 0x1b, 0xfd,
	0x31, 0xa5, 0x8c, 0x22, 0x89, 0xbb, 0x91, 0x15,
	0xad, 0x0a, 0x73, 0x94, 0x65, 0xae, 0xca, 0xfc,
	0x7e, 0xae, 0x85, 0x45, 0xe7, 0xd7, 0x8f, 0x89,
	0x8e, 0x44, 0x62, 0x7a, 0xe0, 0xee, 0x78, 0xbd,
	0x88, 0x62, 0x8e, 0xb8, 0x35, 0x7a, 0xa9, 0x12,
	0x56, 0x2d, 0xe4, 0xbb, 0x2d, 0xc8, 0x98, 0x0e,
	0x35, 0x9e, 0xea, 0x14, 0x43, 0x80, 0xde, 0x9e,
	0x2e, 0xf8, 0xf9, 0xcd, 0x61, 0xa6, 0x22, 0xc8,
	0x77, 0xfc, 0x32, 0x71, 0x4b, 0xcb, 0x00, 0x9b,
	0x9f, 0x85, 0x02, 0x92, 0xfa, 0x84, 0xd8, 0xd8,
	0x09, 0xa4, 0x08, 0xc1, 0x96, 0xd3, 0x9a, 0x27,
	0xa3, 0x82, 0x05, 0xd0, 0x78, 0x18, 0x0a, 0x7d,
	0xb6, 0x95, 0xc0, 0x27, 0x3e, 0x76, 0x77, 0xf3,
	0xd9, 0x62, 0x8e, 0x77, 0xa0, 0x1f, 0x9e, 0x41,
	0xdb, 0x24, 0xaa, 0xdd, 0x8f, 0x94, 0x7b, 0x1f,
	0xff, 0xaa, 0xfd, 0xe2, 0x19, 0xc2, 0x71, 0x80,
	0x0a, 0xda, 0xc5, 0x98, 0x57, 0xde, 0x4e, 0xfb,
	0x38, 0xe6, 0x9b, 0xe5, 0xa6, 0x1f, 0x7d, 0x2c,
	0x41, 0x5f, 0x4d, 0x13, 0xb8, 0x0c, 0xac, 0x9a,
	0x7d, 0xc0, 0x7c, 0x44, 0x64, 0x1e, 0xbc, 0x7d,
	0x16, 0xaa, 0x45, 0xda, 0x62, 0x4e, 0x64, 0x69,
	0xd6, 0x71, 0xd9, 0x64, 0x2a, 0x5d, 0x20, 0x34,
	0xb0, 0xcb, 0x9d, 0xd3, 0x69, 0xd6, 0x60, 0xad,
	0x78, 0x72, 0xb8, 0x36, 0x17, 0xe7, 0xaf, 0x0a,
	0x11, 0x84, 0x43, 0x32, 0x38, 0x43, 0xe5, 0xc5,
	0x1b, 0xf4, 0x48, 0xb6, 0x0e, 0x72, 0x48, 0x2f,
	0x9b, 0xe3, 0xce, 0x27, 0xcd, 0x66, 0x28, 0x5c,
	0x2a, 0xd7, 0x28, 0x52, 0x6e, 0x86, 0x03, 0x60,
	0x7b, 0xbd, 0xbd, 0x53, 0xfb, 0x7d, 0xa1, 0xba,
	0x6a, 0x46, 0x0c, 0xf3, 0x1a, 0xbf, 0xa7, 0xa2,
	0x46, 0x87, 0x40, 0xaa, 0x7d, 0x76, 0x36, 0x85,
	0xa5, 0xbf, 0x0b, 0xd4, 0x56, 0x4c, 0x37, 0xe3,
	0x60, 0x93, 0xdc, 0x3b, 0xca, 0x34, 0x78, 0xcf,
	0xdb, 0x0c, 0x9d, 0x5c, 0x52, 0xb2, 0xd5, 0x7b,
	0xbb, 0x4e, 0xe1, 0xa9, 0x2a, 0xc6, 0x42, 0xf5,
	0x21, 0x9c, 0x15, 0xae, 0xb9, 0x08, 0x3a, 0xc4,
	0x50, 0x7e, 0x0e, 0xb6, 0xc3, 0xfe, 0xf4, 0xd9,
	0x1a, 0x97, 0x30, 0x9f, 0x51, 0x2c, 0xac, 0xd2,
	0x13, 0x93, 0x62, 0x56, 0xcb, 0x34, 0xf3, 0xca,
	0x26, 0xc6, 0x32, 0xbe, 0xf6, 0xd5, 0x1e, 0x5b,
	0x3a, 0x5c, 0x31, 0x08, 0xa1, 0x47, 0x6b, 0x75,
	0x95, 0x8e, 0x3d, 0xbf, 0x2e, 0x81, 0x02, 0x0d,
	0x17, 0x66, 0x6f, 0x04, 0xe0, 0x1e, 0x03, 0x27,
	0xd3, 0xcf, 0x45, 0xc6, 0x08, 0xdb, 0xdf, 0x83,
	0xd5, 0xc2, 0x7f, 0xe8, 0x5f, 0x4a, 0x36, 0x0a,
	0x6d, 0x3c, 0x91, 0x8e, 0x52, 0xf3, 0xdd, 0x62,
	0xff, 0x78, 0x87, 0xd9, 0x4c, 0xad, 0x5c, 0x9f,
};

/*
 * IV method encblkno8, blkno 3.
 */
static const uint8_t bf_cbc_128_encblkno8_vec3_ctxt[SECSIZE] = {
	0xa3, 0x9a, 0x21, 0xbd, 0x1c, 0x97, 0x4f, 0xe2,
	0x7d, 0x63, 0xfe, 0x52, 0x07, 0xac, 0x81, 0x75,
	0x15, 0x04, 0x8c, 0xc7, 0x7c, 0x11, 0x8d, 0x53,
	0x17, 0x87, 0xe8, 0x5d, 0xb1, 0xe8, 0xcb, 0x43,
	0xe2, 0x4b, 0xce, 0x9b, 0xf7, 0x51, 0x0a, 0xee,
	0x0a, 0x49, 0xae, 0x9a, 0xbd, 0x33, 0x3f, 0x0b,
	0xd4, 0xe8, 0x57, 0x77, 0xb1, 0xe1, 0xa5, 0x22,
	0x83, 0xbf, 0x7c, 0x63, 0x6c, 0x3b, 0x25, 0xde,
	0x2c, 0x6a, 0x28, 0x66, 0x0e, 0xc4, 0x8d, 0x66,
	0x66, 0xb6, 0xc6, 0xb5, 0x38, 0x40, 0x84, 0x81,
	0xec, 0x03, 0xdb, 0xbf, 0xe1, 0x8f, 0xf4, 0xb3,
	0xc4, 0x50, 0x24, 0xa2, 0x48, 0x66, 0x73, 0xed,
	0x4b, 0x00, 0x12, 0xd5, 0x15, 0x5f, 0xfb, 0xd9,
	0x6c, 0x76, 0x3b, 0xa2, 0x68, 0x41, 0xc1, 0x96,
	0x50, 0xff, 0x8a, 0x83, 0xac, 0x02, 0x42, 0xcb,
	0xed, 0x93, 0xbf, 0xd9, 0x51, 0x46, 0x50, 0xd1,
	0xeb, 0x02, 0x61, 0x64, 0xa7, 0x0e, 0x95, 0xc2,
	0x95, 0x5a, 0x93, 0xd9, 0x77, 0x17, 0xa4, 0xc7,
	0x51, 0x42, 0xa3, 0xd4, 0x32, 0x4e, 0x4f, 0xe1,
	0xaa, 0x6d, 0xab, 0x08, 0xd9, 0xe9, 0xfe, 0x72,
	0xf3, 0x2c, 0xfb, 0x43, 0xdf, 0x88, 0x44, 0x94,
	0x0b, 0x5c, 0x85, 0x54, 0xe3, 0x13, 0xe2, 0x10,
	0x64, 0xa7, 0xcf, 0xe3, 0x2a, 0x3e, 0xfe, 0xd1,
	0x67, 0xcd, 0xd1, 0x66, 0x06, 0x26, 0x2f, 0x6f,
	0x6f, 0x44, 0xe7, 0xf4, 0xac, 0xe4, 0x58, 0x2f,
	0x61, 0xad, 0x64, 0xc6, 0x0c, 0xf0, 0x9a, 0x3b,
	0x85, 0x1f, 0x3c, 0xea, 0x8e, 0x84, 0xbb, 0x1a,
	0x51, 0x19, 0x3c, 0x6f, 0x5b, 0xf5, 0x6c, 0xb1,
	0x8c, 0x91, 0x25, 0x89, 0x3a, 0x45, 0xb5, 0x35,
	0x13, 0x74, 0xec, 0x68, 0x44, 0xb8, 0xfd, 0xd6,
	0x20, 0x78, 0x7b, 0xe3, 0xe0, 0x82, 0xb7, 0x4a,
	0x38, 0xb6, 0xe4, 0x1b, 0xbf, 0xd9, 0xd3, 0xe9,
	0xbf, 0xd7, 0xdc, 0x8e, 0x90, 0x7d, 0xcb, 0x39,
	0xc4, 0x6c, 0xa4, 0x70, 0x15, 0xf7, 0xa2, 0x16,
	0x04, 0x9b, 0xc0, 0x85, 0x04, 0x1e, 0x69, 0x73,
	0xf3, 0xbd, 0x64, 0x95, 0xfb, 0x21, 0xcb, 0xca,
	0x72, 0xd4, 0x33, 0xe5, 0x11, 0xc3, 0x46, 0xa6,
	0xd2, 0x4c, 0x8a, 0xbb, 0xba, 0x45, 0xcc, 0x45,
	0xdb, 0x8a, 0xec, 0xef, 0x8c, 0x04, 0xcc, 0xeb,
	0x42, 0xad, 0xc3, 0x87, 0xe2, 0x59, 0x3b, 0xed,
	0x2a, 0x11, 0x74, 0xc1, 0x28, 0x8a, 0xc2, 0x17,
	0xca, 0x5a, 0x88, 0xcc, 0x17, 0x2c, 0x03, 0xf6,
	0xcd, 0xd6, 0x92, 0xbd, 0x68, 0x26, 0x41, 0x40,
	0x15, 0x3e, 0x54, 0xf5, 0xb7, 0x4a, 0x82, 0x68,
	0xca, 0x27, 0xed, 0xed, 0x25, 0xd6, 0x0d, 0x0f,
	0x86, 0x62, 0xf0, 0x86, 0x5b, 0xed, 0x94, 0x0b,
	0xd0, 0xec, 0xc7, 0xfd, 0x9c, 0x8a, 0xdf, 0x4f,
	0x65, 0x7e, 0x63, 0x40, 0xeb, 0xe4, 0x79, 0xcb,
	0x67, 0xc6, 0x0e, 0x45, 0xf9, 0xb1, 0x48, 0x27,
	0x16, 0xfc, 0x99, 0x76, 0xac, 0xd0, 0xbc, 0xe6,
	0x9b, 0x29, 0x2d, 0xa5, 0x6c, 0x88, 0x45, 0x7a,
	0x01, 0xf3, 0xe3, 0x15, 0xfb, 0x29, 0xd4, 0x3b,
	0x9a, 0xa8, 0xc6, 0x98, 0x92, 0x19, 0x16, 0xba,
	0xdc, 0x41, 0x70, 0x40, 0x51, 0xfb, 0x7f, 0xb5,
	0xe4, 0x3f, 0x3f, 0x73, 0xb0, 0xb3, 0xd7, 0x6d,
	0x3e, 0x4d, 0x6b, 0x9e, 0x42, 0x8e, 0xbb, 0xd7,
	0xb5, 0x26, 0xa9, 0x19, 0xf5, 0x68, 0xf3, 0x8d,
	0x35, 0x91, 0x06, 0x48, 0xfa, 0x0e, 0x7d, 0xe2,
	0xd3, 0x71, 0x75, 0x44, 0xbd, 0xe6, 0xe6, 0xd6,
	0x36, 0x43, 0x64, 0x3a, 0xd5, 0x97, 0xfa, 0xc0,
	0x10, 0xf7, 0x6c, 0x26, 0xf1, 0xb4, 0xbc, 0xf5,
	0xf6, 0xa3, 0xec, 0x0a, 0xb5, 0x34, 0x55, 0x1a,
	0x67, 0xcb, 0xec, 0x2c, 0x2e, 0x2e, 0x74, 0xed,
	0xfc, 0x85, 0x53, 0x01, 0x87, 0xa7, 0xa0, 0x1f,
};

const struct testvec bf_cbc_128_8_vectors[] = {
	{
		.blkno = 0,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_128_encblkno8_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_128_encblkno8_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_128_encblkno8_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_128_encblkno8_vec3_ctxt,
	},
};

/*
 * IV method encblkno1, blkno 0.
 */
static const uint8_t bf_cbc_256_encblkno1_vec0_ctxt[SECSIZE] = {
	0x14, 0xec, 0xa9, 0xa0, 0x51, 0x9f, 0x5e, 0xb5,
	0x81, 0x6f, 0xa2, 0xbf, 0x5e, 0xf7, 0x91, 0xad,
	0xc5, 0x1a, 0x7e, 0xe6, 0x7a, 0x82, 0x4a, 0xba,
	0x54, 0x60, 0xcb, 0xc3, 0x2f, 0x69, 0x5c, 0xd9,
	0x1e, 0x58, 0xa1, 0x88, 0xa1, 0xe5, 0xa8, 0x52,
	0xdf, 0xf3, 0x8d, 0x5e, 0x2f, 0x81, 0x54, 0xab,
	0x67, 0xb5, 0x05, 0x63, 0x20, 0x10, 0x98, 0xf5,
	0xa5, 0xc3, 0x9e, 0x6d, 0x80, 0x4d, 0xb6, 0x82,
	0x80, 0x5e, 0xb3, 0xc5, 0xd8, 0x77, 0x94, 0xa0,
	0xb8, 0x67, 0xb8, 0x2d, 0x9b, 0x11, 0x3c, 0x24,
	0xbd, 0xb7, 0x0b, 0x1d, 0xeb, 0x1d, 0x6c, 0xab,
	0x3f, 0x8c, 0x91, 0xa0, 0x3a, 0xa6, 0x0c, 0x5a,
	0x88, 0xa0, 0xb5, 0xea, 0x49, 0x58, 0xfb, 0x37,
	0x7c, 0x94, 0xc4, 0x22, 0x35, 0x84, 0xda, 0xd1,
	0x1b, 0x4a, 0x42, 0xa1, 0xd4, 0x90, 0xcd, 0xfb,
	0x77, 0x29, 0xd2, 0xe3, 0x89, 0xec, 0x9e, 0x6a,
	0x4b, 0xbc, 0xc0, 0xfa, 0xb8, 0xdd, 0x5c, 0x2b,
	0xc5, 0x49, 0xb1, 0x6d, 0x6f, 0x2c, 0xb5, 0x50,
	0xd1, 0xd4, 0x9b, 0x15, 0x1c, 0xd7, 0x44, 0xf3,
	0x2e, 0x1f, 0x46, 0xee, 0x38, 0x40, 0xaa, 0x73,
	0xca, 0xf2, 0xc3, 0x83, 0xe2, 0xff, 0xd6, 0xc7,
	0x20, 0xea, 0x70, 0x95, 0x48, 0x58, 0x29, 0x6e,
	0xac, 0x10, 0x75, 0x69, 0x1d, 0xb2, 0x08, 0x3e,
	0x68, 0x43, 0xff, 0x69, 0x1e, 0x88, 0x0a, 0x34,
	0x40, 0xae, 0xb9, 0xf4, 0xb9, 0x3f, 0xa5, 0xd2,
	0xfb, 0xa0, 0xfd, 0x10, 0xa5, 0xbb, 0xd7, 0x22,
	0x8c, 0xd1, 0xf5, 0xc4, 0x11, 0xc6, 0x1e, 0xb5,
	0xfc, 0x90, 0x84, 0xa2, 0x49, 0x38, 0x64, 0x92,
	0x6e, 0xf2, 0xaa, 0xed, 0xe8, 0x9d, 0xac, 0x86,
	0xb7, 0xb3, 0xd9, 0x98, 0x11, 0x8f, 0x51, 0x33,
	0x84, 0x06, 0x40, 0x26, 0x3f, 0xe1, 0xb3, 0x4a,
	0x76, 0x53, 0x68, 0x8b, 0xfe, 0x6f, 0xcd, 0x66,
	0x92, 0x24, 0x42, 0xf4, 0x11, 0x02, 0x01, 0x00,
	0xaa, 0x15, 0x35, 0x42, 0xab, 0x6f, 0x2b, 0x3b,
	0x9a, 0x23, 0x73, 0x18, 0xa8, 0x9b, 0x43, 0x4b,
	0xfb, 0xef, 0x07, 0x75, 0xd7, 0xd6, 0x08, 0x94,
	0xe3, 0x2d, 0xd9, 0xd4, 0x8e, 0x6b, 0x7c, 0xe0,
	0xae, 0xef, 0xcb, 0x5c, 0x46, 0x39, 0x64, 0x34,
	0x48, 0x77, 0x2c, 0x87, 0x68, 0x57, 0xef, 0xba,
	0xd3, 0x3b, 0xb8, 0x68, 0xc5, 0x65, 0x73, 0x44,
	0x0b, 0xef, 0xc7, 0x5e, 0xe6, 0xa2, 0xba, 0x24,
	0x8c, 0x67, 0xa0, 0xf4, 0xef, 0x18, 0x8c, 0x72,
	0x5b, 0x81, 0x8c, 0x81, 0x4f, 0x9a, 0xed, 0x46,
	0x5d, 0x05, 0x9a, 0xdc, 0x01, 0xbe, 0xe8, 0x3f,
	0xb7, 0x5c, 0x8b, 0x2f, 0x92, 0x2c, 0x93, 0x54,
	0x68, 0xfa, 0xd4, 0x27, 0x81, 0xab, 0xa9, 0xfd,
	0x20, 0x21, 0x1b, 0x3a, 0x6e, 0x6b, 0x02, 0x57,
	0x6e, 0xd6, 0x7b, 0x7e, 0x5d, 0x84, 0x47, 0x69,
	0x86, 0x7b, 0x8f, 0x8b, 0xff, 0xb5, 0xcd, 0xc1,
	0x03, 0x18, 0x23, 0x7f, 0x23, 0x2e, 0x3a, 0x48,
	0xe2, 0xf6, 0xb1, 0x78, 0x13, 0x81, 0xbb, 0x80,
	0x91, 0x89, 0x54, 0x7d, 0x1f, 0x1a, 0xd5, 0x35,
	0xad, 0x56, 0x6a, 0x0f, 0xeb, 0x4d, 0x00, 0xdf,
	0xe0, 0xf3, 0x7c, 0xd3, 0x2c, 0x5a, 0x48, 0x39,
	0xa1, 0xc1, 0xfa, 0x34, 0x5f, 0xf9, 0x0b, 0xcd,
	0x1f, 0x21, 0xc6, 0x46, 0xb3, 0xd8, 0x45, 0xc5,
	0x37, 0xf7, 0xd0, 0xda, 0x27, 0x0f, 0xec, 0xec,
	0x05, 0x81, 0x6f, 0x97, 0xca, 0x6d, 0xfa, 0x71,
	0xc9, 0x59, 0x84, 0xc3, 0x0d, 0x55, 0x12, 0xbf,
	0xe1, 0xd2, 0x7c, 0x51, 0x65, 0x8c, 0xc3, 0x8a,
	0x73, 0x2f, 0x1c, 0xd8, 0x13, 0x4a, 0xd1, 0x78,
	0xb2, 0xc8, 0x19, 0x09, 0xce, 0x7b, 0xb6, 0x77,
	0xcc, 0xc3, 0xe6, 0xee, 0x3a, 0x82, 0xf9, 0xc6,
	0x5a, 0x36, 0x46, 0xc0, 0x25, 0xee, 0xaf, 0x78,
};

/*
 * IV method encblkno1, blkno 1.
 */
static const uint8_t bf_cbc_256_encblkno1_vec1_ctxt[SECSIZE] = {
	0x0b, 0xb6, 0x26, 0x92, 0x1d, 0x74, 0xc2, 0x10,
	0xb5, 0x99, 0x5f, 0x62, 0x7f, 0x3b, 0x49, 0x10,
	0xc1, 0x20, 0x9f, 0x38, 0x25, 0x0f, 0x59, 0xde,
	0xe4, 0xc8, 0xb5, 0x27, 0xb1, 0xec, 0x96, 0x40,
	0xe8, 0x05, 0x15, 0x40, 0x96, 0xe0, 0xff, 0xaf,
	0x53, 0x73, 0xa1, 0xa6, 0x73, 0x03, 0xcf, 0x1f,
	0x87, 0x48, 0x7d, 0x81, 0x0e, 0x35, 0x23, 0x7b,
	0xde, 0x12, 0xd8, 0xcd, 0x0a, 0xcb, 0x03, 0xc5,
	0x07, 0xf7, 0x7a, 0x04, 0xf3, 0xda, 0x7d, 0x3b,
	0x73, 0xc6, 0x31, 0xbc, 0x24, 0xde, 0x23, 0x05,
	0x3a, 0xdc, 0xe2, 0x96, 0x85, 0x06, 0xeb, 0x89,
	0xb0, 0x49, 0x3c, 0x79, 0x8e, 0xcf, 0x49, 0x0e,
	0x34, 0x04, 0xa5, 0xcf, 0x45, 0x56, 0xb6, 0xc2,
	0xf1, 0xf1, 0xab, 0x7c, 0x8a, 0xfc, 0xeb, 0xa0,
	0x8a, 0xe6, 0x73, 0xb4, 0xc3, 0x0c, 0x03, 0x5f,
	0x03, 0x53, 0x6f, 0x69, 0xa2, 0xd0, 0xa7, 0x48,
	0xc4, 0x11, 0x88, 0x75, 0xe1, 0xf3, 0xd0, 0x72,
	0x98, 0x6f, 0x84, 0xa6, 0xa8, 0x35, 0xbb, 0xac,
	0x4d, 0xac, 0x55, 0x88, 0x85, 0x86, 0x5a, 0xd9,
	0xb6, 0x57, 0xf9, 0x40, 0xf6, 0x7f, 0x1b, 0x4e,
	0x87, 0xc0, 0x56, 0x8a, 0x2f, 0x3a, 0xe5, 0xa6,
	0x67, 0x68, 0x21, 0x2b, 0xea, 0xfa, 0xee, 0x47,
	0xa0, 0x34, 0x56, 0x7d, 0xa0, 0x3d, 0x58, 0xd7,
	0xff, 0xa2, 0xb6, 0x03, 0x52, 0x16, 0xa5, 0x15,
	0x65, 0xdb, 0xe1, 0x1b, 0xdf, 0x69, 0xb1, 0x48,
	0x6a, 0xdf, 0xc1, 0x00, 0x07, 0xdc, 0x46, 0x4b,
	0x59, 0xcf, 0x15, 0x6b, 0xee, 0x4f, 0x72, 0x77,
	0x6e, 0xbf, 0x47, 0x0e, 0x84, 0x0b, 0xb1, 0xac,
	0x85, 0xce, 0x2b, 0x47, 0x7a, 0xcc, 0x30, 0x0c,
	0x2f, 0x10, 0x27, 0xaa, 0x83, 0x3f, 0x17, 0x39,
	0x84, 0x45, 0x8c, 0xb7, 0x31, 0xb3, 0x7c, 0xcd,
	0xed, 0x86, 0x7d, 0xa9, 0x06, 0x25, 0x1f, 0xe3,
	0x9a, 0x9b, 0x92, 0xdd, 0x07, 0x63, 0x3b, 0x51,
	0x32, 0x2e, 0xae, 0xdf, 0xad, 0xd4, 0x54, 0x5d,
	0x71, 0x36, 0xe9, 0xda, 0x70, 0xe9, 0xec, 0x75,
	0x0b, 0xbb, 0xcc, 0x5d, 0xc5, 0x45, 0x8e, 0x56,
	0x12, 0x87, 0x95, 0x0f, 0x0f, 0x5b, 0x22, 0xc2,
	0xe9, 0x71, 0xf2, 0x7e, 0x7b, 0xc2, 0xce, 0x1f,
	0xb4, 0x43, 0xa5, 0xf1, 0x80, 0x03, 0xd9, 0x44,
	0x3e, 0x97, 0xd6, 0x32, 0x80, 0x99, 0x6b, 0x5b,
	0x25, 0x8b, 0x73, 0x0c, 0x21, 0xda, 0x87, 0x29,
	0x57, 0x1e, 0xa3, 0x1f, 0xc1, 0xb2, 0xd6, 0xa4,
	0x72, 0x64, 0x4a, 0x6b, 0x6f, 0x4d, 0xa8, 0x03,
	0x59, 0x6f, 0xce, 0x8a, 0xd6, 0x1c, 0x63, 0x30,
	0x60, 0xd1, 0x55, 0xc5, 0x44, 0x9a, 0xa8, 0x69,
	0x9f, 0xc7, 0xbe, 0xca, 0x92, 0x83, 0xe9, 0xea,
	0x51, 0x00, 0x5a, 0xdc, 0xbb, 0xbd, 0x5d, 0xf2,
	0x6d, 0x3c, 0x09, 0xde, 0x68, 0x33, 0x5f, 0x5c,
	0x80, 0x8e, 0x22, 0x93, 0x28, 0x5b, 0x77, 0xae,
	0xcd, 0x0d, 0x08, 0xab, 0x94, 0xd6, 0x12, 0x72,
	0x3f, 0xd2, 0xb3, 0xff, 0x87, 0x0a, 0x6f, 0x72,
	0xa7, 0xff, 0xc1, 0xdc, 0x8a, 0x64, 0xdf, 0xeb,
	0x0e, 0x63, 0x71, 0x42, 0x88, 0x2b, 0x13, 0x17,
	0xf2, 0x3b, 0xf9, 0xbb, 0xc9, 0xcc, 0x32, 0x1f,
	0x12, 0x7f, 0xa0, 0x8e, 0x77, 0x31, 0x42, 0x46,
	0x3d, 0xb6, 0xa9, 0x14, 0x6e, 0x02, 0x5a, 0x4f,
	0xf1, 0x5b, 0x91, 0x7e, 0x93, 0xea, 0x94, 0xf1,
	0xcf, 0x0e, 0x10, 0xf8, 0xc2, 0x55, 0x87, 0x68,
	0xf9, 0x49, 0xfa, 0xeb, 0x0f, 0x2c, 0xd7, 0xd8,
	0x26, 0x1a, 0x5b, 0x1a, 0x42, 0x06, 0xea, 0x8a,
	0xb6, 0xec, 0x6e, 0xb0, 0x00, 0xb9, 0x3b, 0x50,
	0xe8, 0x9e, 0xc2, 0x51, 0x4f, 0x03, 0xcd, 0x9f,
	0x36, 0x27, 0xca, 0xa2, 0x98, 0x87, 0x5a, 0xae,
	0xd8, 0x87, 0x76, 0xb6, 0xb6, 0x19, 0x7d, 0x75,
};

/*
 * IV method encblkno1, blkno 2.
 */
static const uint8_t bf_cbc_256_encblkno1_vec2_ctxt[SECSIZE] = {
	0x91, 0x45, 0x4e, 0xe8, 0xad, 0xe3, 0x95, 0x0f,
	0x40, 0x35, 0x21, 0x21, 0x77, 0x62, 0x1d, 0x65,
	0xe4, 0x93, 0x11, 0xd2, 0x20, 0xa4, 0xe4, 0x53,
	0x44, 0xff, 0x60, 0xe9, 0x34, 0xb2, 0x33, 0x87,
	0x3d, 0xb0, 0xd8, 0x37, 0x7e, 0x0e, 0x9a, 0x53,
	0x92, 0xeb, 0xee, 0x16, 0x41, 0x25, 0xe3, 0x80,
	0x0c, 0x53, 0xd8, 0x1f, 0xf0, 0x99, 0xcb, 0x31,
	0xd1, 0x00, 0x82, 0x03, 0xcc, 0xa9, 0x5c, 0x8d,
	0x1a, 0xbb, 0x03, 0x81, 0x80, 0x0d, 0x5d, 0x4a,
	0x96, 0x74, 0x79, 0xf4, 0xa7, 0x46, 0x97, 0x42,
	0x5e, 0xb6, 0x8d, 0xc1, 0x95, 0x1d, 0x98, 0x4d,
	0xe5, 0xe9, 0x70, 0x1a, 0x5a, 0xad, 0xf4, 0x3d,
	0xe1, 0xa7, 0x25, 0xc7, 0xfa, 0x0a, 0x75, 0x98,
	0x2f, 0xef, 0x16, 0x2e, 0xf2, 0x02, 0x8a, 0x4c,
	0x1f, 0x5a, 0xbb, 0x06, 0x1b, 0x4e, 0x50, 0xfb,
	0x03, 0xed, 0x2a, 0x53, 0xdc, 0x2a, 0x65, 0xad,
	0x57, 0x84, 0x48, 0xdb, 0xd2, 0x9b, 0xc0, 0x01,
	0x5f, 0x7d, 0x3e, 0x84, 0xe5, 0x15, 0x7d, 0xc5,
	0x60, 0x4b, 0x18, 0xa1, 0xf3, 0x00, 0x82, 0xd3,
	0x39, 0x2a, 0x1f, 0x8f, 0x6a, 0xb7, 0xeb, 0x76,
	0xfb, 0xf0, 0x5e, 0x66, 0xd8, 0xf1, 0x85, 0xa8,
	0x17, 0xdc, 0x6a, 0xee, 0x53, 0xd9, 0x72, 0x27,
	0xd1, 0x47, 0x73, 0x97, 0x2c, 0xd7, 0xd2, 0xb8,
	0xcd, 0xbe, 0x7b, 0xcc, 0xcc, 0x7d, 0x82, 0x10,
	0x05, 0x5d, 0xff, 0xb0, 0xe0, 0x3a, 0xda, 0x1b,
	0x39, 0x7b, 0x11, 0x30, 0x4f, 0xe7, 0xf9, 0xa6,
	0x43, 0x56, 0x01, 0xe4, 0xed, 0x1a, 0x22, 0x5b,
	0x53, 0x6b, 0x34, 0x58, 0x21, 0x3f, 0x0d, 0xca,
	0x95, 0x24, 0x9a, 0xb0, 0x03, 0xe3, 0x97, 0xf5,
	0x9b, 0xcb, 0x10, 0x6f, 0x1d, 0x8a, 0x8b, 0xaa,
	0x14, 0x0a, 0x89, 0x92, 0xa1, 0x07, 0xb1, 0x35,
	0x40, 0x7f, 0xb0, 0xc3, 0x9a, 0x2a, 0x1f, 0x94,
	0x6c, 0x8f, 0xd8, 0x40, 0x52, 0xec, 0x0e, 0xbf,
	0x91, 0x27, 0xbd, 0x65, 0x25, 0xf2, 0x36, 0xe8,
	0x8f, 0x49, 0x08, 0xa6, 0x8f, 0x82, 0xb3, 0x47,
	0xe9, 0xa4, 0xa6, 0x8e, 0xfb, 0x30, 0xb2, 0x4c,
	0xad, 0x76, 0x65, 0x25, 0xdb, 0x60, 0xa8, 0xeb,
	0xb9, 0xf9, 0x9a, 0x9c, 0x9c, 0x12, 0xab, 0xeb,
	0x4b, 0x96, 0xa5, 0xc3, 0x58, 0x9b, 0x68, 0x2c,
	0x41, 0xac, 0xe5, 0x03, 0xbc, 0xee, 0xb8, 0x05,
	0xf7, 0xe6, 0xb1, 0x07, 0xde, 0x46, 0x28, 0xc1,
	0x2c, 0x15, 0xa2, 0x34, 0xea, 0xe7, 0xc3, 0x36,
	0xe6, 0x18, 0x20, 0x4e, 0x20, 0x3f, 0x32, 0xa8,
	0x29, 0x05, 0xf6, 0xa5, 0xf2, 0xa3, 0xeb, 0x7a,
	0x25, 0x5e, 0x14, 0x1f, 0xd0, 0xe1, 0x8e, 0xfb,
	0x28, 0xc5, 0xa2, 0x42, 0xed, 0x4c, 0x12, 0x15,
	0x2a, 0x08, 0xfb, 0x0b, 0xfb, 0x94, 0x64, 0xc0,
	0x8b, 0xbb, 0xbb, 0x2c, 0xef, 0xab, 0x0b, 0x4c,
	0x27, 0x40, 0x94, 0x3e, 0x93, 0x77, 0x98, 0xcc,
	0x64, 0xe3, 0xba, 0x22, 0x95, 0xd7, 0xc1, 0xe3,
	0xa7, 0xcd, 0xf9, 0x25, 0xdc, 0xc4, 0xd2, 0xee,
	0x5b, 0x53, 0x72, 0x59, 0x8b, 0xea, 0xbf, 0xde,
	0x2b, 0x35, 0xd5, 0x27, 0x57, 0x2e, 0x13, 0xa7,
	0x50, 0x2d, 0xa5, 0xd5, 0x43, 0x0b, 0x49, 0x87,
	0xd0, 0xbd, 0xdd, 0xec, 0x4b, 0xd1, 0x8b, 0xf6,
	0xf6, 0xd0, 0x97, 0xcb, 0x8d, 0x58, 0x35, 0x27,
	0xa5, 0x7e, 0x4a, 0xda, 0x93, 0xa4, 0x1e, 0x39,
	0x53, 0x59, 0x87, 0xfe, 0x82, 0x09, 0xda, 0x03,
	0x33, 0xcf, 0x94, 0x60, 0xb1, 0x0c, 0xa1, 0x0e,
	0xd6, 0xaa, 0xb0, 0x09, 0x96, 0x8b, 0x72, 0x15,
	0xfb, 0xb0, 0x7d, 0x06, 0xf5, 0x2d, 0x64, 0xcd,
	0x03, 0xf0, 0xfa, 0xed, 0x6f, 0x43, 0xe3, 0xf3,
	0x33, 0xaf, 0x65, 0x82, 0x1d, 0xad, 0x03, 0x62,
	0xbe, 0x12, 0x14, 0x85, 0x66, 0x45, 0x03, 0x79,
};

/*
 * IV method encblkno1, blkno 3.
 */
static const uint8_t bf_cbc_256_encblkno1_vec3_ctxt[SECSIZE] = {
	0x65, 0xbb, 0x30, 0x13, 0x9a, 0x26, 0xe9, 0x3d,
	0x54, 0x28, 0x77, 0xf1, 0x3f, 0x9b, 0xe1, 0x3e,
	0x70, 0x90, 0x6f, 0x16, 0xfc, 0x2f, 0x27, 0xb3,
	0xc6, 0x3c, 0x38, 0x31, 0x11, 0xe1, 0x3b, 0x72,
	0x22, 0x1c, 0x01, 0xc5, 0xa6, 0x70, 0x16, 0x4d,
	0xd2, 0xbd, 0xcb, 0x15, 0x00, 0x22, 0xab, 0x65,
	0x6a, 0x96, 0x5e, 0x45, 0x7f, 0xfd, 0x76, 0x18,
	0x43, 0x25, 0x75, 0x73, 0xa4, 0x24, 0xe4, 0x5b,
	0xca, 0xf7, 0x6f, 0xfe, 0xc6, 0x4c, 0x81, 0x77,
	0xe5, 0x1b, 0xb4, 0x7b, 0x64, 0xc2, 0x0d, 0x2b,
	0xa9, 0x76, 0xae, 0x5d, 0xab, 0x81, 0x77, 0xa6,
	0x48, 0xe9, 0x0c, 0x6d, 0x93, 0xbd, 0x7e, 0xdc,
	0x39, 0x89, 0x72, 0xd2, 0x07, 0x87, 0x44, 0xa6,
	0x7d, 0x08, 0x54, 0xc6, 0x90, 0x1d, 0xa2, 0xd7,
	0xfd, 0xd5, 0x59, 0x67, 0xd9, 0x5f, 0x5c, 0xbc,
	0x60, 0xc7, 0xf6, 0x98, 0xad, 0x21, 0xdf, 0xde,
	0x27, 0xca, 0x73, 0x2f, 0x56, 0xb7, 0xd7, 0x54,
	0x6e, 0xc3, 0x09, 0x6f, 0x1b, 0x82, 0x6f, 0xf8,
	0x1b, 0xb2, 0x54, 0x47, 0x54, 0x55, 0x67, 0x1c,
	0x4a, 0x10, 0x44, 0xb8, 0x8e, 0x85, 0x71, 0x70,
	0x02, 0x49, 0xdd, 0x4f, 0xd4, 0xd4, 0x8a, 0x60,
	0x04, 0x17, 0x84, 0xbf, 0xb4, 0x0a, 0x6f, 0xc1,
	0xec, 0x1f, 0x5d, 0x79, 0x59, 0x15, 0x7b, 0x87,
	0xa2, 0xfe, 0x81, 0xa1, 0x0b, 0x74, 0xfa, 0xef,
	0xce, 0x96, 0xec, 0x4b, 0xd8, 0x8e, 0xe9, 0xc2,
	0x6b, 0x15, 0xd9, 0xe4, 0x1d, 0x81, 0xb2, 0x12,
	0xb9, 0x8b, 0x64, 0x3d, 0xfa, 0xf6, 0x29, 0x25,
	0x88, 0x4f, 0xfa, 0x56, 0x34, 0x85, 0xa6, 0xbe,
	0xf7, 0x9f, 0x54, 0xc4, 0xb7, 0x17, 0xd5, 0x00,
	0x2d, 0x06, 0xca, 0xf0, 0xec, 0xf9, 0x52, 0x62,
	0x12, 0xef, 0xc0, 0x57, 0xd1, 0xf3, 0xf2, 0xb1,
	0x3d, 0xc5, 0x69, 0x04, 0x95, 0xaf, 0xc6, 0x54,
	0x18, 0x08, 0x2f, 0xe2, 0xc2, 0xdb, 0x28, 0x63,
	0x7c, 0xf5, 0xba, 0xa4, 0xdf, 0xbd, 0xdd, 0xac,
	0x98, 0xec, 0x9e, 0x07, 0x48, 0xee, 0xb9, 0x6f,
	0x40, 0xba, 0x08, 0xd5, 0x74, 0x97, 0x34, 0x98,
	0x7a, 0x80, 0xc5, 0x78, 0x69, 0x11, 0xd9, 0xcb,
	0x3b, 0x6f, 0xe7, 0xb7, 0x78, 0xb0, 0x5e, 0x02,
	0xaf, 0x6c, 0xef, 0x36, 0x00, 0xca, 0x97, 0x1a,
	0x01, 0x2e, 0xe8, 0xc0, 0x8b, 0xc6, 0x78, 0xf4,
	0x2d, 0x60, 0x2c, 0x04, 0x3f, 0x0b, 0xca, 0x7e,
	0xf1, 0x2e, 0x67, 0x8f, 0x9d, 0xa7, 0xaa, 0xab,
	0xcf, 0xb3, 0x84, 0x9e, 0x14, 0x35, 0x15, 0x3b,
	0x88, 0x9a, 0x33, 0x5d, 0x68, 0x82, 0x29, 0x53,
	0x94, 0x18, 0x0d, 0x14, 0x9e, 0x5f, 0xc1, 0x32,
	0x0a, 0x95, 0x6e, 0xa3, 0x82, 0x4b, 0x58, 0x0f,
	0x9c, 0xf0, 0x26, 0x4b, 0x2f, 0x02, 0x60, 0x85,
	0xdd, 0x2c, 0xb8, 0x87, 0x8b, 0x14, 0x9c, 0x54,
	0x0a, 0x5a, 0x02, 0xbe, 0xe2, 0x71, 0xcc, 0x07,
	0xae, 0x67, 0x00, 0xa8, 0xd4, 0x09, 0x7c, 0xee,
	0x0d, 0x29, 0x17, 0x67, 0x96, 0x68, 0x41, 0xfa,
	0x72, 0x29, 0x98, 0x2b, 0x23, 0xd1, 0xa9, 0x89,
	0x1c, 0xcc, 0xaf, 0x88, 0xdb, 0xb5, 0x1e, 0xb1,
	0xae, 0x17, 0x5f, 0x29, 0x8d, 0x1c, 0x0a, 0x5c,
	0xb9, 0xa7, 0x59, 0x8b, 0x91, 0x41, 0xee, 0x89,
	0xe1, 0x0e, 0x7e, 0x0a, 0xee, 0xbc, 0x35, 0xab,
	0xf1, 0x5a, 0x58, 0x03, 0xa2, 0xcf, 0x33, 0xa3,
	0x74, 0x82, 0xd3, 0xa0, 0x32, 0xfc, 0x3b, 0x9c,
	0xdf, 0xc0, 0x3a, 0x76, 0xe1, 0xea, 0xf0, 0x6d,
	0xc8, 0xe7, 0x97, 0xec, 0x03, 0xc1, 0x72, 0x94,
	0xe5, 0xc4, 0x04, 0x2a, 0x38, 0xb4, 0xef, 0x47,
	0x1d, 0xf9, 0xb8, 0x0a, 0xa9, 0x45, 0xc1, 0x63,
	0xf8, 0x32, 0xdb, 0x5d, 0xb1, 0xa2, 0x80, 0x8c,
	0x23, 0xd3, 0x60, 0xfb, 0xf8, 0x84, 0x57, 0x8b,
};

const struct testvec bf_cbc_256_1_vectors[] = {
	{
		.blkno = 0,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_256_encblkno1_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_256_encblkno1_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_256_encblkno1_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_256_encblkno1_vec3_ctxt,
	},
};

/*
 * IV method encblkno8, blkno 0.
 */
static const uint8_t bf_cbc_256_encblkno8_vec0_ctxt[SECSIZE] = {
	0x90, 0x3d, 0xd2, 0xc0, 0xd3, 0x75, 0xe8, 0x04,
	0x34, 0x66, 0x76, 0x25, 0x70, 0xde, 0x41, 0xf1,
	0x35, 0x63, 0x5c, 0xac, 0x8f, 0x28, 0x17, 0xd3,
	0xae, 0x43, 0xfd, 0x8a, 0xb8, 0xc2, 0xd3, 0x88,
	0xef, 0xfa, 0x21, 0xeb, 0xb4, 0x33, 0x75, 0x69,
	0x7d, 0xe9, 0x27, 0x1c, 0x03, 0xcf, 0x90, 0xcf,
	0xf0, 0xaa, 0x3e, 0x01, 0x41, 0x24, 0xdc, 0x01,
	0x5a, 0xe8, 0x92, 0xea, 0xde, 0x87, 0xbf, 0x75,
	0xe4, 0x0b, 0xf7, 0xa3, 0x42, 0x27, 0xc4, 0xde,
	0x01, 0x6f, 0x5c, 0xec, 0x6d, 0x37, 0x38, 0xd7,
	0x4f, 0x85, 0xc0, 0xc2, 0x3a, 0xa7, 0x2d, 0x15,
	0xd5, 0x1c, 0xee, 0x15, 0x13, 0x7a, 0x4e, 0x33,
	0xc7, 0x59, 0x93, 0x73, 0xe9, 0xf5, 0x39, 0xb0,
	0xad, 0x8b, 0x61, 0xca, 0x4d, 0x31, 0xaa, 0x44,
	0xf6, 0x5c, 0xb7, 0x99, 0xe5, 0x92, 0x3d, 0x47,
	0x79, 0x9b, 0x29, 0x16, 0xe1, 0x2c, 0x30, 0x8b,
	0x1e, 0x17, 0xf0, 0x91, 0x59, 0x00, 0xab, 0x6d,
	0x4d, 0xa7, 0x4d, 0x96, 0xc3, 0xe4, 0x3d, 0x17,
	0x5c, 0x8e, 0xfc, 0x59, 0x48, 0xd4, 0xdd, 0xf4,
	0xea, 0x3a, 0x68, 0xc8, 0xb1, 0x74, 0x69, 0xcb,
	0x49, 0x1a, 0xec, 0x24, 0x4e, 0x7e, 0xe9, 0xba,
	0x75, 0xfb, 0x52, 0x85, 0x75, 0xe9, 0x41, 0x9a,
	0xc6, 0x40, 0x9c, 0x92, 0x3f, 0xe8, 0x99, 0x81,
	0x84, 0x14, 0x2b, 0x62, 0x94, 0xba, 0x56, 0x00,
	0xde, 0x88, 0xe1, 0x35, 0xea, 0x25, 0x88, 0xd1,
	0xce, 0xd4, 0xfc, 0xde, 0xee, 0x1a, 0xfd, 0xb1,
	0xa7, 0x46, 0x9d, 0x0c, 0x99, 0xa6, 0xab, 0x23,
	0x55, 0x2f, 0x46, 0xd7, 0xb3, 0xcd, 0x2c, 0xab,
	0x93, 0x3e, 0xdb, 0xe2, 0x34, 0x69, 0x1a, 0x56,
	0x21, 0x92, 0x56, 0xf4, 0x05, 0xe1, 0x24, 0xee,
	0x4d, 0x5e, 0x89, 0xeb, 0x23, 0x26, 0xdc, 0x14,
	0xde, 0x56, 0x3f, 0x0c, 0x15, 0x3e, 0x42, 0x71,
	0x9b, 0xe8, 0xc5, 0xfd, 0x5e, 0x4a, 0xb6, 0xd9,
	0xa0, 0x18, 0x5d, 0xbd, 0xef, 0x80, 0xb4, 0xf6,
	0x1e, 0x56, 0x2a, 0x57, 0x13, 0xba, 0x24, 0xa4,
	0x6f, 0x4c, 0xd4, 0xaa, 0x31, 0x5e, 0x69, 0x26,
	0xd2, 0xee, 0xef, 0x7f, 0x83, 0x9c, 0x8e, 0x6a,
	0x17, 0xe3, 0xda, 0xf4, 0x59, 0xad, 0x26, 0x83,
	0x53, 0x4c, 0x0d, 0x62, 0xe5, 0x9a, 0x30, 0xc5,
	0x0a, 0xa8, 0xb1, 0x3b, 0x1b, 0x41, 0x5c, 0x74,
	0x42, 0x7b, 0x0c, 0x9e, 0x3c, 0x12, 0x04, 0x46,
	0xc3, 0xc7, 0x10, 0xea, 0xf1, 0x3b, 0xb1, 0x01,
	0xfe, 0x1b, 0xe8, 0xf8, 0x42, 0xbc, 0xe9, 0x11,
	0x9b, 0x63, 0x29, 0x99, 0x18, 0x79, 0x9e, 0xd2,
	0xbf, 0x9d, 0x93, 0x4d, 0x16, 0x0f, 0x14, 0x41,
	0xb6, 0x0c, 0xa5, 0x07, 0x13, 0x29, 0x21, 0x0d,
	0x8f, 0xf9, 0x0a, 0x0a, 0x54, 0xaf, 0xa6, 0x22,
	0x25, 0x79, 0x07, 0xe5, 0x3e, 0x49, 0x6b, 0x12,
	0x9e, 0xfc, 0x91, 0xdb, 0xf3, 0x7c, 0xdf, 0x03,
	0x9c, 0x78, 0xa7, 0xc3, 0x5e, 0x14, 0xde, 0xb5,
	0x6a, 0x7b, 0x3b, 0xe3, 0x4f, 0x8a, 0x49, 0xce,
	0xc9, 0x14, 0x29, 0x96, 0x84, 0xca, 0xe1, 0x49,
	0x41, 0x73, 0xb3, 0x2a, 0xbe, 0x37, 0xb6, 0x2d,
	0xff, 0xf2, 0x8e, 0x3d, 0x02, 0xeb, 0xd4, 0xd3,
	0x15, 0x8f, 0xc0, 0x00, 0x91, 0xd5, 0xe7, 0x76,
	0xf5, 0x6e, 0x81, 0x38, 0x38, 0x07, 0xa6, 0xe8,
	0x72, 0x14, 0x3b, 0x36, 0xef, 0xbc, 0x5b, 0x26,
	0xb0, 0x60, 0x25, 0x49, 0x7e, 0xfc, 0xd8, 0x3b,
	0x63, 0xdc, 0x7f, 0x80, 0xd5, 0x43, 0x78, 0xbb,
	0xf1, 0xf9, 0x3e, 0x75, 0x1d, 0x58, 0xb2, 0xc7,
	0xb6, 0x52, 0xfb, 0xe7, 0x42, 0xef, 0x87, 0xfd,
	0x3a, 0x02, 0x7a, 0xf9, 0xbc, 0xa8, 0x2f, 0xd6,
	0xc1, 0x5f, 0xa4, 0x57, 0x62, 0x83, 0x82, 0x8e,
	0x1e, 0xbb, 0x85, 0xf7, 0x1b, 0x2e, 0xe2, 0xb0,
};

/*
 * IV method encblkno8, blkno 1.
 */
static const uint8_t bf_cbc_256_encblkno8_vec1_ctxt[SECSIZE] = {
	0x2b, 0xf6, 0xd4, 0x61, 0x3b, 0x1f, 0x5b, 0xe9,
	0x32, 0xab, 0x27, 0xa1, 0xaf, 0x5e, 0xf4, 0xa7,
	0xaa, 0xc5, 0x2a, 0x3c, 0x0d, 0x34, 0xd8, 0xb3,
	0xfd, 0xb2, 0xca, 0xfb, 0x3c, 0x38, 0x96, 0x71,
	0x47, 0xaa, 0xa7, 0x2f, 0x48, 0x2c, 0x39, 0x88,
	0x2d, 0xc6, 0xb1, 0xf7, 0xc7, 0x2d, 0xda, 0xe9,
	0xfb, 0x4f, 0x9b, 0x1f, 0xe1, 0x0c, 0x24, 0x89,
	0xfe, 0x70, 0xe0, 0xb2, 0x51, 0x89, 0x51, 0xa9,
	0xae, 0xd1, 0x92, 0x4f, 0x56, 0x5c, 0x2a, 0xf4,
	0xbc, 0x4e, 0x77, 0x4a, 0xb8, 0xab, 0x02, 0x76,
	0xef, 0x69, 0xfb, 0x5e, 0x06, 0xb5, 0xff, 0x31,
	0xce, 0x2b, 0xfc, 0x48, 0x4c, 0x82, 0xe9, 0x3c,
	0x61, 0x69, 0x68, 0x1d, 0xb1, 0xc6, 0x40, 0x10,
	0xd7, 0x0e, 0xd2, 0x26, 0x33, 0x5b, 0x0b, 0xe7,
	0xc2, 0xbe, 0xf4, 0x24, 0x1a, 0xa6, 0x70, 0x31,
	0xa7, 0x15, 0x76, 0xc7, 0x90, 0x8d, 0x60, 0xe0,
	0xee, 0x5b, 0x73, 0xa9, 0xe1, 0xe1, 0xaf, 0xf0,
	0x5e, 0x6f, 0x32, 0x98, 0x92, 0xbe, 0x24, 0x81,
	0x26, 0x9f, 0xb8, 0x67, 0xd0, 0xca, 0x9d, 0x8f,
	0x14, 0xc8, 0x81, 0x2e, 0x57, 0x1c, 0x3a, 0xe0,
	0xdb, 0x49, 0xad, 0x47, 0x51, 0x07, 0x7d, 0xec,
	0xbc, 0xa5, 0x8f, 0xdf, 0x84, 0xe4, 0xdf, 0x76,
	0x8f, 0x0b, 0xef, 0xc4, 0x41, 0xd5, 0x7c, 0xf5,
	0x3c, 0x21, 0x62, 0xc0, 0x1f, 0xbd, 0x39, 0xbe,
	0xe5, 0x75, 0x64, 0xcd, 0xa4, 0xa0, 0x03, 0xf4,
	0x8a, 0x16, 0x3e, 0xde, 0x79, 0x9a, 0x96, 0xff,
	0xf2, 0xbe, 0x88, 0xfd, 0xac, 0xc1, 0x9d, 0x5b,
	0xbf, 0x2f, 0xde, 0xf0, 0x26, 0x2f, 0xc9, 0x45,
	0xbd, 0x26, 0xa5, 0x2c, 0x3c, 0x12, 0x8b, 0xc0,
	0xc8, 0x7a, 0x71, 0xbb, 0xc3, 0xe9, 0xf6, 0x15,
	0x01, 0x6e, 0x94, 0x37, 0xbc, 0xc5, 0x9a, 0x93,
	0x6f, 0x9c, 0x04, 0x7e, 0xe9, 0xb2, 0xba, 0xe8,
	0x86, 0xa1, 0x9c, 0x1f, 0x4b, 0x77, 0x6f, 0x99,
	0x2d, 0x8f, 0x23, 0x34, 0x32, 0x1a, 0x82, 0x2d,
	0x32, 0x41, 0x3e, 0xb8, 0x6a, 0x67, 0xa9, 0x81,
	0xd5, 0x1b, 0x76, 0x9f, 0xd1, 0xb0, 0x06, 0xaf,
	0x10, 0x9e, 0x00, 0x2e, 0xb3, 0x80, 0xde, 0xae,
	0xf2, 0x96, 0x12, 0x5b, 0xe7, 0xc7, 0x25, 0xb7,
	0xd9, 0x1c, 0x04, 0xe5, 0x05, 0xaf, 0x77, 0xfa,
	0x6d, 0xa8, 0x04, 0x74, 0xf8, 0x9c, 0x09, 0xcf,
	0xe2, 0xc2, 0xd1, 0xb6, 0xac, 0xed, 0xb4, 0xbc,
	0x2e, 0xc2, 0xf6, 0x3c, 0xc4, 0x47, 0xc8, 0x81,
	0x3a, 0x50, 0x3c, 0x5c, 0x7c, 0x86, 0x17, 0x22,
	0xe7, 0xa3, 0xff, 0x73, 0x5e, 0x91, 0xbf, 0xb3,
	0x59, 0x07, 0xb7, 0xa4, 0xd5, 0x1b, 0x5c, 0xce,
	0x56, 0xde, 0x5f, 0xae, 0x89, 0xcb, 0x6a, 0xfe,
	0xaf, 0xe7, 0xd4, 0x34, 0x8d, 0x18, 0x22, 0x4d,
	0xd9, 0x91, 0xa6, 0xec, 0x97, 0x0e, 0x29, 0x4d,
	0xf6, 0xd8, 0xb3, 0x50, 0x1c, 0xc9, 0x66, 0x9d,
	0x2e, 0x5e, 0x27, 0xce, 0x36, 0xcb, 0x47, 0x35,
	0x41, 0x16, 0x0c, 0x4e, 0x73, 0x90, 0x52, 0xc5,
	0x65, 0xb8, 0x0c, 0xdc, 0x36, 0x8d, 0xdc, 0xca,
	0x97, 0x0b, 0xbc, 0xcb, 0x79, 0xc7, 0x4c, 0xd2,
	0x21, 0x5c, 0xbd, 0xeb, 0xea, 0xfb, 0x87, 0xe1,
	0xe0, 0x75, 0x39, 0xb9, 0x84, 0x1e, 0xa7, 0xfe,
	0x7d, 0x41, 0x75, 0x15, 0x88, 0x98, 0xd4, 0x80,
	0x42, 0x57, 0xb5, 0x65, 0xbf, 0xb8, 0xbd, 0x19,
	0x28, 0xd8, 0xa7, 0x6c, 0xe7, 0xc1, 0x00, 0xdc,
	0xde, 0xcb, 0x30, 0x3d, 0x29, 0x5e, 0xa6, 0x9c,
	0xbb, 0xb8, 0xec, 0x28, 0x23, 0x36, 0x23, 0x27,
	0xee, 0xdd, 0x24, 0x7d, 0x9a, 0xc9, 0xb5, 0x3c,
	0x7a, 0x3f, 0x1d, 0xd9, 0x32, 0x47, 0xc0, 0x4d,
	0x86, 0x9b, 0x2d, 0xa9, 0x5c, 0x93, 0x90, 0x51,
	0x70, 0xe6, 0x8f, 0x35, 0x96, 0xe0, 0x11, 0x00,
};

/*
 * IV method encblkno8, blkno 2.
 */
static const uint8_t bf_cbc_256_encblkno8_vec2_ctxt[SECSIZE] = {
	0x28, 0x93, 0x0f, 0x23, 0xfb, 0xa4, 0x9e, 0xe5,
	0x11, 0x38, 0x2b, 0xbd, 0x8d, 0x2d, 0xdb, 0x11,
	0xfa, 0xac, 0x74, 0x28, 0x95, 0x29, 0xf2, 0x92,
	0x8d, 0x00, 0x8a, 0x8a, 0x04, 0x92, 0x92, 0x33,
	0x8c, 0x4b, 0x29, 0x8e, 0xde, 0x59, 0xf4, 0x72,
	0xae, 0x2f, 0xe4, 0xe9, 0xd4, 0xe4, 0xb6, 0x69,
	0xc5, 0x1b, 0xbe, 0x02, 0x85, 0x4b, 0x24, 0x1e,
	0x79, 0xb9, 0x22, 0x29, 0x4c, 0x60, 0x8c, 0xc5,
	0x03, 0x1d, 0x79, 0xfe, 0x3d, 0x9e, 0x47, 0xb6,
	0xf7, 0x17, 0x65, 0x24, 0x15, 0x5d, 0x75, 0x4d,
	0xdb, 0xbc, 0x68, 0x67, 0x3b, 0xc3, 0x5a, 0x7e,
	0x9d, 0x98, 0x67, 0xe4, 0x8f, 0x9a, 0xd1, 0x31,
	0xe0, 0x26, 0xa1, 0x68, 0xbe, 0x53, 0x73, 0x7c,
	0xfd, 0xce, 0x6c, 0xd6, 0x1f, 0x51, 0xe6, 0x84,
	0x54, 0x7a, 0xe3, 0x7f, 0x7c, 0x8f, 0x2f, 0x88,
	0x58, 0xfe, 0x5e, 0x82, 0x11, 0xc9, 0xa5, 0x89,
	0xa4, 0x49, 0x92, 0x21, 0x0f, 0x03, 0xdb, 0x16,
	0xc4, 0xc0, 0x80, 0xb7, 0x16, 0x4c, 0x29, 0xbe,
	0x18, 0xfa, 0x2d, 0xdf, 0x4a, 0x23, 0x34, 0x9a,
	0x27, 0xea, 0xed, 0x95, 0x25, 0x14, 0xa8, 0x2e,
	0x17, 0x59, 0x04, 0xb0, 0x5c, 0x6d, 0xc7, 0xeb,
	0xed, 0xf6, 0x73, 0xae, 0x18, 0x0e, 0x4b, 0xec,
	0xc6, 0xb7, 0x39, 0xe7, 0x62, 0xf0, 0x84, 0x30,
	0x10, 0xb6, 0xf3, 0x27, 0x6d, 0xfe, 0x32, 0xe7,
	0xfe, 0xff, 0x43, 0xba, 0x89, 0xfe, 0x24, 0xa8,
	0x0e, 0x7c, 0xf2, 0x23, 0x9d, 0x66, 0x6f, 0x9c,
	0xe6, 0x88, 0xbc, 0x3f, 0x44, 0x4c, 0x73, 0x13,
	0x77, 0x95, 0x6f, 0xcb, 0xc8, 0xa5, 0x7a, 0xa9,
	0xeb, 0xe1, 0x0f, 0x9e, 0x25, 0xbe, 0x99, 0x1a,
	0x99, 0x7f, 0xbb, 0xec, 0x89, 0x91, 0x3e, 0x52,
	0xb5, 0xac, 0xc9, 0xd3, 0xea, 0xb0, 0xf2, 0x0c,
	0xc8, 0x58, 0x4b, 0x93, 0xa3, 0x9f, 0xad, 0x5a,
	0x80, 0x4e, 0x02, 0x20, 0x9d, 0xac, 0x4b, 0xe0,
	0x59, 0x4d, 0xb5, 0x51, 0x07, 0xf5, 0xa6, 0xb3,
	0xc9, 0x20, 0x58, 0x7e, 0x45, 0xec, 0x58, 0xea,
	0x49, 0xbb, 0x03, 0xf5, 0x6c, 0xdd, 0xcc, 0xa3,
	0x13, 0x21, 0x79, 0xc9, 0xc2, 0x92, 0x60, 0xd5,
	0xb2, 0x3b, 0x74, 0xbc, 0x57, 0xa6, 0x70, 0x36,
	0x75, 0xf5, 0x01, 0xd1, 0xb2, 0xe9, 0xfd, 0xc5,
	0x93, 0x5e, 0x60, 0x6b, 0xfd, 0xd2, 0x56, 0xc0,
	0x1f, 0xe4, 0xcd, 0x4c, 0xfa, 0xc4, 0xd8, 0xc2,
	0x2d, 0xf9, 0x9f, 0x82, 0x0f, 0x40, 0x7b, 0xad,
	0x35, 0x63, 0x95, 0x7e, 0x49, 0x4a, 0xfe, 0x8f,
	0xaa, 0x57, 0x3a, 0x0c, 0x59, 0x69, 0xe7, 0xb8,
	0xfc, 0x71, 0x5c, 0x4f, 0x18, 0x12, 0xe5, 0xef,
	0xdb, 0x1f, 0x66, 0x9a, 0xe2, 0x1a, 0x92, 0x1f,
	0xfe, 0x20, 0x81, 0xe5, 0x83, 0x97, 0xfb, 0xaf,
	0xeb, 0x31, 0x6c, 0x81, 0xf2, 0x2f, 0xf4, 0x41,
	0xf1, 0xd9, 0x61, 0xfb, 0x36, 0x4e, 0xab, 0xc5,
	0x8b, 0x9c, 0x37, 0xea, 0x88, 0xeb, 0x1e, 0x4c,
	0x84, 0x1a, 0xac, 0x4c, 0x19, 0x39, 0x51, 0x53,
	0xe0, 0x50, 0xb0, 0xdf, 0xe6, 0xc5, 0xbb, 0x80,
	0x06, 0x30, 0x14, 0xf5, 0x0e, 0x73, 0xd2, 0xdb,
	0x19, 0x45, 0x30, 0xdc, 0xd0, 0x4d, 0xe6, 0xd6,
	0x0e, 0x2d, 0x77, 0xa3, 0xb3, 0x27, 0xda, 0x99,
	0x62, 0x88, 0x35, 0xba, 0x64, 0x15, 0xec, 0xaf,
	0x70, 0x97, 0x94, 0x81, 0x30, 0x6d, 0x63, 0x42,
	0x71, 0x3e, 0x06, 0xec, 0x50, 0x96, 0x87, 0x59,
	0xe7, 0x26, 0x9c, 0xcf, 0xc7, 0xe7, 0x62, 0x82,
	0x3b, 0xd7, 0xfe, 0xb4, 0x48, 0x45, 0x9d, 0x54,
	0x18, 0x15, 0x13, 0x74, 0x92, 0x6f, 0x43, 0xb3,
	0xa9, 0x82, 0xd4, 0xc2, 0xef, 0x61, 0x9d, 0x5e,
	0x1d, 0xc6, 0x80, 0xd3, 0xe9, 0xdd, 0x52, 0x9c,
	0x4d, 0x04, 0x05, 0xa0, 0x43, 0x36, 0xb6, 0x89,
};

/*
 * IV method encblkno8, blkno 3.
 */
static const uint8_t bf_cbc_256_encblkno8_vec3_ctxt[SECSIZE] = {
	0x60, 0x91, 0x19, 0x89, 0xee, 0xac, 0x12, 0xe5,
	0x60, 0x6c, 0xfd, 0xf9, 0xe4, 0xc6, 0xc9, 0xbd,
	0x75, 0xe1, 0xa5, 0xe6, 0xf4, 0xed, 0x4b, 0xf4,
	0x16, 0xf4, 0x57, 0xa4, 0xe6, 0x60, 0x8e, 0x1b,
	0x97, 0x0e, 0xd5, 0x09, 0xa1, 0x3c, 0x6c, 0xe8,
	0x91, 0xaf, 0x19, 0x96, 0x2a, 0x7b, 0x77, 0x30,
	0xc7, 0x99, 0xe6, 0xaa, 0xb0, 0xe2, 0x29, 0x1c,
	0x39, 0x54, 0x1c, 0x0a, 0x4a, 0x51, 0xa2, 0xa2,
	0x22, 0x96, 0x05, 0x8b, 0x12, 0x80, 0x16, 0x28,
	0xdc, 0xfa, 0x22, 0x90, 0xa4, 0x33, 0xb1, 0x84,
	0x13, 0x52, 0x5d, 0xb5, 0xd4, 0xe8, 0x60, 0x18,
	0x6a, 0xb8, 0x1b, 0xdb, 0xb4, 0x69, 0xf6, 0x09,
	0x95, 0x71, 0xdd, 0x43, 0x21, 0xad, 0x7e, 0xf7,
	0x8e, 0x7a, 0x0d, 0xf0, 0x52, 0x54, 0xb8, 0xdc,
	0x7d, 0x72, 0x29, 0x97, 0x2b, 0x9c, 0x2c, 0xef,
	0xc3, 0x26, 0x68, 0x72, 0xae, 0x29, 0x0f, 0x67,
	0xbf, 0xea, 0x92, 0x27, 0xd1, 0xba, 0x8d, 0x32,
	0x8b, 0x26, 0x91, 0x30, 0x88, 0xcc, 0x47, 0xaf,
	0x54, 0x8c, 0x88, 0x88, 0x2f, 0x59, 0x76, 0x34,
	0x43, 0x35, 0x44, 0xc3, 0x16, 0x28, 0x62, 0x4b,
	0xbb, 0x47, 0x99, 0x7c, 0x26, 0x51, 0xe2, 0x7d,
	0xd8, 0x2c, 0x35, 0xf4, 0x24, 0xf1, 0x5b, 0x01,
	0xcc, 0x9a, 0x54, 0xd8, 0xc1, 0x73, 0x85, 0x83,
	0xdd, 0x0d, 0xd5, 0x75, 0xac, 0x67, 0x68, 0x59,
	0x3e, 0x6e, 0x9a, 0x4a, 0x7f, 0xbd, 0x85, 0xeb,
	0x36, 0x3e, 0xfd, 0x03, 0xfe, 0x2b, 0xe6, 0x97,
	0x16, 0x6b, 0xd0, 0x22, 0xb1, 0x00, 0xcc, 0x7c,
	0x03, 0xb1, 0x7c, 0x23, 0x7a, 0xca, 0x5e, 0x0b,
	0xba, 0x37, 0xa6, 0x08, 0x5b, 0xa6, 0x2b, 0x57,
	0x58, 0x0b, 0x5a, 0x58, 0x91, 0x3c, 0xf9, 0x46,
	0x05, 0x03, 0x0a, 0x9b, 0xca, 0x2d, 0x71, 0xe2,
	0xbb, 0x1e, 0xd3, 0xc5, 0xc2, 0xb4, 0xde, 0x7b,
	0xbb, 0x8b, 0x45, 0x39, 0xf5, 0x3d, 0xa2, 0xe5,
	0xb1, 0x40, 0x3b, 0x9e, 0x47, 0x93, 0xf9, 0x9c,
	0x50, 0x5c, 0x9b, 0x8d, 0x18, 0x47, 0xd3, 0xe8,
	0x61, 0xbc, 0x93, 0xdc, 0xf7, 0x20, 0x5a, 0x00,
	0x0e, 0xb8, 0xee, 0x5e, 0x83, 0x06, 0x48, 0x06,
	0x91, 0x08, 0x9e, 0x9c, 0x73, 0x6d, 0xb9, 0x31,
	0x62, 0xdc, 0x8a, 0x37, 0x17, 0x47, 0x2f, 0x0f,
	0xc0, 0x02, 0x02, 0xf3, 0x06, 0x26, 0x6c, 0x9d,
	0x96, 0x9f, 0xb0, 0xb3, 0x3b, 0x72, 0x18, 0x59,
	0xf4, 0xb7, 0x26, 0xcc, 0xa4, 0x46, 0xdb, 0x51,
	0xad, 0xed, 0xd8, 0x3a, 0xc4, 0x3a, 0x09, 0x30,
	0x72, 0xd9, 0x2c, 0xfe, 0x5f, 0xa8, 0x46, 0x75,
	0xf7, 0xba, 0x46, 0x1e, 0x7e, 0x4c, 0xd6, 0xdd,
	0x92, 0x2b, 0x23, 0xc6, 0x59, 0x19, 0xda, 0x9a,
	0x01, 0x9c, 0x5c, 0xc1, 0xaa, 0xcf, 0x6d, 0xd0,
	0xa5, 0x06, 0xc7, 0x5e, 0x6b, 0x60, 0x64, 0x9f,
	0xfe, 0xa8, 0x3f, 0x64, 0xa8, 0xed, 0xf8, 0x62,
	0xd7, 0x6d, 0x34, 0x41, 0x3e, 0x5e, 0x74, 0xc7,
	0xe6, 0x62, 0xb1, 0x5a, 0xec, 0x6a, 0xc1, 0x71,
	0x19, 0xf3, 0xf1, 0xe7, 0x46, 0x13, 0xd6, 0xb6,
	0x5a, 0xf4, 0xca, 0x3f, 0xe3, 0xa1, 0x1f, 0xe4,
	0xda, 0xd6, 0x0c, 0x62, 0x6a, 0x33, 0x42, 0x99,
	0x6f, 0x5d, 0x3a, 0xe0, 0xe7, 0xfa, 0x2d, 0x47,
	0x4a, 0xec, 0xaa, 0x71, 0xb5, 0xeb, 0x62, 0xb8,
	0x31, 0x34, 0x07, 0x44, 0xa2, 0x18, 0xec, 0x76,
	0xf7, 0x77, 0x56, 0x86, 0xc7, 0xe0, 0x1e, 0x8e,
	0xec, 0x16, 0x2b, 0xeb, 0xff, 0xaa, 0xba, 0x83,
	0x1f, 0xdc, 0x32, 0x23, 0x27, 0xea, 0xea, 0x0c,
	0x3a, 0x5f, 0x2b, 0xb4, 0xee, 0x0c, 0xf2, 0x73,
	0xbb, 0x59, 0x9b, 0x73, 0xf7, 0xfa, 0xe1, 0x1f,
	0x3b, 0xdb, 0x40, 0x29, 0xf0, 0x6c, 0xbe, 0x8f,
	0x2e, 0xd0, 0x83, 0xf7, 0xe8, 0x2a, 0x81, 0x82,
};

const struct testvec bf_cbc_256_8_vectors[] = {
	{
		.blkno = 0,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_256_encblkno8_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_256_encblkno8_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_256_encblkno8_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_256_encblkno8_vec3_ctxt,
	},
};

/*
 * IV method encblkno1, blkno 0.
 */
static const uint8_t bf_cbc_448_encblkno1_vec0_ctxt[SECSIZE] = {
	0xb0, 0x1b, 0x43, 0xc9, 0x84, 0x53, 0x51, 0x72,
	0x1f, 0x93, 0x62, 0x94, 0x13, 0x1f, 0xe3, 0xc1,
	0x30, 0xee, 0xc4, 0x35, 0x98, 0xb8, 0x11, 0x9b,
	0xd1, 0x23, 0xfa, 0x2d, 0xc1, 0xe6, 0xf3, 0x8f,
	0x8b, 0x05, 0x8d, 0xc5, 0x1d, 0xc3, 0x29, 0x1a,
	0xff, 0x08, 0xb0, 0x97, 0x88, 0x50, 0x8f, 0x41,
	0x66, 0xc6, 0x70, 0x37, 0xca, 0xb2, 0xcd, 0x91,
	0x89, 0x41, 0x1b, 0x42, 0xce, 0x02, 0xec, 0xe8,
	0x9b, 0xf3, 0x50, 0x95, 0x5f, 0xba, 0xda, 0xd5,
	0x0c, 0x2f, 0x29, 0x2a, 0x14, 0x96, 0x4a, 0x60,
	0x04, 0x43, 0x62, 0x80, 0x89, 0xcf, 0xfd, 0xb6,
	0xc7, 0x2d, 0xb2, 0xac, 0xce, 0x51, 0xf5, 0xd1,
	0xcd, 0x41, 0x59, 0x8b, 0xf5, 0x02, 0x2d, 0xe8,
	0xc4, 0xee, 0xe9, 0x40, 0x6f, 0xcd, 0xbe, 0x3b,
	0xd0, 0x8d, 0x3e, 0x6c, 0x42, 0x2a, 0x17, 0xfd,
	0x78, 0xf6, 0xb2, 0xde, 0x69, 0x81, 0xbb, 0xae,
	0xd9, 0x1f, 0x49, 0xa6, 0x4c, 0x5a, 0xe9, 0x94,
	0x49, 0x9b, 0x7e, 0x32, 0x6d, 0x11, 0xea, 0x88,
	0xda, 0xf0, 0xc7, 0x8d, 0x07, 0x98, 0xab, 0xc7,
	0xf3, 0xcf, 0x2e, 0xda, 0x27, 0x44, 0x68, 0xc7,
	0xdc, 0xda, 0x00, 0xd0, 0x6d, 0x64, 0x5b, 0x39,
	0x46, 0x3d, 0x98, 0x24, 0xee, 0x3b, 0x36, 0x23,
	0x62, 0xf6, 0x8e, 0xb9, 0x8d, 0xc1, 0x9a, 0x7b,
	0xd7, 0x03, 0x6b, 0xb7, 0x81, 0x19, 0xec, 0x2f,
	0x0c, 0x0b, 0x32, 0x5a, 0xb3, 0x25, 0xf5, 0xcc,
	0xa3, 0x60, 0xe6, 0x4e, 0x03, 0xcd, 0xdc, 0x67,
	0xe5, 0x26, 0xda, 0xe0, 0x1f, 0x33, 0x99, 0xc3,
	0x43, 0x8c, 0x9c, 0x1a, 0x85, 0xb1, 0x15, 0x04,
	0xc1, 0x06, 0xd1, 0x2d, 0xc9, 0x67, 0x72, 0xe7,
	0xd6, 0x6d, 0x1f, 0x22, 0x56, 0x56, 0xfa, 0x8a,
	0xd7, 0x16, 0x37, 0x3f, 0x3e, 0x67, 0xa5, 0xb7,
	0xee, 0x3e, 0xd2, 0x38, 0xd5, 0xce, 0xa9, 0x62,
	0x82, 0x17, 0xae, 0xae, 0x62, 0xe6, 0xb7, 0xf2,
	0x73, 0xf1, 0xc1, 0xb7, 0xe9, 0x62, 0x79, 0x3b,
	0x3b, 0x3f, 0xaa, 0x0d, 0x42, 0x03, 0x35, 0x3f,
	0x5d, 0xa4, 0xba, 0x02, 0x23, 0x65, 0x40, 0x0e,
	0x61, 0x31, 0xc9, 0xd5, 0x79, 0x36, 0x76, 0x7c,
	0x21, 0x4a, 0x75, 0xb2, 0xa0, 0x2b, 0xc1, 0xb8,
	0xba, 0xf8, 0x10, 0x7a, 0x85, 0x32, 0x81, 0xbf,
	0x2d, 0x58, 0x3c, 0x22, 0x2f, 0x0b, 0xce, 0x03,
	0x12, 0xce, 0x13, 0xce, 0x4a, 0x81, 0x77, 0x1d,
	0x68, 0x99, 0xc0, 0x25, 0xeb, 0xd9, 0x80, 0x0d,
	0xe7, 0x17, 0xc8, 0x41, 0xbd, 0x4b, 0x9c, 0xdd,
	0x41, 0x74, 0x90, 0x2a, 0x65, 0x92, 0x05, 0xff,
	0x06, 0x24, 0xeb, 0x89, 0xd8, 0x41, 0xaa, 0x41,
	0xac, 0x8a, 0x31, 0xc6, 0x82, 0x44, 0x12, 0x5a,
	0xd9, 0x1c, 0xca, 0x76, 0x2b, 0x4e, 0xe8, 0x18,
	0x7e, 0x50, 0xf5, 0x8c, 0x7c, 0x11, 0xe7, 0x0e,
	0xfd, 0x08, 0x5e, 0x64, 0x8b, 0x5b, 0x9f, 0x94,
	0x8b, 0x0d, 0x83, 0x7e, 0xef, 0x89, 0x30, 0x4a,
	0x55, 0xce, 0x10, 0x5f, 0x15, 0xd2, 0xe7, 0x07,
	0x0c, 0x34, 0x92, 0xda, 0xae, 0x84, 0x26, 0x28,
	0x36, 0x1a, 0x96, 0xc6, 0xf1, 0x41, 0x2b, 0xb6,
	0x01, 0xfe, 0x20, 0x05, 0x43, 0x80, 0x45, 0xdf,
	0x5c, 0xc2, 0x96, 0xc8, 0x41, 0x68, 0x87, 0x78,
	0xbc, 0xf8, 0x34, 0xfb, 0x19, 0x61, 0xab, 0x7f,
	0x15, 0x56, 0x7f, 0x1a, 0xf5, 0x08, 0xf1, 0x1c,
	0x59, 0x70, 0x92, 0x3a, 0xda, 0x1a, 0xfd, 0xfc,
	0x4d, 0xe1, 0x12, 0x61, 0xc1, 0xd8, 0xdb, 0x63,
	0x6e, 0x6b, 0x19, 0x96, 0x68, 0x17, 0x9b, 0xf5,
	0xa9, 0x5d, 0x2c, 0xaf, 0xad, 0xc6, 0x26, 0x9e,
	0x09, 0xcb, 0x67, 0x4e, 0x50, 0x7d, 0x2f, 0xae,
	0x4e, 0x73, 0xd9, 0x5a, 0xaa, 0x5d, 0x54, 0x20,
	0x7b, 0x77, 0xcf, 0xf8, 0xad, 0x88, 0x6b, 0xc8,
};

/*
 * IV method encblkno1, blkno 1.
 */
static const uint8_t bf_cbc_448_encblkno1_vec1_ctxt[SECSIZE] = {
	0x8b, 0x2a, 0xcf, 0x7d, 0x38, 0x1b, 0xaa, 0x33,
	0x1c, 0xe6, 0xa1, 0x37, 0x6e, 0x9e, 0xb3, 0x48,
	0x2a, 0xb8, 0x61, 0x11, 0x00, 0xe5, 0x48, 0xea,
	0xb5, 0x9f, 0x6c, 0xa4, 0xdf, 0x8d, 0x5a, 0xd8,
	0x03, 0x55, 0x4d, 0x07, 0x7d, 0x5f, 0x1b, 0x18,
	0xd1, 0x86, 0x52, 0xc1, 0x13, 0xda, 0x99, 0x23,
	0xeb, 0xab, 0xb2, 0x93, 0x40, 0x7c, 0x6a, 0x8a,
	0xaa, 0xf8, 0xf1, 0x66, 0xf1, 0x10, 0x1d, 0xcd,
	0x25, 0xc7, 0x84, 0x55, 0x02, 0x1d, 0xc0, 0x3c,
	0xba, 0xf4, 0xbf, 0xe2, 0xe4, 0xc3, 0x57, 0xdc,
	0x0d, 0xfd, 0xeb, 0xb3, 0x7d, 0x31, 0x82, 0x6b,
	0x5c, 0x0e, 0x92, 0xa5, 0x42, 0x8b, 0x7f, 0x36,
	0x74, 0x4d, 0xfd, 0x2f, 0xd7, 0x19, 0x0d, 0x23,
	0xa7, 0x36, 0xe8, 0xe4, 0xe5, 0xff, 0xc8, 0x0d,
	0xe1, 0x48, 0x25, 0x79, 0xa9, 0x22, 0xac, 0x72,
	0x86, 0x28, 0xcb, 0x63, 0xa6, 0xa0, 0x46, 0x08,
	0x53, 0xb8, 0x59, 0xab, 0x0f, 0x8f, 0xb1, 0x78,
	0xf8, 0x4e, 0x6d, 0x1a, 0xb5, 0xdd, 0x12, 0x02,
	0x57, 0x55, 0xf8, 0xab, 0x78, 0x7d, 0x75, 0x61,
	0x81, 0x20, 0xd6, 0x4b, 0x7d, 0x76, 0x05, 0xc9,
	0x56, 0xf2, 0xe9, 0x3f, 0xb6, 0xb5, 0x02, 0x2b,
	0x1c, 0x29, 0xbf, 0x07, 0xe9, 0x5f, 0x9b, 0x18,
	0x38, 0x26, 0xa0, 0x09, 0xde, 0x24, 0x5b, 0x37,
	0x72, 0x74, 0xf4, 0x9f, 0x86, 0x28, 0x89, 0xb6,
	0x0c, 0x95, 0x24, 0x2f, 0x88, 0x80, 0x6a, 0xc7,
	0x3a, 0xdc, 0x89, 0xb0, 0xa3, 0xfe, 0x6e, 0x38,
	0xec, 0x0b, 0x1d, 0xbc, 0xd5, 0x90, 0x48, 0xfb,
	0xb1, 0x54, 0xac, 0x6e, 0x35, 0xb0, 0x71, 0x9e,
	0x57, 0x07, 0x81, 0x90, 0xcb, 0x63, 0xb2, 0x7f,
	0x4c, 0x81, 0xe1, 0x58, 0xda, 0x27, 0xef, 0x77,
	0xe7, 0xde, 0x96, 0x83, 0x2a, 0xb1, 0x6b, 0x08,
	0x62, 0x89, 0xdc, 0x3a, 0x3f, 0x08, 0xff, 0xdc,
	0x50, 0x3e, 0xc1, 0xe4, 0x33, 0x8b, 0xad, 0x19,
	0x90, 0x0b, 0x8e, 0xc3, 0x55, 0x77, 0xf0, 0xc2,
	0x24, 0xf9, 0x0c, 0x99, 0x84, 0xb2, 0xcc, 0x23,
	0x8c, 0xab, 0x79, 0x0d, 0xff, 0x75, 0x3a, 0xe0,
	0xc9, 0xe8, 0x1e, 0x15, 0x02, 0xd5, 0x67, 0x8e,
	0x32, 0xe3, 0x1f, 0xda, 0xfb, 0x88, 0xeb, 0xa5,
	0x23, 0xea, 0x1d, 0xaa, 0xc3, 0x62, 0x7f, 0x27,
	0x38, 0x2a, 0xf6, 0xa2, 0x6a, 0x0e, 0x05, 0xff,
	0xe4, 0x63, 0x70, 0xec, 0xf8, 0x25, 0x96, 0x08,
	0xcb, 0x22, 0x2a, 0xaa, 0xbc, 0x45, 0x04, 0xb9,
	0xbc, 0x64, 0x07, 0x09, 0x31, 0xee, 0x5f, 0x9a,
	0xb1, 0x71, 0x85, 0x10, 0x60, 0xfb, 0x3c, 0x56,
	0xeb, 0xfe, 0x91, 0xab, 0x3f, 0x09, 0x76, 0xba,
	0x3c, 0xfc, 0xa6, 0x0d, 0xce, 0x9f, 0x59, 0xd4,
	0x83, 0x8d, 0x98, 0xf5, 0x0d, 0x60, 0x1f, 0xd1,
	0x10, 0x61, 0x77, 0x0d, 0xd1, 0xcd, 0xc2, 0xc2,
	0x2c, 0x7d, 0xf6, 0x15, 0x16, 0x90, 0xc9, 0xc5,
	0x1e, 0xe9, 0xf5, 0x7b, 0xb4, 0x49, 0x47, 0x91,
	0x6a, 0x94, 0x26, 0x94, 0xb3, 0xb5, 0xa7, 0x9e,
	0xcb, 0xb1, 0x9c, 0xb7, 0x5d, 0x25, 0x3c, 0x2d,
	0x8c, 0xa8, 0xa1, 0xb1, 0x79, 0x8e, 0x60, 0xa2,
	0x3e, 0x04, 0xfa, 0x3f, 0xb4, 0x43, 0xde, 0x0a,
	0xde, 0xf4, 0x58, 0xe7, 0xd1, 0x04, 0x1d, 0xb7,
	0x1d, 0xa6, 0xcb, 0x49, 0x41, 0x30, 0xb5, 0x41,
	0xb5, 0x14, 0x19, 0xe4, 0xeb, 0x2c, 0xe2, 0xf0,
	0x66, 0x59, 0xe3, 0xc6, 0xb0, 0xd0, 0x1e, 0xaa,
	0x9f, 0xa5, 0x0b, 0xb8, 0xfd, 0xae, 0x62, 0xcf,
	0x3e, 0xe2, 0xea, 0x79, 0xc5, 0x3e, 0xcf, 0xf0,
	0x40, 0x00, 0x8c, 0x81, 0x97, 0xed, 0xac, 0xf9,
	0x61, 0x75, 0x4e, 0xd7, 0xb7, 0xb2, 0x02, 0x14,
	0x04, 0xf6, 0xbf, 0x25, 0xbe, 0x78, 0x2a, 0xea,
	0xd2, 0x61, 0xf2, 0x7e, 0x45, 0x6b, 0x20, 0xca,
};

/*
 * IV method encblkno1, blkno 2.
 */
static const uint8_t bf_cbc_448_encblkno1_vec2_ctxt[SECSIZE] = {
	0x35, 0x34, 0x49, 0x29, 0x12, 0x3f, 0xc6, 0x2f,
	0xc9, 0x3d, 0xc9, 0x54, 0x46, 0xf3, 0x26, 0xf9,
	0x5e, 0x45, 0xb4, 0xba, 0x6d, 0x0f, 0x98, 0x53,
	0x8b, 0x7a, 0x00, 0x47, 0xb6, 0xbd, 0x70, 0x89,
	0x60, 0x8e, 0x52, 0x0b, 0xe3, 0x0a, 0xd8, 0x0e,
	0x48, 0xf3, 0xcc, 0x66, 0x8d, 0x71, 0xaa, 0x0e,
	0xc2, 0x68, 0x03, 0x05, 0xf9, 0xef, 0x1d, 0x14,
	0x5a, 0x85, 0x88, 0x70, 0x77, 0xcf, 0xe3, 0xdf,
	0x18, 0xe5, 0xfb, 0xea, 0xe1, 0xe8, 0xe0, 0x25,
	0xb2, 0x14, 0x61, 0x5d, 0x2f, 0xce, 0x61, 0xec,
	0x68, 0xc8, 0x06, 0x60, 0x41, 0xc1, 0xe3, 0x0a,
	0x5e, 0x96, 0x15, 0x9b, 0x2c, 0x5b, 0xfd, 0xba,
	0x17, 0x2e, 0x50, 0xb6, 0x68, 0x39, 0x21, 0x56,
	0x31, 0x2e, 0xb5, 0x29, 0xff, 0x4a, 0x12, 0x34,
	0x02, 0x54, 0xb1, 0x7f, 0xd8, 0x35, 0xec, 0x79,
	0x9e, 0xef, 0x62, 0xf3, 0x4b, 0x58, 0x96, 0xf1,
	0x83, 0x26, 0x57, 0x3d, 0x55, 0xb9, 0xb0, 0xa7,
	0x48, 0x65, 0x06, 0xee, 0x14, 0x88, 0xb5, 0x58,
	0xc8, 0x4d, 0x6e, 0xd8, 0x44, 0x76, 0x21, 0x16,
	0xa4, 0xdf, 0x68, 0x4b, 0xff, 0x69, 0x23, 0x66,
	0x18, 0x75, 0xe6, 0x29, 0xea, 0x95, 0x87, 0x1c,
	0xed, 0x2d, 0xbd, 0xbf, 0x22, 0x56, 0x11, 0xd5,
	0x59, 0x90, 0x24, 0xd7, 0xae, 0xda, 0x99, 0x49,
	0xe3, 0x23, 0x03, 0x24, 0x3b, 0x02, 0x49, 0x1d,
	0xa5, 0x57, 0x2f, 0xea, 0xd1, 0x6a, 0x17, 0x53,
	0x0f, 0xb3, 0xa9, 0x64, 0x8a, 0xdb, 0x62, 0x17,
	0xad, 0x5c, 0x7d, 0x56, 0x14, 0x0f, 0xfb, 0x14,
	0xbe, 0x7e, 0xa1, 0xa8, 0x27, 0xf0, 0xf0, 0x3a,
	0xe7, 0xc5, 0x26, 0x98, 0x9d, 0x29, 0xf7, 0xfd,
	0x43, 0x13, 0x34, 0xe2, 0xb8, 0x0b, 0x14, 0xe9,
	0x79, 0x66, 0x7a, 0xf2, 0xed, 0x79, 0x37, 0x16,
	0x75, 0x2b, 0xf2, 0x99, 0xa1, 0xba, 0xf1, 0xc1,
	0x61, 0x8d, 0x78, 0x46, 0x3a, 0x67, 0x58, 0x6a,
	0x55, 0x0d, 0x2e, 0x08, 0x47, 0xfc, 0x94, 0x2f,
	0x65, 0xa1, 0x1a, 0xfe, 0x05, 0xfa, 0x41, 0x00,
	0x6c, 0x42, 0xbc, 0x65, 0x37, 0xd9, 0x25, 0x9c,
	0xf4, 0x83, 0x8a, 0xdb, 0x91, 0x96, 0xc5, 0xa9,
	0x02, 0x44, 0xbc, 0x9a, 0x26, 0x9c, 0xd1, 0xfa,
	0x06, 0x8b, 0xd6, 0x40, 0x8f, 0x0a, 0xd2, 0x5b,
	0xd8, 0x57, 0xd5, 0x61, 0x1b, 0x86, 0xa6, 0x49,
	0x15, 0xe4, 0x06, 0x6c, 0x48, 0x24, 0xb8, 0xe3,
	0x23, 0xed, 0xcf, 0x39, 0x44, 0x4d, 0xf0, 0x4e,
	0x89, 0x44, 0x0d, 0x3a, 0xe6, 0x1b, 0x7c, 0x39,
	0xfd, 0x79, 0x0e, 0x78, 0xc7, 0xf6, 0xa3, 0x91,
	0x18, 0x2a, 0xfb, 0x92, 0x48, 0xcc, 0x8b, 0xbb,
	0x33, 0x07, 0x42, 0xf5, 0xd0, 0x01, 0x8b, 0x12,
	0xd9, 0x5e, 0x9d, 0xe4, 0x13, 0x99, 0x11, 0x18,
	0x86, 0x8a, 0xb7, 0xa6, 0xe2, 0x38, 0x34, 0x12,
	0x67, 0xd6, 0x4b, 0xc0, 0x23, 0x56, 0xba, 0x53,
	0xbe, 0x20, 0xe5, 0xec, 0x16, 0xf9, 0x74, 0x92,
	0x62, 0xfc, 0xb9, 0xe8, 0xa3, 0xbf, 0x3b, 0x06,
	0x76, 0xa5, 0xf5, 0x56, 0x81, 0x72, 0x50, 0xc8,
	0x55, 0x80, 0x7d, 0xe1, 0x46, 0x92, 0xa1, 0xeb,
	0x41, 0xaf, 0xce, 0x52, 0xb1, 0xb3, 0x51, 0xf2,
	0xba, 0x03, 0xb4, 0xcb, 0x16, 0xd2, 0x92, 0x3d,
	0x0c, 0x9b, 0xe9, 0xd9, 0x5d, 0xcf, 0x79, 0x05,
	0xbd, 0xe0, 0x44, 0x39, 0xf0, 0x35, 0x2d, 0x7a,
	0x31, 0x3b, 0x24, 0xb3, 0xb4, 0xa5, 0x08, 0xf5,
	0xac, 0x51, 0xf1, 0x09, 0x52, 0x14, 0xc7, 0xb5,
	0xe0, 0x65, 0x47, 0x30, 0xdd, 0xa4, 0xfd, 0x71,
	0x68, 0xa5, 0x4b, 0x00, 0x72, 0xe2, 0xc4, 0xa9,
	0x5d, 0x21, 0x6d, 0x83, 0x4e, 0x88, 0xaa, 0x76,
	0x07, 0xf0, 0xf7, 0x36, 0xa2, 0x5e, 0xd5, 0x4c,
	0x7f, 0x9b, 0x74, 0x89, 0x0a, 0x18, 0xdc, 0x9a,
};

/*
 * IV method encblkno1, blkno 3.
 */
static const uint8_t bf_cbc_448_encblkno1_vec3_ctxt[SECSIZE] = {
	0xc3, 0xd3, 0xae, 0x7e, 0x4f, 0xbe, 0x0d, 0x50,
	0xd6, 0x63, 0x2c, 0xa2, 0xfd, 0x07, 0xf1, 0x33,
	0x2a, 0x15, 0x8f, 0xd7, 0x63, 0xb6, 0x5f, 0x04,
	0x69, 0x90, 0xa3, 0x1a, 0xd3, 0xdd, 0xe0, 0x70,
	0xb1, 0xcd, 0xd5, 0xe0, 0x75, 0xd2, 0x31, 0x38,
	0xcc, 0x65, 0xbb, 0xc3, 0x3b, 0xc6, 0xc9, 0x33,
	0x43, 0x9c, 0x32, 0x69, 0x95, 0x10, 0x74, 0x36,
	0x3a, 0x05, 0x9c, 0x26, 0x2f, 0x80, 0x20, 0x92,
	0x74, 0x31, 0xc0, 0xf4, 0xb1, 0x42, 0x58, 0xc8,
	0x3e, 0xaa, 0xd4, 0xba, 0xba, 0x4b, 0x5f, 0x47,
	0x1a, 0x9e, 0x43, 0xaf, 0x25, 0x64, 0x0c, 0x2e,
	0xa3, 0xf5, 0xde, 0x6e, 0x28, 0x5e, 0xb9, 0x9e,
	0xc9, 0xdf, 0x85, 0xda, 0xc9, 0xa8, 0x30, 0xf9,
	0x44, 0x9b, 0x16, 0xcb, 0x4b, 0x47, 0x6f, 0x11,
	0x5c, 0xd7, 0xc9, 0xb7, 0x9c, 0x50, 0x04, 0x3b,
	0x2f, 0x13, 0xab, 0xb6, 0x72, 0xe7, 0x11, 0x29,
	0x35, 0xf3, 0xae, 0x5e, 0x2a, 0xa7, 0x1a, 0xac,
	0xb7, 0x4c, 0x7b, 0x69, 0x89, 0xfc, 0xff, 0x37,
	0x24, 0xf8, 0x65, 0xc0, 0x87, 0x89, 0x69, 0x1e,
	0xa6, 0x7e, 0xe7, 0xb3, 0xb7, 0xa8, 0x42, 0x73,
	0x83, 0xdb, 0x56, 0x4f, 0xce, 0xb9, 0x6e, 0x38,
	0x40, 0x24, 0xb8, 0xdd, 0xab, 0x25, 0x3b, 0xc6,
	0x58, 0xed, 0xc7, 0x2b, 0xe0, 0x11, 0x8b, 0x62,
	0xe8, 0x4f, 0xcf, 0xba, 0x2e, 0xd7, 0x6d, 0xf0,
	0x14, 0xa5, 0xee, 0x24, 0xd3, 0x3a, 0xb4, 0xf2,
	0xdc, 0x0d, 0x79, 0xc6, 0x14, 0x52, 0x14, 0x4b,
	0xd1, 0x8c, 0x18, 0xef, 0x1f, 0xd8, 0xe7, 0x60,
	0xf3, 0x28, 0xce, 0xf1, 0x59, 0xc8, 0x43, 0x02,
	0x0e, 0x08, 0x72, 0xe5, 0x7d, 0x5b, 0xc5, 0x80,
	0xfb, 0xca, 0x2a, 0x63, 0x8d, 0x3c, 0x54, 0x04,
	0x1e, 0xdf, 0x94, 0x53, 0xf8, 0x44, 0xe5, 0xc2,
	0x5c, 0x36, 0xc9, 0x75, 0x1c, 0xa2, 0x98, 0x3d,
	0xd6, 0xee, 0x38, 0xf9, 0xab, 0x2d, 0x1a, 0xdb,
	0x87, 0x2c, 0x86, 0xfd, 0xf9, 0xb2, 0x4d, 0x21,
	0xb7, 0xc0, 0x8f, 0x75, 0x21, 0x53, 0xc7, 0xd7,
	0x3e, 0xc6, 0x6c, 0x98, 0x2c, 0x6d, 0x44, 0x13,
	0x40, 0xea, 0xaa, 0x84, 0xd9, 0x47, 0xfd, 0x65,
	0x2c, 0x3d, 0xb0, 0x76, 0xe7, 0xdd, 0xd7, 0x06,
	0x8a, 0x79, 0xa7, 0x6c, 0x3a, 0x2d, 0x32, 0xf7,
	0xae, 0xe7, 0xd2, 0xb1, 0xf2, 0xe0, 0x3a, 0x10,
	0x19, 0xa1, 0x79, 0x7b, 0x76, 0x0b, 0xeb, 0xf3,
	0x01, 0x6e, 0x9f, 0xa1, 0x5d, 0x16, 0x09, 0xec,
	0x6b, 0x64, 0xe3, 0x96, 0xb8, 0x89, 0x99, 0x8c,
	0x77, 0xcf, 0x3f, 0x37, 0x42, 0x51, 0x82, 0x5a,
	0x63, 0x89, 0x12, 0x12, 0x59, 0x38, 0xe3, 0xcf,
	0xa0, 0xda, 0xbc, 0x69, 0x75, 0x48, 0x7c, 0x3e,
	0x56, 0x75, 0x12, 0x10, 0xc5, 0x96, 0x5b, 0x34,
	0x52, 0x2e, 0xce, 0xba, 0xd2, 0x7f, 0x8f, 0x1c,
	0xbd, 0x35, 0x3b, 0x74, 0x5b, 0x6c, 0xfe, 0xa7,
	0xf1, 0x4a, 0x07, 0x95, 0xff, 0xa8, 0xa0, 0x2a,
	0x85, 0xec, 0xd7, 0x56, 0x3b, 0x28, 0x2e, 0x09,
	0x50, 0x43, 0xbd, 0x49, 0x22, 0xdc, 0x78, 0x0e,
	0x7e, 0x68, 0x78, 0xd5, 0x57, 0xc2, 0xd6, 0x7c,
	0x2f, 0xd0, 0x4a, 0x62, 0x16, 0x38, 0x04, 0x23,
	0x43, 0x21, 0xaa, 0xe1, 0x96, 0x0e, 0xa9, 0x22,
	0xe7, 0x2c, 0xb8, 0x5e, 0x8f, 0xc6, 0x2d, 0xbd,
	0x99, 0x25, 0x33, 0xb8, 0x47, 0x99, 0x8d, 0xf5,
	0x22, 0x08, 0x9b, 0xd5, 0xad, 0x83, 0x67, 0xec,
	0x05, 0x89, 0xda, 0xd6, 0xe4, 0xe2, 0xd5, 0xef,
	0x7f, 0x61, 0x1e, 0x03, 0x81, 0x03, 0xb1, 0x98,
	0x5a, 0x29, 0x69, 0x13, 0xb2, 0xe6, 0xe1, 0x2c,
	0x66, 0x88, 0x39, 0x90, 0xf9, 0xae, 0x5d, 0x71,
	0xfe, 0x07, 0x30, 0x7d, 0xba, 0xa9, 0x37, 0xb5,
	0xff, 0x2c, 0xa9, 0xe6, 0x95, 0x48, 0xb2, 0xc8,
};

const struct testvec bf_cbc_448_1_vectors[] = {
	{
		.blkno = 0,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_448_encblkno1_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_448_encblkno1_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_448_encblkno1_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_448_encblkno1_vec3_ctxt,
	},
};

/*
 * IV method encblkno8, blkno 0.
 */
static const uint8_t bf_cbc_448_encblkno8_vec0_ctxt[SECSIZE] = {
	0x91, 0xb4, 0x8f, 0x78, 0x34, 0xbe, 0x03, 0xe9,
	0x4b, 0xca, 0xf4, 0xfa, 0x27, 0x99, 0xa0, 0xd0,
	0xa0, 0x85, 0xf9, 0xca, 0xcc, 0x2e, 0x0d, 0x41,
	0x91, 0xdb, 0xf9, 0x71, 0xcd, 0x49, 0xf7, 0x66,
	0x9b, 0x0c, 0x70, 0x49, 0x36, 0x72, 0xf4, 0xc0,
	0x17, 0xdd, 0xd8, 0x6a, 0xd2, 0x17, 0xfe, 0x50,
	0xa3, 0x2b, 0xa8, 0x7c, 0x9c, 0x12, 0x8c, 0x08,
	0x3d, 0xb5, 0x20, 0xc0, 0x83, 0xc1, 0xf5, 0x34,
	0x4b, 0xa5, 0xcc, 0x7c, 0xc6, 0x9f, 0x4b, 0x7f,
	0x09, 0x0c, 0x04, 0xe3, 0x02, 0xfe, 0xe2, 0x5a,
	0x45, 0xee, 0x4a, 0xcb, 0xc0, 0xe1, 0xf1, 0xae,
	0x1a, 0x22, 0x95, 0xb9, 0x30, 0xe8, 0x2d, 0x90,
	0xe5, 0x6a, 0x2f, 0x0a, 0x59, 0x15, 0xfc, 0x60,
	0xa5, 0x91, 0x95, 0x1a, 0xa2, 0xf5, 0x87, 0xa3,
	0x97, 0x45, 0x1b, 0xfb, 0x78, 0x62, 0x8b, 0xb6,
	0x86, 0xc6, 0x94, 0x9a, 0x9d, 0x09, 0x46, 0xc9,
	0x1f, 0x5f, 0x2e, 0x6c, 0xed, 0x5e, 0xe2, 0xab,
	0xca, 0x30, 0xdb, 0x13, 0x37, 0x8d, 0xb7, 0xc6,
	0xce, 0x74, 0xe3, 0xe2, 0x64, 0x7b, 0x74, 0x3e,
	0x9f, 0x18, 0x4e, 0xe2, 0x2b, 0x52, 0x08, 0x60,
	0x2b, 0x6d, 0x17, 0x1a, 0xff, 0xae, 0xfb, 0xea,
	0x59, 0x1d, 0x45, 0xe8, 0xf9, 0x0c, 0x56, 0xb8,
	0xb5, 0xc7, 0x8f, 0xa7, 0x9e, 0x67, 0x87, 0xbb,
	0xe2, 0x37, 0x56, 0x49, 0xed, 0x75, 0x27, 0x54,
	0xb5, 0x16, 0x34, 0xb6, 0xa6, 0xe0, 0x27, 0x48,
	0x91, 0xfb, 0x80, 0x4f, 0x5e, 0xef, 0x40, 0x6d,
	0x28, 0x8f, 0x2e, 0x56, 0xab, 0x6c, 0x7d, 0xde,
	0xbd, 0xa5, 0xa6, 0x47, 0xe2, 0xc9, 0xb0, 0xc5,
	0x69, 0x5d, 0x57, 0x18, 0x59, 0x08, 0x66, 0x1b,
	0xdc, 0x4f, 0xa4, 0xe2, 0xb5, 0xff, 0x72, 0x4d,
	0x25, 0x79, 0x38, 0x8f, 0xca, 0x23, 0x81, 0x31,
	0x1d, 0xeb, 0x7b, 0x7e, 0x04, 0xc3, 0xa6, 0xac,
	0x12, 0x92, 0x95, 0x44, 0x08, 0xae, 0x25, 0xb5,
	0xc2, 0x39, 0x88, 0x28, 0xc5, 0xc7, 0x3c, 0xfe,
	0x95, 0xe1, 0x1a, 0x27, 0xbe, 0xfd, 0x0a, 0xc7,
	0xd1, 0x02, 0x1a, 0xae, 0x03, 0xfb, 0xce, 0x79,
	0xe6, 0xea, 0x4a, 0xa1, 0x0e, 0x68, 0x61, 0x2f,
	0xfa, 0x7a, 0xb8, 0xda, 0xd1, 0x4a, 0xbe, 0x88,
	0xd2, 0xa0, 0x6b, 0x90, 0xc3, 0x37, 0x18, 0x77,
	0x2e, 0xc8, 0x44, 0x2e, 0x9d, 0xa2, 0x4a, 0xc7,
	0x54, 0xb2, 0x38, 0x9b, 0x60, 0x5e, 0x5b, 0xb0,
	0x31, 0x4c, 0x10, 0xf0, 0x1d, 0x8c, 0xab, 0x9b,
	0x25, 0x86, 0x05, 0xfc, 0xd9, 0x39, 0x11, 0x2b,
	0x7e, 0x07, 0xfd, 0x6b, 0xb3, 0x28, 0x57, 0x66,
	0x94, 0xc6, 0xfc, 0x48, 0x71, 0xeb, 0x7b, 0x9a,
	0x26, 0x8f, 0x9f, 0x97, 0xb6, 0x83, 0x9a, 0xdc,
	0x5d, 0x84, 0x96, 0x9d, 0xe3, 0xa5, 0x12, 0x97,
	0x8c, 0x87, 0xa6, 0x2a, 0xa8, 0x15, 0x9a, 0xb0,
	0x0e, 0x31, 0x55, 0xf7, 0x1b, 0x5c, 0x9c, 0xaf,
	0x13, 0x20, 0x13, 0x50, 0xaf, 0xc9, 0xf4, 0xd9,
	0x47, 0x16, 0xcc, 0x9d, 0xec, 0xa9, 0x2d, 0xfc,
	0x3d, 0x5d, 0x48, 0x89, 0x86, 0x91, 0x6f, 0x1a,
	0xa5, 0xf0, 0xb6, 0x9b, 0xa9, 0x08, 0xf2, 0x05,
	0xad, 0xe5, 0xe0, 0xd8, 0x2e, 0x09, 0xbe, 0x3a,
	0xf2, 0xdc, 0xeb, 0x14, 0xe6, 0x8c, 0x21, 0x20,
	0xbb, 0x42, 0x6a, 0xba, 0x55, 0x79, 0x54, 0x8b,
	0xa8, 0x43, 0x14, 0x28, 0x66, 0xd2, 0x44, 0x65,
	0x9c, 0xa9, 0xdc, 0x3b, 0x3f, 0xab, 0x36, 0xb4,
	0xbc, 0x9d, 0x22, 0x92, 0x24, 0xa8, 0x29, 0x18,
	0x98, 0x3a, 0xbe, 0xa1, 0xb4, 0xf8, 0x7f, 0xea,
	0x2f, 0x58, 0x69, 0xce, 0xee, 0x99, 0x76, 0x2c,
	0x53, 0x84, 0xf5, 0x9e, 0xa3, 0x98, 0xe6, 0x94,
	0x39, 0xfa, 0xfa, 0x6d, 0x66, 0xe2, 0x39, 0x2b,
	0x4b, 0x84, 0x14, 0x89, 0x18, 0xa2, 0x03, 0xdd,
};

/*
 * IV method encblkno8, blkno 1.
 */
static const uint8_t bf_cbc_448_encblkno8_vec1_ctxt[SECSIZE] = {
	0x90, 0x3d, 0xee, 0x58, 0xf7, 0xe3, 0xc4, 0x18,
	0xaf, 0xfa, 0xaf, 0xed, 0x9a, 0x45, 0xe7, 0xa8,
	0xbe, 0xcd, 0x0f, 0xb9, 0x53, 0x18, 0xc6, 0x14,
	0xbe, 0xbb, 0x90, 0x1c, 0x5c, 0x60, 0x93, 0xcb,
	0x62, 0xb3, 0xdb, 0x31, 0x19, 0x39, 0xce, 0x3f,
	0xe4, 0x8f, 0x44, 0xc1, 0x10, 0x80, 0x05, 0x6b,
	0x77, 0xe3, 0xe1, 0xe5, 0xd2, 0x6d, 0x93, 0xf2,
	0xb3, 0x81, 0x03, 0xf8, 0xbc, 0x7b, 0x5a, 0x71,
	0x15, 0x16, 0x3a, 0x2f, 0x03, 0xbb, 0x67, 0x6a,
	0xd9, 0xf5, 0x63, 0x6f, 0x3d, 0x75, 0x1b, 0x0a,
	0x4b, 0x9d, 0x04, 0x11, 0x7e, 0xe8, 0x3e, 0x2d,
	0x04, 0x8f, 0xbf, 0x8a, 0xb2, 0x35, 0x76, 0xc5,
	0xcc, 0x6d, 0x9e, 0x99, 0x71, 0x13, 0xf6, 0x5e,
	0xeb, 0x74, 0x96, 0x8a, 0x29, 0x38, 0x0b, 0x25,
	0x4b, 0x89, 0xa9, 0x43, 0x3c, 0x2f, 0x03, 0x14,
	0x8d, 0x0f, 0xe3, 0xe7, 0x01, 0xd1, 0x2e, 0x14,
	0x08, 0x51, 0xba, 0x06, 0x39, 0x76, 0x35, 0xbc,
	0x14, 0xa6, 0x16, 0x36, 0x47, 0xcc, 0x48, 0xe0,
	0xd6, 0xd7, 0x07, 0xb0, 0xf0, 0x30, 0x6c, 0xf8,
	0x68, 0x9d, 0x6c, 0x4b, 0x69, 0x33, 0x78, 0x0e,
	0x4a, 0xfa, 0x97, 0xfb, 0x0c, 0x0d, 0x0a, 0xc3,
	0x4b, 0x7b, 0x77, 0x77, 0x18, 0x9a, 0x74, 0x85,
	0x2f, 0xae, 0xc7, 0x90, 0x45, 0x4b, 0xa2, 0x06,
	0x3e, 0xa2, 0x48, 0xe7, 0x6c, 0x86, 0x65, 0x78,
	0x97, 0x0b, 0x97, 0xc1, 0x70, 0x91, 0x12, 0x79,
	0xae, 0xf0, 0x2b, 0x44, 0xe9, 0x84, 0x8d, 0x78,
	0x53, 0xf8, 0x3a, 0xf5, 0x9f, 0x27, 0x3d, 0x49,
	0x69, 0xd1, 0x18, 0xa4, 0xb2, 0xd0, 0xbb, 0xf2,
	0x57, 0x76, 0xb7, 0x77, 0x16, 0x2f, 0xf8, 0x0c,
	0xa5, 0x86, 0x43, 0x0d, 0x2d, 0xfe, 0x84, 0xc6,
	0xbb, 0x58, 0x81, 0x47, 0x3d, 0xa3, 0x93, 0xb0,
	0x50, 0xfc, 0x25, 0xf7, 0xc5, 0x05, 0xe5, 0xf2,
	0xb3, 0x79, 0x12, 0xe4, 0x37, 0x71, 0x2d, 0xe8,
	0xa5, 0x0b, 0xce, 0x67, 0x51, 0x4f, 0xab, 0xc7,
	0x7b, 0x3b, 0xac, 0x78, 0x97, 0x82, 0x02, 0xf4,
	0x62, 0x20, 0x1b, 0x8b, 0xac, 0x07, 0x3b, 0xd7,
	0x0b, 0x99, 0x27, 0x85, 0x52, 0x7a, 0x79, 0x18,
	0xfb, 0x81, 0x3a, 0x05, 0x76, 0x6c, 0x3c, 0x6a,
	0x35, 0xe1, 0x2b, 0x03, 0x48, 0x70, 0x1a, 0xa8,
	0x30, 0x33, 0x61, 0xe2, 0xd8, 0x99, 0x86, 0x7f,
	0xfd, 0xe0, 0x4a, 0xe4, 0x62, 0xa1, 0xac, 0xcb,
	0xb8, 0x8a, 0xf3, 0xaa, 0xd6, 0x61, 0x9d, 0xc5,
	0xdb, 0xf5, 0x53, 0x39, 0x1d, 0xd7, 0xf8, 0x96,
	0xc6, 0x2b, 0xca, 0xbf, 0x83, 0x4e, 0x89, 0x63,
	0x53, 0x6f, 0x17, 0xaa, 0xf3, 0x61, 0x9b, 0x75,
	0x8c, 0x5a, 0xf8, 0x21, 0x84, 0x52, 0xb8, 0x76,
	0xbc, 0xf5, 0x9b, 0xd6, 0x98, 0x26, 0x58, 0xec,
	0xdd, 0xa8, 0xf1, 0xee, 0x9e, 0x14, 0x24, 0x94,
	0x7c, 0xb6, 0x45, 0x8b, 0xc7, 0x85, 0x50, 0x4e,
	0x30, 0xd7, 0x51, 0x8c, 0x33, 0xeb, 0xeb, 0x5d,
	0x52, 0x58, 0x43, 0xcb, 0x25, 0x4a, 0x77, 0x34,
	0xe6, 0x70, 0x5b, 0x6f, 0x8f, 0xe8, 0x07, 0xee,
	0x76, 0x4a, 0xad, 0xab, 0x11, 0x8a, 0x1b, 0x92,
	0x60, 0x79, 0xb8, 0xe0, 0x88, 0xa7, 0x3a, 0xe0,
	0x15, 0xf6, 0x57, 0xf0, 0xe8, 0x87, 0xda, 0xf8,
	0x90, 0x4e, 0xe7, 0xb3, 0xb4, 0xe7, 0x73, 0x5c,
	0xd3, 0x69, 0xfb, 0x23, 0x4f, 0x4f, 0xc8, 0xd2,
	0xfb, 0xf5, 0xf5, 0x76, 0x80, 0xb6, 0xb7, 0xe2,
	0xb7, 0x96, 0x1a, 0x97, 0x12, 0x40, 0x16, 0x86,
	0xd6, 0x66, 0xf5, 0x46, 0x9e, 0x04, 0x62, 0xaa,
	0x9c, 0xc9, 0x45, 0x39, 0x5c, 0xd3, 0x58, 0x40,
	0xb4, 0x32, 0xd8, 0x6c, 0x19, 0xfa, 0xa5, 0x5f,
	0x1e, 0x83, 0x5a, 0x32, 0x5e, 0x1d, 0xab, 0xa9,
	0x48, 0x1b, 0x1b, 0x37, 0x66, 0xf8, 0x67, 0xd7,
};

/*
 * IV method encblkno8, blkno 2.
 */
static const uint8_t bf_cbc_448_encblkno8_vec2_ctxt[SECSIZE] = {
	0xb3, 0x2d, 0x0d, 0x90, 0x7a, 0x50, 0xb6, 0x29,
	0x9d, 0xd4, 0x12, 0x1a, 0xc9, 0x71, 0x56, 0xc4,
	0xce, 0x3d, 0x74, 0xf9, 0x91, 0xe4, 0x79, 0x8a,
	0x42, 0x34, 0x1a, 0xd4, 0x84, 0xaa, 0x01, 0x85,
	0x13, 0xe8, 0xab, 0xaa, 0xb6, 0x81, 0x82, 0xd6,
	0x32, 0xd6, 0x9f, 0x63, 0x69, 0x55, 0x10, 0x5f,
	0x2e, 0xb3, 0x51, 0x74, 0x80, 0x5c, 0xe9, 0x8d,
	0x27, 0xd8, 0x41, 0x2d, 0x27, 0x07, 0xb4, 0x13,
	0xcd, 0xc0, 0xb3, 0x96, 0xaa, 0x59, 0x9c, 0x3e,
	0x16, 0x9e, 0x18, 0xa5, 0x92, 0x52, 0x59, 0x19,
	0x19, 0x62, 0xeb, 0xd9, 0xdf, 0x9d, 0x10, 0x77,
	0xfb, 0x3c, 0xf1, 0xeb, 0x70, 0xf6, 0x6a, 0x9b,
	0xad, 0x99, 0x35, 0xd7, 0xf8, 0xde, 0x2b, 0x52,
	0x85, 0xdd, 0x36, 0x07, 0x3a, 0x88, 0xa6, 0xbb,
	0x98, 0x7a, 0xe3, 0xb0, 0xe6, 0xae, 0x33, 0x5d,
	0x47, 0x0c, 0x6c, 0xd4, 0x84, 0x33, 0x7e, 0xf4,
	0xea, 0xfd, 0xf1, 0x1c, 0xdb, 0x8b, 0xfe, 0x01,
	0x77, 0xa3, 0x07, 0x21, 0x09, 0xac, 0x0c, 0xe4,
	0x63, 0xaa, 0x3d, 0xb3, 0x05, 0xf8, 0x73, 0x03,
	0x69, 0x53, 0x2f, 0xd3, 0x53, 0x29, 0xf2, 0x02,
	0x60, 0x41, 0xed, 0xc7, 0xe9, 0x2e, 0xba, 0x54,
	0xa4, 0xfe, 0x26, 0xf0, 0xd9, 0x2c, 0x58, 0x2d,
	0x90, 0x94, 0x19, 0xf9, 0xe5, 0xaa, 0xe2, 0x13,
	0x9a, 0x67, 0x90, 0x44, 0x61, 0x36, 0xd0, 0x3f,
	0xe7, 0xe8, 0x7e, 0x47, 0x2b, 0x84, 0x97, 0xe2,
	0x0b, 0x8a, 0xfa, 0x2f, 0xbc, 0x1a, 0x70, 0xed,
	0xb2, 0x93, 0x36, 0x0e, 0xe6, 0xb1, 0xa2, 0x5a,
	0x04, 0x04, 0x8d, 0x2f, 0x82, 0xdb, 0x49, 0x5e,
	0x1b, 0x90, 0xc1, 0x27, 0x2a, 0x2f, 0x64, 0x5b,
	0xce, 0x35, 0x06, 0x0d, 0xb6, 0x05, 0x50, 0x5a,
	0x90, 0xc8, 0x21, 0xd0, 0xf0, 0xb9, 0xf3, 0x43,
	0x25, 0xd8, 0xb9, 0x86, 0xb7, 0xd0, 0x29, 0x75,
	0xdc, 0xf6, 0xf6, 0x14, 0x5e, 0x1f, 0xad, 0x54,
	0x28, 0x63, 0x92, 0xcd, 0xcd, 0x75, 0xb2, 0xe8,
	0xf3, 0x1f, 0xfd, 0x88, 0x72, 0x04, 0xab, 0xe2,
	0x77, 0x21, 0xc1, 0xb4, 0x87, 0xa7, 0x98, 0x86,
	0x2d, 0x7f, 0x01, 0x24, 0xed, 0x3a, 0x23, 0x83,
	0xb7, 0xa4, 0xb7, 0x08, 0x4e, 0xee, 0x10, 0x77,
	0x29, 0x22, 0xab, 0xea, 0x4a, 0x68, 0x1f, 0x7f,
	0xd6, 0xe6, 0x90, 0x11, 0xbc, 0x23, 0x2e, 0x47,
	0x78, 0xea, 0x1d, 0x33, 0x7c, 0x02, 0x09, 0x55,
	0x7a, 0xbc, 0xa7, 0x3d, 0x9a, 0xd0, 0x4d, 0x40,
	0x6c, 0xc2, 0x99, 0xc2, 0xe9, 0x0e, 0xcf, 0x06,
	0x82, 0x61, 0x5c, 0x76, 0xca, 0xef, 0x8f, 0xd3,
	0x78, 0x2a, 0xae, 0x39, 0x29, 0x4e, 0xc1, 0x2c,
	0xb1, 0xbc, 0xcd, 0x76, 0x4f, 0x25, 0xf0, 0x5b,
	0x78, 0x3a, 0xa4, 0x39, 0x52, 0x5a, 0xc7, 0xcf,
	0x17, 0x3d, 0xb0, 0x39, 0x63, 0xf9, 0xf9, 0xfb,
	0x6f, 0x35, 0xb7, 0xf1, 0x49, 0x10, 0xd8, 0x12,
	0x8d, 0xa2, 0xdf, 0xc5, 0x26, 0x37, 0xd1, 0xb8,
	0xef, 0xdc, 0x04, 0x1b, 0x0d, 0x60, 0xbf, 0xbc,
	0xc0, 0xff, 0x56, 0x8b, 0xd3, 0x6e, 0x71, 0xfc,
	0x87, 0x00, 0x86, 0x10, 0x78, 0x3b, 0xce, 0x8b,
	0xe8, 0x5f, 0x8c, 0xce, 0x03, 0xa2, 0x89, 0x8c,
	0x16, 0x00, 0x0e, 0xd8, 0x53, 0xaf, 0x7f, 0x77,
	0x78, 0x40, 0x5e, 0x5e, 0xd1, 0x7d, 0xf8, 0x41,
	0xa8, 0x1e, 0xa5, 0xe5, 0xe9, 0xd6, 0x17, 0x2c,
	0x2f, 0x1b, 0xff, 0xef, 0xf5, 0x53, 0x31, 0xf3,
	0x5b, 0xe4, 0x84, 0x7c, 0xe2, 0x45, 0x3c, 0x82,
	0x5b, 0xf6, 0x03, 0x35, 0xdd, 0x03, 0x22, 0xbe,
	0x77, 0x9c, 0x6a, 0x7d, 0xc8, 0x29, 0x41, 0x53,
	0xbb, 0xab, 0x6e, 0xa5, 0x00, 0xaf, 0x3b, 0x1d,
	0x76, 0x12, 0xac, 0x44, 0x5c, 0x7d, 0xd2, 0x3b,
	0x3a, 0x95, 0xb0, 0xa9, 0x4f, 0x27, 0x76, 0x17,
};

/*
 * IV method encblkno8, blkno 3.
 */
static const uint8_t bf_cbc_448_encblkno8_vec3_ctxt[SECSIZE] = {
	0x8e, 0xc4, 0x56, 0x64, 0x1e, 0x2a, 0x0d, 0x60,
	0x54, 0x5c, 0xcd, 0xe0, 0x6d, 0xa7, 0x4c, 0x30,
	0x7e, 0x85, 0x21, 0xdf, 0xaa, 0xb2, 0x32, 0xde,
	0xc0, 0xc6, 0x56, 0xe0, 0x43, 0xc2, 0x3e, 0x6c,
	0x8c, 0x62, 0x35, 0xaa, 0xf9, 0xba, 0xc9, 0x52,
	0x38, 0x72, 0x06, 0xcc, 0x02, 0xa2, 0xb8, 0x85,
	0xf7, 0xcc, 0xe6, 0x8c, 0x86, 0x8f, 0x9c, 0xd6,
	0x1f, 0xf8, 0x24, 0x9d, 0xca, 0xe8, 0xed, 0x3c,
	0x80, 0x0b, 0xaf, 0x0c, 0x78, 0x4f, 0x5b, 0x2a,
	0x0f, 0xfe, 0xe5, 0xe6, 0x12, 0x8a, 0xff, 0xc7,
	0x6a, 0x97, 0xd9, 0xcb, 0xc8, 0x6a, 0x83, 0x12,
	0xa1, 0x12, 0x84, 0xc8, 0x72, 0x1c, 0xb7, 0x68,
	0x23, 0x24, 0xff, 0x5a, 0x78, 0x99, 0x9d, 0xb8,
	0x03, 0x70, 0x0a, 0x09, 0xa1, 0x3d, 0xfe, 0xe0,
	0xc5, 0x1b, 0xea, 0x58, 0xbc, 0x52, 0x70, 0xa2,
	0x4e, 0xcc, 0x43, 0xfb, 0xb7, 0xc4, 0xbd, 0xb6,
	0xa9, 0x1e, 0xff, 0xf6, 0x69, 0xaa, 0xab, 0xa4,
	0xd7, 0x07, 0x0d, 0xda, 0x41, 0x4b, 0xe3, 0xa5,
	0xef, 0x94, 0x9f, 0xb4, 0xd8, 0xd1, 0x41, 0xd0,
	0x9e, 0xa0, 0x0a, 0x70, 0xdb, 0xb8, 0x5e, 0x27,
	0xc6, 0x08, 0x38, 0x6a, 0x31, 0xe3, 0xa0, 0xd6,
	0x90, 0xad, 0x19, 0x0c, 0x7e, 0x1d, 0x21, 0xc8,
	0x66, 0x30, 0x73, 0x8e, 0x06, 0x97, 0xec, 0xc6,
	0xfe, 0x5c, 0xc6, 0xc0, 0xd1, 0x5c, 0x5f, 0xf8,
	0x01, 0xb3, 0xac, 0x18, 0x66, 0x1e, 0x04, 0xaf,
	0xa7, 0xd3, 0x6d, 0x10, 0x74, 0xa1, 0x9a, 0x36,
	0x10, 0xa0, 0xd6, 0x28, 0x61, 0x93, 0x98, 0x14,
	0x67, 0x6f, 0x7d, 0x52, 0x86, 0x48, 0x17, 0x99,
	0x53, 0xa3, 0xee, 0xe5, 0x93, 0xf6, 0x80, 0xe1,
	0x52, 0xf0, 0x39, 0x48, 0x5c, 0x20, 0x05, 0xd9,
	0xae, 0xa2, 0xe3, 0x25, 0x4e, 0x60, 0x84, 0xf8,
	0xad, 0xd6, 0xf6, 0x95, 0x8e, 0x95, 0xd0, 0x49,
	0x1c, 0x65, 0xd0, 0xc8, 0xa5, 0x26, 0xc0, 0xdf,
	0x32, 0xbe, 0xbc, 0xb7, 0x6d, 0xe5, 0x5e, 0x6d,
	0x38, 0x7d, 0x03, 0xd5, 0x94, 0x7a, 0x14, 0x2b,
	0x02, 0xe0, 0x09, 0x00, 0x50, 0xf1, 0x52, 0x69,
	0x06, 0x33, 0x4b, 0x5b, 0xa2, 0xbc, 0x2e, 0xa9,
	0x1a, 0xb7, 0xca, 0xa8, 0xb4, 0xa2, 0x5b, 0xcd,
	0x35, 0xe4, 0x03, 0xdd, 0x8f, 0x66, 0x3a, 0x34,
	0xc6, 0x2a, 0xd6, 0x90, 0xa2, 0xef, 0xe2, 0xfa,
	0x7c, 0xc1, 0x6c, 0x21, 0xd0, 0xfd, 0x96, 0x92,
	0xb5, 0x99, 0xe4, 0xb7, 0x66, 0xd4, 0xf2, 0x50,
	0x22, 0xef, 0x66, 0x1e, 0x5f, 0x62, 0xd1, 0x77,
	0x87, 0x52, 0xed, 0x40, 0x69, 0xfd, 0xab, 0x66,
	0xe4, 0x0e, 0x2b, 0xa8, 0x67, 0x4c, 0x6d, 0xce,
	0xb0, 0x61, 0x8e, 0x6c, 0xc5, 0x34, 0xab, 0x03,
	0x3e, 0x8a, 0xe5, 0x2b, 0xa2, 0xa4, 0x04, 0xa2,
	0x01, 0x81, 0x79, 0x72, 0xfc, 0x19, 0xbd, 0x38,
	0x39, 0xee, 0xb3, 0x95, 0xc5, 0x6f, 0xed, 0xaa,
	0x6e, 0xca, 0xeb, 0xc6, 0xaf, 0xeb, 0x76, 0xb4,
	0xd7, 0xc3, 0x1b, 0x65, 0x99, 0xc6, 0xa3, 0xe8,
	0x96, 0x5e, 0xc1, 0x0c, 0xd2, 0xf8, 0x65, 0xcf,
	0x42, 0xc5, 0x8f, 0x52, 0x5d, 0x90, 0x21, 0x55,
	0xec, 0x9d, 0x93, 0x81, 0xb7, 0x9a, 0xa4, 0x35,
	0xe7, 0xef, 0xef, 0x2d, 0x4c, 0x02, 0xf7, 0x2b,
	0x26, 0xe0, 0x9e, 0x3a, 0x31, 0xfd, 0x94, 0xb3,
	0xa7, 0x8a, 0x93, 0xf3, 0xe1, 0x77, 0x79, 0xdf,
	0xcf, 0x1f, 0x99, 0x55, 0x20, 0xc3, 0x7d, 0x8a,
	0xbc, 0xff, 0x63, 0x64, 0x87, 0xa9, 0x42, 0x63,
	0xc9, 0x67, 0x7e, 0x51, 0x99, 0x9c, 0xcb, 0x47,
	0xa9, 0xc8, 0x5e, 0x83, 0x87, 0x55, 0x7c, 0x45,
	0x3a, 0xb4, 0xfe, 0x97, 0x24, 0x17, 0x1d, 0x5e,
	0xdf, 0xe0, 0xe8, 0x17, 0xa6, 0x31, 0x99, 0xeb,
	0xb7, 0xb5, 0xd5, 0xd7, 0x7c, 0x2f, 0x22, 0x26,
};

const struct testvec bf_cbc_448_8_vectors[] = {
	{
		.blkno = 0,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_448_encblkno8_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_448_encblkno8_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_448_encblkno8_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = bf_cbc_ptxt,
		.ctxt = bf_cbc_448_encblkno8_vec3_ctxt,
	},
};

static int
open_disk(const char *devpath, const char *imgpath, size_t size)
{
	int fd;

	fd = open(imgpath, O_CREAT | O_RDWR | O_TRUNC, 0600);
	if (fd < 0)
		return -1;

	if (ftruncate(fd, size) < 0)
		goto fail;

	if (rump_pub_etfs_register_withsize(devpath,
	    imgpath, RUMP_ETFS_BLK, 0, size) < 0) {
		goto fail;
	}

	unlink(imgpath);
	return fd;
fail:
	close(fd);
	unlink(imgpath);
	return -1;
}

static int
open_cgd(int devno)
{
	char devpath[32];

	sprintf(devpath, "/dev/rcgd%d%c", devno, getrawpartition() + 'a');

	return rump_sys_open(devpath, O_RDWR, 0);
}

static int
configure_cgd(int fd, const char *dkpath, const char *alg,
    const char *ivmethod, const char *key, size_t keylen)
{
	struct cgd_ioctl ci;

	memset(&ci, 0, sizeof(ci));
	ci.ci_disk = dkpath;
	ci.ci_alg = alg;
	ci.ci_ivmethod = ivmethod;
	ci.ci_keylen = 8 * keylen - 8; /* Exclude the NUL terminator. */
	ci.ci_key = key;
	ci.ci_blocksize = 64;

	return rump_sys_ioctl(fd, CGDIOCSET, &ci);
}

static int
unconfigure_cgd(int fd)
{
	struct cgd_ioctl ci;

	return rump_sys_ioctl(fd, CGDIOCCLR, &ci);
}

static int
write_testvec(int cgdfd, const struct testvec *tv)
{

	if (rump_sys_lseek(cgdfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		return -1;

	if (rump_sys_write(cgdfd, tv->ptxt, SECSIZE) != SECSIZE)
		return -1;

	return 0;
}

static int
read_testvec(int cgdfd, const struct testvec *tv)
{
	char *buf;
	int res = -1;

	buf = malloc(SECSIZE);
	if (buf == NULL)
		return -1;

	if (rump_sys_lseek(cgdfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		goto fail;

	if (rump_sys_read(cgdfd, buf, SECSIZE) != SECSIZE)
		goto fail;

	res = memcmp(buf, tv->ptxt, SECSIZE);
fail:
	free(buf);
	return res;
}

static int
check_testvec(int dkfd, const struct testvec *tv)
{
	char *buf;
	int res = -1;

	buf = malloc(SECSIZE);
	if (buf == NULL)
		return -1;

	if (lseek(dkfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		goto fail;

	if (read(dkfd, buf, SECSIZE) != SECSIZE)
		goto fail;

	res = memcmp(buf, tv->ctxt, SECSIZE);
fail:
	free(buf);
	return res;
}

ATF_TC(cgd_bf_cbc_128_encblkno1);
ATF_TC_HEAD(cgd_bf_cbc_128_encblkno1, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test blowfish-cbc with 128 bits key, ivmethod encblkno1");
}

ATF_TC_BODY(cgd_bf_cbc_128_encblkno1, tc)
{
	const char imgpath[] = "blowfish-cbc-128-encblkno1.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno1",
	    bf_cbc_128_key, sizeof(bf_cbc_128_key)));

	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_128_1_vectors[0]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_128_1_vectors[1]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_128_1_vectors[2]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_128_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno1",
	    bf_cbc_128_key, sizeof(bf_cbc_128_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_128_1_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_128_1_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_128_1_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_128_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_128_1_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_128_1_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_128_1_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_128_1_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_bf_cbc_128_encblkno8);
ATF_TC_HEAD(cgd_bf_cbc_128_encblkno8, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test blowfish-cbc with 128 bits key, ivmethod encblkno8");
}

ATF_TC_BODY(cgd_bf_cbc_128_encblkno8, tc)
{
	const char imgpath[] = "blowfish-cbc-128-encblkno8.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno8",
	    bf_cbc_128_key, sizeof(bf_cbc_128_key)));

	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_128_8_vectors[0]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_128_8_vectors[1]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_128_8_vectors[2]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_128_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno8",
	    bf_cbc_128_key, sizeof(bf_cbc_128_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_128_8_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_128_8_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_128_8_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_128_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_128_8_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_128_8_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_128_8_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_128_8_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_bf_cbc_256_encblkno1);
ATF_TC_HEAD(cgd_bf_cbc_256_encblkno1, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test blowfish-cbc with 256 bits key, ivmethod encblkno1");
}

ATF_TC_BODY(cgd_bf_cbc_256_encblkno1, tc)
{
	const char imgpath[] = "blowfish-cbc-256-encblkno1.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno1",
	    bf_cbc_256_key, sizeof(bf_cbc_256_key)));

	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_256_1_vectors[0]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_256_1_vectors[1]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_256_1_vectors[2]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_256_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno1",
	    bf_cbc_256_key, sizeof(bf_cbc_256_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_256_1_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_256_1_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_256_1_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_256_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_256_1_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_256_1_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_256_1_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_256_1_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_bf_cbc_256_encblkno8);
ATF_TC_HEAD(cgd_bf_cbc_256_encblkno8, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test blowfish-cbc with 256 bits key, ivmethod encblkno8");
}

ATF_TC_BODY(cgd_bf_cbc_256_encblkno8, tc)
{
	const char imgpath[] = "blowfish-cbc-256-encblkno8.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno8",
	    bf_cbc_256_key, sizeof(bf_cbc_256_key)));

	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_256_8_vectors[0]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_256_8_vectors[1]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_256_8_vectors[2]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_256_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno8",
	    bf_cbc_256_key, sizeof(bf_cbc_256_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_256_8_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_256_8_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_256_8_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_256_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_256_8_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_256_8_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_256_8_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_256_8_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_bf_cbc_448_encblkno1);
ATF_TC_HEAD(cgd_bf_cbc_448_encblkno1, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test blowfish-cbc with 448 bits key, ivmethod encblkno1");
}

ATF_TC_BODY(cgd_bf_cbc_448_encblkno1, tc)
{
	const char imgpath[] = "blowfish-cbc-448-encblkno1.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno1",
	    bf_cbc_448_key, sizeof(bf_cbc_448_key)));

	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_448_1_vectors[0]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_448_1_vectors[1]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_448_1_vectors[2]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_448_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno1",
	    bf_cbc_448_key, sizeof(bf_cbc_448_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_448_1_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_448_1_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_448_1_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_448_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_448_1_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_448_1_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_448_1_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_448_1_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_bf_cbc_448_encblkno8);
ATF_TC_HEAD(cgd_bf_cbc_448_encblkno8, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test blowfish-cbc with 448 bits key, ivmethod encblkno8");
}

ATF_TC_BODY(cgd_bf_cbc_448_encblkno8, tc)
{
	const char imgpath[] = "blowfish-cbc-448-encblkno8.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno8",
	    bf_cbc_448_key, sizeof(bf_cbc_448_key)));

	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_448_8_vectors[0]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_448_8_vectors[1]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_448_8_vectors[2]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &bf_cbc_448_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "blowfish-cbc", "encblkno8",
	    bf_cbc_448_key, sizeof(bf_cbc_448_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_448_8_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_448_8_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_448_8_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &bf_cbc_448_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_448_8_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_448_8_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_448_8_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &bf_cbc_448_8_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TP_ADD_TCS(tp)
{

	ATF_TP_ADD_TC(tp, cgd_bf_cbc_128_encblkno1);
	ATF_TP_ADD_TC(tp, cgd_bf_cbc_128_encblkno8);
	ATF_TP_ADD_TC(tp, cgd_bf_cbc_256_encblkno1);
	ATF_TP_ADD_TC(tp, cgd_bf_cbc_256_encblkno8);
	ATF_TP_ADD_TC(tp, cgd_bf_cbc_448_encblkno1);
	ATF_TP_ADD_TC(tp, cgd_bf_cbc_448_encblkno8);

	return atf_no_error();
}
