/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - Integrating C++ with Tcl
 * 
 *	FILE: "tcl_obj_lazy.cc"
 *									  created: 29/12/96 {7:46:15 pm} 
 *								  last update: 09/06/98 {14:09:59 PM} 
 *	Author:	Vince Darley
 *	E-mail:	<darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www:	<http://www.fas.harvard.edu/~darley/>
 *	
 *	See	header file	for	further	information
 * ###################################################################
 */

#include "tcl_obj.h"

tcl_obj::tcl_obj(Tcl_Interp* interp, const Tcl_Obj* obj):_interp(interp),
	_obj(),_status(TCL_OK),_list_mode(0) {
	if(obj) {
		lzobj.append((Tcl_Obj*)obj);
	}
}

tcl_obj::tcl_obj(const Tcl_Obj* obj):_interp(0),
	_obj(),_status(TCL_OK),_list_mode(0) {
	if(obj) {
		lzobj.append((Tcl_Obj*)obj);
	} 
}

tcl_obj::tcl_obj(const tcl_obj& t):_interp(t._interp),
	_obj(t._obj),_status(TCL_OK),_list_mode(t._list_mode) {
	lzobj = t.lzobj;
}

/* 
 * void	tcl_obj::ensure_interp_synchronised(void) const	{
 *	   if(interpreter())
 *		   if(Tcl_GetObjResult(interpreter()) != _obj)
 *			   Tcl_SetObjResult(interpreter(),(Tcl_Obj*)_obj);
 * }
 */

const tcl_obj& tcl_obj::operator=(const tcl_obj& t) {
	_interp = t._interp;
	lzobj = t.lzobj;
	_obj = t._obj;
	_status = t.status();
	_list_mode = t._list_mode;
	return *this;
}

tcl_obj::~tcl_obj(void) {
	if(_interp) {
		//cpptcl_delete_tcl_obj(*this);
	}	
}

/* 
 * -------------------------------------------------------------------------
 * 
 * "tcl_obj::_make_up_to_date" --
 * 
 *  Could optimise this for the case where o isn't set at all, and
 *  the list contains a single element.
 * -------------------------------------------------------------------------
 */
void tcl_obj::_make_up_to_date(tcl_strobj& o) const {
	if(lzobj.isNonEmpty()) {
		for(olist_pos<tcl_strobj> p(lzobj);p;p++) {
			if(list_mode()) {
				o.lappend(p.item());
			} else {
				o.append(p.item());
			}
		}
		lzobj.remove_all_contents();
	}
#if 0
	// for debugging purposes, can keep the string representation up to date
	int llen;
	const char* ss = Tcl_GetStringFromObj((Tcl_Obj*)o,&llen);
#endif
}

tcl_obj& result(tcl_obj& o) {
	o._status = TCL_OK;
	o.push_into_result();
	return o;
}

tcl_obj& cmd(tcl_obj& o) {
	o._make_up_to_date();
	o << "[" << lappend;
	return o;
}

tcl_obj& end_cmd(tcl_obj& o) {
	o._make_up_to_date();
	o << "]" << lappend;
	return o;
}

tcl_obj& eval(tcl_obj& o) {
	o._make_up_to_date();
	if(Tcl_GetObjResult(o.interpreter()) == (Tcl_Obj*)o) {
		Tcl_SetObjResult(o.interpreter(),NULL);
	} 
#if 0
	// for debugging purposes, can keep the string representation up to date
	int llen;
	const char* ss = Tcl_GetStringFromObj(o._obj,&llen);
#endif
	o._status = Tcl_EvalObj(o.interpreter(),o);
	o.clear_value();
	return o;
}

tcl_obj& endl(tcl_obj& o) {
	return o.endl();
}

tcl_obj& tcl_error(tcl_obj& o) {
	o._status = TCL_ERROR;
	o.push_into_result();
	return o;
}

tcl_obj& get_result(tcl_obj& o) {
	o.pull_from_result();
	return o;
}

tcl_obj& tcl_obj::endl(void) {		
	#ifdef MAC_TCL
	*this << '\r';
	#else
	*this << '\n';
	#endif
	return *this;
}

void tcl_obj::pull_from_result(void) {
	_clear_working();
	_obj = Tcl_GetObjResult(interpreter());
}

void tcl_obj::push_into_result(void) {
	_make_up_to_date();
	if(interpreter()) {
		_push_obj_into_result();
	}
}

void tcl_obj::_push_obj_into_result(void) {
	if(_obj != Tcl_GetObjResult(interpreter())) {
		Tcl_SetObjResult(interpreter(),_obj);
	}
	_obj.clear();
}

tcl_obj& discard(tcl_obj& o) {
	o._clear_working();
	//Tcl_ResetObjResult(o.interpreter());
	return o;
}

tcl_obj& lappend(tcl_obj& o) {
	if(!o.list_mode()) {
		tcl_strobj tmp;
		o._make_up_to_date(tmp);
		o << list_mode_on << (Tcl_Obj*)tmp << list_mode_off;
	}
	return o;
}

tcl_obj& to_int(tcl_obj& o) {
	o._make_up_to_date();
	long i;
	return o >> i;
}

tcl_obj& to_double(tcl_obj& o) {
	o._make_up_to_date();
	double i;
	return o >> i;
}

tcl_obj& to_bool(tcl_obj& o) {
	o._make_up_to_date();
	bool i;
	return o >> i;
}

tcl_obj& tcl_obj::operator>> (long& i) {
	_make_up_to_date();
	_status = Tcl_GetLongFromObj(_interp,_obj,&i);
	return (*this);
}
tcl_obj& tcl_obj::operator>> (int& i) {
	_make_up_to_date();
	_status = Tcl_GetIntFromObj(_interp,_obj,&i);
	return (*this);
}
tcl_obj& tcl_obj::operator>> (double& i) {
	_make_up_to_date();
	_status = Tcl_GetDoubleFromObj(_interp,_obj,&i);
	return (*this);
}

tcl_obj& tcl_obj::operator>> (bool& i) {
	_make_up_to_date();
	register int iptr;
	_status = Tcl_GetBooleanFromObj(_interp,_obj,&iptr);
	i = (iptr ? true : false);
	return (*this);
}

void tcl_obj::add_precise_string(const char* const& i, int len) {
	lzobj.append(tcl_strobj(i,len));
}

/// Get the string value with length
const char* tcl_obj::get_precise_string(int& len) const {
	_make_up_to_date();
	return Tcl_GetStringFromObj(_obj,&len);
}

const char* tcl_obj::str(void) {
	_make_up_to_date();
	int len;
	return Tcl_GetStringFromObj(_obj,&len);
}

tcl_obj& tcl_obj::operator<< (const tcl_obj& i) {
	if((list_mode() == i.list_mode()) && i._obj.is_empty()) {
		lzobj += i.lzobj;
	} else {
		i._make_up_to_date();
		lzobj.append((Tcl_Obj*)i);
	}
	return *this;
}
const tcl_obj& tcl_obj::operator= (const long& i) {
	lzobj.remove_all_contents();
	lzobj.append(Tcl_NewLongObj(i));
	return *this;
}
const tcl_obj& tcl_obj::operator= (const int& i) {
	lzobj.remove_all_contents();
	lzobj.append(Tcl_NewIntObj(i));
	return *this;
}
const tcl_obj& tcl_obj::operator= (const char& i) {
	lzobj.remove_all_contents();
	lzobj.append(Tcl_NewStringObj((char*)&i,1));
	return *this;
}
const tcl_obj& tcl_obj::operator= (const char* const& i) {
	lzobj.remove_all_contents();
	lzobj.append(Tcl_NewStringObj((char*)i,-1));
	return *this;
}

const tcl_obj& tcl_obj::operator= (const double& i) {
	lzobj.remove_all_contents();
	lzobj.append(Tcl_NewDoubleObj(i));
	return *this;
}

const tcl_obj& tcl_obj::operator= (const bool& i) {
	lzobj.remove_all_contents();
	lzobj.append(Tcl_NewBooleanObj(i));
	return *this;
}

tcl_obj& tcl_obj::ResetResult(void)  {
	if(_interp)
		Tcl_ResetResult(_interp);
    return *this;
}

void tcl_obj::_clear_working(void) {
	lzobj.remove_all_contents();
}

void tcl_obj::clear_value(void) {
	_clear_working();
	_obj.clear();
}

Tcl_Obj* tcl_obj::result(void) { 
	return Tcl_GetObjResult(interpreter());
}

const Tcl_Obj* tcl_obj::result(void) const { 
	return Tcl_GetObjResult(interpreter());
}

int tcl_obj::PackageLibraryInit(const char* libVarName, const char* envVarName, 
				const char* pkgName, 
				const char* pkgInitFile, 
				const char* version, 
				const char* prettyPkgName,
				const char* compiledLocation) {
    /*
     *  Set up the library and load the init file.
     */	
    if(Tcl_GetVar(_interp, (char*) libVarName, TCL_GLOBAL_ONLY) != NULL) {
	return TCL_OK;
    }
    if(prettyPkgName ==0) 
	prettyPkgName = pkgName;
    if(compiledLocation)
	Tcl_SetVar(_interp,(char*) libVarName, (char*)compiledLocation, TCL_GLOBAL_ONLY);
    else
	Tcl_SetVar(_interp,(char*) libVarName, "", TCL_GLOBAL_ONLY);

    /* 
     * If you are on MacOS, and get library files from the resource fork 
     * then you will find your library variable is not set.  You must decide
     * what to do with it.
    */
    (*this) << "set " << pkgName << "dirs {}\n"
	    << "if [info exists env(" << envVarName << ")] {\n"
	    << "	lappend " << pkgName << "dirs $env(" << envVarName << ")\n"
	    << "}\n"
	    << "if [info exists env(EXT_FOLDER)] {\n"
	    << "	lappend " << pkgName << "dirs [file join $env(EXT_FOLDER) \"Tool Command Language\" lib " << pkgName << version << "]\n"
	    << "}\n"
	    << "lappend " << pkgName << "dirs ${" << libVarName << "}\n"
	    << "unset " << libVarName << "\n"
	    << "lappend " << pkgName << "dirs [file join [file dirname [info library]] " << pkgName << version << "]\n"
	    << "set parentDir [file dirname [file dirname [info nameofexecutable]]]\n"
	    << "lappend " << pkgName << "dirs [file join $parentDir lib " << pkgName << version << "]\n"
#if 0
			<< "if {![regexp {.*[ab][12345]} $patchLevel lib]} {\n"
			<< "	set lib " << pkgName << version << "\n"
			<< "}\n"
			<< "lappend " << pkgName << "dirs [file join [file dirname $parentDir] $lib library]\n"
#endif
	    << "lappend " << pkgName << "dirs [file join [file dirname $parentDir] " << pkgName << " library]\n"
	    << "lappend " << pkgName << "dirs [file join $parentDir library]\n"
	    << "foreach " << pkgName << "i $" << pkgName << "dirs {\n"
	    << "	set " << libVarName << " $" << pkgName << "i\n"
	    << "	if ![catch {uplevel #0 source \\{[file join $" << pkgName << "i " << pkgInitFile << "]\\}}] {\n"
	    << "		break\n"
	    << "	} else {\n"
	    << "		unset " << libVarName << "\n"
	    << "	}\n"
	    << "}\n"
	    << "unset " << pkgName << "i\n"
	    << "if ![info exists " << libVarName << "] {\n"
	    << "	if [catch {uplevel #0 source -rsrc " << pkgInitFile << "}] {\n"
	    << "		set " << libVarName << " {}\n"
	    << "		set msg \"Can't find a usable " << pkgInitFile << " in the following directories: \n\"\n"
	    << "		append msg \"    $" << pkgName << "dirs\n\"\n"
	    << "		append msg \"This probably means that " << prettyPkgName << " wasn't installed properly\n\"\n"
	    << "		append msg \"or you need to set your " << envVarName << " environment variable.\n\"\n"
	    << "		error $msg\n"
	    << "	}\n"
	    << "}" << eval;
    return _status;
}

#ifdef USE_TCL_STUBS
#include <tclInt.h>
#else
// Nabbed two declarations from 'tclInt.h' for the moment
extern "C" Tcl_Namespace *	Tcl_CreateNamespace (Tcl_Interp *interp,
			    char *name, ClientData clientData,
			    Tcl_NamespaceDeleteProc *deleteProc);
extern "C" Tcl_Namespace *	Tcl_FindNamespace (Tcl_Interp *interp,
			    char *name, Tcl_Namespace *contextNsPtr,
			    int flags);
#endif


int tcl_obj::MakeNamespace(const char* name) {
    Tcl_Namespace* ns;
    /*
     *  Find or create the namespace.
     */
    ns = Tcl_FindNamespace(_interp, (char*)name, (Tcl_Namespace*)NULL, 
						   TCL_GLOBAL_ONLY);
								
    if (ns == NULL) {
    	ns = Tcl_CreateNamespace(_interp, (char*)name, (ClientData)NULL, 
								 (Tcl_NamespaceDeleteProc*)NULL);
    }
    if (ns == NULL) {
		(*this) << " (cannot initialize " << name << " namespace)" << tcl_error;
        return TCL_ERROR;
    }
    return TCL_OK;
}

#include "cpptcl_metaobject.h"

tcl_base* tcl_obj::FindObject(const char* name) const {
	Tcl_CmdInfo info;
	if (!Tcl_GetCommandInfo(_interp, (char*) name, &info))
		return (tcl_base*) NULL;
	else
		return (tcl_base*) (info.objClientData);
}

tcl_base* tcl_obj::FindObject(Tcl_Obj* name) const {
	int len;
	return FindObject(Tcl_GetStringFromObj(name,&len));
}

tcl_base* tcl_obj::FindObject(const char* name, const char* type) const {
	tcl_base* obj = FindObject(name);	
	if(tcl_base::metaobject->is_of_type(obj->type(),type)) {
		return obj;
	} else {
		return 0;
	}
}

tcl_base* tcl_obj::FindObject(Tcl_Obj* name, const char* type) const {
	int len;
	return FindObject(Tcl_GetStringFromObj(name,&len),type);
}
